/*
	Lander 2
	by Sven Doezema

	TODO:
	- levels
	- sound fx
	- music

	- occational crash when exiting program (very rare, perhaps only when MODE_DIE --- VERY VERY VERY RARE -- fixed?)
*/



#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <allegro.h>
#include "data.h"
#include "shared.h"
#include "collide.h"




#define LANDER2_MAJOR_VERSION		0
#define LANDER2_MINOR_VERSION		0
#define LANDER2_VERSION_STRING		"BETA VERSION"


#define LANDER_PAN_W	SCREEN_W/256 * (player_x - camera_x + PLAYER_W/2)

#ifndef MIN
#define MIN(a,b)	((a) < (b)) ? (a) : (b)
#endif
#ifndef MAX
#define MAX(a,b)	((a) > (b)) ? (a) : (b)
#endif
#ifndef SWAP
#define SWAP(a,b)	(a) = (a) ^ (b); (b) = (a) ^ (b); (a) = (a) ^ (b);
#endif
#define center_sprite(dest, src)	draw_sprite(((BITMAP*)dest),((BITMAP*)src),((BITMAP*)dest)->w/2 - ((BITMAP*)src)->w/2, ((BITMAP*)dest)->h/2 - ((BITMAP*)src)->h/2);


#define RIGHT_PANEL_W	16
#define BOTTOM_PANEL_H	48


#define MODE_GAME		0
#define MODE_DIE		1
#define MODE_MENU		3
#define MODE_LEVEL_TRANSITION	4


#define DIFFICULTY_EASY	0
#define DIFFICULTY_MEDIUM	1
#define DIFFICULTY_HARD	2


#define TILE_SPECIAL_LAND_START	14
#define TILE_GAS2	14
#define TILE_GAS5	15
#define TILE_GAS10	16
#define TILE_GAS100	17
#define TILE_END_OF_LEVEL	18
#define TILE_SPECIAL_LAND_END	18





#define COLOR_BLACK0				0
#define COLOR_BLACK					1
#define COLOR_WHITE					15
#define COLOR_VELOCITY_BAR			24
#define COLOR_VELOCITY_ZONE_ACTIVE	25
#define COLOR_VELOCITY_ZONE_DEAD	26
#define COLOR_HEAT_LEFT				27
#define COLOR_HEAT_USED				28
#define COLOR_FUEL_LEFT				29
#define COLOR_FUEL_USED				30
#define COLOR_CREDIT_HEADING		16
#define COLOR_CREDIT_NAME			25
#define COLOR_EXPLOSION_START		64




volatile int animateFlag=FALSE;


int done = FALSE,
	direction = 1,
	fullscreen = TRUE,
	inAGame,
	deaths, lastDeaths, lastLevel,
	difficulty, lastDifficulty,
	playerLives,
	current_song,
	currentLevel = 1,
	score = 0,
	gameMode = MODE_GAME,
	last_flame,				// last time a flame particle was shot out (they should shoot only 10 times a second)
	onGround = FALSE,
	lastSuccessfulX,		// x/y of last gas station landing
	lastSuccessfulY,
	soundOn=TRUE,
	musicOn=TRUE,
	ticksAtDeath,
	ticksAtTransitionStart;
float player_x, player_y, player_dy = 0.0, camera_x_at_death;
float player_frame=0.0;
float targetVerticalAcceleration = 0.0f, targetHorizontalAcceleration = 0.0f, verticalAcceleration, horizontalAcceleration;
float fuel,maxFuel,heat,maxHeat;



// variables for explosion when you crash
#define CRASH_X	0
#define CRASH_Y	1
#define CRASH_ANGLE	2
#define CRASH_SPIN	3
#define CRASH_DSPIN 4
#define CRASH_ELEMENTS	5
#define CRASH_CHUNKS	8	// how many chunks fly when you crash
float crashChunk[CRASH_CHUNKS][CRASH_ELEMENTS];
float time_at_death;

int last_tick_check;
int rocketIsPlaying = FALSE, playRocket = FALSE, rocketFrequency=0;
BITMAP *col_bmp;










void update_timer(void)
{
	TICKS++;
}
END_OF_FUNCTION(update_timer);




// updates 10 times a second
void animate_tiles(void)
{
	animateFlag = TRUE;
}
END_OF_FUNCTION(animate_tiles);



#define MAX_TRANSITION_STARS	2048
typedef struct tTransitionStar
{
	float x, y, distance;
	char intensity, size;
} tTransitionStar;
tTransitionStar *transitionStar;



tTransitionStar *allocateTransitionStar(void)
{
	int n;
	tTransitionStar *star = malloc(sizeof(*star) * MAX_TRANSITION_STARS);


	for(n=0; n<MAX_TRANSITION_STARS; n++)
	{
		star[n].x = rand()%(SCREEN_W*2);
		star[n].y = rand()%(SCREEN_H*2);
		star[n].distance = (rand()%512)/256.0f;
		star[n].intensity = rand()%16;
		star[n].size = rand()%2;
	}


	return star;
}


void destroyTransitionStar(tTransitionStar *star)
{
	fLogfile("Destroying transition stars\n");
	if(star)
		free(star);
}

void drawTransitionStar(tTransitionStar *star)
{
	int n;
	//fLogfile("Drawing transition stars...\n");

	for(n=0; n<MAX_TRANSITION_STARS; n++)
	{
		circlefill(buffer,star[n].x -= 1.0f*star[n].distance * TIME_SINCE_LAST_FRAME, star[n].y -= 1.0f*star[n].distance * TIME_SINCE_LAST_FRAME, star[n].size, star[n].intensity);
		if(star[n].x < 0.0f)
		{
			star[n].x += SCREEN_W*2;
		}
		if(star[n].y < 0.0f)
		{
			star[n].y += SCREEN_H*2;
		}
	}
}


typedef struct tBackgroundStar
{
	int x, y;
	char intensity, size;
	float distance;
} tBackgroundStar;
tBackgroundStar star[(LEVEL_W*LEVEL_H)/32];

void backgroundStarInit(void)
{
	int n;
	for(n=0; n<(LEVEL_W * LEVEL_H) / 32; n++)
	{
		star[n].distance = 1.5f + (rand()%128/32.0f);
		star[n].x = rand()%(int)((LEVEL_W*TILE_W)/star[n].distance + SCREEN_W);
		star[n].y = rand()%(int)((LEVEL_H*TILE_H)/star[n].distance + SCREEN_H);
		star[n].size = rand()%2;
		star[n].intensity = rand()%16;
	}
}



typedef struct tKeyHandler
{
	int MAX_ACCELERATION,
		MED_ACCELERATION,
		MIN_ACCELERATION,
		LEFT_ACCELERATION,
		RIGHT_ACCELERATION,
		MAX_LR_ACCELERATION;
} tKeyHandler;
tKeyHandler *keyHandle;


tKeyHandler *initKeyHandler(void)
{
	tKeyHandler *keyHandle = malloc(sizeof(*keyHandle));;
	keyHandle->MAX_ACCELERATION = get_config_int("keys","max_accel", KEY_Q);
	keyHandle->MED_ACCELERATION = get_config_int("keys","med_accel", KEY_A);
	keyHandle->MIN_ACCELERATION = get_config_int("keys","min_accel", KEY_Z);
	keyHandle->LEFT_ACCELERATION = get_config_int("keys","left_accel", KEY_LEFT);
	keyHandle->RIGHT_ACCELERATION = get_config_int("keys","right_accel", KEY_RIGHT);
	keyHandle->MAX_LR_ACCELERATION = get_config_int("keys","max_lr_accel", KEY_W);

	return keyHandle;
}

void destroyKeyHandler(tKeyHandler *keyHandle)
{
	if(keyHandle) free(keyHandle);
}



// this is a horribly written function, perhaps I should start over from scratch...
void customizeKeys(tKeyHandler *keyHandler, void (*draw_func)(void))
{
	int n,o;
	static char *questions[6] = {
		"Maximum Upward Acceleration",
		"Medium Upward Acceleration",
		"Minimum Upward Acceleration",
		"Leftward Acceleration",
		"Rightward Acceleration",
		"Maximum Total Acceleration"};
		int custom_key[6]={keyHandler->MAX_ACCELERATION, keyHandler->MED_ACCELERATION, keyHandler->MIN_ACCELERATION, keyHandler->LEFT_ACCELERATION, keyHandler->RIGHT_ACCELERATION, keyHandler->MAX_LR_ACCELERATION};


START_CUSTOM_KEYS:
	clear_keybuf();
	for(n=0; n<6; n++)
	{
		while(!keypressed())
		{
			clear(buffer);
			draw_func();
			for(o=0; o<6; o++)
			{
				textprintf(buffer,font,SCREEN_W/5,SCREEN_H/5 + o*10, COLOR_WHITE,"Press the key for: %s", questions[o]);
				textprintf(buffer,font,SCREEN_W-SCREEN_W/5, SCREEN_H/5 + o*10, COLOR_WHITE, keyName[custom_key[o]]);
			}
			textprintf(buffer,font,SCREEN_W/5,SCREEN_H/5 + n*10, COLOR_CREDIT_HEADING,"Press the key for: %s", questions[n]);
			blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
		}
		custom_key[n] = readkey() >> 8;
		textprintf(buffer,font,SCREEN_W-SCREEN_W/5, SCREEN_H/5 + n*10, COLOR_WHITE, keyName[custom_key[n]]);
		blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
		clear_keybuf();
	}

	while(!keypressed())
	{
		clear(buffer);
		draw_func();
		for(o=0; o<6; o++)
		{
			textprintf(buffer,font,SCREEN_W/5,SCREEN_H/5 + o*10, COLOR_WHITE,"Press the key for: %s", questions[o]);
			textprintf(buffer,font,SCREEN_W-SCREEN_W/5, SCREEN_H/5 + o*10, COLOR_WHITE, keyName[custom_key[o]]);
		}
		textprintf_centre(buffer,font,SCREEN_W/2, SCREEN_H - SCREEN_H/5, COLOR_CREDIT_HEADING, "Are keys okay?  (Y/N)");
		blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
	}
	n = readkey() >> 8;

	if(n == KEY_Y)
	{
		fLogfile("Setting new keys to handler...");
		keyHandler->MAX_ACCELERATION = custom_key[0];
		keyHandler->MED_ACCELERATION = custom_key[1];
		keyHandler->MIN_ACCELERATION = custom_key[2];
		keyHandler->LEFT_ACCELERATION = custom_key[3];
		keyHandler->RIGHT_ACCELERATION = custom_key[4];
		keyHandler->MAX_LR_ACCELERATION = custom_key[5];

		set_config_int("keys","max_accel",custom_key[0]);
		set_config_int("keys","med_accel",custom_key[1]);
		set_config_int("keys","min_accel",custom_key[2]);
		set_config_int("keys","left_accel",custom_key[3]);
		set_config_int("keys","right_accel",custom_key[4]);
		set_config_int("keys","max_lr_accel",custom_key[5]);
		fLogfile("good!\n");
		return;
	}
	else goto START_CUSTOM_KEYS;
}


typedef struct tParticle
{
	float x,y,dx,dy;
	float color;	// index in a color ramp
	float angle;
	float radius;
} tParticle;


typedef struct tParticleSource
{
	int numParticles;
	int maxInstances;
	int maxRadius;
	int avgLife; // how any ms particle will live for
	tParticle *particle;
} tParticleSource;
tParticleSource *particles;



// initialises a particle source
tParticleSource *particleSourceAllocate(int numParticles, int maxInstances, int maxRadius, int avgLife)
{
	tParticleSource *p;

	p = malloc(sizeof(*p));
	if(!p) FATAL_ERROR("Couldn't allocate particle source!");

	p->particle = malloc(sizeof(tParticle) * maxInstances *	numParticles);
	if(!p->particle) FATAL_ERROR("Couldn't allocate particles!");


	p->numParticles = numParticles;
	p->maxInstances = maxInstances;
	p->maxRadius = maxRadius;
	p->avgLife = avgLife;

	return p;
}


// destroys a particle source
void particleSourceDestroy(tParticleSource *p)
{
	if(p)
	{
		if(p->particle) free(p->particle);
		free(p);
	}
}


// adds particles to the list
void particleSourceAdd(tParticleSource *p, int strength, int x, int y, float angle)
{
	int n, added=0;
	for(n=0; n<p->maxInstances * p->numParticles; n++)
	{
		if(added >= strength * (1.5f * TIME_SINCE_LAST_FRAME)) return;
		if(p->particle[n].radius < 1.0f)
		{
			p->particle[n].angle = angle + (-15 + rand()%30);
			p->particle[n].radius = MIN(p->maxRadius/(10.0f/strength),((rand()+1)%p->maxRadius) -	((rand()+1)%(int)(p->maxRadius/(50.0f/strength) + 1)));
			p->particle[n].x = x + (-5 + rand()%10);
			p->particle[n].y = y + (-5 + rand()%10);
			p->particle[n].color = COLOR_EXPLOSION_START;
			p->particle[n].dx = fixtof(fixcos(ftofix(p->particle[n].angle))) * (strength/5.0f);
			p->particle[n].dy = fixtof(fixsin(ftofix(p->particle[n].angle))) * (strength/5.0f);
			added++;
		}
	}
}



// this function originally generated the fire ramp, but not it generates most of the palette
// it should not be used in release builds, but a static copy of what's generated should be used instead.
#define RAMP(a,b,c)		( fabs( (a) - (b) ) / (c) )
void buildFireParticleRamp(void)
{
	// start blue, yellow, red, orange, dark orange in 4 phases
	// 80,172,248 -> 253,249,74 -> 255,6,6 -> 255,168,6 -> 214,116,18

	int m,n,o, colors[5][4]={
		{80, 172,248, 12},	//blue
		{253,249,74,  24},	//yellow
		{255,6,  6,   32},	//red
		{255,168,6,   48},	//kinda orange
		{214,116,18,  64}	//darker orange
	};
	float rStep, gStep, bStep;
	RGB c={0,0,0};

	// 32-63 reserved entries -- black for now
	for(n=32;n<64;n++)
	{
		set_color(n,&c);
	}


	// 1 - 16 black to white ramp
	rStep = RAMP(0,255,15.0f);
	for(n=0; n<16; n++)
	{
		c.r = MIN(255,MAX(0,(n*rStep)/4.0f));
		c.g = MIN(255,MAX(0,(n*rStep)/4.0f));
		c.b = MIN(255,MAX(0,(n*rStep)/4.0f));
		set_color(n,&c);
	}


	// fire ramp
	for(m=0; m<4; m++)
	{
		static int total_count = 0;
		o=0;
		rStep = RAMP(colors[m][0], colors[m+1][0], colors[m][3]);
		gStep = RAMP(colors[m][1], colors[m+1][1], colors[m][3]);
		bStep = RAMP(colors[m][2], colors[m+1][2], colors[m][3]);
		for(n=total_count; n<total_count + colors[m][3]; n++)
		{
			c.r = colors[m][0] > colors[m+1][0] ? colors[m][0] - MIN(255,MAX(0,(o*rStep)/4.0f)) : colors[m][0] + MIN(255,MAX(0,(o*rStep)/4.0f));
			c.g = colors[m][1] > colors[m+1][1] ? colors[m][1] - MIN(255,MAX(0,(o*gStep)/4.0f)) : colors[m][1] + MIN(255,MAX(0,(o*gStep)/4.0f));
			c.b = colors[m][2] > colors[m+1][2] ? colors[m][2] - MIN(255,MAX(0,(o*bStep)/4.0f)) : colors[m][2] + MIN(255,MAX(0,(o*bStep)/4.0f));
			set_color(n + COLOR_EXPLOSION_START, &c);
			o++;
		}
		total_count += colors[m][3];
	}

/*
	rStep = RAMP(253,255,24.0f);
	gStep = RAMP(249,6,  24.0f);
	bStep = RAMP(74, 6,  24.0f);
	o=0;
	for(n=12; n<36; n++)
	{
		c.r = (253.0f + (o*rStep))/4.0f;
		c.g = (249.0f - (o*gStep))/4.0f;
		c.b = (74.0f - (o*bStep))/4.0f;
		set_color(n + COLOR_EXPLOSION_START, &c);
		o++;
	}

	rStep = RAMP(255,255,28.0f);
	gStep = RAMP(6,  168,28.0f);
	bStep = RAMP(6,  6,  28.0f);
	o=0;
	for(n=36; n<64; n++)
	{
		c.r = (255.0f + (0*rStep))/4.0f;
		c.g = (6.0f + (o*gStep))/4.0f;
		c.b = (6.0f + (o*bStep))/4.0f;
		set_color(n + COLOR_EXPLOSION_START, &c);
		o++;
	}


	rStep = RAMP(255,214,84.0f);
	gStep = RAMP(168,116,84.0f);
	bStep = RAMP(6,  18, 84.0f);
	o=0;
	for(n=108; n<192; n++)
	{
		c.r = (255.0f - (o*rStep))/4.0f;
		c.g = (168.0f - (o*gStep))/4.0f;
		c.b = (6.0f + (o*bStep))/4.0f;
		set_color(n + COLOR_EXPLOSION_START, &c);
		o++;
	}
*/
}

// draws and then updates the particles in the given list -- call once per frame for accurate results
void particleSourceDraw(tParticleSource *p)
{
	int n;


	for(n=0; n<p->maxInstances * p->numParticles; n++)
	{
		if(p->particle[n].radius > 0)
		{
			circlefill(buffer,p->particle[n].x - camera_x, p->particle[n].y - camera_y, p->particle[n].radius, MIN(255,p->particle[n].color));
			p->particle[n].dy -= 10.0f * GRAVITY * TIME_SINCE_LAST_FRAME;
			p->particle[n].y += p->particle[n].dy * TIME_SINCE_LAST_FRAME;
			p->particle[n].x += p->particle[n].dx * TIME_SINCE_LAST_FRAME;
			p->particle[n].radius -= (float)p->avgLife/10000.0f * TIME_SINCE_LAST_FRAME;
			p->particle[n].color += 1.0f * TIME_SINCE_LAST_FRAME;
		}
	}
}






void playerDie(void)
{
	int n;
	gameMode = MODE_DIE;
	playRocket = FALSE;
	deaths++;
	lastDeaths = deaths;

	for(n=0; n<CRASH_CHUNKS; n++)
	{
		crashChunk[n][CRASH_X] = player_x;
		crashChunk[n][CRASH_Y] = player_y;
		crashChunk[n][CRASH_ANGLE] = rand()%256;
		crashChunk[n][CRASH_SPIN] = rand()%256;
		crashChunk[n][CRASH_DSPIN] = (rand()%256/32.0f) - 4.0f;
	}


	play_sample(data[S_CRASH].dat,255,LANDER_PAN_W,rand()%500 + 750,FALSE);

	

	for(n=0; n<128; n++)
	{
		particleSourceAdd(particles,10,player_x + PLAYER_W/2,player_y + PLAYER_H/2,rand()%256);
	}
	ticksAtDeath = TICKS;

	clear_to_color(buffer, COLOR_WHITE);
	clear_keybuf();
}


void setVolume(float s, float m)
{
	if(soundOn==FALSE) s = 0.0f;
	if(musicOn==FALSE) m = 0.0f;

	set_volume(s,m);
}


void changeVolume(float sound, float music)
{
	soundVolume += sound * TIME_SINCE_LAST_FRAME;
	musicVolume += music * TIME_SINCE_LAST_FRAME;

	if(soundVolume < 0.0f) soundVolume = 0.0f;
	else if(soundVolume > 255.0f) soundVolume = 255.0f;

	if(musicVolume < 0.0f) musicVolume = 0.0f;
	else if(musicVolume > 255.0f) musicVolume = 255.0f;

	setVolume(soundVolume, musicVolume);
}




int bounding_collision(int left1, int top1, int right1, int bottom1, int left2, int top2, int right2, int bottom2)
{
	if(bottom1 < top2) return FALSE;
	if(top1 > bottom2) return FALSE;
	if(right1 < left2) return FALSE;
	if(left1 > right2) return FALSE;
	return TRUE;
}











void doScores(void)
{
	fLogfile("doScores\n");
	clear(buffer);
	textprintf_centre(buffer,data[F_HUGE_0].dat,SCREEN_W/2,5,COLOR_WHITE,"High Scores");


	textprintf_centre(buffer,data[F_SMALL_0].dat,SCREEN_W/2,SCREEN_H-20,COLOR_WHITE,"Press any key to return to menu");
	blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
	clear_keybuf();
	readkey();
	clear_keybuf();
	clear(screen);

}



void init(void)
{
	fLogfile("init\n");


	// set player default location
	camera_x = camera_y = heat = verticalAcceleration = horizontalAcceleration = 0.0f;
	fuel = maxFuel = maxHeat = 100.0f;
	gameMode = MODE_GAME;

	destroyLevel(level);
	level = allocateLevel();
	loadLevel(currentLevel);
	player_x = lastSuccessfulX = startX;
	player_y = lastSuccessfulY = startY;

	backgroundStarInit();

	
	onGround = FALSE;
	text_mode(-1);
	done = FALSE;
//	current_song++;
//	if(current_song > 4) current_song = 0;
//	play_midi(data[M_TRACK_1 + current_song].dat,TRUE);
}




void fQuit(void (*draw_func)(void))
{
	fLogfile("fQuit\n");
	clear_keybuf();
	if(doMenu(draw_func,
		data[F_HUGE_0].dat,"Really Quit?",
		2,data[F_SMALL_0].dat,"Yes\0No") == 0)
	{
		// save all settings
		set_config_int("sound","sound_on",soundOn);
		set_config_int("sound","music_on",musicOn);
		set_config_int("sound","sound_vol",soundVolume);
		set_config_int("sound","music_vol",musicVolume);
		set_config_int("etc","x",lastDeaths);
		set_config_int("etc","l",lastLevel);
		set_config_int("etc","d",lastDifficulty);

		
		destroy_bitmap(buffer);
		unload_datafile(data);

		particleSourceDestroy(particles);
		destroyLevel(level);
		destroyTileset();
		destroyKeyHandler(keyHandle);
		destroyTransitionStar(transitionStar);

		set_palette(default_palette);
		set_gfx_mode(GFX_TEXT,0,0,0,0);
		allegro_exit();
		exit(0);
	}
}




void keycheck(void)
{
//	fLogfile("KC\n");


	if(gameMode == MODE_GAME)
	{
		playRocket = FALSE;
		if(fuel > 0.0f)
		{
			rocketFrequency = 500;
	
			if(key[keyHandle->MAX_ACCELERATION])
			{
				rocketFrequency += 300;
				playRocket = TRUE;
				heat += 0.20f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.05f * TIME_SINCE_LAST_FRAME;
				targetVerticalAcceleration = 0.125f;
				particleSourceAdd(particles,5,player_x + PLAYER_W/2,player_y + PLAYER_H + 4,64);
				onGround = FALSE;
			}
			else if(key[keyHandle->MED_ACCELERATION])
			{
				rocketFrequency += 146;
				playRocket = TRUE;
				heat += 0.12f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.03f * TIME_SINCE_LAST_FRAME;
				targetVerticalAcceleration = 0.075f;
				particleSourceAdd(particles,3,player_x + PLAYER_W/2,player_y + PLAYER_H + 3,64);
				onGround = FALSE;
			}
			else if(key[keyHandle->MIN_ACCELERATION])
			{
				rocketFrequency += 75;
				playRocket = TRUE;
				heat += 0.04f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.01f * TIME_SINCE_LAST_FRAME;
				targetVerticalAcceleration = 0.025f;
				particleSourceAdd(particles,2,player_x + PLAYER_W/2,player_y + PLAYER_H + 1,64);
				onGround = FALSE;
			}
			else if(key[keyHandle->MAX_LR_ACCELERATION])
			{
				rocketFrequency += 850;
				playRocket = TRUE;
				heat += 0.52f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.13f * TIME_SINCE_LAST_FRAME;
				targetVerticalAcceleration = 0.195f;
				particleSourceAdd(particles,5,player_x + PLAYER_W/2,player_y + PLAYER_H,64);
				particleSourceAdd(particles,4,player_x + PLAYER_W + 3,player_y + 20,27);
				particleSourceAdd(particles,4,player_x - 2,player_y + 20,101);
				onGround = FALSE;
			}
			else
			{
				targetVerticalAcceleration = 0.0f;
			}
			if(key[keyHandle->LEFT_ACCELERATION])
			{
				rocketFrequency += 275;
				playRocket = TRUE;
				heat += 0.16f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.04f * TIME_SINCE_LAST_FRAME;
				if(!onGround)	// a little hack so that movement is restricted 
				{
					targetHorizontalAcceleration += -0.10f;
					targetVerticalAcceleration += 0.035f;
				}
				particleSourceAdd(particles,4,player_x + PLAYER_W + 3,player_y + 20,27);
				onGround = FALSE;
			}
			if(key[keyHandle->RIGHT_ACCELERATION])
			{
				rocketFrequency += 275;
				playRocket = TRUE;
				heat += 0.16f * TIME_SINCE_LAST_FRAME;
				fuel -= 0.04f * TIME_SINCE_LAST_FRAME;
				targetHorizontalAcceleration += 0.10f;
				targetVerticalAcceleration += 0.035f;
				particleSourceAdd(particles,4,player_x - 2,player_y + 20,101);
				onGround = FALSE;
			}
		}	// if you got fuel
	} // if gameMode == MODE_GAME
	else if(gameMode == MODE_DIE)
	{
		if(TICKS - ticksAtDeath > 1000 && keypressed())
		{
			player_x = lastSuccessfulX;
			player_y = lastSuccessfulY;
			heat = targetHorizontalAcceleration = targetVerticalAcceleration = horizontalAcceleration = verticalAcceleration = 0.0f;
			fuel = maxFuel;
			onGround = FALSE;
			gameMode = MODE_GAME;
		}
	}
	else if(gameMode == MODE_LEVEL_TRANSITION)
	{
		if(TICKS - ticksAtTransitionStart > 1000 && keypressed())
		{
			gameMode = MODE_GAME;
			destroyLevel(level);
			level = allocateLevel();
			currentLevel++;
			lastLevel = currentLevel;
			loadLevel(currentLevel);
			fuel = maxFuel;
			onGround = FALSE;
			player_x = startX;
			player_y = startY;
			lastSuccessfulX = lastSuccessfulY = heat = camera_x = camera_y = targetVerticalAcceleration = targetHorizontalAcceleration = verticalAcceleration = horizontalAcceleration = 0.0f;

			set_config_int("etc","l",currentLevel);
			set_config_int("etc","x",deaths);
			set_config_int("etc","d",difficulty);
		}
	}



	//if(key[KEY_F2]) toggleSound();
#ifdef ALLEGRO_DEBUG
	if(key[KEY_F11]) BITMAP_DUMP();
#endif
	if(key[KEY_F3]) changeVolume(-0.5,0);	// sound down
	if(key[KEY_F4]) changeVolume(0.5,0);	// sound up
	if(key[KEY_F7]) changeVolume(0,-0.5);	// music down
	if(key[KEY_F8]) changeVolume(0,0.5);	// music up
	if(key[KEY_F5])
	{
//		current_song++;
//		if(current_song > 4)
//			current_song = 0;
//		stop_midi();
//		play_midi(data[M_TRACK_1 + current_song].dat,TRUE);
	}
//	if(key[KEY_F6]) stop_midi();

	
	// change the screen to windowed & fullscreen
	if((key[KEY_ALT] || key[KEY_ALTGR]) && key[KEY_ENTER])
	{
		clear_keybuf();
		if(fullscreen)
		{
			set_color_depth(color_depth);
			if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0)
				if(set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, 640, 480, 0, 0) != 0)
				{
					set_gfx_mode(GFX_TEXT,0,0,0,0);
					allegro_message("I'm deeply sorry, but you got hit by some funky bug setting a windowed mode :\\");
				}
			fullscreen = FALSE;
			set_palette(data[P_GAME].dat);
		}
		else
		{
			set_color_depth(color_depth);
			if(set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, 640, 480, 0, 0) != 0)
				if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0)
				{
					set_gfx_mode(GFX_TEXT,0,0,0,0);
					allegro_message("I'm deeply sorry, but you got hit by some funky bug setting a fullscreen mode :\\");
				}
			fullscreen = TRUE;
			set_palette(data[P_GAME].dat);
		}
		clear_keybuf();
	}
}


void updateClock(void)
{
	float temp;
	int n;

	// timer junk
CALCULATE_TIME_SINCE_LAST_FRAME:
	TIME_SINCE_LAST_FRAME = (TICKS - last_tick_check) / 10.0f;
	last_tick_check = TICKS;
	temp = 100.0f / TIME_SINCE_LAST_FRAME;
	if(temp > 0.0f && temp < 500.0f)
	{
		fpsRecord[fpsIndex] = temp;
		fpsIndex++;
	}


	fps = 0.0f;
	for(n=0; n<256; n++) fps += fpsRecord[n];
	fps = fps/256.0f;

	// If the time since the last frame is over 1/5 of a second, something bad is happening...
	if(TIME_SINCE_LAST_FRAME > 20.0f)
	{
		//textprintf_centre(screen,font,SCREEN_W/2,SCREEN_H/2,COLOR_WHITE,"Your computer is too slow!");
		goto CALCULATE_TIME_SINCE_LAST_FRAME;
	}
}


void update(void)
{
	int x,y;


//	updateClock();

	//fLogfile("ANIMATE\n");

	// animate tiles the silly way -- not yet used in this game
	if(animateFlag)
	{
		int n=0;
		BITMAP *temp;
		while(animatedTiles[tileset][n] != -1)
		{
			temp = tile[animatedTiles[tileset][n]];
			tile[animatedTiles[tileset][n]] = tile[animatedTiles[tileset][n]+animatedTiles[tileset][n+2]];
			tile[animatedTiles[tileset][n]+animatedTiles[tileset][n+2]] = temp;
			animatedTiles[tileset][n+2]++;
			if(animatedTiles[tileset][n+2] > animatedTiles[tileset][n+1]-1) animatedTiles[tileset][n+2] = 1;
			n+=3;
		}
		animateFlag = FALSE;
	}



	
	
	if(gameMode == MODE_GAME)
	{
		//fLogfile("UPDATE GRAVITY\n");
		if(!onGround)
		{
			verticalAcceleration -= GRAVITY * TIME_SINCE_LAST_FRAME;
			verticalAcceleration += (targetVerticalAcceleration / 30.0f) * TIME_SINCE_LAST_FRAME;
			if(fabs(targetVerticalAcceleration) > 0.000001f) targetVerticalAcceleration = (targetVerticalAcceleration/10.0f) * TIME_SINCE_LAST_FRAME;

			horizontalAcceleration += (targetHorizontalAcceleration / 30.0f) * TIME_SINCE_LAST_FRAME;
			if(fabs(targetHorizontalAcceleration) > 0.000001f) targetHorizontalAcceleration = (targetHorizontalAcceleration/10.0f) * TIME_SINCE_LAST_FRAME;

			player_x += horizontalAcceleration * TIME_SINCE_LAST_FRAME;
			player_y -= verticalAcceleration * TIME_SINCE_LAST_FRAME;
		}



		// update heat
		if(heat > maxHeat)
		{
			playerDie();
			heat = maxHeat;
		}
		else if(heat > 0.0001f)
		{
			heat -= 0.1 * TIME_SINCE_LAST_FRAME;
			if(heat <= 0.0f) heat = 0.0001f; // the ten thousandths place won't be seen, and div by zero is bad...
		}
		

		// make sure fuel is above 0
		if(fuel < 0.0f)
		{
			fuel = 0.0f;
		}




		// check collisions
		// draw the tiles to the temp bitmap
		clear(col_bmp);
		for(y=MAX(player_y/TILE_H - 1,0); y<MIN((player_y + PLAYER_H) / TILE_H, LEVEL_H); y++)
		{
			for(x=MAX(player_x/TILE_W - 1,0); x<MIN((player_x + PLAYER_W)/TILE_W,LEVEL_W); x++)
			{
				if(level->interactive->data[y][x] > 0) draw_sprite(col_bmp,tile[(int)level->interactive->data[y][x]],x*TILE_W - player_x,y*TILE_H - player_y);
			}
		}

		// make a new mask
		mk_spr_mask(col_bmp, 1);

		// check :)
		if((verticalAcceleration <= 0.0f
		&& verticalAcceleration > -0.17f)
		&& (horizontalAcceleration < 0.09f
		&& horizontalAcceleration > -0.09f))
		{
			if((getpixel(col_bmp, 0, PLAYER_H-1) != COLOR_BLACK0
			&& getpixel(col_bmp, PLAYER_W-1, PLAYER_H-1) != COLOR_BLACK0 )
			&& !onGround && check_pp_collision(0,0,0,1,0,0))
			{
				onGround = TRUE;
				verticalAcceleration = 0.0f;
				horizontalAcceleration = 0.0f;
				targetVerticalAcceleration = 0.0f;
				targetHorizontalAcceleration = 0.0f;
				player_x = (int)player_x;
				player_y = (int)player_y;
			}
			else if(onGround)
			{
				int y1Var = (player_y + PLAYER_H) / TILE_H;
				int x1Var = (player_x + PLAYER_W - 1) / TILE_W;
				int x2Var = (player_x) / TILE_W;
				// check if tile under both right and left legs is a gas tile
				if(level->interactive->data[y1Var][x1Var] >= TILE_SPECIAL_LAND_START
				&& level->interactive->data[y1Var][x2Var] >= TILE_SPECIAL_LAND_START
				&& level->interactive->data[y1Var][x1Var] <= TILE_SPECIAL_LAND_END
				&& level->interactive->data[y1Var][x2Var] <= TILE_SPECIAL_LAND_END)
				{
					lastSuccessfulX = player_x;
					lastSuccessfulY = player_y;

					if(level->interactive->data[y1Var][x1Var] == TILE_GAS2
					&& level->interactive->data[y1Var][x2Var] == TILE_GAS2)
					{
						fuel += 0.02f * TIME_SINCE_LAST_FRAME;
						if(fuel > maxFuel) fuel = maxFuel;
					}
					else if(level->interactive->data[y1Var][x1Var] == TILE_GAS5
						 && level->interactive->data[y1Var][x2Var] == TILE_GAS5)
					{
						fuel += 0.05f * TIME_SINCE_LAST_FRAME;
						if(fuel > maxFuel) fuel = maxFuel;
					}
					else if(level->interactive->data[y1Var][x1Var] == TILE_GAS10
						 && level->interactive->data[y1Var][x2Var] == TILE_GAS10)
					{
						fuel += 0.10f * TIME_SINCE_LAST_FRAME;
						if(fuel > maxFuel) fuel = maxFuel;
					}
					else if(level->interactive->data[y1Var][x1Var] == TILE_GAS100
						 && level->interactive->data[y1Var][x2Var] == TILE_GAS100)
					{
						fuel = maxFuel;
					}
					else if(level->interactive->data[y1Var][x1Var] == TILE_END_OF_LEVEL
						 && level->interactive->data[y1Var][x2Var] == TILE_END_OF_LEVEL)
					{
						gameMode = MODE_LEVEL_TRANSITION;
						ticksAtTransitionStart = TICKS;
					}
				}
				// check if on an end-of-level tile
			}
			else if(check_pp_collision(0,0,0,1,0,0))
			{
				playerDie();
			}
		}
		else if(check_pp_collision(0,0,0,1,0,0))
		{
			playerDie();
		}
	




		// sound rocket
		if(playRocket && !rocketIsPlaying)
		{
			play_sample(data[S_ROCKET].dat,255,LANDER_PAN_W,rocketFrequency,TRUE);
			rocketIsPlaying = TRUE;
		}
		if(rocketIsPlaying) adjust_sample(data[S_ROCKET].dat,255,LANDER_PAN_W,rocketFrequency,TRUE);
		if(!playRocket)
		{
			stop_sample(data[S_ROCKET].dat);
			rocketIsPlaying = FALSE;
		}
	} // if gameMode == MODE_GAME


	


	else if(gameMode == MODE_LEVEL_TRANSITION)
	{
		clear_keybuf();
	}
}


void draw(void)
{
	int n,x,y;

//	fLogfile("DR\n");
	updateClock();
	
	if(gameMode != MODE_LEVEL_TRANSITION)
	{
		// update camera x axis
		if(player_x - camera_x < SCREEN_W/3)
		{
			camera_x = MAX(0.0f, player_x - (SCREEN_W/3));
		}
		else if(player_x - camera_x > (2*SCREEN_W)/3)
		{
			camera_x = MIN((LEVEL_W * TILE_W) + RIGHT_PANEL_W - SCREEN_W, player_x - ((2*SCREEN_W)/3));
		}
	
		// update camera y axis
		if(player_y - camera_y < SCREEN_H/3)
		{
			camera_y = MAX(0.0f, player_y - SCREEN_H/3);
		}
		else if(player_y - camera_y > 2*SCREEN_H/3)
		{
			camera_y = MIN((LEVEL_H * TILE_H) + BOTTOM_PANEL_H - SCREEN_H, player_y - 2*SCREEN_H/3);
		}

	
	
	
		// draw stars
		for(n=0; n<(LEVEL_W * LEVEL_H) / 32; n++)
		{
			circlefill(buffer,star[n].x - camera_x/star[n].distance, star[n].y - camera_y/star[n].distance, star[n].size, star[n].intensity);
		}
	
		// draw background
		if(difficulty==DIFFICULTY_EASY)
		{
			for(y=MAX(camera_y/TILE_H - 1,0); y<MIN((camera_y + SCREEN_H) / TILE_H, LEVEL_H); y++)
			{
				for(x=MAX(camera_x/TILE_W - 1,0); x<MIN((camera_x + SCREEN_W)/TILE_W,LEVEL_W); x++)
				{
					if(level->background->data[y][x] > 0)
						draw_sprite(buffer,tile[(int)level->background->data[y][x]],x*TILE_W - camera_x,y*TILE_H - camera_y);
				}
			}
		}

	
		// draw interactive layer
		{
			for(y=MAX(camera_y/TILE_H - 1,0); y<MIN((camera_y + SCREEN_H) / TILE_H, LEVEL_H); y++)
			{
				for(x=MAX(camera_x/TILE_W - 1,0); x<MIN((camera_x + SCREEN_W)/TILE_W,LEVEL_W); x++)
				{
					if(level->interactive->data[y][x] > 0) draw_sprite(buffer,tile[(int)level->interactive->data[y][x]],x*TILE_W - camera_x,y*TILE_H - camera_y);
				}
			}
		}



		//draw "foreground"
		if(difficulty==DIFFICULTY_HARD)
		{
			for(y=MAX(camera_y/TILE_H - 1,0); y<MIN((camera_y + SCREEN_H) / TILE_H, LEVEL_H); y++)
			{
				for(x=MAX(camera_x/TILE_W - 1,0); x<MIN((camera_x + SCREEN_W)/TILE_W,LEVEL_W); x++)
				{
					if(level->foreground->data[y][x] >0) draw_sprite(buffer,tile[(int)level->foreground->data[y][x]],x*TILE_W - camera_x,y*TILE_H - camera_y);
				}
			}
		}



		// draw lander
		if(gameMode == MODE_GAME)
		{
			// draw_sprite(buffer,data[B_LANDER].dat,player_x - camera_x, player_y - camera_y);
			draw_sprite(buffer,data[B_LANDER_BODY].dat,player_x - camera_x + 1,player_y - camera_y);
			// draw fire
			particleSourceDraw(particles);
			draw_sprite(buffer,data[B_LANDER_LEGS].dat,player_x - camera_x,player_y - camera_y + 12);
		}
		else if(gameMode == MODE_DIE)
		{
			particleSourceDraw(particles);

			
			for(n=0;n<2;n++)rotate_sprite(buffer,data[B_CHUNK0].dat, (crashChunk[n][CRASH_X] += fixtof(fixcos(ftofix(crashChunk[n][CRASH_ANGLE]))) * (horizontalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_x, (crashChunk[n][CRASH_Y] += fixtof(fixsin(ftofix(crashChunk[n][CRASH_ANGLE]))) * (verticalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_y, ftofix(crashChunk[n][CRASH_SPIN] += crashChunk[n][CRASH_DSPIN] * TIME_SINCE_LAST_FRAME));
			for(n=2;n<4;n++)rotate_sprite(buffer,data[B_CHUNK1].dat, (crashChunk[n][CRASH_X] += fixtof(fixcos(ftofix(crashChunk[n][CRASH_ANGLE]))) * (horizontalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_x, (crashChunk[n][CRASH_Y] += fixtof(fixsin(ftofix(crashChunk[n][CRASH_ANGLE]))) * (verticalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_y, ftofix(crashChunk[n][CRASH_SPIN] += crashChunk[n][CRASH_DSPIN] * TIME_SINCE_LAST_FRAME));

			for(n=4; n<CRASH_CHUNKS; n++)
				rotate_sprite(buffer,data[B_CHUNK0 - 2 + n].dat,
					(crashChunk[n][CRASH_X] += fixtof(fixcos(ftofix(crashChunk[n][CRASH_ANGLE]))) * (horizontalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_x,
					(crashChunk[n][CRASH_Y] += fixtof(fixsin(ftofix(crashChunk[n][CRASH_ANGLE]))) * (verticalAcceleration+0.7f) * TIME_SINCE_LAST_FRAME) - camera_y,
					ftofix(crashChunk[n][CRASH_SPIN] += crashChunk[n][CRASH_DSPIN] * TIME_SINCE_LAST_FRAME));

			
			// Make the ship get blowed up
			particleSourceAdd(particles,fuel/10.0f,player_x + PLAYER_W/2,player_y + PLAYER_H/2,rand()%256);
			fuel -= 0.35f * TIME_SINCE_LAST_FRAME;
			if(fuel < 0.0f) fuel = 0.0001f;

			// update explosion location & camera
			verticalAcceleration -= (verticalAcceleration/100.0f) * TIME_SINCE_LAST_FRAME;
			//if(fabs(targetVerticalAcceleration) > 0.000001f) targetVerticalAcceleration = (targetVerticalAcceleration/10.0f) * TIME_SINCE_LAST_FRAME;

			horizontalAcceleration -= (horizontalAcceleration/100.0f) * TIME_SINCE_LAST_FRAME;
			//if(fabs(targetHorizontalAcceleration) > 0.000001f) targetHorizontalAcceleration = (targetHorizontalAcceleration/10.0f) * TIME_SINCE_LAST_FRAME;

			player_x += horizontalAcceleration * TIME_SINCE_LAST_FRAME;
			player_y -= verticalAcceleration * TIME_SINCE_LAST_FRAME;
		}


		// velocity
		// right of screen
		rectfill(buffer,SCREEN_W-16,0,SCREEN_W,SCREEN_H-32,COLOR_VELOCITY_BAR);
		if(verticalAcceleration <= 0.0f && verticalAcceleration > -0.17f) n = COLOR_VELOCITY_ZONE_ACTIVE;
		else n = COLOR_VELOCITY_ZONE_DEAD;
		rectfill(buffer,SCREEN_W-16, SCREEN_H/2 - 48, SCREEN_W, SCREEN_H/2 - 32, n);

		// bottom of screen
		rectfill(buffer,0,SCREEN_H-48,SCREEN_W,SCREEN_H-33,COLOR_VELOCITY_BAR);
		if(horizontalAcceleration > -0.09f && horizontalAcceleration < 0.09f) n = COLOR_VELOCITY_ZONE_ACTIVE;
		else n = COLOR_VELOCITY_ZONE_DEAD;
		rectfill(buffer,SCREEN_W/2 - 16, SCREEN_H-48, SCREEN_W/2, SCREEN_H-33, n);

		// lines
		line(buffer,SCREEN_W-16, SCREEN_H/2 - 48 - (verticalAcceleration*100), SCREEN_W, SCREEN_H/2 - 48- (verticalAcceleration*100), COLOR_BLACK);
		line(buffer,SCREEN_W/2 + (horizontalAcceleration*100) - 8,SCREEN_H-48,SCREEN_W/2 + (horizontalAcceleration*100) - 8,SCREEN_H-33, COLOR_BLACK);



		// fuel, heat
		rectfill(buffer,0,SCREEN_H-32,heat*(SCREEN_W/maxHeat),SCREEN_H-17,COLOR_HEAT_LEFT);
		rectfill(buffer,heat*(SCREEN_W/maxHeat),SCREEN_H-32,SCREEN_W,SCREEN_H-17,COLOR_HEAT_USED);
		rectfill(buffer,0,SCREEN_H-16,fuel*(SCREEN_W/maxFuel),SCREEN_H,COLOR_FUEL_LEFT);
		rectfill(buffer,fuel*(SCREEN_W/maxFuel),SCREEN_H-16,SCREEN_W,SCREEN_H,COLOR_FUEL_USED);
		textprintf(buffer,font,0,SCREEN_H-28,COLOR_BLACK,"Heat: %0.2f%%",100.0f*heat/maxHeat);
		textprintf(buffer,font,0,SCREEN_H-12,COLOR_BLACK,"Fuel: %0.2f%%",100.0f*fuel/maxFuel);


		// debug text
#ifdef ALLEGRO_DEBUG
		textprintf(buffer,font,0,0,COLOR_WHITE,"%0.0f x%0.0f y%0.0f dx%0.2f dy%0.2f diff%d sx%d sy%d",fps,player_x,player_y,horizontalAcceleration,verticalAcceleration,difficulty,startX,startY);
#endif

		if(gameMode == MODE_DIE)
		{
			textprintf_centre(buffer,data[F_SMALL_0].dat,SCREEN_W/2,SCREEN_H-BOTTOM_PANEL_H - text_height(data[F_SMALL_0].dat),COLOR_WHITE,"Press any key to continue.");
		}
	}// gameMode != MODE_LEVEL_TRANSITION

	else if(gameMode == MODE_LEVEL_TRANSITION)
	{
		drawTransitionStar(transitionStar);
		// draw large lander
		center_sprite(buffer,data[B_ADVANCE].dat);
		// draw text
		textprintf_centre(buffer,data[F_HUGE_0].dat,SCREEN_W/2,25, COLOR_WHITE, "Now Approaching Level %d",currentLevel+1);
		textprintf_centre(buffer,data[F_SMALL_0].dat,SCREEN_W/2,SCREEN_H-20, COLOR_WHITE,"Press any key to continue...");
	}


	// blit screen
//	vsync();
	if(key[KEY_F12]) fScreenshot("LAND");
}




// very hacked together..
void doOptions(void (*draw_func)(void))
{
	static const int maxItems = 6;
	static char *item[6] = {"Sound","Sound Volume","Music","Music Volume","Customize keys","Return to menu"};
	int selected = 0,n,k;
	char temp[80];



	while(1)
	{
//		updateClock();
		clear_keybuf();
		while(!keypressed())
		{
			clear(buffer);
			draw_func();
			textprintf_centre(buffer,data[F_HUGE_0].dat,SCREEN_W/2,10,COLOR_WHITE, "Options");
	
			for(n=0; n<maxItems; n++)
			{
				textprintf(buffer,data[F_SMALL_0].dat,SCREEN_W/10, text_height(data[F_HUGE_0].dat) + 25 + n*48, COLOR_WHITE, item[n]);
			}

			draw_sprite(buffer,data[B_ARROW].dat,0,text_height(data[F_HUGE_0].dat) + 25 + selected*48 - 8);
			if(soundOn) sprintf(temp,"On");
			else sprintf(temp,"Off");
			textprintf(buffer,data[F_SMALL_0].dat,SCREEN_W/2,text_height(data[F_HUGE_0].dat) + 25 + 0*48, COLOR_WHITE,temp);

			if(musicOn) sprintf(temp,"On");
			else sprintf(temp,"Off");
			textprintf(buffer,data[F_SMALL_0].dat,SCREEN_W/2,text_height(data[F_HUGE_0].dat) + 25 + 2*48, COLOR_WHITE,temp);

			textprintf(buffer,data[F_SMALL_0].dat,SCREEN_W/2,text_height(data[F_HUGE_0].dat) + 25 + 1*48, COLOR_WHITE, "%0.0f%%",soundVolume/255.0f * 100.0f);
			textprintf(buffer,data[F_SMALL_0].dat,SCREEN_W/2,text_height(data[F_HUGE_0].dat) + 25 + 3*48, COLOR_WHITE, "%0.0f%%",musicVolume/255.0f * 100.0f);

			blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
		}

		k = readkey() >> 8;


		switch(k)
		{
			case KEY_ESC:
			{
				return;
			}
			case KEY_ENTER:
			{
				switch(selected)
				{
				case 0:	// sound on/off
					if(soundOn == TRUE) soundOn = FALSE;
					else soundOn = TRUE;
					break;

				case 1:	// volume (do nothing)
					changeVolume(1.0f * TIME_SINCE_LAST_FRAME, 0.0f);
					break;

				case 2: // music on/off
					if(musicOn == TRUE) musicOn = FALSE;
					else musicOn = TRUE;
					break;

				case 3: // volume (do nothing)
					changeVolume(0.0f,1.0f * TIME_SINCE_LAST_FRAME);
					break;

				case 4: // customize keys
					customizeKeys(keyHandle, draw_func);
					break;

				case 5:
					return;
				}
				break;
			}
			case KEY_UP:
			{
				selected--;
				if(selected < 0) selected = 0;
				break;
			}
			case KEY_DOWN:
			{
				selected++;
				if(selected == maxItems) selected = maxItems - 1;
				break;
			}
			case KEY_LEFT:
			{
				switch(selected)
				{
				case 0:	// sound on/off
					if(soundOn == TRUE) soundOn = FALSE;
					else soundOn = TRUE;
					break;

				case 1:	// sound volume
					changeVolume(-1.0f * TIME_SINCE_LAST_FRAME,0.0f);
					break;

				case 2:	// music on/off
					if(musicOn == TRUE) musicOn = FALSE;
					else musicOn = TRUE;
					break;

				case 3:	// music volume
					changeVolume(0.0f,-1.0f * TIME_SINCE_LAST_FRAME);
					break;

				default:
					break;
				}
				break;
			}
			case KEY_RIGHT:
			{
				switch(selected)
				{
				case 0:	// sound on/off
					if(soundOn == TRUE) soundOn = FALSE;
					else soundOn = TRUE;
					break;

				case 1: // sound volume
					changeVolume(1.0f * TIME_SINCE_LAST_FRAME,0.0f);
					break;

				case 2:	// music on/off
					if(musicOn == TRUE) musicOn = FALSE;
					else musicOn = TRUE;
					break;

				case 3:	// music volume
					changeVolume(0.0f,1.0f * TIME_SINCE_LAST_FRAME);
					break;

				default:
					break;
				}
				break;
			}
			case KEY_F12:
			{
				fScreenshot("LAND");
				break;
			}

		}
		clear_keybuf();
	}
}




typedef struct tCredit
{
	char *heading;
	int people;
	char *name;
	char **addr;
} tCredit;

tCredit credits[] =
{
	{"Engine Programming",1,"Steven Wallace",NULL},
	{"Editor Programming",1,"Steven Wallace",NULL},
	{"2-Dimentional Art",1,"Steven Wallace",NULL},
	{"3-Dimentional Art",1,"Steven Wallace",NULL},
	{"Level Design",1,"Steven Wallace",NULL},
	{"Special Thanks",8,
		"Cori Crawford\0"
		"Chris \"CMG\" Moyer-Grice\0"
		"Chris Fiorello\0"
		"Stella Carr\0"
		"Jen Krakovich\0"
		"Ellen Roggie\0"
		"Becky Schafer\0"
		"Jeff Babajtis",
		NULL
	},

	{"And a shout to all my homies on campus",69,
		"Cori\0"
		"Dora\0"
		"Christine\0"
		"Jen\0"
		"Kasie\0"
		"Asha\0"
		"Becky\0"
		"Ellen\0"
		"Erin\0"
		"Jamie\0"
		"Matt\0"
		"Greg\0"
		"Dan\0"
		"Kurt\0"
		"James (who's no longer with us)\0"
		"Christian\0"
		"Joe\0"
		"Matt\0"
		"Jesse\0"
		"Warren\0"
		"Lucas\0"
		"Chris\0"
		"Dave\0"
		"Nick\0"
		"Brent\0"
		"Jeff\0"
		"Heidi\0"
		"Sarah\0"
		"Victoria\0"
		"Flora\0"
		"Jen\0"
		"Beth\0"
		"Rachel\0"
		"Rachel\0"
		"Raquel\0"
		"Diana\0"
		"Aimee\0"
		"Heidi\0"
		"Steve\0"
		"Stella\0"
		"Kristin\0"
		"Ryan\0"
		"Becky\0"
		"Chris\0"
		"Josh\0"
		"Tommy\0"
		"Adam\0"
		"Chris\0"
		"Matthias\0"
		"Caleb\0"
		"Micky\0"
		"Will\0"
		"Rob\0"
		"Ben\0"
		"Jen\0"
		"Jessica\0"
		"Mike\0"
		"Jake\0"
		"Laura\0"
		"Shannon\0"
		"Kristin\0"
		"Nate\0"
		"Kate\0"
		"Hannah\0"
		"Hannah\0"
		"BJ\0"
		"Barry\0"
		"Ethan\0"
		"Christina",
		NULL
	},

	{"Also....",3,
		"Ben Davis (No, I haven't finished DUMBMP3 yet)\0"
		"Icicled (Moo!)\0"
		"LoomSoft (College is too fun!)\0",NULL
	},

	{"College professors who I ignored and chose to program instead",5,
		"Dr. Dave Perkins\0"
		"Dr. Benjamin Lipscomb\0"
		"Dr. John Tyson\0"
		"Dr. Daniel Chamberlain\0"
		"Dr. Wei Hu\0",NULL
	},

	{"Can't forget last semester's professors",5,
		"Dr. Jake Jacobs\0"
		"Dr. Dave Perkins\0"
		"Dr. Eckley\0"
		"Dr. Wardwell\0"
		"Dr. Swanson\0",
		NULL
	},

	{"And a SUPER special thanks to my favorite professor of all...",4,
		"Professor Doezema\0"
		"I love you man\0"
		"(no, really, I do!)\0"
		"Western Civ is the most interesting class in the world!\0", NULL
	},

	{"Also...", 3,
		"The 17th Century Bohemian Protestants who throwed\0"
		"two governors and a secretary out a window, who then\0"
		"fell 70 feet into a pile 'o' manure.",
		NULL
	},

	{"Other people, who without their work this game would not exist:", 19,
		"Einstein (e = MC Hammer)\0"
		"Sophocles (Evil influencer of the dirty old man Freud)\0"
		"Sigmond Frued (very very dirty old man)\0"
		"Aristotle (Who said nonsense that sounded like wisdom)\0"
		"George \"Dubya\" Bush (Duuuuuuuuuh)\0"
		"Walt Disney (Mikey Mouse is dope yo!)\0"
		"Isaac Newton (Optiks was the best book I've ever read!)\0"
		"Micky Dolenz (I took the last train to Clarksville, but...)\0"
		"Peter, Paul, and Mary (Don't puff the magic dragon)\0"
		"King Louis XIV of France (who's get mad sexy legs!)\0"
		"Big Al (I'm too cheap to buy your food, but...)\0"
		"All my homies with the last name \"Dietel\" (NOT!)\0"
		"Wesley Willis (Rock on dude!)\0"
		"Samuel Colt (Great inventor, he ... NO!  Peace!)\0"
		"Martin Luther King (I have a dream, it's to live in the gutter...)\0"
		"Isaac Jacobs (Ima run from tha cops, foo!)\0"
		"\0"
		"..and other people, too...",
		NULL
	},

	{"",30,"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",NULL},


	// Version information
	{"Lander 2 "LANDER2_VERSION_STRING, 2,
		"(c) Copyright 2003 Chedda Cheeze\0"
		"Built on " __DATE__ " at " __TIME__ " using " ALLEGRO_PLATFORM_STR "\0",
		NULL
	},

	{"",90,
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
		NULL
	},

	{"Hmmm.....",31,
		"The credits are still going\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",NULL
	},
	{"Oh yeah...",31,
		"...press any key to return to the menu.\0"
		"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",NULL
	},
	{"You're not leaving, eh?",456,
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"There isn't any fun special message here,\0you're wasting your time.\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"Someone here has too much time\0on their hands and it isn't me...\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"\"Froedrick?\"  \"Eyegore!\"\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0"
		"CHEESE CAKE!"
/*30*/	"\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
		NULL
	},
	{"Tip:", 1,
		"You's Nuts",
		NULL
	},


	{NULL, 0, NULL, NULL}	// marks end of credits
};



// make this function more efficient!!!!!!! (ie only draw what's on the screen .. hmm .. I suppose it does that check, right?
void doCredits(tCredit *credit, void (*draw_func)(void))
{
	int TICKS_AT_START = TICKS;
	int n,i,a,o, currentOffset;
	float yOffset=0.0f;

	// setup the memory for the credits
	for(n=0; credit[n].heading != NULL; n++)
	{
		credit[n].addr = (char**)malloc(sizeof(char *) * credit[n].people);
		credit[n].addr[0] = credit[n].name;

		a = 1;
		for(i=0; a<(int)credit[n].people; ++i)
		{
			if(credit[n].name[i] == '\0')
			{
				credit[n].addr[a] = credit[n].name + i + 1;
				++a;
			}
		}
	}


	// scroll credits upward
	clear_keybuf();
	while(TICKS - TICKS_AT_START < 1000 || !keypressed())
	{
		clear(buffer);
		draw_func();		// should contain an updateClock call (not really, but... )

		// display the junk
		currentOffset = 0;
		for(n=0; credit[n].heading != NULL; n++)
		{
			textprintf(buffer,font,SCREEN_W/10, SCREEN_H - yOffset + currentOffset, COLOR_CREDIT_HEADING, credit[n].heading);
			currentOffset += 10;
			for(o=0; o<credit[n].people; o++)
			{
				textprintf(buffer,font,SCREEN_W/7 , SCREEN_H - yOffset + currentOffset, COLOR_CREDIT_NAME, credit[n].addr[o]);
				currentOffset += 10;
			}
			currentOffset += 20;
		}

		yOffset += 0.25f * TIME_SINCE_LAST_FRAME;
		blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
	}
}





void doTitleScreen(void)
{
	updateClock();

	drawTransitionStar(transitionStar);

	// draw large lander
	center_sprite(buffer,data[B_ADVANCE].dat);

	// version
	textprintf(buffer,font,3,SCREEN_H-10,COLOR_WHITE,"%s",LANDER2_VERSION_STRING);

	// copyright
	textprintf_right(buffer,font,SCREEN_W,SCREEN_H-10,COLOR_WHITE,"(c) Copyright 2003 Chedda Cheeze");

#ifdef ALLEGRO_DEBUG
	textprintf(buffer,font,0,0,COLOR_WHITE,"%0.0f",fps);
#endif
}






void doGame(int newGame)
{
	fLogfile("doGame\n");

	START_GAME:
	if(newGame)
	{
		currentLevel = 1;
		deaths = 0;
		difficulty = doMenu(&doTitleScreen,
			data[F_HUGE_0].dat,"Select Difficulty",
			4,data[F_SMALL_0].dat,
			"Wimp\0Average\0Masochist\0Cancel");
		if(difficulty == 3) return;
		lastDifficulty = difficulty;
		set_config_int("etc","c",TRUE);
	}
	else
	{
		currentLevel = lastLevel;
		deaths = lastDeaths;
		difficulty = lastDifficulty;
	}
	
	init();

	RESUME_GAME:
	last_tick_check = TICKS = 0;
	TIME_SINCE_LAST_FRAME = 0.0f;

	fLogfile("Main game loop starting or resuming\n");
	clear_keybuf();
	while(!key[KEY_ESC] && done == FALSE)
	{
		keycheck();
		update();
		clear(buffer);
		draw();

		//	vsync();	// this is usually not implemented properly on modern hardware :-/
		blit(buffer,screen,0,0,0,0,SCREEN_W,SCREEN_H);
	}
	fLogfile("Main game loop over\n");
	if(key[KEY_ESC])
	{
		int menu;
		MENU_THINGY:
		clear_keybuf();
		draw();
		menu = doMenu(&draw,
			data[F_HUGE_0].dat,"Lander 2",
			8,data[F_SMALL_0].dat,
			"Resume Game\0Get Blowed Up\0New Game\0End Game\0Options\0Credits\0High Score\0Quit");
		switch(menu)
		{
			case 0:	//resume
				ticksAtDeath = -900;
				goto RESUME_GAME;
				break;

			case 1: // blow me up
				TICKS = last_tick_check = 0;
				TIME_SINCE_LAST_FRAME = 0.0f;
				playerDie();
				goto RESUME_GAME;
				break;

			case 2:	//new
				clear_keybuf();

				if(doMenu(&draw, data[F_HUGE_0].dat, "Current game will be lost!", 2, data[F_SMALL_0].dat,"New game\0Cancel") == 0)
				{
					newGame = TRUE;
					goto START_GAME;
				}
				else
					goto MENU_THINGY;
				break;

			case 3: // exit to title
				clear_keybuf();
				// anything need to be done to close the game out?

				if(doMenu(&draw, data[F_HUGE_0].dat, "Really end game?", 2, data[F_SMALL_0].dat,"Yes\0No") == 0)
					break;
				else
					goto MENU_THINGY;

			case 4: // options
				doOptions(&draw);
				goto MENU_THINGY;
				break;

			case 5: // credits
				doCredits(credits,&draw);
				goto MENU_THINGY;

			case 6:	//hiscore
				doScores();
				goto MENU_THINGY;
				break;

			case 7:	//quit
				fQuit(&draw);
				goto MENU_THINGY;
				break;
		}
	}
	
	fLogfile("Game over must've occured...\n");
}




void doIntro(void)
{
	float w=0.0f,h=0.0f, y=0.0f;
	int last_reference_point;
	BITMAP *bmp = data[B_COMPANY_LOGO].dat;

	set_palette(data[P_COMPANY_LOGO].dat);
	clear(screen);
	clear_keybuf();

	
	// zoom in on logo
	while(w < SCREEN_W)
	{
		updateClock();
		if(keypressed()) return;
		stretch_blit(bmp,screen,0,0,SCREEN_W,SCREEN_H, SCREEN_W/2 - w/2, SCREEN_H/2 - h/2, w, h); 
		w += SCREEN_W/180.0f * TIME_SINCE_LAST_FRAME;
		h += SCREEN_H/180.0f * TIME_SINCE_LAST_FRAME;
	}

	// wait 1/2 second
	last_reference_point = TICKS;
	while(TICKS - last_reference_point < 500)
	{
		updateClock();
		if(keypressed()) return;
	}

	// scroll downward
	while(y < SCREEN_H)
	{
		updateClock();
		if(keypressed()) return;

		rectfill(screen,0,0,SCREEN_W,y,0);
		blit(bmp,screen,0,0,0,y,SCREEN_W,SCREEN_H-y+1);
		y += 2.0 * TIME_SINCE_LAST_FRAME;
	}

	destroy_bitmap(bmp);
}




int main(int argc, char *argv[])
{
	char temp[80];
	allegro_init();
	sprintf(temp,"Lander 2 %s",LANDER2_VERSION_STRING);
	set_window_title(temp);

	set_config_file("lander2.ini");

	lastLevel = get_config_int("etc","l",1);
	lastDeaths = get_config_int("etc","x",0);
	lastDifficulty = get_config_int("etc","d",1);
	inAGame = get_config_int("etc","c",FALSE);


	install_keyboard();
	keyHandle = initKeyHandler();
	install_timer();
	install_mouse();

	install_sound(DIGI_AUTODETECT,MIDI_AUTODETECT,0);
	soundVolume = get_config_int("sound","sound_vol",100);
	musicVolume = get_config_int("sound","music_vol",100);
	soundOn = get_config_int("sound","sound_on",TRUE);
	musicOn = get_config_int("sound","music_on",TRUE);
	setVolume(soundVolume,musicVolume);

	fLogfileFlush();


	if(argc > 1)
	{
		currentLevel = atoi(argv[1]);
	}


	SetGFXMode(GFX_AUTODETECT_FULLSCREEN,640,480);
	particles = particleSourceAllocate(15,250,TILE_W/4,1000);
	transitionStar = allocateTransitionStar();


	packfile_password("fruitpunch");
	data = load_datafile("data.dat");
	if(!data)
	{
		set_gfx_mode(GFX_TEXT,0,0,0,0);
		allegro_message("Error loading datafile!");
		allegro_exit();
		exit(1);
	}
	packfile_password(NULL);
	buffer = create_bitmap(SCREEN_W,SCREEN_H);
	col_bmp = create_bitmap(PLAYER_W,PLAYER_H);
	mk_spr_mask(data[B_LANDER].dat,0);
	current_song = rand()%5;

	LOCK_FUNCTION(update_timer);
	LOCK_VARIABLE(TICKS);
	LOCK_FUNCTION(animate_tiles);
	LOCK_VARIABLE(animateFlag);
	install_int_ex(&update_timer,MSEC_TO_TIMER(1));
	install_int_ex(&animate_tiles,BPS_TO_TIMER(10));


	doIntro();
	clear(screen);
	set_palette(data[P_GAME].dat);
//	buildFireParticleRamp();
	START_MENU:
	{
		int menu;
		clear_keybuf();
		menu = doMenu(&doTitleScreen, data[F_HUGE_0].dat, "LANDER 2",
			6, data[F_SMALL_0].dat, "Resume Last Game\0New Game\0Options\0Credits\0High Score\0Quit");
		switch(menu)
		{
			case 0: // resume
				if(inAGame == FALSE)
					doGame(TRUE);
				else
					doGame(FALSE);
				goto START_MENU;
				break;

			case 1:	//new
				doGame(TRUE);
				goto START_MENU;
				break;

			case 2:
				doOptions(&doTitleScreen);
				goto START_MENU;
				break;

			case 3:
				doCredits(credits, &doTitleScreen);
				goto START_MENU;
				break;

			case 4:	//hiscore
				doScores();
				goto START_MENU;
				break;

			case 5:	//quit
				fQuit(&doTitleScreen);
				goto START_MENU;
				break;
		}
	}


// program should never exit this way...
	return 2;
}
END_OF_MAIN();
