#include <allegro.h>
#include <stdlib.h>
#include <iostream>
#include <iomanip>
#include "metronome.h"

struct option
{
    char const *name;
    char const *pattern;
    int bpm;
    char const *comment; 
};

option options[] =
{
    { "reel",      "2111",         380,   "standard reel" },
    { "reelslow",  "2111",         300,   "standard reel, slow" },
    { "jig",       "211",          360,   "standard jig" },
    { "jigslow",   "211",          300,   "standard jig, slow" },
    { "slipjig",   "201201211",    350,   "slip jig (emphasis on 1 and 2)" },
    { "slide",     "201101",       400,   "slide (emphasis on 1)" },
    { "march",     "2010",         480,   "march" },
    { "hornpipe",  "200010100010", 1050,  "hornpipe (2/3 -> 1/3)" },
    { "hornpipe2", "20011001",     700,   "hornpipe (3/4 -> 1/4) ugly!" },
    { "5/8",       "21211",        300,   "5/8 tunes" },
    { "7/8",       "2112121",      330,   "7/8 tunes" },
    { "9/8",       "211211211",    365,   "9/8, without emphasis" },
    { "11/8",      "21112111211",  300,   "11/8 tunes (4/4/3)" },
    { "13/8",      "211121112112121", 300,   "13/8 tunes (4/4/3/2/2)" },
    { "sparse1",   "2000",         380,   "reel, only first beat" },
    { "sparse2",   "200",          360,   "jig, only first beat" },
    { 0,           0,              0,       0 }
};

void usage(char const *progname)
{
    cerr << "usage: " << progname << " [options]" << endl
	 << endl
	 << "\twhere <options> is one or more of" << endl
	 << "\t\tname     : a predefined rythm" << endl
	 << "\t\t-b bpm   : beats per minute" << endl
	 << "\t\t-p pat   : emphasis pattern" << endl
	 << "\t\t-h       : display this usage message" << endl
	 << endl
	 << "\tThe speed must be specified in beats per minute, an integer number" << endl
	 << "\tgreater than zero. This specifies the duration of a single unit in" << endl
	 << "\tthe emphasis pattern. The pattern must be specified as a string" << endl
	 << "\tconsisting only of the characters '0', '1' or '2'. The meaning of" << endl
	 << "\tthese numbers is:" << endl
	 << "\t\t 0: skip this beat" << endl
	 << "\t\t 1: play a soft beat" << endl
	 << "\t\t 2: play a loud beat (emphasis)" << endl
	 << endl
	 << "\tAn example command line for typical no-brain rock music could be:" << endl
	 << "\t\t" << progname << " -b 400 -p 2121" << endl
	 << endl
	 << "\tThe following set of rythms has been predefined, specify them using" << endl
	 << "\tthe name parameter on the commandline. Note that pattern and bpm" << endl
	 << "\tsettings can be overruled with the -b and -p options. Options will" << endl
	 << "\toverrule any previous options on the command line!" << endl
	 << endl;
    
    
    int i = 0;
    cerr << "\t" << setiosflags(ios::left)
	 << setw(12) << "name"
	 << setw(16) << "pattern"
	 << setw(8) << "bps"
	 << "description" << endl
	 << endl;
    while (options[i].name)
    {
	cerr << "\t"
	     << setiosflags(ios::left)
	     << setw(12) << options[i].name
	     << setw(16) << options[i].pattern
	     << setw(8) << options[i].bpm
	     << options[i].comment
	     << endl;
	i++;
    }
    cerr << endl;
}

int set_options(Metronome &m, int argc, char **argv)
{
    int
	speed_set = 0,
	pattern_set = 0;
    
    for (int a = 1; a < argc; a++)
    {
	// option starting with -: -b, -p
	if (argv[a][0] == '-')
	{
	    char opt = argv[a][1];

	    // check if we have a next argument
	    if (++a < argc)
	    {
		// we have a 'next' argument
		switch (opt)
		{
		    case 'b':
			if (!m.set_bpm(strtol(argv[a], 0, 0)))
			{
			    cerr << "Invalid bpm value specified." << endl;
			    return 0;
			}
			speed_set = 1;
			break;
		    case 'p':
			if (!m.set_pattern(argv[a]))
			{
			    cerr << "Invalid pattern spacified." << endl;
			    return 0;
			}
			pattern_set = 1;
			break;
		    default:
			usage(argv[0]);
			return 0;
			break;
		}
	    }
	    else
	    {
		// no next argument
		usage(argv[0]);
		return 0;
	    }
	}
	else
	{
	    // predefined rythm
	    int i = 0;
	    while (options[i].name)
	    {
		if (!strcmp(options[i].name, argv[a]))
		{
		    if (!m.set_bpm(options[i].bpm)
			||
			!m.set_pattern(options[i].pattern))
		    {
			cerr << "Error setting predefined rythm '"
			     << options[i].name << "'." << endl;
			return 0;
		    }
		    speed_set = 1;
		    pattern_set = 1;
		    break;
		}
		i++;
	    }

	    if (!options[i].name)
	    {
		cerr << "Invalid pattern name specified." << endl;
		return 0;
	    }
	}
    }

    // check if both speed and pattern have been defined
    if (!speed_set)
    {
	cerr << "Using default bpm setting: 380." << endl;
	m.set_bpm(380);
    }

    if (!pattern_set)
    {
	cerr << "Using default pattern: 2111 (reel)." << endl;
	m.set_pattern("2111");
    }
    
    return 1;
}

int main(int argc, char **argv)
{
    Metronome m;
    SAMPLE *smp;

    // install allegro
    // install timer
    // install keyboard
    // initialize sound
    allegro_init();
    install_timer();
    install_keyboard();
    
    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, 0) < 0)
    {
	cerr << "Could not install digital sound driver." << endl;
	return 1;
    }

    // load sample
    smp = load_wav("move.wav");
    if (!smp)
    {
	cerr << "Could not load sample 'move.wav'." << endl;
	return 1;
    }

    // read options
    if (!set_options(m, argc, argv))
	return 0;
    
    // set metronome parameters
    m.set_sample(smp);
    
    // start metronome
    m.start();

    // play
    while (!key[KEY_ESC])
    {
	m.play();
    }

    // stop metronome
    m.stop();

    // kill sample
    destroy_sample(smp);
    
    return 0;
}
END_OF_MAIN();
