/*
 *    Example program for the Allegro library, by Shawn Hargreaves.
 *
 *    Modified for Matrix Tracer Texture Demo ,By Karthik Kumar V .
 *   
 *    Requires : AlFont Add-On Library .
 *    Requires : mrs2.ttf to be in same directory .
 *    Suggested Resolution : 1024 x 768 x 16 or higher .
 *   
 *    This program demonstrates how to use the 3d matrix functions.
 *    It isn't a very elegant or efficient piece of code, but it does 
 *    show the stuff in action. I'll leave it to you to design a proper 
 *    model structure and rendering pipeline: after all, the best way to 
 *    do that sort of stuff varies hugely from one game to another.
 */

#include <stdio.h>

#include "allegro.h"

#include "alfont.h"              /* An Extra */
#include "mdef.h"                /* Another Extra */

#define DO_M_BG                  /* Draw Matrix BG */
#define DO_BOX_ENDS              /* Draw Ends of Boxes */

#define NUM_SHAPES         8     /* number of bouncing cubes */

#define NUM_VERTICES       8     /* a cube has eight corners */
#define NUM_FACES          6     /* a cube has six faces */

#define ELEN               32    /* edge of a cube */ 
#define NHP                31    /* (higher/equal power of 2 ,nearest to ELEN) - 1 */ 
                                 
#define ttfname FMRS        /* For Symbols */

typedef struct VTX
{
   fixed x, y, z;
} VTX;


typedef struct QUAD              /* four vertices makes a quad */
{
   VTX *vtxlist;
   int v1, v2, v3, v4;
} QUAD;


typedef struct SHAPE             /* store position of a shape */
{
   fixed x, y, z;                /* x, y, z position */
   fixed rx, ry, rz;             /* rotations */
   fixed dz;                     /* speed of movement */
   fixed drx, dry, drz;          /* speed of rotation */
} SHAPE;


VTX points[] =                   /* a cube, centered on the origin */
{
   /* vertices of the cube */
   { -ELEN << 16, -ELEN << 16, -ELEN << 16 },
   { -ELEN << 16,  ELEN << 16, -ELEN << 16 },
   {  ELEN << 16,  ELEN << 16, -ELEN << 16 },
   {  ELEN << 16, -ELEN << 16, -ELEN << 16 },
   { -ELEN << 16, -ELEN << 16,  ELEN << 16 },
   { -ELEN << 16,  ELEN << 16,  ELEN << 16 },
   {  ELEN << 16,  ELEN << 16,  ELEN << 16 },
   {  ELEN << 16, -ELEN << 16,  ELEN << 16 },
};


QUAD faces[] =                   /* group the vertices into polygons */
{
   { points, 0, 3, 2, 1 },
   { points, 4, 5, 6, 7 },
   { points, 0, 1, 5, 4 },
   { points, 2, 3, 7, 6 },
   { points, 0, 4, 7, 3 },
   { points, 1, 2, 6, 5 }
};


SHAPE shapes[NUM_SHAPES];        /* a list of shapes */


/* somewhere to put translated vertices */
VTX output_points[NUM_VERTICES * NUM_SHAPES];
QUAD output_faces[NUM_FACES * NUM_SHAPES];

/* changed from enum */
int wireframe=0,flat=1,gcol=2,grgb=3,atex=4,ptex=5,atex_mask=6,ptex_mask=7,
    atex_lit=8,ptex_lit=9,atex_mask_lit=10,ptex_mask_lit=11,atex_trans=12,ptex_trans=13,
    atex_mask_trans=14,ptex_mask_trans=15,last_mode=16,render_mode;

int render_type[] =
{
   0,
   POLYTYPE_FLAT,
   POLYTYPE_GCOL,
   POLYTYPE_GRGB,
   POLYTYPE_ATEX,
   POLYTYPE_PTEX,
   POLYTYPE_ATEX_MASK,
   POLYTYPE_PTEX_MASK,
   POLYTYPE_ATEX_LIT,
   POLYTYPE_PTEX_LIT,
   POLYTYPE_ATEX_MASK_LIT,
   POLYTYPE_PTEX_MASK_LIT,
   POLYTYPE_ATEX_TRANS,
   POLYTYPE_PTEX_TRANS,
   POLYTYPE_ATEX_MASK_TRANS,
   POLYTYPE_PTEX_MASK_TRANS
};


char *mode_desc[] =
{
   "Wireframe",
   "Flat shaded",
   "Single color gouraud shaded",
   "Gouraud shaded",
   "Texture mapped",
   "Perspective correct texture mapped",
   "Masked texture mapped",
   "Masked persp. correct texture mapped",
   "Lit texture map",
   "Lit persp. correct texture map",
   "Masked lit texture map",
   "Masked lit persp. correct texture map",
   "Transparent texture mapped",
   "Transparent perspective correct texture mapped",
   "Transparent masked texture mapped",
   "Transparent masked persp. correct texture mapped",
};

/* Draw Loop */
char dtbuf[2]={"$"};
int di,dj,dx,dy;

ALFONT_FONT *alfy; /* For Font Symbols */

BITMAP *texture;   /* For Shape Texture */

/* Returns New Character */
char retchar(char orig,int flag)
{
    if(flag||((rand()%8)>5)) /* Condition to Generate Code .*/
        return prtstr[(rand()%prtlen)];
    return orig;
}

/* Tracer Init */
void trainit(int i)
{
    int j;
    tracers[i].pclen=((rand()%MAXLDSP)+MINLEN);
    tracers[i].x=(i*(SYMS+1));
    tracers[i].y=((rand()%(SCREEN_H+(tracers[i].pclen*(SYMS+1))))-(tracers[i].pclen*(SYMS+1)));
    tracers[i].ysp=((rand()%MAXDSP)+MINSP);
    for(j=0;j<tracers[i].pclen-1;j++)
    {
        tracers[i].string[j]=retchar(' ',1);
    }
    tracers[i].string[tracers[i].pclen-1]=' ';
}

/* Texture Loop */
void dotex(BITMAP *texture,int cflag,int uflag,int cliph)
{
    if(cflag)clear_bitmap(texture); 
    for(di=0;di<NUMTRACES;di++)
    {
        dx=tracers[di].x;
        dy=tracers[di].y;
        for(dj=(tracers[di].pclen-1);dj>=0;dj--)
        {
                dtbuf[0]=tracers[di].string[dj];
                tracers[di].string[dj]=retchar(tracers[di].string[dj],0);
                alfont_textout(texture,alfy,dtbuf,dx,dy,greens[((dj*NUMGRAD)/(tracers[di].pclen-1))]);
                dy+=(SYMS+1);
        }
        if(uflag)
        {
                tracers[di].y+=tracers[di].ysp;
                if(tracers[di].y>=cliph){trainit(di);tracers[di].y=-(tracers[di].pclen*(SYMS+1));} //Independant of This ....
        }
        }
 }

/* Post Texture Loop */
void doposttex(BITMAP *texture)
{
rect(texture,0,0,texture->w-1,texture->h-1,greens[NUMGRAD]);
}

/* initialise shape positions */
void init_shapes(void)
{
   int c;

   for (c=0; c<NUM_SHAPES; c++) {
      shapes[c].x = ((rand() & 255) - 128) << 16;
      shapes[c].y = ((rand() & 255) - 128) << 16;
      shapes[c].z = 768 << 16;
      shapes[c].rx = 0;
      shapes[c].ry = 0;
      shapes[c].rz = 0;
      shapes[c].dz =  ((rand() & 255) - 8) << 12;
      shapes[c].drx = ((rand() & NHP) - 16) << 12;
      shapes[c].dry = ((rand() & NHP) - 16) << 12;
      shapes[c].drz = ((rand() & NHP) - 16) << 12;
   }
}

/* update shape positions */
void animate_shapes(void)
{
   int c;

   for (c=0; c<NUM_SHAPES; c++) {
      shapes[c].z += shapes[c].dz;

      if ((shapes[c].z > itofix(1024)) ||
	  (shapes[c].z < itofix(192)))
	 shapes[c].dz = -shapes[c].dz;

      shapes[c].rx += shapes[c].drx;
      shapes[c].ry += shapes[c].dry;
      shapes[c].rz += shapes[c].drz;
   }
}

/* translate shapes from 3d world space to 2d screen space */
void translate_shapes(void)
{
   int c, d;
   MATRIX matrix; //Ain't Related to "The Matrix" ;)
   VTX *outpoint = output_points;
   QUAD *outface = output_faces;

   for (c=0; c<NUM_SHAPES; c++) {
      /* build a transformation matrix */
      get_transformation_matrix(&matrix, itofix(1),
				shapes[c].rx, shapes[c].ry, shapes[c].rz,
				shapes[c].x, shapes[c].y, shapes[c].z);

      /* output the vertices */
      for (d=0; d<NUM_VERTICES; d++) {
	 apply_matrix(&matrix, points[d].x, points[d].y, points[d].z, &outpoint[d].x, &outpoint[d].y, &outpoint[d].z);
	 persp_project(outpoint[d].x, outpoint[d].y, outpoint[d].z, &outpoint[d].x, &outpoint[d].y);
      }

      /* output the faces */
      for (d=0; d<NUM_FACES; d++) {
	 outface[d] = faces[d];
	 outface[d].vtxlist = outpoint;
      }

      outpoint += NUM_VERTICES;
      outface += NUM_FACES;
   }
}



/* draw a line (for wireframe display) */
void wire(BITMAP *b, VTX *v1, VTX *v2)
{
   int col = MID(128, 255 - fixtoi(v1->z+v2->z) / 16, 255);
   line(b, fixtoi(v1->x), fixtoi(v1->y), fixtoi(v2->x), fixtoi(v2->y), palette_color[col]);
}



/* draw a quad */
void draw_quad(BITMAP *b, VTX *v1, VTX *v2, VTX *v3, VTX *v4, int mode)
{
   int col;

   /* four vertices */
   V3D vtx1 = { 0, 0, 0, 0,      0,      0 };
   V3D vtx2 = { 0, 0, 0, ELEN<<16, 0,      0 };
   V3D vtx3 = { 0, 0, 0, ELEN<<16, ELEN<<16, 0 };
   V3D vtx4 = { 0, 0, 0, 0,        ELEN<<16, 0 };

   vtx1.x = v1->x;   vtx1.y = v1->y;   vtx1.z = v1->z;
   vtx2.x = v2->x;   vtx2.y = v2->y;   vtx2.z = v2->z;
   vtx3.x = v3->x;   vtx3.y = v3->y;   vtx3.z = v3->z;
   vtx4.x = v4->x;   vtx4.y = v4->y;   vtx4.z = v4->z;

   /* cull backfaces */
   if ((mode != POLYTYPE_ATEX_MASK) && (mode != POLYTYPE_PTEX_MASK) &&
       (mode != POLYTYPE_ATEX_MASK_LIT) && (mode != POLYTYPE_PTEX_MASK_LIT) &&
       (polygon_z_normal(&vtx1, &vtx2, &vtx3) < 0))
      return;

   /* set up the vertex color, differently for each rendering mode */
   switch (mode) {

      case POLYTYPE_FLAT:
	 col = MID(128, 255 - fixtoi(v1->z+v2->z) / 16, 255);
	 vtx1.c = vtx2.c = vtx3.c = vtx4.c = palette_color[col];
	 break;

      case POLYTYPE_GCOL:
	 vtx1.c = palette_color[0xD0];
	 vtx2.c = palette_color[0x80];
	 vtx3.c = palette_color[0xB0];
	 vtx4.c = palette_color[0xFF];
	 break;

      case POLYTYPE_GRGB:
	 vtx1.c = 0x000000;
	 vtx2.c = 0x7F0000;
	 vtx3.c = 0xFF0000;
	 vtx4.c = 0x7F0000;
	 break;

      case POLYTYPE_ATEX_LIT:
      case POLYTYPE_PTEX_LIT:
      case POLYTYPE_ATEX_MASK_LIT:
      case POLYTYPE_PTEX_MASK_LIT:
	 vtx1.c = MID(0, 255 - fixtoi(v1->z) / 4, 255);
	 vtx2.c = MID(0, 255 - fixtoi(v2->z) / 4, 255);
	 vtx3.c = MID(0, 255 - fixtoi(v3->z) / 4, 255);
	 vtx4.c = MID(0, 255 - fixtoi(v4->z) / 4, 255);
	 break; 
   }

   /* draw the quad */
     quad3d(b, mode, texture, &vtx1, &vtx2, &vtx3, &vtx4);
}

/* callback for qsort() */
int quad_cmp(const void *e1, const void *e2)
{
   QUAD *q1 = (QUAD *)e1;
   QUAD *q2 = (QUAD *)e2;

   fixed d1 = q1->vtxlist[q1->v1].z + q1->vtxlist[q1->v2].z +
	      q1->vtxlist[q1->v3].z + q1->vtxlist[q1->v4].z;

   fixed d2 = q2->vtxlist[q2->v1].z + q2->vtxlist[q2->v2].z +
	      q2->vtxlist[q2->v3].z + q2->vtxlist[q2->v4].z;

   return d2 - d1;
}



/* draw the shapes calculated by translate_shapes() */
void draw_shapes(BITMAP *b)
{
   int c;
   QUAD *face = output_faces;
   VTX *v1, *v2, *v3, *v4;

   /* depth sort */
   qsort(output_faces, NUM_FACES * NUM_SHAPES, sizeof(QUAD), quad_cmp);
    
   /* Do texture Loop */
   if (render_mode >= atex&&render_mode <= ptex_mask_trans)
   {
   dotex(texture,1,1,b->h); //As b->h > texture->h ....!
   #ifdef DO_M_BG 
         dotex(b,0,0,b->h);
   #endif /* DO_M_BG */
   #ifdef DO_BOX_ENDS 
      doposttex(texture);
   #endif /* DO_BOX_ENDS */
   }
   
   for (c=0; c < NUM_FACES * NUM_SHAPES; c++) {
      /* find the vertices used by the face */
      v1 = face->vtxlist + face->v1;
      v2 = face->vtxlist + face->v2;
      v3 = face->vtxlist + face->v3;
      v4 = face->vtxlist + face->v4;

      /* draw the face */
      if (render_mode == wireframe) {
	 wire(b, v1, v2);
	 wire(b, v2, v3);
	 wire(b, v3, v4);
	 wire(b, v4, v1);
      }
      else {
 	  draw_quad(b, v1, v2, v3, v4, render_type[render_mode]);
      }

      face++;
   }
}

/* RGB -> color mapping table. Not needed, but speeds things up */
RGB_MAP rgb_table;


/* lighting color mapping table */
COLOR_MAP light_table;

/* transparency color mapping table */
COLOR_MAP trans_table;



void print_progress(int pos)
{
   #ifdef ALLEGRO_CONSOLE_OK

      if ((pos & 3) == 3) {
	 printf("*");
	 fflush(stdout);
      }

   #endif
}

int main(void)
{
   BITMAP *buffer;
   PALETTE pal;
   int i,c, w, h, bpp;
   int last_retrace_count;

   allegro_init();
   install_keyboard();
   install_mouse();
   install_timer();
   if(alfont_init()!=ALFONT_OK)
   {
   allegro_message("Unable to initialize font system");
   allegro_exit();return 1;
   }
   alfy = alfont_load_font(ttfname);
   if(!alfy)
   {
   allegro_message("Unable to get font : %s\n", ttfname);
   alfont_exit();allegro_exit();return 1;
   }
   
   /* color 0 = black */
   pal[0].r = pal[0].g = pal[0].b = 0;

   /* copy the desktop palette */
   for (c=1; c<64; c++)
      pal[c] = desktop_palette[c];

   /* make a red gradient */
   for (c=64; c<96; c++) {
      pal[c].r = (c-64)*2;
      pal[c].g = pal[c].b = 0;
   }

   /* make a green gradient */
   for (c=96; c<128; c++) {
      pal[c].g = (c-96)*2;
      pal[c].r = pal[c].b = 0;
   }

   /* set up a greyscale in the top half of the palette */
   for (c=128; c<256; c++)
      pal[c].r = pal[c].g = pal[c].b = (c-128)/2;

   /* build rgb_map table */
   #ifdef ALLEGRO_CONSOLE_OK
      printf("Generating rgb_map table:\n");
      printf("<................................................................>\r<");
   #endif

   create_rgb_table(&rgb_table, pal, print_progress);
   rgb_map = &rgb_table;

   /* build a lighting table */
   #ifdef ALLEGRO_CONSOLE_OK
      printf("\n\n");
      printf("Generating lighting table:\n");
      printf("<................................................................>\r<");
   #endif

   create_light_table(&light_table, pal, 0, 0, 0, print_progress);
   color_map = &light_table;

   /* build a transparency table */
   #ifdef ALLEGRO_CONSOLE_OK
      printf("\n\n");
      printf("Generating transparency table:\n");
      printf("<................................................................>\r<");
   #endif

   /* textures are 25% transparent (75% opaque) */
   create_trans_table(&trans_table, pal, 192, 192, 192, print_progress);

   #ifdef ALLEGRO_CONSOLE_OK
      printf("\n");
   #endif

   /* set up the truecolor blending functions */
   /* textures are 25% transparent (75% opaque) */
   set_trans_blender(0, 0, 0, 192);

   /* set the graphics mode */
   if (set_gfx_mode(GFX_SAFE, 320, 200, 0, 0) != 0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to set any graphic mode\n%s\n", allegro_error);
      return 1;
   }
   set_palette(desktop_palette);

   c = GFX_AUTODETECT;
   w = SCREEN_W;
   h = SCREEN_H;
   bpp = bitmap_color_depth(screen);
   if (!gfx_mode_select_ex(&c, &w, &h, &bpp)) {
      allegro_exit();
      return 1;
   }

   set_color_depth(bpp);

   if (set_gfx_mode(c, w, h, 0, 0) != 0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Error setting graphics mode\n%s\n", allegro_error);
      return 1;
   }

   set_palette(pal);
   text_mode(-1);
   
   /* building stuff */
   #ifdef ALLEGRO_CONSOLE_OK
      printf("\n\n");
      printf("Generating matrix stuff:\n");
      printf("<................................................................>\r<");
   #endif

   /* Tracer Palette Initialize */
   for(i=0;i<=NUMGRAD;i++)
      greens[NUMGRAD-i]=makecol((int)((i*i*0.38)/NUMGRAD),(int)(((i*i*0.99)/NUMGRAD)+(2*i))/3,(int)((i*i*0.69)/NUMGRAD));
   
   /* make a bitmap for use as a texture map */
   texture = create_bitmap(NHP+1,NHP+1);
   
   /* Font Initialize */
   alfont_set_font_size(alfy,SYMS);

   /* Tracer Initialize */
   NUMTRACES=(SCREEN_W/(SYMS+1)); //As (buffer->w=SCREEN_W) > texture->w ....!

   /* Variable Initialize */
   for(;prtstr[prtlen];prtlen++);
   
   /* Randomize */
   srand(time(NULL));
   
   /* Tracer Initialize */
   for(i=0;i<NUMTRACES;i++){trainit(i);}

   /* double buffer the animation */
   buffer = create_bitmap(SCREEN_W, SCREEN_H);

   /* set up the viewport for the perspective projection */
   set_projection_viewport(0, 0, SCREEN_W, SCREEN_H);

   /* initialise the render mode */
   render_mode=wireframe;

   /* initialise the bouncing shapes */
   init_shapes();

   last_retrace_count = retrace_count;

   for (;;) {
      clear_bitmap(buffer);
      while (last_retrace_count < retrace_count) {
	       animate_shapes();
	       last_retrace_count++;
      }

     translate_shapes();
     
     draw_shapes(buffer);
 
     textprintf(buffer, font, 0, 0, palette_color[192], "%s, %d bpp", mode_desc[render_mode], bitmap_color_depth(screen));
     textout(buffer, font, "Press a key to change", 0, 12, palette_color[192]);

      vsync();
      blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H); 

      if (keypressed()) {
	 if ((readkey() & 0xFF) == 27)
	    break;
	 else {
	    render_mode++;
	    if (render_mode >= last_mode) {
	       render_mode = wireframe;
	       color_map = &light_table;
	    }
	    if (render_type[render_mode] >= POLYTYPE_ATEX_TRANS)
	       color_map = &trans_table;
	  }
      }
   }
   
   destroy_bitmap(texture);
   destroy_bitmap(buffer);
   alfont_exit();
   allegro_exit();
   return 0;
}

END_OF_MAIN();
