#include "metronome.h"
#include <allegro.h>

// global timer flag. make static so it is invisible outside of this file.
// this means, unfortunately, that you can have only 1 instance of
// metronome at any time!
static int timer_flag = 0;

Metronome::Metronome()
{
    // set default values
    bpm = 180;
    pattern = "211";
    pattern_pos = 0;
    sample = 0;
}

Metronome::~Metronome()
{
    // since we're using a <string>, no need to destroy it.
    stop();    // kill timer
}

void Metronome::start()
{
    // install the timer (time in milliseconds)
    install_int(timer_proc, 60 * 1000 / bpm);
}

void Metronome::stop()
{
    // stop the timer (remove interrupt)
    remove_int(timer_proc);
}

// call this in a tight loop to have the metronome make noise
// it will check the timer flag. If flag is set, the sample is played
// and 1 is returned. If the flag is not set, 0 is returned.
int Metronome::play()
{
    if (!sample)
	return 0;
    
    if (timer_flag)
    {
	char c = pattern[pattern_pos];
	
	// volume depends on pattern.
	// if '0', don't play.
	// if '1', play softly (50).
	// if '2', play loud (255)
	// pan       = 128:   centre stereo
	// frequency = 1000:  original recording frequency.
	switch (c)
	{
	    case '1':
		play_sample(sample, 50, 128, 1000, 0);
		break;
	    case '2':
		play_sample(sample, 255, 128, 1000, 0);
		break;
	    default:
		break;
	}

	// reset the flag
	timer_flag = 0;

	// move to next pattern position (wrap around)
	pattern_pos = (pattern_pos + 1) % pattern.length();

	// return success
	return 1;
    }

    return 0;
}

void Metronome::timer_proc()
{
    // just set a flag
    // this is an interrupt driven function, so try to do
    // as little as possible to prevent problems with memory locking.
    timer_flag = 1;
}

// 0: failure
// 1: succes
int Metronome::set_pattern(string const &npat)
{
    // check the new pattern
    if (npat.length() <= 0)
	return 0;

    for (unsigned int i = 0; i < npat.length(); i++)
    {
	char c = npat[i];
	
	if (c != '0' && c != '1' && c != '2')
	    return 0;
    }
    
    pattern = npat;
    pattern_pos = 0;
    
    return 1;
}

// set and retrieve pattern
string const & Metronome::get_pattern() const
{
    return pattern;
}

int Metronome::get_pattern_length() const
{
    return pattern.length();
}

// set and retrieve the sample
void Metronome::set_sample(SAMPLE *smp)
{
    sample = smp;
}

SAMPLE *Metronome::get_sample() const
{
    return sample;
}

// set and retrieve pattern position
int Metronome::set_pattern_pos(int npp)
{
    if (npp >= 0 && npp < (int)pattern.length())
    {
	pattern_pos = npp;
	return 1;
    }

    return 0;
}

int Metronome::get_pattern_pos() const
{
    return pattern_pos;
}

// set and retrieve beats per minute
int Metronome::set_bpm(int nbpm)
{
    if (nbpm > 0)
    {
	bpm = nbpm;
	return 1;
    }

    return 0;
}

int Metronome::get_bpm() const
{
    return bpm;
}
