//      Copyright (C) 2015  E.J.M. Martens
//
//      This program is free software; you can redistribute it and/or
//      modify it under the terms of the GNU General Public License
//      as published by the Free Software Foundation; either version 2
//      of the License, or (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.


/**  \mainpage Star - Trek   V3
*    \author    Edwin (E.J.M.) Martens 2015 - 2016
*    \section  INTRODUCTION
* @verbatim


 \++================================|                    _=_
  \_________________________________/              ___/==+++==\___
               """\__      \"""       |======================================/
                     \__    \_          / ..  . _/--===+_____+===--""
                        \__   \       _/.  .. _/         `+'
                           \__ \   __/_______/                      \ /
                          ___-\_\-'---==+____|                  ---==O=-
                    __--+" .    . .        "==_                     / \
                    /  |. .  ..     -------- | \
                    "==+_    .   .  -------- | /
                         ""\___  . ..     __=="
                               """"--=--""


      Credits go to Paramount pictures for the star - trek concept and universe.


 @endverbatim
*
* This game is actually a very advanced version of the BASIC STAR-TREK text game.
* see https://en.wikipedia.org/wiki/Star_Trek_%28text_game%29.
* Many years ago before the WebBrowser wars there was a book SpecTrek (Walther Schoonenberg).
* It tought how to write a bit more ´graphical´ version of the STAR-TREK text game.
* It was still turn based and the ´graphics´ where reprogrammed character codes in a text screen (wich is impossible with a PC).
* The book was for the ZX Spectrum computer, hence itś name. Reading the book I ported everyting to GW-Basic and improved on it.
* A second ( real time) version, still with a text command interface, but with true sprites was written in Delphi 3.0 using Delphi X.
* The third version (The second real time version) was written in C++ using Allegro 4.x and OpenLayer, and a big 50 * 50 universe.
* This version (V3)  ( third of the real time versions) has a whopping 200 * 200 sectors universe, a sector is now what a Quadrant was
* in the text version, making it more in line with the startrek universe. It uses the great Allegro 5 library wich enables a lot of cool features.
* Ships can now Cloak, and actually use it in a smart way.
*
*
* \section  BASIC WORKINGS:
*
* At the start of a new game, a random universe is created. Size is defined by UNIVERSE_SIZE in types.h
* A sector is the actual "Playing field" you see in the main view. This is where the action happens.
* Every time the Enterprise enters a new sector, the old sector is stored in the universe, saving everything that happened in that sector.
* See Universe.h and universe.cpp. And a new "playing field" is built, reading the sector at the new sector position of the enterprise.
* A sector has a size of SECTORSIZE * SECTORSIZE (10000 * 10000) pixels. So when the Enterprise passes the borders, the Sector position changes
* and the procedure described above takes place.
* A sector is built by adding TSprite objects: SpaceObjects, ships, etc. ( all derived from TSprite in Engine.h) to the Engine.
* When an object is added to the engine, the engine OWNS that object and also takes care of destruction !
*
* All ships are derived from TShip, which has a LOT of basic ship functionality. Adding a new ship class to the game is quite easy
* The Universe will be dynamic, with rules for moving ships to other sectors, and results of "Simulated battles" and other events.
*
*
* \section THE GAME
* You are placed in the role of James Tiberius Kirk, captain of the USS Enterprise NCC-1701.
* The Enterprise is a big spaceship with a lot of systems and equipment. Everything on your ship uses energy to function.
* This energy is taken from the WARPCORE energy buffer. The Warp core generates energy by a matter-antimatter reaction producing a huge amount of energy.
* Under normal circumstances more than enough to power everything on your ship and enough to make a warp jump through multiple sectors.
* However If the WARPCORE is damaged the energy output will decrease.
* Energy is also needed to maintain a containment field around the WARPCORE. If the warpcore is damaged and there is no energy left the warpcore will fail.
* Resulting in the destruction of your ship !
* Damaged systems can be repaired at the Engineering screen. The repair crew can only fix one item at a time.
* At a federation starbase repairs can be made much quicker.
* The Navigation screen shows the Known universe, revealing more when new sectors are explored or when scanned with a probe (probes not implenemted yet)
*
* The game is far from finished yet. However you can already explore a HUGE universe
* with Klingon and Romulan empires. For testing purposes A Klingon Task force will invade your home sector at the start of a game.
* To disable this.. go to Universe.cpp and comment out:
*
*    a_Sector.m_nKlingonBC  = 2;
*    a_Sector.m_nKlingonBOP = 2;
*    a_Sector.m_nFederation1 = 3;
*
* In Universe::GenerateHomeSector(Sector & a_Sector).
*
*
* \section COMBAT
* The Enterprise has 2 weapons, Photon torpedoes and Phasers.
* PHOTON TORPEDOES torpedoes explode when they hit something resulting in massive damage over all systems.
* The damage is not dependent on the distance, there is only a maximum range.
* you only have a certain amount of torpedoes, they can be replenished at a starbase only !
* When COMPUTER and SENSORS are in good shape, they will precalculate a firing solution for a selected target.
*
* PHASERS only consume energy. Over time, the Phaserbanks are charged with energy from the warpcore buffer.
* The damage a phaser does is dependend on the distance to the target. The further away, the less damage.
* PHASERS are much more precise and can take out one single system.
*
* The Enterprise is also equipped with shields. When hit, the shield energy will drop using energy to block the incoming weapon.
* The shieldsystem will immediately begin to recharge the shield, taking energy from the WARPCORE buffer.
* If the shieldsystem is damaged,it will become less efficient in charging the shield.
* (Converting Warpcore energy into shield energy will be less efficient)
*
*
* \section CONTROLS
*
* The standard controls are:
*
*       Arrow UP     --->  Speed up
*       Arrow DOWN   --->  Slow Down
*       Arrow LEFT   --->  Turn Left
*       Arrow RIGHT  --->  Turn Right
*       T            --->  Select nearest Enemy Target
*       F            --->  Select nearest Friend
*       TAB          --->  Fire front photon tube
*       L SHIFT      --->  Fire aft photon tube
*       P            --->  Launch a probe
*       SPACE        --->  Fire Phaser
*       1..5         --->  Warp 1 tm Warp 5
*       T            --->  target nearest enemy
*       F            --->  target nearest friend
*       D            --->  Dock at starbase (when in range)
*       R            --->  Release from dock.
*       F1           --->  Main screen
*       F2           --->  Engineering
*       F3           --->  Astro Navigation
*
*       F5           --->  Navigation Exploration
*       F6           --->  Navigation Tactical mode
*       F7           --->  Navigation Political mode
*
*       F12          --->  Take a screenshot saved in screenshot folder
*
**/

/** @file StartrekV3.cpp
    Main source file, containing the game loop and setup and initialization functions.
    Also basic game rules like sector bounds checking and game state switching is implemented here.
*/



//TODO (Edwin#2#) NPC Warp effect when leaving or entering the current sector ( warp effect sprite)
//TODO (Edwin#2#) NPC-Player communication and correct behaviour
//TODO (Edwin#3#) Spock analysis, click object to let spock tell about it


#include <stdio.h>
#include <iostream>
#include <time.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_native_dialog.h>
#include <allegro5/allegro_image.h>
#include <allegro5/allegro_primitives.h>
#include <allegro5/allegro_audio.h>
#include <allegro5/allegro_acodec.h>
#include <map>
#include <vector>

#include "version.h"
//AutoVersion::MAJOR //<< version Text
#include "utils.h"
#include "FontManager.h"
#include "SoundManager.h"
#include "Paralax.h"
#include "Engine.h"
#include "Bullet.h"

//#include "Smoke.h"
//#include "Explosion.h"
#include "Animation.h"

#include "Universe.h"

#include "Enterprise.h"
#include "Romulan_BOP.h"
#include "Klingon_BC.h"
#include "Klingon_BOP.h"
#include "SpaceObject.h"
#include "Starbase.h"
#include "Federation_Ship.h"
#include "KeyMapper.h"
#include "Communication.h"
#include "Probe.h"
#include "Menu.h"
#include "SaveSlots.h"


//FILE  * g_pFile;
char   g_szFileName[] = "MyGame.sav"; // temporary hardcoded savefile


#ifdef DEBUG
char   g_szSector[20];  // sector position storage for debug version
#endif // DEBUG

bool g_blGodMode = false;  // GOD cheat mode for testing the game


///General save path
ALLEGRO_PATH * g_pSavePath = NULL;

/// Path for screenshots
ALLEGRO_PATH * g_pScreenshotPath = NULL;

/// The display for graphics...
ALLEGRO_DISPLAY * g_pDisplay = NULL;

/// The event queue
ALLEGRO_EVENT_QUEUE *	g_pEventQueue = NULL;

// Other Globals
bool g_blQuit = false;          // if true quit program
bool g_blEscape = false;

/// game logic cycles per second
double g_nCyclesPerSecond = 60;

/// last polled system time to calculate deltatime
double g_dLastTime;

/// screenmode.. what screen should be visible [Main] [Engineering] [Navigation] etc
int g_nScreenMode = MODE_MAINSCREEN;

/// Gamestate stack [Menu] [Intro] [Game]
vector<int> g_vGameState;

// screen properties
int g_nScreenWidth;
int g_nScreenHeight;


// Global pointers
ALLEGRO_TIMER * g_pTimer =0;
/// Pointer to universe Object.
Universe      * g_pUniverse         = NULL;
/// Pointer to Keymapper.
TKeyMapper    * g_pKeyMapper        = NULL;
/// Pointer to Game Engine.
TEngine       * g_pEngine           = NULL;
/// Pointer Paralax system.
Paralax       * g_pParalax          = NULL;
/// Pointer to The enterprise ( The player !)
TEnterprise   * g_pEnterprise       = NULL;
/// Pointer to The Communication System.
Communication * g_pCommunication    = NULL;
/// Pointer to the menu system
TMenu * g_pMenu = NULL;

SlotManager * g_pSlotManager = NULL;


//Forward declarations
/// Get the desktop resolution
void get_desktop_resolution(int a_nAdapter, int *a_pWidth, int * a_pHeighth);
/// This function creates a new universe
void NewUniverse();
/// When the Enterprise enters a new sector This function builds a new playing field
/// from the sector you just entered.
void NewSector(int a_nX, int a_nY);
/// Checks if the enterprise leaves this sector
void CheckSectorBounds();
///  Invoke a keymapping functionality so you can customize the controls. still a Rough implementation
void DoKeymap();
///  invoke menu ( not implemented yet )
void DoMenu();
/// The actual game loop
void DoGame();
/// Intro screen
void DoIntro();
/// screen for loading a game
void DoLoadScreen();
/// screen for saving a game
void DoSaveScreen();
/// start a new game
void NewGame();
/// save current game
void SaveGame();
/// setup load screen
void SetupLoadScreen();
/// quit program
void Quit();
/// return to running game
void ReturnToGame();
/// Save key mapping
void SaveMapping();
/// Load key Mapping
void LoadMapping();
/// get the date and time
const string currentDateTime();



// Menu event handlers

void NewGame()
{

     g_pEngine->Clear(false);
     // create the Enterprise
	 g_pEnterprise = new TEnterprise();
	 // add the enterprise to the game engine
	 g_pEngine->Add(g_pEnterprise);
	 Log("Enterprise created");

	 // Create the communication system
     delete g_pCommunication;
     g_pCommunication = new Communication(150,5);
     Log("Communication created");

	 // create a new universe, this should move to a menu function in the future
	 NewUniverse();
	 Log("New universe generated");

	 // Build a new sector
	 NewSector(g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);
	 Log("New Sector");

     // set enterprise position and move camera to it
     g_pEnterprise->SetPosition(g_pUniverse->m_nSectorCenter, g_pUniverse->m_nSectorCenter);
     g_pEngine->SetOrigin(g_pUniverse->m_nSectorCenter, g_pUniverse->m_nSectorCenter);

     // push gamestate GS_GAME on the gamestate stack
     g_vGameState.push_back(GS_GAME);
     Log("GAME STATE TO GAME");

     #ifdef DEBUG
            sprintf(g_szSector,"SECTOR: %d,%d",g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);
     #endif // DEBUG
     al_hide_mouse_cursor(g_pDisplay);
     g_dLastTime = al_get_time();

	 if (g_pMenu != NULL)
     {
         g_pMenu->ClearItems();
         g_pMenu->AddMenuItem("Return to game",ReturnToGame );
         g_pMenu->AddMenuItem("New Game",NewGame);
         g_pMenu->AddMenuItem("Load",SetupLoadScreen);
         g_pMenu->AddMenuItem("Save",SaveGame);
	     g_pMenu->AddMenuItem("Map Keys",DoKeymap);
         g_pMenu->AddMenuItem("Quit",Quit);
     }

}


void SaveMapping()
{
    ALLEGRO_PATH * pSavePath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);
    al_set_path_filename(pSavePath,"keys.ini");
    ofstream SaveFile(al_path_cstr(pSavePath,ALLEGRO_NATIVE_PATH_SEP),ios::out);
    if (SaveFile)
    {
       g_pKeyMapper->Save(SaveFile);
       SaveFile.close();
    }
}

void LoadMapping()
{
   ALLEGRO_PATH * pLoadPath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);
   al_set_path_filename(pLoadPath,"keys.ini");
   ifstream LoadFile(al_path_cstr(pLoadPath,ALLEGRO_NATIVE_PATH_SEP),ios::in);
   if (LoadFile)
   {
      g_pKeyMapper->Load(LoadFile);
      LoadFile.close();
   }
   else
   {
      // Set default
      g_pKeyMapper->SetDefaultMapping();
   }
}



void LoadGame(string a_strLoadName)
{
    delete g_pSlotManager;
    a_strLoadName+=".UNI";
    al_set_path_filename(g_pSavePath, a_strLoadName.c_str());
    ifstream LoadFile( al_path_cstr(g_pSavePath,ALLEGRO_NATIVE_PATH_SEP),
                       ios::in | ios::binary);

    if (LoadFile)
       {
          g_pEngine->Load(LoadFile);
          g_pUniverse = new Universe();
          g_pUniverse->Load(LoadFile);
          LoadFile.close();
          g_pEnterprise->SetDefaultNavigationOrigin();
          // Create the communication system
          delete g_pCommunication;
          g_pCommunication = new Communication(150,5);
          Log("Communication created");
          g_vGameState.pop_back();
          g_vGameState.push_back(GS_GAME);
          Log("GAME STATE TO GAME");
          al_hide_mouse_cursor(g_pDisplay);
          g_dLastTime = al_get_time();

#ifdef DEBUG
          sprintf(g_szSector,"SECTOR: %d,%d",g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);
#endif // DEBUG
          string strInfo = "Game "+ a_strLoadName +" loaded...";
          g_pCommunication->AddMessage(9, CREW_KIRK,strInfo.c_str());

       }
       else
       {
          g_pCommunication->AddMessage(9, CREW_KIRK,"There was an error loading your game !");
          g_vGameState.pop_back();
       }




    if (g_pMenu != NULL)
    {
        g_pMenu->ClearItems();
        g_pMenu->AddMenuItem("Return to game",ReturnToGame );
        g_pMenu->AddMenuItem("New Game",NewGame);
        g_pMenu->AddMenuItem("Load",SetupLoadScreen);
        g_pMenu->AddMenuItem("Save",SaveGame);
	    g_pMenu->AddMenuItem("Map Keys",DoKeymap);
        g_pMenu->AddMenuItem("Quit",Quit);
    }

    al_destroy_path(g_pSavePath);
    g_pSavePath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);
    al_append_path_component(g_pSavePath, "Save");
}


void SaveGame()
{
    time_t t = time(0);
    string strFileName = currentDateTime();
    strFileName+=".UNI";
    al_set_path_filename(g_pSavePath, strFileName.c_str());
    ofstream SaveFile(al_path_cstr(g_pSavePath,ALLEGRO_NATIVE_PATH_SEP),
                      ios::out | ios::binary);
    if (SaveFile)
    {
        g_pUniverse->StoreSector(g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY,g_pEngine);
        g_pEngine->Save(SaveFile);
        g_pUniverse->Save(SaveFile);
        SaveFile.close();
        g_pCommunication->AddMessage(9, CREW_KIRK,"Game saved !");
    }
    else
    {
        g_pCommunication->AddMessage(9, CREW_KIRK,"There was an error saving your game !");
    }

    al_destroy_path(g_pSavePath);
    g_pSavePath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);
    al_append_path_component(g_pSavePath, "Save");
    ReturnToGame();
}


void SetupLoadScreen()
{
    g_pSlotManager = new SlotManager(true,g_pSavePath,LoadGame);
    Log("Gamestate to GS_LOAD");
    g_vGameState.push_back(GS_LOAD);
}


void Quit()
{
    g_vGameState.clear();
}

void ReturnToGame()
{
     g_vGameState.push_back(GS_GAME);
     Log("GAME STATE TO GAME");
     al_hide_mouse_cursor(g_pDisplay);
     g_dLastTime = al_get_time();
}



void DoKeymap()
{
    if (g_vGameState.back() != GS_KEYMAP)
    {
        g_vGameState.push_back(GS_KEYMAP);
        Log("Gamestate to GS_KEYMAP");
    }
}




/** \brief
 *  Initializes all Classes using resources for multiple instances.
 *  Sprite classes for example
 * \return
 *  returns true on success , false on failure
 */
bool InitObjects()
{
	if (!TBullet::Init())
	{
       Log("Could not initialize Bullets");
       return false;
	}

	if (!TAnimation::Init())
    {
        Log("Could not initialize Animations");
        return false;
    }

	if (!TEnterprise::Init())
	{
	    Log("Could not initialize Enterprise");
	    return false;
	}

	if (!TRomulanBop::Init())
    {
        Log("Could not initialize Romulan BOP");
        return false;
    }
	if (!TKlingonBC::Init())
	{
	    Log("Could not initialize KlingonBC");
	    return false;
	}

	if (!TKlingonBOP::Init())
	{
	    Log("Could not initialize KlingonBOP");
	    return false;
	}

    if (!TSpaceObject::Init())
    {
        Log("Could not initialize Space Object");
        return false;
    }
	if (!TStarbase::Init())
	{
	    Log("Could not initialize Starbase");
	    return false;
	}

	if (!TFederation_Ship::Init())
    {
        Log("Could not initialize Federation ship");
        return false;
    }

	if (!Communication::Init())
    {
        Log("Could not initialize Communication");
        return false;
    }

    if (!TProbe::Init())
    {
        Log("Could not initialize Probes");
        return false;
    }
    Log("Objects initialized succesfully");
	return true;
}


/** \brief
 *  Some classes better be DeInited when using multiple resources.
 *  Allegro frees them anyway but Valgrind doesn't like it that way.
 */
void DeInitObjects()
{
    TStarbase::DeInit();
    TSpaceObject::DeInit();
    TAnimation::DeInit();
}





/** \brief
 *  Setup and initialize Allegro, the game Engine, Sprite Classes, Sound manager etc.
 * \return
 * returns true on success, false on failure
 */
bool Setup()
{

	if(al_init())
	{
        InitLog(); // Rude log system for debugging
        Log("Allegro Initialized");

        g_pSavePath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);
        al_append_path_component(g_pSavePath, "Save");
        g_pScreenshotPath = al_get_standard_path(ALLEGRO_RESOURCES_PATH);;
        al_append_path_component(g_pScreenshotPath, "screenshot");

#ifdef DEBUG
        strcpy(g_szSector,"");
#endif // DEBUG

		g_nScreenWidth  = 1024;
	    g_nScreenHeight =  768;
		//al_set_new_display_option(ALLEGRO_VSYNC, 1, ALLEGRO_SUGGEST);
#ifdef DEBUG
		al_set_new_display_flags(ALLEGRO_WINDOWED);
#else
		if ( al_show_native_message_box(g_pDisplay,
                                        "SETUP",
                                        "Screen Mode",
                                        "Go Fullscreen ?",
                                        NULL,
                                        ALLEGRO_MESSAGEBOX_YES_NO) == 1)
        {
            al_set_new_display_flags(ALLEGRO_FULLSCREEN);
        }
        else
        {
            al_set_new_display_flags(ALLEGRO_WINDOWED);
        }
#endif
		g_pDisplay = al_create_display(g_nScreenWidth, g_nScreenHeight);
		if (g_pDisplay != NULL)
		{
			Log("Display created");

			// Install allegro modules
			al_init_primitives_addon();
			al_init_image_addon();
			al_install_keyboard();
			al_install_mouse();
			Log("Add ons installed");

			// Init Sound manager
			SoundManager::Init_SoundManager(RESERVED_SAMPLES);
			Log("SoundManager Initializes");
            // Load the sounds
			SoundManager::LoadSounds();
            Log("Sounds loaded");


			// install event queue
			g_pEventQueue = al_create_event_queue();
			al_register_event_source(g_pEventQueue, al_get_keyboard_event_source());
			al_register_event_source(g_pEventQueue, al_get_mouse_event_source());
			al_register_event_source(g_pEventQueue, al_get_display_event_source(g_pDisplay));
			Log("Event sources registered");


			// Init Fontmanager
            FontManager::Init_FontManager();
			Log("Fontmanager Initializes");
			FontManager::LoadFonts();

           // Initialize the Sprite Object classes
			if (!InitObjects())
			{
				Log("Could not init Objects");
				return false;
			}


			// set quit to false
			g_blQuit = false;

			// create the menu
			g_pMenu = new TMenu();
			if (g_pMenu != NULL)
            {
               g_pMenu->AddMenuItem("New Game",NewGame);
               g_pMenu->AddMenuItem("Load",SetupLoadScreen);
               g_pMenu->AddMenuItem("Map Keys",DoKeymap);
			   g_pMenu->AddMenuItem("Quit",Quit);
            }
            else
            {
                Log("Could not create menu");
				return false;
            }





            // Create game engine
            g_pEngine = new TEngine(g_nScreenWidth,g_nScreenHeight);
            Log("Game engine created");

			// create Paralax system
			g_pParalax = new Paralax(g_nScreenWidth,g_nScreenHeight,300,5);
			Log("Paralax created");


            // create the keymapper
            g_pKeyMapper = new TKeyMapper(g_pEngine->m_clGREEN, g_pEngine->m_clYELLOW,FontManager::GetFont(FONT::TEXT));
            Log("Key Mapper created");

            // push gamestate GS_GAME on the gamestate stack
            g_vGameState.push_back(GS_MENU);
            Log("GAME STATE TO MENU");



			// Get rid of the mouse
			al_show_mouse_cursor(g_pDisplay);

			// Initialize DeltaTime, and create the timer
			g_dLastTime = al_get_time();
			g_pTimer=al_create_timer(1.000/g_nCyclesPerSecond);
            al_register_event_source(g_pEventQueue, al_get_timer_event_source(g_pTimer));


            LoadMapping();
			// start the timer
			al_start_timer(g_pTimer);
		}
		else
		{
			// Display creation failed
			return false;
		}
	}
	else
	{
		// Allegro could not be initialized
		return false;
	}

    // all went well
	return true;
}


/** \brief
 *  Gets the desktop resolution
 *  Currently not in use !
 *
 * \param int a_nAdapter
 * Number of the screen adapter in use
 * \param int *a_pWidth
 * OUT ScreenWidth
 * \param int *a_pHeighth
 * OUT ScreenHeight
 *
 */
void get_desktop_resolution(int a_nAdapter, int *a_pWidth, int * a_pHeighth)
{
  ALLEGRO_MONITOR_INFO info;
  al_get_monitor_info(a_nAdapter, &info);

  *a_pWidth = info.x2 - info.x1;
  *a_pHeighth = info.y2 - info.y1;
}




/** \brief
 *  Builds a new universe and sets Enterprise home position
 *  This creates a new game !
 */
void NewUniverse()
{
    Log("Delete old universe");
    delete g_pUniverse;
    Log("Create new universe Object");
    g_pUniverse = new Universe();
    Log("Generate random universe");
    g_pUniverse->Generate();
    Log("Set enterprise position");
    g_pEnterprise->m_nSectorPositionX = g_pUniverse->GetHomeX();
    g_pEnterprise->m_nSectorPositionY = g_pUniverse->GetHomeY();
    g_pEnterprise->SetDefaultNavigationOrigin();
    g_dLastTime = al_get_time();
}


/** \brief
 *  Clears the Engine, keeping the Enterprise and
 *  builds a new Playingfield from a sector in the universe.
 * \param int a_nX
 * X position of new sector
 * \param int a_nY
 * Y position of new sector
 */

void NewSector(int a_nX, int a_nY)
{
    g_pEngine->Clear(true);
    g_pUniverse->BuildSector(a_nX,a_nY,g_pEngine);
    g_pEnterprise->SetDefaultNavigationOrigin();
}




/** \brief
 *  Shutdown and free, Engine, Paralax sprite Objects, allegro etc.
 */
void ShutDown()
{
    delete g_pEngine;
    delete g_pParalax;

    DeInitObjects();
    SoundManager::DeInit_SoundManager();
    al_destroy_path(g_pSavePath);
    al_destroy_path(g_pScreenshotPath);
	al_destroy_event_queue(g_pEventQueue);
	al_destroy_timer(g_pTimer);
	al_destroy_display(g_pDisplay);
	al_uninstall_system();
}



/** \brief
 *  Render the correct screen according to the gamestate
 */

void Render()
{
    al_clear_to_color(g_pEngine->m_clBLACK);
    if (!g_vGameState.empty())
    {
        switch(g_vGameState.back())
        {
            case GS_INTRO:

            break;

            case GS_MENU:
                 DoMenu();
            break;

            case GS_LOAD:
                DoLoadScreen();
            break;

            case GS_SAVE:
                DoSaveScreen();
            break;

            case GS_KEYMAP:
                g_pKeyMapper->DrawMapping();
            break;

            case GS_GAME:
                 DoGame();
            break;
        }
        al_flip_display();
	}
	else
    {
        // exit game
    }
}

/** \brief
 *  Process a game cycle
 *  Calls the AI function of the game objects
 *  Invokes the next Communication message.
 */
void GameCycle()
{
	g_pEngine->Do_ai();
	if (g_pEnterprise!=NULL)
    {
       g_pCommunication->Next();
    }

}


/** \brief
 *  Checks if the Enterprise crossed the borders of a sector
 *  If so it stores the old sector in the universe
 *  and calls void NewSector(int a_nX, int a_nY)
 */
void CheckSectorBounds()
{
    if ((g_pEnterprise!=NULL)&&(!g_pEnterprise->m_blDestroyed))
    {
        bool blNewSector = false;
        int XX = 0;
        int YY = 0;
        double secsize = double(SECTORSIZE);

        // check sector boundaries
        if (g_pEnterprise->GetX() < 0)
        {
            XX = -1;
            g_pEnterprise->SetX(secsize + g_pEnterprise->GetX());
            blNewSector = true;
        }
        else if (g_pEnterprise->GetX() > secsize)
        {
            XX = 1;
            g_pEnterprise->SetX(g_pEnterprise->GetX()-secsize );
            blNewSector = true;
        }
        if (g_pEnterprise->GetY() < 0)
        {
            YY= -1;
            g_pEnterprise->SetY(secsize + g_pEnterprise->GetY());
            blNewSector = true;
        }
        else if (g_pEnterprise->GetY() > secsize)
        {
            YY = 1;
            g_pEnterprise->SetY(g_pEnterprise->GetY()-secsize);
            blNewSector = true;
        }

        // in a new sector, store the old one and build a new...
        if (blNewSector)
        {
            g_pUniverse->StoreSector(g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY,g_pEngine);
            g_pEnterprise->m_nSectorPositionX += XX;
            g_pEnterprise->m_nSectorPositionY += YY;

            if (g_pEnterprise->m_nSectorPositionX==UNIVERSE_SIZE)
            {
                g_pEnterprise->m_nSectorPositionX =0;
            }
            else if (g_pEnterprise->m_nSectorPositionX<0)
            {
                g_pEnterprise->m_nSectorPositionX=UNIVERSE_SIZE-1;
            }

            if (g_pEnterprise->m_nSectorPositionY==UNIVERSE_SIZE)
            {
                g_pEnterprise->m_nSectorPositionY =0;
            }
            else if (g_pEnterprise->m_nSectorPositionY<0)
            {
                g_pEnterprise->m_nSectorPositionY =UNIVERSE_SIZE-1;
            }
            NewSector(g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);

#ifdef DEBUG
            sprintf(g_szSector,"SECTOR: %d,%d",g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);
#endif // DEBUG

            g_pEngine->SetOrigin(g_pEnterprise->GetX(),g_pEnterprise->GetY());
            g_pEnterprise->LooseTarget();
        }
    }
}


///////////////////////////////////////////////////////////////////////////////////////
//     EVENT HANDLERS
///////////////////////////////////////////////////////////////////////////////////////


/** \brief
 *  KeyDown event handler
 * \param int a_nKey
 * The key code
 */
void KeyDown(int a_nKey)
{
    if (g_vGameState.back() == GS_KEYMAP)
    {
        if (!g_pKeyMapper->MapKey(a_nKey))
        {
            SaveMapping();
        }
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_0))
    {
        g_pEngine->m_blKeys[WARP_0] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_1))
    {
        g_pEngine->m_blKeys[WARP_1] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_2))
    {
        g_pEngine->m_blKeys[WARP_2] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_3))
    {
        g_pEngine->m_blKeys[WARP_3] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_4))
    {
        g_pEngine->m_blKeys[WARP_4] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,WARP_5))
    {
        g_pEngine->m_blKeys[WARP_5] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,UP))
	{
	    g_pEngine->m_blKeys[UP] = true;

	    if ((g_pEnterprise!=NULL)&&(g_nScreenMode == MODE_NAVIGATION))
        {
            g_pEnterprise->DecNavY();
        }
	}
	else if (g_pKeyMapper->IsKey(a_nKey,DOWN))
	{
	    g_pEngine->m_blKeys[DOWN] = true;

	    if ((g_pEnterprise!=NULL)&&(g_nScreenMode == MODE_NAVIGATION))
        {
            g_pEnterprise->IncNavY();
        }
	}
	else if (g_pKeyMapper->IsKey(a_nKey,LEFT))
	{
	    g_pEngine->m_blKeys[LEFT] = true;

	    if ((g_pEnterprise!=NULL)&&(g_nScreenMode == MODE_NAVIGATION))
        {
            g_pEnterprise->DecNavX();
        }
	}
	else if (g_pKeyMapper->IsKey(a_nKey,RIGHT))
	{
	    g_pEngine->m_blKeys[RIGHT] = true;

	    if ((g_pEnterprise!=NULL)&&(g_nScreenMode == MODE_NAVIGATION))
        {
            g_pEnterprise->IncNavX();
        }
	}
    else if (g_pKeyMapper->IsKey(a_nKey,PHASER))
	{
	    g_pEngine->m_blKeys[PHASER] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,TARGET_UP))
	{
	    if ((g_pEnterprise!=NULL)&&(!g_pEnterprise->m_blDestroyed))
	    {
	        g_pEnterprise->LockUp();
	    }
	}
	else if (g_pKeyMapper->IsKey(a_nKey,TARGET_DOWN))
	{
	    if ((g_pEnterprise!=NULL)&&(!g_pEnterprise->m_blDestroyed))
	    {
            g_pEnterprise->LockDown();
	    }
	}
	else if (g_pKeyMapper->IsKey(a_nKey,PHOTON))
	{
	    g_pEngine->m_blKeys[PHOTON] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,PHOTON_BACK))
	{
	    g_pEngine->m_blKeys[PHOTON_BACK] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,PROBE))
    {
        g_pEngine->m_blKeys[PROBE] = true;
    }
    else if (g_pKeyMapper->IsKey(a_nKey,DOCK))
	{
	    g_pEngine->m_blKeys[DOCK] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,RELEASE))
	{
	    g_pEngine->m_blKeys[RELEASE] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,SEEK_TARGET))
	{
	    g_pEngine->m_blKeys[SEEK_TARGET] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,SEEK_FRIEND))
	{
	    g_pEngine->m_blKeys[SEEK_FRIEND] = true;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,SCREEN_MAIN))
	{
	    al_hide_mouse_cursor(g_pDisplay);
		g_nScreenMode = MODE_MAINSCREEN;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,SCREEN_ENGINEERING))
	{
	    al_show_mouse_cursor(g_pDisplay);
		g_nScreenMode = MODE_ENGINEERING;
	}
	else if (g_pKeyMapper->IsKey(a_nKey,SCREEN_NAVIGATION))
	{
	    al_show_mouse_cursor(g_pDisplay);
		g_nScreenMode = MODE_NAVIGATION;
	}
	else if ((g_pKeyMapper->IsKey(a_nKey,NAVIGATION_VIEW_TOPO))&&(g_pEnterprise!= NULL)&& (! g_pEnterprise->m_blDestroyed))
	{
        g_pEnterprise->SetNavigationMode(NAV_TOPO);
	}
	else if ((g_pKeyMapper->IsKey(a_nKey,NAVIGATION_VIEW_TACTICAL))&&(g_pEnterprise!= NULL)&& (! g_pEnterprise->m_blDestroyed))
	{
        g_pEnterprise->SetNavigationMode(NAV_TACTICAL);
	}
	else if ((g_pKeyMapper->IsKey(a_nKey,NAVIGATION_VIEW_OCCUPATION))&&(g_pEnterprise!= NULL)&& (! g_pEnterprise->m_blDestroyed))
	{
        g_pEnterprise->SetNavigationMode(NAV_OCCUPATION);
	}
    else
    {
        // handle fixed keys
        switch(a_nKey)
        {

#ifdef DEBUG
            case ALLEGRO_KEY_SLASH:
            g_blGodMode = ! g_blGodMode;
            break;
#endif // DEBUG


            case ALLEGRO_KEY_F10:
                {
                 ofstream SaveFile(g_szFileName,ios::out | ios::binary);
                 if (SaveFile)
                 {
                    g_pUniverse->StoreSector(g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY,g_pEngine);
                    g_pEngine->Save(SaveFile);
                    g_pUniverse->Save(SaveFile);
                    SaveFile.close();
                 }
                 else
                 {
                    // handle error
                 }
                }
            break;

            case ALLEGRO_KEY_F11:
                {
                ifstream LoadFile(g_szFileName,ios::in | ios::binary);
                if (LoadFile)
                {
                    g_pEngine->Load(LoadFile);
                    g_pUniverse->Load(LoadFile);
                    LoadFile.close();
                    g_pEnterprise->SetDefaultNavigationOrigin();
#ifdef DEBUG
                    sprintf(g_szSector,"SECTOR: %d,%d",g_pEnterprise->m_nSectorPositionX,g_pEnterprise->m_nSectorPositionY);
#endif // DEBUG

                }
                else
                {
                    // handle error
                }
                }
            break;

            case ALLEGRO_KEY_F12:
                screenshot(al_path_cstr(g_pScreenshotPath,ALLEGRO_NATIVE_PATH_SEP),NULL);
            break;
        }
    }
}



/** \brief
 *  KeyUp event handler
 * \param int a_nKey
 * The key code
 */
void KeyUp(int a_nKey)
{
        if (g_pKeyMapper->IsKey(a_nKey,WARP_0))
        {
            g_pEngine->m_blKeys[WARP_0] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,WARP_1))
        {
            g_pEngine->m_blKeys[WARP_1] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,WARP_2))
        {
            g_pEngine->m_blKeys[WARP_2] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,WARP_3))
        {
            g_pEngine->m_blKeys[WARP_3] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,WARP_4))
        {
            g_pEngine->m_blKeys[WARP_4] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,WARP_5))
        {
            g_pEngine->m_blKeys[WARP_5] = false;
        }
        if (g_pKeyMapper->IsKey(a_nKey,UP))
        {
            g_pEngine->m_blKeys[UP] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,DOWN))
        {
            g_pEngine->m_blKeys[DOWN] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,LEFT))
        {
            g_pEngine->m_blKeys[LEFT] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,RIGHT))
        {
            g_pEngine->m_blKeys[RIGHT] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,PHASER))
        {
            g_pEngine->m_blKeys[PHASER] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,PHOTON))
        {
            g_pEngine->m_blKeys[PHOTON] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,PHOTON_BACK))
        {
            g_pEngine->m_blKeys[PHOTON_BACK] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,PROBE))
        {
            g_pEngine->m_blKeys[PROBE] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,DOCK))
        {
            g_pEngine->m_blKeys[DOCK] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,RELEASE))
        {
            g_pEngine->m_blKeys[RELEASE] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,SEEK_TARGET))
        {
            g_pEngine->m_blKeys[SEEK_TARGET] = false;
        }
        else if (g_pKeyMapper->IsKey(a_nKey,SEEK_FRIEND))
        {
            g_pEngine->m_blKeys[SEEK_FRIEND] = false;
        }
        else
        {
            // handle fixed keys
            switch(a_nKey)
            {
                case ALLEGRO_KEY_ESCAPE:
                g_vGameState.pop_back();
                if (g_vGameState.back()==GS_MENU)
                {
                    al_show_mouse_cursor(g_pDisplay);
                }
                break;
            }
        }
}

/** \brief
 *  mouse event handler
 *  \param ALLEGRO_MOUSE_EVENT * mouse_event
 *
 */
void Mouse(ALLEGRO_MOUSE_EVENT * mouse_event)
{
    switch(g_vGameState.back())
    {

            case GS_MENU:
            if (g_pMenu!=NULL)
            {
                g_pMenu->ProcessMouseEvent(mouse_event);
            }
            break;

            case GS_LOAD:
            if (g_pSlotManager!=NULL)
            {
                g_pSlotManager->ProcessMouseEvent(mouse_event);
            }

            break;

            case GS_SAVE:
            if (g_pSlotManager!=NULL)
            {
                g_pSlotManager->ProcessMouseEvent(mouse_event);
            }
            break;

    }

}

/** \brief
 *  MouseButtonDown event handler
 *  \param ALLEGRO_MOUSE_EVENT * mouse_event
 *
 */
void MouseButtonDown(ALLEGRO_MOUSE_EVENT * mouse_event)
{
    switch(g_vGameState.back())
    {

            case GS_MENU:
            if (g_pMenu!=NULL)
            {
                g_pMenu->ProcessMouseEvent(mouse_event);
            }
            break;

            case GS_LOAD:
            if (g_pSlotManager!=NULL)
            {
                g_pSlotManager->ProcessMouseEvent(mouse_event);
            }
            break;

            case GS_SAVE:
            if (g_pSlotManager!=NULL)
            {
                g_pSlotManager->ProcessMouseEvent(mouse_event);
            }
            break;

    }
}

/** \brief
 *  MouseButtonUp event handler
 *  \param ALLEGRO_MOUSE_EVENT * mouse_event
 *
 */
void MouseButtonUp(ALLEGRO_MOUSE_EVENT * mouse_event)
{

}

/** \brief
 *  Timer event handler
 *  \param ALLEGRO_TIMER_EVENT * timer_event
 *
 */
void Timer(ALLEGRO_TIMER_EVENT * timer_event)
{
	if (!g_vGameState.empty())
    {
        switch(g_vGameState.back())
        {
            case GS_INTRO:
            DoIntro();
            break;

            case GS_MENU:

            break;

            case GS_GAME:
            GameCycle();
            break;
        }
    }
	else
    {
        g_blQuit = true;
    }

}




/** \brief
 *  Handle the Introduction screen,
 *  Not implemented yet
 */
void DoIntro()
{

	//	g_blGameRunning =true;
	//	g_nGameState = GS_GAME;
	//	g_nScreenMode = MODE_MAINSCREEN;

}

/** \brief
 *  Handle the Menu,
 *  Not implemented yet
 *    [New Game]
 *    [Load]
 *    [Save]
 *    [Help]
 */
void DoMenu()
{
    if (g_pMenu!=NULL)
    {
        g_pMenu->Draw(FontManager::GetFont(FONT::MENU));
    }
}


void DoLoadScreen()
{
    al_draw_text(FontManager::GetFont(FONT::TEXT),g_pEngine->m_clBLUE, 10, 30, 0,"Load game...");
    if (g_pSlotManager!=NULL)
    {
        g_pSlotManager->Draw(FontManager::GetFont(FONT::TEXT));
    }


}

void DoSaveScreen()
{
    al_draw_text(FontManager::GetFont(FONT::TEXT),g_pEngine->m_clBLUE, 10, 30, 0,"Save game...");
    if (g_pSlotManager!=NULL)
    {
        g_pSlotManager->Draw(FontManager::GetFont(FONT::TEXT));
    }
}



/** \brief
 *  Move everything in the engine.
 *  Free all destroyed objects.
 *  Check the sector bounds
 *  Move the paralax
 *  Draw everything
 */
void DoGame()
{
     double dDeltaTime = al_get_time() - g_dLastTime;
     g_dLastTime = al_get_time();
     g_pEngine->Move(dDeltaTime);
     if ((g_pEnterprise!=NULL) && (g_pEnterprise->m_blDestroyed))
     {
        g_pEnterprise=NULL;
     }
     g_pEngine->Kill();

     if(g_pEnterprise!=NULL)
     {
        CheckSectorBounds();
        g_pEngine->Folow(g_pEnterprise);
     }
     else
     {
        g_nScreenMode = MODE_MAINSCREEN;
     }
     g_pParalax->Move(g_pEngine->GetDx(), g_pEngine->GetDy());

     switch (g_nScreenMode)
     {
        case MODE_MAINSCREEN:
        if (g_pEnterprise!=NULL)
        {
           g_pParalax->Draw(g_pDisplay,g_pEnterprise->GetWarp(),g_pEnterprise->GetAngle());
        }
        else
        {
           g_pParalax->Draw(g_pDisplay);
        }


        g_pEngine->Draw();

        if (g_pEnterprise!=NULL)
        {
            g_pEngine->DrawHud(g_pEnterprise);
            g_pEngine->Draw_sensor(60,g_nScreenHeight-60,g_pEnterprise);
            g_pEnterprise->DrawTargetInfo(0,g_nScreenWidth-140, g_nScreenHeight-120);
#ifdef DEBUG
            al_draw_text(FontManager::GetFont(FONT::DBG),al_map_rgb(255,255,255), 10, 10, 0,g_szSector);
            if (g_blGodMode)
            {
                al_draw_text(FontManager::GetFont(FONT::DBG),al_map_rgb(255,255,255), 10, 30, 0,"GOD MODE !");
            }
#endif // DEBUG
         }
         break;

         case MODE_ENGINEERING:
              g_pEnterprise->draw_engineering();
         break;

         case MODE_NAVIGATION:
              g_pEnterprise->Draw_Navigation(g_pUniverse);
         break;

   }

   if (g_pEnterprise!=NULL)
   {
       g_pCommunication->Draw();
   }
   else
   {
       //g_pCommunication->
   }
}


/** \brief
 *  Program entry
 * \param int argc
 * Number of program arguments ( not used yet)
 * \param char **argv
 *  Argument list ( not used yet)
 */


int main(int argc, char **argv)
{
	try
	{

	if (Setup())
	{
		// Event loop
		while (!g_blQuit)
		{
			ALLEGRO_EVENT ev;
			if (al_get_next_event(g_pEventQueue, &ev))
			{
                switch (ev.type)
				{

				case ALLEGRO_EVENT_KEY_DOWN:
					 KeyDown(ev.keyboard.keycode);
     				break;

				case ALLEGRO_EVENT_KEY_UP:
					 KeyUp(ev.keyboard.keycode);
                     break;

				case ALLEGRO_EVENT_DISPLAY_CLOSE:
					g_blQuit = true;
					break;

				case ALLEGRO_EVENT_MOUSE_AXES:
					 Mouse(&ev.mouse);
					 break;

                case ALLEGRO_EVENT_MOUSE_BUTTON_DOWN:
					 MouseButtonDown(&ev.mouse);
					 break;

                case ALLEGRO_EVENT_MOUSE_BUTTON_UP:
					 MouseButtonUp(&ev.mouse);
					 break;

				case ALLEGRO_EVENT_TIMER:
					 Timer(&ev.timer);
					 break;
                }

			}
			else
			{
				Render();
			}
    	}
	}
	else
	{
		// show error message
		al_show_native_message_box(
        g_pDisplay,
        "ERROR",
        "Setup Failed",
        "Setup failure",
        NULL,
        ALLEGRO_MESSAGEBOX_ERROR);

	}

	}

	catch (A5Exception & except)
	{
        // show error message
        al_show_native_message_box(
        g_pDisplay,
        "ERROR",
        "Game engine exception",
        except.what(),
        NULL,
        ALLEGRO_MESSAGEBOX_ERROR);
    }

    catch (exception & e)
    {
        // show error message
        al_show_native_message_box(
        g_pDisplay,
        "ERROR",
        "Standard exception",
        e.what(),
        NULL,
        ALLEGRO_MESSAGEBOX_ERROR);
    }

	ShutDown();
	return 0;
}

const string currentDateTime()
{
    time_t     now = time(0);
    struct tm  tstruct;
    char       buf[80];
    tstruct = *localtime(&now);
    strftime(buf, sizeof(buf), "%A-%Y-%m-%d-%H-%M-%S", &tstruct);
    return buf;
}

