#include <allegro5/allegro.h>

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "m_config.h"
#include "m_globvars.h"

#define G_WORLD
#include "g_header.h"
#undef G_WORLD

#include "e_editor.h"
#include "g_motion.h"
#include "g_proc.h"
#include "g_packet.h"
#include "g_cloud.h"
#include "g_world.h"
#include "g_proc_new.h"
#include "g_game.h"
#include "i_view.h"
#include "i_input.h"
#include "i_console.h"

#include "g_misc.h"

#include "c_header.h"
#include "t_template.h"

#include "m_maths.h"
#include "i_disp_in.h"
#include "i_background.h"
#include "s_menu.h"

#include "c_prepr.h"

void place_proc_randomly(struct proc_struct* pr);
void init_world_background(void);
void init_hex_block_nodes(int x, int y);
void clear_background_square(int x1, int y1, int x2, int y2);
void clear_background_circle(int centre_block_x, int centre_block_y, int clear_size);
void add_data_well(int block_x, int block_y, int w_init_index);

extern struct world_init_struct w_init;
extern struct template_struct templ [PLAYERS] [TEMPLATES_PER_PLAYER];
extern struct game_struct game;
extern struct view_struct view;
extern struct editorstruct editor;

int load_source_file_into_template(char* filename, int player_index, int player_template);

void init_vision_block(int x, int y);
static void prepare_world_for_game(void);

// This function starts the game, deriving world parameters from a world_init struct.
void start_world(void)
{



 open_template(0, 0);
// open_template_in_editor(&templ[0][0]);

 new_world_from_world_init();

 prepare_world_for_game();

 start_game();

 initialise_view(settings.option [OPTION_WINDOW_W], settings.option [OPTION_WINDOW_H]);

 view.camera_x = w.player[game.user_player_index].spawn_position.x;
 view.camera_y = w.player[game.user_player_index].spawn_position.y;

 initialise_control();

 init_consoles();
 setup_consoles();

// game.phase = GAME_PHASE_PREGAME;
// game.fast_forward = FAST_FORWARD_OFF;
// game.fast_forward_type = FAST_FORWARD_TYPE_SMOOTH;


}



// This function takes a world_init_struct and sets up a worldstruct based on the world_init_struct.
// Currently, any memory allocation failure causes a fatal error. Could change this (but it would require deallocation of a partially allocated world, which isn't implemented yet)
void new_world_from_world_init(void)
{


// char player_name [PLAYERS] [PLAYER_NAME_LENGTH];



 int i, j, p;

 w.players = w_init.players;
 w.command_mode = w_init.command_mode;

 switch(w_init.core_setting)
 {
	 case 0:
	 	w.cores_per_player = 16;
	 	w.procs_per_player = 64;
	 	break;
	 case 1:
	 	w.cores_per_player = 32;
	 	w.procs_per_player = 128;
	 	break;
	 case 2:
	 	w.cores_per_player = 64;
	 	w.procs_per_player = 256;
	 	break;
	 case 3:
	 	w.cores_per_player = 128;
	 	w.procs_per_player = 512;
	 	break;
	 default: // should never happen
			fpr("\nError: g_world.c: new_world_from_world_init(): w_init.core_setting = %i", w_init.core_setting);
			error_call();
			break;
 }


 w.max_procs = w.procs_per_player * w.players;
 w.max_cores = w.cores_per_player * w.players;
// w.gen_limit = w_init.gen_limit;
 w.max_packets = w.players * 200; // not sure about this
 w.max_clouds = CLOUDS; // see new cloud code in g_cloud.c if max_clouds is ever set to anything other than CLOUDS

 w.blocks.x = w_init.map_size_blocks;
 w.blocks.y = w_init.map_size_blocks;

 w.fixed_size.x = block_to_fixed(w.blocks.x);
 w.fixed_size.y = block_to_fixed(w.blocks.y);

 w.w_pixels = al_fixtoi(w.fixed_size.x);
 w.h_pixels = al_fixtoi(w.fixed_size.y);

// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below
 w.block = calloc(w.blocks.x, sizeof(struct block_struct*));
 if (w.block == NULL)
 {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.block");
      error_call();
 }
 for (i = 0; i < w.blocks.x; i ++)
 {
   w.block [i] = calloc(w.blocks.y, sizeof(struct block_struct));
   if (w.block [i] == NULL)
   {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.block");
      error_call();
   }
 }

 for (i = 0; i < w.blocks.x; i ++)
 {
  for (j = 0; j < w.blocks.y; j ++)
  {
   w.block [i][j].block_type = BLOCK_NORMAL;
  }
 }

 for (i = 0; i < w.blocks.x; i ++)
 {
  w.block [i] [0].block_type = BLOCK_SOLID;
  w.block [i] [1].block_type = BLOCK_SOLID;
  w.block [i] [2].block_type = BLOCK_EDGE_UP;
  w.block [i] [w.blocks.y - 1].block_type = BLOCK_SOLID;
  w.block [i] [w.blocks.y - 2].block_type = BLOCK_SOLID;
  w.block [i] [w.blocks.y - 3].block_type = BLOCK_EDGE_DOWN;
 }
 for (i = 0; i < w.blocks.y; i ++)
 {
  w.block [0] [i].block_type = BLOCK_SOLID;
  w.block [1] [i].block_type = BLOCK_SOLID;
  if (i>1&&i<(w.blocks.y-2))
			w.block [2] [i].block_type = BLOCK_EDGE_LEFT;
  w.block [w.blocks.x - 1] [i].block_type = BLOCK_SOLID;
  w.block [w.blocks.x - 2] [i].block_type = BLOCK_SOLID;
  if (i>1&&i<(w.blocks.y-2))
   w.block [w.blocks.x - 3] [i].block_type = BLOCK_EDGE_RIGHT;

 }

 w.block [2] [2].block_type = BLOCK_EDGE_UP_LEFT;
 w.block [2] [w.blocks.y - 3].block_type = BLOCK_EDGE_DOWN_LEFT;
 w.block [w.blocks.x - 3] [2].block_type = BLOCK_EDGE_UP_RIGHT;
 w.block [w.blocks.x - 3] [w.blocks.y - 3].block_type = BLOCK_EDGE_DOWN_RIGHT;

// initialise vision_area
 w.vision_areas_x = w.blocks.x;
 w.vision_areas_y = w.blocks.y;
// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below
 for (p = 0; p < w.players; p ++)
	{
  w.vision_area [p] = calloc(w.vision_areas_x, sizeof(struct vision_area_struct*));
  if (w.vision_area [p] == NULL)
  {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.vision_area");
      error_call();
  }
  for (i = 0; i < w.vision_areas_x; i ++)
  {
   w.vision_area [p] [i] = calloc(w.vision_areas_y, sizeof(struct vision_area_struct));
   if (w.vision_area [p] [i] == NULL)
   {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.vision_area");
      error_call();
   }
  }
	}

 w.core = calloc(w.max_cores, sizeof(struct core_struct));
 if (w.core == NULL)
 {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.core");
      error_call();
 }
// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below


// allocate memory for the proc array.
// use calloc because this isn't remotely time-critical.
 w.proc = calloc(w.max_procs, sizeof(struct proc_struct));
 if (w.proc == NULL)
 {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.proc");
      error_call();
 }
// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below

// now allocate the packet array:
 w.packet = calloc(w.max_packets, sizeof(struct packet_struct));
 if (w.packet == NULL)
 {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.packet");
      error_call();
 }
// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below

// now allocate the cloud array:
 w.cloud = calloc(w.max_clouds, sizeof(struct cloud_struct));
 if (w.cloud == NULL)
 {
      fprintf(stdout, "g_world.c: Out of memory in allocating w.cloud");
      error_call();
 }
// when adding any dynamic memory allocation to this function, remember to free the memory in deallocate_world() below

// setup players
 for (i = 0; i < w.players; i ++)
 {
  w.player[i].active = 1;
  w.player[i].core_index_start = i * w.cores_per_player;
  w.player[i].core_index_end = (i+1) * w.cores_per_player;
  w.player[i].proc_index_start = i * w.procs_per_player;
  w.player[i].proc_index_end = (i+1) * w.procs_per_player;
  strcpy(w.player[i].name, w_init.player_name [i]); // if loading from disk, the name copied here will be updated later

  w.player[i].processes = 0;
  w.player[i].components_current = 0;
  w.player[i].components_reserved = 0;

  w.player[i].output_console = 0;
//  w.player[i].default_print_colour = PRINT_COL_LGREY;
  w.player[i].error_console = 0;

  w.player[i].data = w_init.player_starting_data [i]; // should usually be 300

  w.player[i].random_seed = w_init.game_seed + (w_init.core_setting + w_init.data_wells + w_init.players) * (i + 7); // doesn't need to be particularly random

 }

// w.system_output_console = 0;
// w.system_error_console = 0;

 initialise_world();

 w.allocated = 1;

}




// This function clears an existing world so it can be re-used (or initialises one just created in new_world_from_world_init()).
// The world's basic parameters (e.g. size, number of procs etc) should have been established, so new_world_from_world_init() must have been called first.
void initialise_world(void)
{

 int p, c, i, j;
 struct core_struct* core;
 struct proc_struct* proc;
 struct block_struct* bl;

 w.world_time = BASE_WORLD_TIME; // currently 255 - allows subtraction of small amounts without wrapping the unsigned value
 w.debug_mode_general = 0;

// world_time starts after 0 so that things like deallocation counters can be subtracted from it without running into unsigned int problems

 for (c = 0; c < w.max_cores; c ++)
 {
  core = &w.core [c];
  core->exists = 0;
  core->destroyed_timestamp = 0;
  core->index = c;
 }


 for (p = 0; p < w.max_procs; p ++)
 {
  proc = &w.proc [p];
  proc->exists = 0;
  proc->reserved = 0;
  proc->destroyed_timestamp = 0;
  proc->index = p;
 }

 for (i = 0; i < w.blocks.x; i ++)
 {
  for (j = 0; j < w.blocks.y; j ++)
  {
    bl = &w.block [i] [j];
    bl->tag = 0;
    bl->blocklist_down = NULL;
  }
 }

 w.blocktag = 1; // should probably be 1 more than the value that all of the blocks in the world are set to.

 init_packets();
 init_clouds();

 w.data_wells = 0;

 init_world_background();

}


void add_data_well(int block_x, int block_y, int w_init_index) // w_init_index is index in w_init data well arrays
{

#ifdef SANITY_CHECK
 if (w.data_wells >= DATA_WELLS)
	{
		fpr("\nError: g_world.c: add_data_well(): too many data wells.");
		error_call();
	}
#endif

	w.data_well[w.data_wells].active = 1;
	w.data_well[w.data_wells].block_position.x = block_x;
	w.data_well[w.data_wells].block_position.y = block_y;
	w.data_well[w.data_wells].position.x = al_itofix((block_x * BLOCK_SIZE_PIXELS) + BLOCK_SIZE_PIXELS / 2);
	w.data_well[w.data_wells].position.y = al_itofix((block_y * BLOCK_SIZE_PIXELS) + BLOCK_SIZE_PIXELS / 2);
	w.data_well[w.data_wells].data_max = 192;
	w.data_well[w.data_wells].data = w.data_well[w.data_wells].data_max;
	w.data_well[w.data_wells].last_harvested = 0;
	w.data_well[w.data_wells].last_transferred = 0;

	w.data_well[w.data_wells].reserve_data [0] = w_init.reserve_data [w.data_wells] [0];//2000;//1000 + grand(1000);
	w.data_well[w.data_wells].reserve_data [1] = w_init.reserve_data [w.data_wells] [1];
	w.data_well[w.data_wells].reserve_squares = w_init.reserve_squares [w.data_wells];
	w.data_well[w.data_wells].spin_rate = w_init.data_well_spin_rate [w.data_wells];

// clear_background_square(block_x - 4, block_y - 4, block_x + 4, block_y + 4);
 clear_background_circle(block_x, block_y, 3);

 w.block[block_x][block_y].backblock_type = BACKBLOCK_DATA_WELL;
 w.block[block_x][block_y].backblock_value = w.data_wells;

 w.data_wells ++;

}


void clear_background_square(int x1, int y1, int x2, int y2)
{

	if (x1 < 3)
		x1 = 3;
 if (y1 < 3)
		y1 = 3;
	if (x2 > w.blocks.x - 4)
		x2 = w.blocks.x - 4;
	if (y2 > w.blocks.y - 4)
		y2 = w.blocks.y - 4;

	int i, j;

	for (i = x1; i < x2 + 1; i ++)
	{
		for (j = y1; j < y2 + 1; j ++)
		{
			w.block[i][j].backblock_type = BACKBLOCK_EMPTY;
		}
	}

}


void clear_background_circle(int centre_block_x, int centre_block_y, int clear_size)
{
 int x1 = centre_block_x - clear_size;
 int y1 = centre_block_y - clear_size;
 int x2 = centre_block_x + clear_size + 1;
 int y2 = centre_block_y + clear_size + 1;

	int i, j, k;
	int size_pixels = clear_size * BLOCK_SIZE_PIXELS;
	al_fixed size_fixed = al_itofix(size_pixels);
	al_fixed dist;
	al_fixed centre_fixed_x = block_to_fixed(centre_block_x) + BLOCK_SIZE_FIXED / 2;
	al_fixed centre_fixed_y = block_to_fixed(centre_block_y) + BLOCK_SIZE_FIXED / 2;

	for (i = x1; i < x2; i ++)
	{
		for (j = y1; j < y2; j ++)
		{
			if (i < 3
				|| i >= w.blocks.x - 4
				|| j < 3
				|| j >= w.blocks.y - 4)
				continue;
			dist = distance(al_itofix(centre_block_y - j), al_itofix(centre_block_x - i));
			if (dist < al_itofix(clear_size - 1))
			{
			 w.block[i][j].backblock_type = BACKBLOCK_EMPTY;
			 continue;
			}
			if (dist > al_itofix(clear_size + 1))
			{
//			 w.block[i][j].backblock_type = BACKBLOCK_EMPTY; - do nothing
			 continue;
			}
			if (w.block[i][j].backblock_type == BACKBLOCK_BASIC_HEX)
			{
				int nodes_cleared = 0;

			 for (k = 0; k < BLOCK_NODES; k ++)
			 {
				 dist = distance(block_to_fixed(j) + al_itofix(w.block[i][j].node_y [k]) - centre_fixed_y, block_to_fixed(i) + al_itofix(w.block[i][j].node_x [k]) - centre_fixed_x);
				 if (dist < size_fixed)
					{
						w.block[i][j].node_exists [k] = 0;
						nodes_cleared++;
					}
					 else
				   if (dist < size_fixed + al_itofix(100))
					  {
						  w.block[i][j].node_size [k] *= 160 + al_fixtoi(dist - size_fixed);
						  w.block[i][j].node_size [k] /= 260;
					  }
			 }
			 if (nodes_cleared == BLOCK_NODES)
					w.block[i][j].backblock_type = BACKBLOCK_EMPTY;
			}

		}
	}

}



// This function frees the memory used by the worldstruct
// It doesn't otherwise reinitialise the worldstruct
// Call it only when certain that memory has been allocated for the worldstruct (as otherwise it will try to free already free memory)
void deallocate_world(void)
{

#ifdef SANITY_CHECK
 if (w.allocated == 0)
 {
  fprintf(stdout, "\nError: g_world.c:deallocate_world() called when w.allocated already 0.");
  error_call();
 }
#endif

 int i, p;

// first, free each column of blocks:
 for (i = 0; i < w.blocks.x; i ++)
 {
  free(w.block [i]);
 }

// now free the row of pointers to the columns:
 free(w.block);

// now do the same for vision areas:
 for (p = 0; p < w.players; p ++)
	{
  for (i = 0; i < w.vision_areas_x; i ++)
  {
   free(w.vision_area [p] [i]);
  }
  free(w.vision_area [p]);
 }

// free the rest of the arrays:
 free(w.core);
 free(w.proc);
 free(w.packet);
 free(w.cloud);

 w.allocated = 0;

}

// Once the world data structure has been fully initialised, call this function
//  to prepare the game (by adding things like data wells etc).
static void prepare_world_for_game(void)
{

 int i;

 for (i = 0; i < w.players; i ++)
	{
		w.player[i].spawn_position.x = block_to_fixed(w_init.spawn_position[i].x) + BLOCK_SIZE_FIXED / 2;
		w.player[i].spawn_position.y = block_to_fixed(w_init.spawn_position[i].y) + BLOCK_SIZE_FIXED / 2;
		w.player[i].spawn_angle = w_init.spawn_angle[i];
	}

	for (i = 0; i < w_init.data_wells; i ++)
	{
  add_data_well(w_init.data_well_position[i].x, w_init.data_well_position[i].y, i);
	}

}



void run_world(void)
{

// first run the data wells
 int i, j;

 if ((w.world_time & 255) == 0)
	{
		for (i = 0; i < w.data_wells; i ++)
		{
			if (w.data_well[i].active
				&& w.data_well[i].data < w.data_well[i].data_max) // only draw data from reserve if there's space for it
			{
				int data_transferred;
				for (j = 0; j < DATA_WELL_RESERVES; j++)
				{
					if (w.data_well[i].reserve_data [j] == 0)
						continue;
					data_transferred = w.data_well[i].reserve_squares * DATA_WELL_REPLENISH_RATE;
					if (data_transferred > w.data_well[i].data_max - w.data_well[i].data)
						data_transferred = w.data_well[i].data_max - w.data_well[i].data;
					if (data_transferred > w.data_well[i].reserve_data [j])
						data_transferred = w.data_well[i].reserve_data [j];
					w.data_well[i].reserve_data [j] -= data_transferred;
					w.data_well[i].data += data_transferred;
					w.data_well[i].last_transferred = w.world_time;
				}
			}
		}
	}

// Now run the mission, if there is one:
// if (game.type == GAME_TYPE_

}

// This function creates a world background in the block structures
// it assumes that the block structures have otherwise been set up
void init_world_background(void)
{

	int i, j;

// first just init the whole backblock array
 for (i = 0; i < w.blocks.x; i ++)
 {
  for (j = 0; j < w.blocks.y; j ++)
  {
   w.block [i] [j].backblock_type = BACKBLOCK_BASIC_HEX;
   init_hex_block_nodes(i, j);
   init_vision_block(i, j);
  }
 }

// now the edges:
 for (i = 0; i < w.blocks.x; i ++)
 {

  w.block [i] [0].backblock_type = BACKBLOCK_OUTER;
  w.block [i] [1].backblock_type = BACKBLOCK_EDGE_UP;
  w.block [i] [w.blocks.y - 1].backblock_type = BACKBLOCK_OUTER;
  w.block [i] [w.blocks.y - 2].backblock_type = BACKBLOCK_EDGE_DOWN;
 }
 for (i = 0; i < w.blocks.y; i ++)
 {
  w.block [0] [i].backblock_type = BACKBLOCK_OUTER;
  if (i>0&&i<(w.blocks.y-1))
			w.block [1] [i].backblock_type = BACKBLOCK_EDGE_LEFT;
  w.block [w.blocks.x - 1] [i].backblock_type = BACKBLOCK_OUTER;
  if (i>0&&i<(w.blocks.y-1))
   w.block [w.blocks.x - 2] [i].backblock_type = BACKBLOCK_EDGE_RIGHT;

 }

 w.block [1] [1].backblock_type = BACKBLOCK_EDGE_UP_LEFT;
 w.block [1] [w.blocks.y - 2].backblock_type = BACKBLOCK_EDGE_DOWN_LEFT;
 w.block [w.blocks.x - 2] [1].backblock_type = BACKBLOCK_EDGE_UP_RIGHT;
 w.block [w.blocks.x - 2] [w.blocks.y - 2].backblock_type = BACKBLOCK_EDGE_DOWN_RIGHT;
/*
// now let's clear a space:
 for (i = 8; i < 20; i++)
	{
		for (j = 8; j < 16; j ++)
		{
   w.block [i] [j].backblock_type = BACKBLOCK_EMPTY;
// shouldn't need to worry about hex nodes as they just won't be displayed
		}
	}*/


}

#define NODE_SPACING ((BLOCK_SIZE_PIXELS + 3) / 3)

void init_hex_block_nodes(int x, int y)
{

 int k, l, nd;
 struct block_struct* bl;
// int spacing = (BLOCK_SIZE_PIXELS + 3) / 3;
 int space_size = (NODE_SPACING / 6) * 5;
// int space_min = spacing / 12;

    bl = &w.block [x] [y];
    nd = 0;

    for (k = 0; k < 3; k ++)
    {
     for (l = 0; l < 3; l ++)
     {
//      bl->node_size [nd] = (space_size - grand(24)) / 2;
//      bl->node_x [nd] = (k * spacing) + spacing / 2;
//      bl->node_y [nd] = (l * spacing) + spacing / 2;
      bl->node_exists [nd] = 1;

      bl->node_size [nd] = ((space_size) / 2) - 2 + 2;
      if (grand(4) == 0)
       bl->node_size [nd] += 2;
      if (grand(7) == 0)
       bl->node_size [nd] += 2;
      if (grand(7) == 0)
       bl->node_size [nd] -= 2;
      if (grand(7) == 0)
       bl->node_size [nd] -= 4;
      bl->node_x [nd] = (k * NODE_SPACING) + NODE_SPACING / 2;
      bl->node_y [nd] = (l * NODE_SPACING) + NODE_SPACING / 2;

      if ((y + l) % 2 == 0)
       bl->node_x [nd] += NODE_SPACING / 2;

      bl->node_x_base [nd] = bl->node_x [nd];
      bl->node_y_base [nd] = bl->node_y [nd];

      bl->node_team_col [nd] = 0;
      bl->node_new_colour [nd] = 0;
      bl->node_col_saturation [nd] = 0;
      bl->node_new_saturation [nd] = 0;

      bl->node_disrupt_timestamp [nd] = 0;
      bl->node_pending_explosion_timestamp [nd] = 0;
      bl->node_colour_change_timestamp [nd] = 0;

      nd ++;
     }
    }



}


void disrupt_block_nodes(al_fixed x, al_fixed y, int player_cause, int size)
{

 unsigned int bx = fixed_to_block(x);
 unsigned int by = fixed_to_block(y);

 int i;

 if (bx >= w.blocks.x
  || by >= w.blocks.y) // don't need to check negatives as bx/by are unsigned
  return; // just fail if out of bounds

 struct block_struct* bl = &w.block [bx] [by];

 for (i = 0; i < 9; i ++)
 {
  change_block_node(bl, i, bl->node_x [i] + grand(size) - grand(size), bl->node_y [i] + grand(size) - grand(size), bl->node_size [i] + grand(size) - grand(size));
  change_block_node_colour(bl, i, player_cause);
  bl->node_disrupt_timestamp [i] = w.world_time + NODE_DISRUPT_TIME_CHANGE;
 }

}


void disrupt_single_block_node(al_fixed x, al_fixed y, int player_cause, int size)
{

// we need a slight fine-tuning offset here:
 x -= al_itofix(BLOCK_SIZE_PIXELS / 6);
 y -= al_itofix(BLOCK_SIZE_PIXELS / 6);

 unsigned int bx = fixed_to_block(x);
 unsigned int by = fixed_to_block(y);

//  fprintf(stdout, "\ndisrupt_single_block_node() called: block (%i, %i) (%i, %i) at (%f, %f)", bx, by, bx * BLOCK_SIZE_PIXELS, by * BLOCK_SIZE_PIXELS, al_fixtof(x), al_fixtof(y));


 unsigned int i; // if changed to signed, need to make sure bounds checks below check for negative

 if (bx >= w.blocks.x
  || by >= w.blocks.y) // don't check for negatives as bx/by are unsigned
  return; // just fail if out of bounds

 struct block_struct* bl = &w.block [bx] [by];

 int int_x = al_fixtoi(x);
 int int_y = al_fixtoi(y);

//  fprintf(stdout, "\nxy (%f, %f) int_xy (%i, %i)", al_fixtof(x), al_fixtof(y), int_x, int_y);

 int_x -= BLOCK_SIZE_PIXELS * bx;
 int_x *= 3;
 int_x /= BLOCK_SIZE_PIXELS;

 int_y -= BLOCK_SIZE_PIXELS * by;
 int_y *= 3;
 int_y /= BLOCK_SIZE_PIXELS;

 i = (int_x * 3) + int_y;
//  fprintf(stdout, "\nnode (%i) from source (%i, %i)", i, int_x, int_y);

#ifdef SANITY_CHECK
 if (i >= 9) // i is unsigned so don't need to check for negative
 {
  fprintf(stdout, "\ng_world.c: disrupt_single_block_node(): invalid node (%i) from source (%f, %f)", i, al_fixtof(x), al_fixtof(y));
  error_call();
 }
#endif

// for (i = 0; i < 9; i ++)
 {
  change_block_node(bl, i, bl->node_x [i] + grand(size) - grand(size), bl->node_y [i] + grand(size) - grand(size), bl->node_size [i] + grand(size) - grand(size));
  change_block_node_colour(bl, i, player_cause);
//  bl->node_x [i] += grand(5) - grand(5);
//  bl->node_y [i] += grand(5) - grand(5);
/*
  bl->node_size [i] += grand(size) - grand(size);
  if (bl->node_size [i] < 5)
   bl->node_size [i] = 5;
  if (bl->node_size [i] > 40)
   bl->node_size [i] = 40;*/
  bl->node_disrupt_timestamp [i] = w.world_time + NODE_DISRUPT_TIME_CHANGE;


 }

}

#define NODE_LOCATION_MIN -5
#define NODE_LOCATION_MAX (BLOCK_SIZE_PIXELS+5)
#define NODE_SIZE_MIN 5
#define NODE_SIZE_MAX 40

// Moves the block node to new_x,new_y
void change_block_node(struct block_struct* bl, int i, int new_x, int new_y, int new_size)
{
return;
 bl->node_x [i] = new_x;
 if (bl->node_x [i] < NODE_LOCATION_MIN)
  bl->node_x [i] = NODE_LOCATION_MIN;
   else
   {
    if (bl->node_x [i] > NODE_LOCATION_MAX)
     bl->node_x [i] = NODE_LOCATION_MAX;
   }

 bl->node_y [i] = new_y;
 if (bl->node_y [i] < NODE_LOCATION_MIN)
  bl->node_y [i] = NODE_LOCATION_MIN;
   else
   {
    if (bl->node_y [i] > NODE_LOCATION_MAX)
     bl->node_y [i] = NODE_LOCATION_MAX;
   }

 bl->node_size [i] = new_size;
 if (bl->node_size [i] < NODE_SIZE_MIN)
  bl->node_size [i] = NODE_SIZE_MIN;
   else
   {
    if (bl->node_size [i] > NODE_SIZE_MAX)
     bl->node_size [i] = NODE_SIZE_MAX;
   }

}


// Moves the block node back towards its original position
void align_block_node(struct block_struct* bl, int i)
{

 if (bl->node_x [i] < bl->node_x_base [i])
	{
		bl->node_x [i] ++;
	}
	 else
		{
   if (bl->node_x [i] > bl->node_x_base [i])
		  bl->node_x [i] --;
		}

 if (bl->node_y [i] < bl->node_y_base [i])
	{
		bl->node_y [i] ++;
	}
	 else
		{
   if (bl->node_y [i] > bl->node_y_base [i])
		  bl->node_y [i] --;
		}

	bl->node_size [i] += grand(5) - grand(5);

	if (bl->node_size [i] < 10)
		bl->node_size [i] = 10;

	if (bl->node_size [i] > 22)
		bl->node_size [i] = 22;

}


void change_block_node_colour(struct block_struct* bl, int i, int player_cause)
{

  if (bl->node_team_col [i] == player_cause)
  {
   if (bl->node_col_saturation [i] < BACK_COL_SATURATIONS - 1)
    bl->node_col_saturation [i] ++;
//   fprintf(stdout, "\nBlock node %i colour matches (%i) saturation now %i", i, player_cause, bl->node_col_saturation [i]);
  }
   else
   {
//    fprintf(stdout, "\nBlock node %i colour (%i) changed to (%i)", i, bl->node_team_col [i], player_cause);
    bl->node_team_col [i] = player_cause;
    bl->node_col_saturation [i] = 0;
   }


}


void explosion_affects_block_nodes(al_fixed explosion_x, al_fixed explosion_y, int explosion_size, int player_index)
{

	int i, j, k;
	int centre_block_x = fixed_to_block(explosion_x);
	int centre_block_y = fixed_to_block(explosion_y);
	int explosion_x_pixels = al_fixtoi(explosion_x);
	int explosion_y_pixels = al_fixtoi(explosion_y);

// test values:
	int explosion_size_pixels = explosion_size;
	int explosion_size_blocks = (explosion_size / BLOCK_SIZE_PIXELS) + 1; // this is a radius

	int left_block = centre_block_x - explosion_size_blocks;
	if (left_block < 2)
		left_block = 2;
	int right_block = centre_block_x + explosion_size_blocks + 1;
	if (right_block > w.blocks.x - 3)
		right_block = w.blocks.x - 3;
	int top_block = centre_block_y - explosion_size_blocks;
	if (top_block < 2)
		top_block = 2;
	int bottom_block = centre_block_y + explosion_size_blocks + 1;
	if (bottom_block > w.blocks.y - 3)
		bottom_block = w.blocks.y - 3;

	struct block_struct* blk;
	int node_dist_from_centre;
	int node_world_x, node_world_y;
	int node_explosion_time;

	for (i = left_block; i < right_block; i++)
	{
		for (j = top_block; j < bottom_block; j ++)
		{
			if (w.block[i][j].backblock_type != BACKBLOCK_BASIC_HEX)
				continue;
			blk = &w.block[i][j];
			for (k = 0; k < BLOCK_NODES; k ++)
			{
				if (w.block[i][j].node_exists [k] == 0)
					continue;
				node_world_x = (i * BLOCK_SIZE_PIXELS) + blk->node_x [k];
				node_world_y = (j * BLOCK_SIZE_PIXELS) + blk->node_y [k];
				node_dist_from_centre = hypot(node_world_y - explosion_y_pixels, node_world_x - explosion_x_pixels); // TO DO: replace hypot!!!
				int approaching_edge = explosion_size_pixels - node_dist_from_centre;
				if (approaching_edge < 80
				 && grand(80) > approaching_edge)
						continue;
				if (node_dist_from_centre < explosion_size_pixels)
				{
					node_explosion_time = 34 - (node_dist_from_centre / 10);// + grand(5);
					if (node_explosion_time > 32)
						node_explosion_time = 32;
     blk->node_pending_explosion_timestamp [k] = w.world_time + 32 + (32 - node_explosion_time);
//     blk->node_pending_explosion_player [k] = player_index;
     blk->node_pending_explosion_strength [k] = 1;//explosion_strength_at_node;
     blk->node_colour_change_timestamp [k] = blk->node_pending_explosion_timestamp [k] - 32;
     blk->node_team_col [k] = blk->node_new_colour [k]; // shouldn't do this if replacing another pending explosion!
     blk->node_new_colour [k] = player_index;
     blk->node_col_saturation [k] = blk->node_new_saturation [k]; // shouldn't do this if replacing another pending explosion!
     if (blk->node_new_saturation [k] < BACK_COL_SATURATIONS - 1)
      blk->node_new_saturation [k] ++;

				}
			}
		}
	}


}



void pulse_block_node(al_fixed pulse_x, al_fixed pulse_y)
{

//	int i, j, k;
	int block_x = fixed_to_block(pulse_x);
	int block_y = fixed_to_block(pulse_y);

	struct block_struct* blk = &w.block[block_x][block_y];

	if (blk->backblock_type != BACKBLOCK_BASIC_HEX)
		return;

	int node_index = ((al_fixtoi(pulse_y) & (BLOCK_SIZE_PIXELS-1)) / NODE_SPACING) + ((al_fixtoi(pulse_x) & (BLOCK_SIZE_PIXELS-1)) / NODE_SPACING) * 3;

#ifdef SANITY_CHECK
 if (node_index < 0
		|| node_index > 8)
	{
		fpr("\nError in g_world.c: pulse_block_node(): node_index out of bounds (%i)", node_index);
		error_call();
	}
#endif

 blk->node_pending_explosion_timestamp [node_index] = w.world_time + 32;

// fpr("\n pulse at %i,%i block %i,%i node %i", al_fixtoi(pulse_x), al_fixtoi(pulse_y), block_x, block_y, node_index);

}




void init_vision_block(int x, int y)
{

/*

 w.block[x][y].vision_block_x [0] = 16;
 w.block[x][y].vision_block_y [0] = 32;

 w.block[x][y].vision_block_x [1] = 80;
 w.block[x][y].vision_block_y [1] = 32;

 w.block[x][y].vision_block_x [2] = 48;
 w.block[x][y].vision_block_y [2] = 96;

 w.block[x][y].vision_block_x [3] = 112;
 w.block[x][y].vision_block_y [3] = 96;



 int i;

 for (i = 0; i < 4; i ++)
	{

  w.block[x][y].vision_block_x_shrink [i] = 1.5 + grand(100) * 0.004;
	}

 return;


 for (i = 0; i < 4; i ++)
	{
			w.block[x][y].vision_block_x_shrink [i] = 1;
			w.block[x][y].vision_block_y_shrink [i] = 1;
	}

 i = 0;


 float split_x = grand(64) + 32;
 float split_y = grand(64) + 32;

 w.block[x][y].vision_block_x [i] = grand(48) + 8;
 w.block[x][y].vision_block_y [i] = grand(48) + 8;

 i++;

 w.block[x][y].vision_block_x [i] = grand(48) + 8 + split_x;
 w.block[x][y].vision_block_y [i] = grand(48) + 8;

 i++;

 w.block[x][y].vision_block_x [i] = grand(48) + 8;
 w.block[x][y].vision_block_y [i] = grand(48) + 8 + split_y;

 i++;

 w.block[x][y].vision_block_x [i] = grand(48) + 8 + split_x;
 w.block[x][y].vision_block_y [i] = grand(48) + 8 + split_y;
*/
}


/*

Let's try to get the game to a state where it can be started

Need to:
- make w_init then call start_world
- make a template
- call proc creation functions with the template
- that should be enough?

*/

#include "c_compile.h"
#include "c_header.h"
#include "e_editor.h"
//#define TEST_SOURCE_LINES 30
/*
char test_source [1] [TEST_SOURCE_LINES] [80] =
{
{
"#process",
"core_shape_4, 0,",
"  {object_none, 0},",
"  {object_none, 0},",
"  {object_none, 0},",
"  {object_none, 0}",
"#code",
"",
"int a;"
}
};
*/
/*
struct source_struct
{
// lines in the text array should be null-terminated, although actually they don't have to be as each time text is used bounds-checking is done
 char text [SOURCE_TEXT_LINES] [SOURCE_TEXT_LINE_LENGTH];
//  *** text array must be the same as in source_struct (as the code that converts bcode to source code assumes that it can treat source.text in the same way as source_edit.text)

// int src_file [SOURCE_TEXT_LINES]; // stores the index of the file that the line came from
 int from_a_file; // is 1 if loaded from a file, 0 otherwise (will be 0 if a new empty source file created in the editor, until it's saved)
 char src_file_name [FILE_NAME_LENGTH]; // should be empty if from_a_file == 0
 char src_file_path [FILE_PATH_LENGTH]; // same
};
*/

// Loads one of the pre-set source files into a template and compiles it.
void load_mission_source(char* filename, int player_index, int template_index)
{
	int loaded = load_source_file_into_template(filename, player_index, template_index);

	if (loaded != 1)
	{
		fpr("\nError: failed to load mission file [%s] into template[%i][%i].", filename, player_index, template_index);
// Causes a fatal error on failure as these source files should be known to be valid
		error_call();
	}

	templ[player_index][template_index].locked = 1; // load_source_file_into_template should do enough for it to be okay to lock these templates.

}

// Loads default sources specified in init.txt
// Prints a message (to stdout) on failure, but doesn't break anything
void load_default_source(char* filename, int player_index, int template_index)
{

	fpr("\nLoading default source file [%s] into player %i template %i... ", filename, player_index, template_index);

	int loaded = load_source_file_into_template(filename, player_index, template_index);

 switch(loaded)
 {
 	case -1: // failed to open file
 		fpr(" load failed :(");
 		break;
 	case 0: // failed to compile
 		fpr(" could not compile :(");
 		break;
 	case 1: // success
 		fpr(" loaded.");
 		break;

 }


}

// Returns:
//  -1 failed to load source file
//  0 loaded source file, but failed to compile
//  1 success
int load_source_file_into_template(char* filename, int player_index, int template_index)
{

 int target_esource = (player_index * TEMPLATES_PER_PLAYER) + template_index;

 struct source_struct temp_src;

 if (!load_source_file(filename, &temp_src))
		return -1;
/*	{
		fpr("\n Error: failed to load file [%s]", filename);
		error_call();
	}*/

 open_new_template(&templ[player_index][template_index]);

 source_to_editor(&temp_src, target_esource);
 editor.source_edit[target_esource].from_a_file = 1;
 editor.source_edit[target_esource].saved = 1;

 return compile(&templ[player_index][template_index], templ[player_index][template_index].source_edit, COMPILE_MODE_BUILD);
// returns 1 or 0

}





/*

PLAN for single-player game (maybe also multi)
 - each player starts with a certain amount of data, from which process 0 is built
  - player then starts with a proportion of leftover data - maybe 1/2 or 1/4?


PLAN for single-player autonomous enemies

Builder:
 - basic hex core with:
  - harvest, storage, allocator
  - build
- How does it behave?
 - just gathers data and allocates it.
 - keeps track of harvesters, builders and commanders using targetting memory
  - if < 3 harvesters, builds more
  - if < 1 builders, builds one
  - if < 3 commanders, builds more
  - otherwise, builds attackers.

Templates:
0 - static builder/allocator
1 - mobile builder
2 - harvester
3 - commander
4 - attacker


More advanced versions might have extra components with other objects on them

Mobile builder:
 - wanders until it finds a data well. looks around for friendly builder. If none found, builds one.

mobile harvester:
 - wanders until it finds a data well. harvests data well then returns.
  - can store a few data wells and will go to another if it goes to an empty one.
  - if return fails, wanders until it finds a friendly allocator then reassigns

wandering attacker (can have various designs)
 - just wanders randomly attacking things
 - listens for "follow me" messages and obeys

commander
 - wanders randomly
 - broadcasts "follow me" messages now and then
 - well-protected and probably has repair_other objects


Later:

How to coordinate attacks?
 - Each static builder controls everything built by it.
 - when one of its processes finds an enemy builder/allocator it sends a low priority message to the builder.



BUT FIRST really need to do repair.

OBJECT_TYPE_REPAIR
OBJECT_TYPE_REPAIR_OTHER

standard methods:

repair_component(int component)
repair_auto()
 - repairs most damaged component

restore_component(int component)
 - tries to restore target component.
  - ideally this would try to fix any inner components that the target component requires
  - but maybe implement this later
restore_auto()
 - restores first component in template list
 - not sure whether this should run auto_repair. Maybe not - leave this to user

repair_component_other(int target_index, int component)
repair_auto_other(int target_index)
 - runs repair_auto on target process
repair_auto_scan()
 - first tries to repair self
 - then scans for friendly targets
 - probably repairs closest damaged one? not sure

restore_component_other(int target_index, int component)
restore_auto_other(int target_index)
restore_auto_scan()


repairing costs something like 20 power to repair 10hp each cycle?
multiple repair objects increase power and repair rate proportionally, but can't call them separately

restoring:
 - costs a lot of power - say 120?
 - restored component has minimal hp - say 20?
 - cooldown is determined by data cost of restored component
  - and reduced by number of repair objects

During cooldown can repair, but not restore.


*** alternative approach:

repair_self()
 - finds the first damaged component and repairs it.
restore_self()
 - finds the first destroyed component and repairs it.
  - query: can we assume that a group member with a lower group index will be upstream from a group member with a higher group index?
   - I think so but check

* costs a lot of power - the purpose of repair is to allow processes to retreat and recover. It's not meant to replace an interface.

repair_other(int target_index)
 - runs repair_self() on target
repair_scan()
 - finds nearest damaged friendly process and runs repair_other() on it
restore_other(int target_index)
 - runs restore_self() on target
restore_scan()
 - guess what



*/

