/*

This file contains the music-playing functions.

Every function with a name starting with "sthread" should run in the separate music thread.

*/


#include <allegro5/allegro.h>
#include <math.h>

#include "allegro5/allegro_audio.h"
#include "allegro5/allegro_acodec.h"

#include <stdio.h>
#include <stdlib.h>
//#include <math.h>

#include "m_config.h"
#include "m_globvars.h"

//#include "g_misc.h"
//#include "g_header.h"

#include "x_sound.h"
#include "x_init.h"
#include "x_synth.h"
#include "x_music.h"



extern ALLEGRO_SAMPLE *sample [SAMPLES];
ALLEGRO_SAMPLE *msample [MSAMPLES];
extern ALLEGRO_EVENT_SOURCE sound_event_source;
extern ALLEGRO_EVENT_QUEUE *sound_queue;
extern ALLEGRO_EVENT sound_event;

extern struct sound_configstruct sound_config; // in x_sound.c

float tone [TONES];

extern struct synthstruct synth;



static unsigned int sthread_rand(unsigned int rand_max);
//static void sthread_play_msample(int msample_index, int play_tone, int vol, int pan);
static void sthread_play_scale_sample(int instrument_flip, int instrument, int play_note, int vol, int pan);
static void sthread_new_instrument(int instrument, int scale_type, int instrument_type, int effect, int tone_offset);
static int sthread_random_instrument_type(void);

enum
{
NOTE_0,
NOTE_1,
NOTE_2,
NOTE_3,
NOTE_4,
NOTE_5,
NOTE_6,
NOTE_7,
NOTES
};


/*

The following is a cellular-automata music generator.

It's based on Batuhan Bozkurt's Otomata (see http://www.earslap.com/page/otomata.html )



*/

enum
{
INSTRUMENT_TYPE_BASIC_SINE, // later code assumes this is 0
INSTRUMENT_TYPE_SINE_HARM1, // and this is 1
INSTRUMENT_TYPE_SINE_HARM2, // and 2
INSTRUMENT_TYPE_SINE_SQUARE,
INSTRUMENT_TYPE_SQUARE,
INSTRUMENT_TYPE_SQUARE_HARM1, // must come directly after SQUARE
INSTRUMENT_TYPE_SQUARE_HARM2, // must come directly after SQUARE_HARM1
INSTRUMENT_TYPES

};

#define INSTRUMENT_TYPES_COMMON INSTRUMENT_TYPE_SQUARE
// instrument types below INSTRUMENT_TYPES_COMMON will be more common

enum
{
INSTRUMENT_EFFECT_NONE,
INSTRUMENT_EFFECT_HIGH_PITCH,
INSTRUMENT_EFFECT_VERY_HIGH_PITCH,
INSTRUMENT_EFFECT_SHORT,
INSTRUMENT_EFFECT_LONG,

INSTRUMENT_EFFECTS
};

#define CAM_W 9
#define CAM_H 9
#define CAM_AGENTS 24

enum
{
CADIR_UP,
CADIR_RIGHT,
CADIR_DOWN,
CADIR_LEFT,
CADIRS
};

#define INSTRUMENTS 3

struct cam_agent_struct
{
	int exists;
	int x, y;
	int direction;
	int instrument;
//	int move_x, move_y;
 timestamp collision_time;
// int msample_index;
// int scale_index;

 int echo_tone;
 int echo_strength;
 int echo_pan; // should be from -50 to 50
 int echo_instrument;
 int echo_flip; // 0 or 1 - based on instrument_flip for the agent's instrument at time when echo began

};

struct camstate_struct
{
	int scale_index; // shared by all instruments
	int note_offset; // shared by all instruments
	int counter;
	int tempo;
	int next_change; // should never be shorter than about 25 (to make sure that the instrument flipping doesn't mess up old echoes)
	timestamp total_time;
	int cam_grid [CAM_W] [CAM_H]; // holds index of agent, or -1 if empty
	struct cam_agent_struct agent [CAM_AGENTS];

	int instrument_sample_flip [INSTRUMENTS];
	int instrument_type [INSTRUMENTS];
	int instrument_effect [INSTRUMENTS];
	int base_echo_strength [INSTRUMENTS];

	unsigned int rand_seed;

};
struct camstate_struct camstate;

ALLEGRO_SAMPLE* scale_sample [2] [INSTRUMENTS] [SCALE_TONES];


static void sthread_cam_note(int agent_index, int tone_index);
static void sthread_rotate_cam_agent(int agent_index);
static void sthread_change_camstate(void);

static void sthread_create_samples_for_scale(int flip, int instrument, int scale_type, int instrument_type, int effect, int tone_offset);

// This function is usually only called from the sound thread. reset_music is called from outside (see x_sound.c).
//  However this function is called from the main thread during initialisation (before the sound thread has started)
void init_camstate(unsigned int rand_seed)
{

 camstate.rand_seed = rand_seed;

	camstate.tempo = 1;
	camstate.counter = 8 + camstate.tempo; // 8 is to put a short delay at the start
	camstate.total_time = 16;
	camstate.next_change = 40 + sthread_rand(20);

	int i, j;

	for (i = 0; i < CAM_W; i ++)
	{
	 for (j = 0; j < CAM_H; j ++)
	 {
	 	camstate.cam_grid [i] [j] = -1;
	 }
	}

	for (i = 0; i < CAM_AGENTS; i ++)
	{
		camstate.agent [i].exists = 0;
		camstate.agent [i].collision_time = 0;
		camstate.agent [i].echo_strength = 0;
		camstate.agent [i].instrument = sthread_rand(INSTRUMENTS);

	}


 for (i = 0; i < 12; i ++)
	{
		camstate.agent [i].exists = 1;
		camstate.agent [i].x = sthread_rand(CAM_W);
		camstate.agent [i].y = sthread_rand(CAM_H);

		camstate.agent [i].direction = sthread_rand(CADIRS);

//		camstate.agent [i].scale_index = sthread_rand(2);// + sthread_rand_r(4);

/*
		camstate.agent [i].msample_index = MSAMPLE_NOTE + sthread_rand(3);// + sthread_rand_r(4);
		if (camstate.agent [i].msample_index == MSAMPLE_NOTE2)
			camstate.agent [i].msample_index = MSAMPLE_NOTE3;

		camstate.agent [i].msample_index = MSAMPLE_NOTE3;
*/
	}

//	camstate.scale_index = 1;//SCALE_MINOR;
//	camstate.note_offset = 0;
	camstate.base_echo_strength [0] = 6;//sthread_rand(12);
	camstate.base_echo_strength [1] = 6;//sthread_rand(12);
	camstate.base_echo_strength [2] = 6;//sthread_rand(12);

 camstate.scale_index = sthread_rand(SCALE_NOTE_TYPE);
 camstate.note_offset = 0;

 sthread_new_instrument(0, camstate.scale_index, sthread_random_instrument_type(), sthread_rand(INSTRUMENT_EFFECTS), camstate.note_offset);
 sthread_new_instrument(1, camstate.scale_index, sthread_random_instrument_type(), sthread_rand(INSTRUMENT_EFFECTS), camstate.note_offset);
 sthread_new_instrument(2, camstate.scale_index, sthread_random_instrument_type(), sthread_rand(INSTRUMENT_EFFECTS), camstate.note_offset);


// sthread_new_instrument(0, camstate.scale_index, INSTRUMENT_TYPE_BASIC_SINE, 0, camstate.note_offset);
// sthread_new_instrument(1, camstate.scale_index, INSTRUMENT_TYPE_BASIC_SQUARE, 0, camstate.note_offset);
// sthread_new_instrument(2, camstate.scale_index, INSTRUMENT_TYPE_SINE_HARM2, 0, camstate.note_offset);

/*
 i = 0;

	camstate.agent [i].exists = 1;
	camstate.agent [i].x = 2;
	camstate.agent [i].y = 2;
	camstate.agent [i].direction = 0;


 i ++;
	camstate.agent [i].exists = 1;
	camstate.agent [i].x = 3;
	camstate.agent [i].y = 2;
	camstate.agent [i].direction = 0;

 i ++;
	camstate.agent [i].exists = 1;
	camstate.agent [i].x = 4;
	camstate.agent [i].y = 2;
	camstate.agent [i].direction = 0;

 i ++;
	camstate.agent [i].exists = 1;
	camstate.agent [i].x = 3;
	camstate.agent [i].y = 3;
	camstate.agent [i].direction = 2;
*/

}

// returns a random new instrument, weighted towards the sine wave ones
static int sthread_random_instrument_type(void)
{

	if (sthread_rand(3) == 0)
		return sthread_rand(INSTRUMENT_TYPES);

 return sthread_rand(INSTRUMENT_TYPES_COMMON);

}


void sthread_run_camstate(void)
{
//return;
//#define BASE_ECHO_STRENGTH 10

	int i;
/*	for (i = 0; i < CAM_AGENTS; i ++)
	{


		if (!camstate.agent[i].exists)
			continue;

		camstate.agent[i].echo_strength --;
fpr("\n str %i ", camstate.agent[i].echo_strength);
		if (camstate.agent[i].echo_strength > 0
			&& camstate.agent[i].echo_strength < BASE_ECHO_STRENGTH-1)
//			&& (camstate.agent[i].echo_strength&1))
		{
    sthread_play_msample(camstate.agent[i].msample_index, scale [camstate.scale_index] [camstate.agent[i].echo_tone], 20 + camstate.agent[i].echo_strength * 20, -50 + camstate.agent[i].echo_pan + ((camstate.agent[i].echo_strength&2) * 100));
		}
	}*/

	camstate.counter --;

	if (camstate.counter > 0)
		return;

//	camstate.total_time ++;
/*
	static int play_note = 0;
	static int play_instrument = 0;

 sthread_play_scale_sample(1, play_instrument, play_note, 100, 0);
 play_instrument ++;
 if (play_instrument == 3)
	{
  play_note ++;
  play_note %= 9;
  play_instrument = 0;
	}

 camstate.counter = 3;
 return;
*/

	camstate.counter = camstate.tempo;
	camstate.total_time ++;
	camstate.next_change --;


	for (i = 0; i < CAM_AGENTS; i ++)
	{


		if (!camstate.agent[i].exists
			 || camstate.agent[i].collision_time == camstate.total_time)
			continue;


		switch(camstate.agent [i].direction)
		{
		 case CADIR_UP:
		 	if (camstate.agent [i].y == 0)
				{
     sthread_cam_note(i, camstate.agent[i].x);
					camstate.agent[i].direction = CADIR_DOWN;
					break;
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = -1;
				camstate.agent[i].y --;
				if (camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] != -1)
				{
					camstate.agent[i].direction = CADIR_RIGHT;
					sthread_rotate_cam_agent(camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y]);
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = i;
				break;
		 case CADIR_DOWN:
		 	if (camstate.agent [i].y == CAM_H - 1)
				{
     sthread_cam_note(i, camstate.agent[i].x);
					camstate.agent[i].direction = CADIR_UP;
					break;
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = -1;
				camstate.agent[i].y ++;
				if (camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] != -1)
				{
					camstate.agent[i].direction = CADIR_LEFT;
					sthread_rotate_cam_agent(camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y]);
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = i;
				break;
		 case CADIR_LEFT:
		 	if (camstate.agent [i].x == 0)
				{
     sthread_cam_note(i, camstate.agent[i].y);
					camstate.agent[i].direction = CADIR_RIGHT;
					break;
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = -1;
				camstate.agent[i].x --;
				if (camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] != -1)
				{
					camstate.agent[i].direction = CADIR_UP;
					sthread_rotate_cam_agent(camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y]);
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = i;
				break;
		 case CADIR_RIGHT:
		 	if (camstate.agent [i].x == CAM_W - 1)
				{
     sthread_cam_note(i, camstate.agent[i].y);
					camstate.agent[i].direction = CADIR_LEFT;
					break;
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = -1;
				camstate.agent[i].x ++;
				if (camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] != -1)
				{
					camstate.agent[i].direction = CADIR_DOWN;
					sthread_rotate_cam_agent(camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y]);
				}
				camstate.cam_grid [camstate.agent[i].x] [camstate.agent[i].y] = i;
				break;
		}


		camstate.agent[i].echo_strength --;

		if (camstate.agent[i].echo_strength > 0
//			&& camstate.agent[i].echo_strength < BASE_ECHO_STRENGTH-1
			&& (camstate.agent[i].echo_strength&1))
		{
//    sthread_play_scale_sample(camstate.agent[i].echo_flip, camstate.agent[i].echo_instrument, camstate.agent[i].echo_tone, 5 + camstate.agent[i].echo_strength * 10, -50 + camstate.agent[i].echo_pan + ((camstate.agent[i].echo_strength&2) * 50));

    sthread_play_scale_sample(camstate.agent[i].echo_flip, camstate.agent[i].echo_instrument, camstate.agent[i].echo_tone, 25 + camstate.agent[i].echo_strength * 10, -50 + camstate.agent[i].echo_pan + ((camstate.agent[i].echo_strength&2) * 50));

		}

	}

	if (camstate.next_change <= 0)
	{
		sthread_change_camstate();
		camstate.next_change = 40 + sthread_rand(120);
	}


}

static unsigned int sthread_rand(unsigned int rand_max)
{

 camstate.rand_seed = camstate.rand_seed * 1103515245 + 12345;
 return (unsigned int)(camstate.rand_seed / 65536) % rand_max;

}

static void sthread_cam_note(int agent_index, int tone_index)
{

 camstate.agent[agent_index].echo_pan = -50 + (camstate.agent[agent_index].x * 100) / CAM_W;
// sthread_play_msample(camstate.agent[agent_index].msample_index, scale [camstate.scale_index] [tone_index] + camstate.note_offset, 100, camstate.agent[agent_index].echo_pan);
 sthread_play_scale_sample(camstate.instrument_sample_flip[camstate.agent[agent_index].instrument], camstate.agent[agent_index].instrument, tone_index, 100, camstate.agent[agent_index].echo_pan);

 camstate.agent[agent_index].echo_strength = camstate.base_echo_strength [camstate.agent[agent_index].instrument];//BASE_ECHO_STRENGTH / 2;
 camstate.agent[agent_index].echo_tone = tone_index;//scale [camstate.scale_index] [tone_index] + camstate.note_offset;
 camstate.agent[agent_index].echo_instrument = camstate.agent[agent_index].instrument;
 camstate.agent[agent_index].echo_flip = camstate.instrument_sample_flip[camstate.agent[agent_index].instrument];

}

static void sthread_rotate_cam_agent(int agent_index)
{
	camstate.agent[agent_index].direction ++;
	camstate.agent[agent_index].direction %= CADIRS;
	camstate.agent[agent_index].collision_time = camstate.total_time;
}

// makes some kind of random change to camstate.
static void sthread_change_camstate(void)
{

 int changes = 1;

 if (sthread_rand(4) == 0)
		changes += sthread_rand(5);
// there may not be this many actual changes, as some of the change types result in this function returning (to avoid double-flipping an instrument)

	int change_type;
	int i;

 while(changes > 0)
	{
		change_type = sthread_rand(5);

		switch(change_type)
		{
		 case 0: // scale change - applies to all instruments
		 	{
		 		camstate.scale_index = sthread_rand(SCALE_NOTE_TYPE);
     for (i = 0; i < INSTRUMENTS; i ++)
					{
						sthread_new_instrument(i, camstate.scale_index, camstate.instrument_type [i], camstate.instrument_effect [i], camstate.note_offset);
					}
		 	}
		 	return; // return, not break
		 case 1: // note offset change - applies to all instruments
		 	{
		 		camstate.note_offset = sthread_rand(9);
     for (i = 0; i < INSTRUMENTS; i ++)
					{
						sthread_new_instrument(i, camstate.scale_index, camstate.instrument_type [i], camstate.instrument_effect [i], camstate.note_offset);
					}
		 	}
		 	return; // return, not break
		 case 2: // one instrument is changed
				{
						sthread_new_instrument(sthread_rand(INSTRUMENTS), camstate.scale_index, sthread_random_instrument_type(), sthread_rand(INSTRUMENT_EFFECTS), camstate.note_offset);
				}
				return; // return, not break
			case 3: // echo strength change
				camstate.base_echo_strength [sthread_rand(INSTRUMENTS)] = 3 + sthread_rand(5);
				break;
			case 4: // cam agent direction change
				{
					int agent_index = sthread_rand(CAM_AGENTS);
				 camstate.agent[agent_index].direction = sthread_rand(CADIRS);
// doesn't matter if agent isn't active
				}
				break;
//			case 3:

// remember to change sthread_rand() call above when adding more cases!



		}

		changes --;
	}

// may not reach this far

}


static void sthread_new_instrument(int instrument, int scale_type, int instrument_type, int effect, int tone_offset)
{

	int new_flip = camstate.instrument_sample_flip [instrument] ^ 1;

 sthread_create_samples_for_scale(new_flip, instrument, scale_type, instrument_type, effect, tone_offset);

 camstate.instrument_sample_flip [instrument] = new_flip;
 camstate.instrument_type [instrument] = instrument_type;
 camstate.instrument_effect [instrument] = effect;

}


float scale_sample_data [2] [INSTRUMENTS] [SCALE_TONES] [SYNTH_SAMPLE_MAX_SIZE];

#define SCALE_NOTE_TYPE 2
/*
Just temperament for a scale starting with C
 - unfortunately just temperament doesn't seem to work so well for this kind of music
#define RATIO_CS (float) (16.0/15)
#define RATIO_D (float) (9.0/8)
#define RATIO_DS (float) (6.0/5)
#define RATIO_E (float) (5.0/4)
#define RATIO_F (float) (4.0/3)
#define RATIO_FS (float) (45.0/32)
#define RATIO_G (float) (3.0/2)
#define RATIO_GS (float) (8.0/5)
#define RATIO_A (float) (5.0/3)
#define RATIO_AS (float) (16.0/9)
#define RATIO_B (float) (15.0/8)
*/
/*
Equal temperament:
*/
#define RATIO_CS 1.059463
#define RATIO_D 1.122462
#define RATIO_DS 1.189207
#define RATIO_E 1.259921
#define RATIO_F 1.334840
#define RATIO_FS 1.414214
#define RATIO_G 1.498307
#define RATIO_GS 1.587401
#define RATIO_A 1.681793
#define RATIO_AS 1.781797
#define RATIO_B 1.887749


float scale_ratio [SCALE_NOTE_TYPE] [SCALE_TONES] =
{

	{
		1, // C
		RATIO_E,
		RATIO_G,
		RATIO_A,
		RATIO_AS,
		2,
		RATIO_E * 2,
		RATIO_G * 2,
		RATIO_A * 2,


	},

	{
		1, // C
		RATIO_G,
		RATIO_GS,
		RATIO_AS,
		2,
		RATIO_D * 2,
		RATIO_DS * 2,
		RATIO_G * 2,
		RATIO_GS * 2,

	},
/*
	{
		1, // C
		RATIO_D,
		RATIO_E,
		RATIO_G,
		RATIO_A,
		2,
		RATIO_D * 2,
		RATIO_E * 2,
		RATIO_G * 2
	},*/
/*
	{
		1, // C
		RATIO_D,
		RATIO_F,
		RATIO_G,
		RATIO_AS,
		2,
		RATIO_D * 2,
		RATIO_F * 2,
		RATIO_AS * 2
	},
*/

};

float temp_buffer [SYNTH_SAMPLE_MAX_SIZE];


static void sthread_create_samples_for_scale(int flip, int instrument, int scale_type, int instrument_type, int effect, int tone_offset)
{

	int i;

 float* current_buffer;
 int buffer_length = 8000;

 int attack_length = 50; // counted from start
 int decay_length = 200; // counted from end of attack
 int release_length = 4500; // counted back from end

 float base_freq;


 base_freq = 125 * pow(1.059463, tone_offset); // should this be something else? I don't know how it maps to Hz or whatever

 switch(effect)
 {
// 	case INSTRUMENT_EFFECT_LOW_PITCH:
//   base_freq /= 2; break;
 	case INSTRUMENT_EFFECT_VERY_HIGH_PITCH: // * 4 doesn't sound that great
 	case INSTRUMENT_EFFECT_HIGH_PITCH:
   base_freq *= 2; break;
// 	case INSTRUMENT_EFFECT_VERY_HIGH_PITCH:
//   base_freq *= 4; break;
  case INSTRUMENT_EFFECT_SHORT:
			buffer_length = 5000;
			release_length = 1500;
			break;
  case INSTRUMENT_EFFECT_LONG:
			buffer_length = 10000; // must not be longer than SYNTH_SAMPLE_MAX_SIZE
			break;
 }

//instrument_type = INSTRUMENT_TYPE_SQUARE + sthread_rand(3);

 switch(instrument_type)
 {

 	case INSTRUMENT_TYPE_BASIC_SINE:
 	case INSTRUMENT_TYPE_SINE_HARM1:
	 case INSTRUMENT_TYPE_SINE_HARM2:
	  for (i = 0; i < SCALE_TONES; i ++)
	  {

    current_buffer = scale_sample_data [flip] [instrument] [i];

    init_waveform(current_buffer, buffer_length);



//    add_waveform_sine(current_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i], 0, 1);
    add_waveform_sine(current_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i], instrument_type, 0, 1);

/*
    if (instrument_type >= INSTRUMENT_TYPE_SINE_HARM1)
				{
			  init_waveform(temp_buffer, buffer_length);
     add_waveform_sine(temp_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i] * 2, 0, 0.4);

     add_waveforms(current_buffer, temp_buffer, buffer_length);
				}

    if (instrument_type >= INSTRUMENT_TYPE_SINE_HARM2)
				{
			  init_waveform(temp_buffer, buffer_length);
     add_waveform_sine(temp_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i] * 4, 0, 0.4);

     add_waveforms(current_buffer, temp_buffer, buffer_length);
				}
*/
    apply_adsr(current_buffer, buffer_length,
												   attack_length, // attack length
												   decay_length, 0.5, // decay length, target amplification after decay
												   release_length); // release length (counted back from end


    float amplification = 0.3 - (instrument_type * 0.1);
    amplify(current_buffer, buffer_length, amplification);
     scale_sample [flip] [instrument] [i] = finish_sample(current_buffer, buffer_length); // finish_sample exits on failure to create sample

	 }
	 break;

	 case INSTRUMENT_TYPE_SINE_SQUARE:
	  for (i = 0; i < SCALE_TONES; i ++)
	  {

    current_buffer = scale_sample_data [flip] [instrument] [i];
    init_waveform(current_buffer, buffer_length);

    add_waveform_sine(current_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i], 1, 0, 1);

//    if (i % 3 == 0)
//				{

			  init_waveform(temp_buffer, buffer_length);
     add_waveform_square(temp_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i], 0, 0, 0.3);
/*    apply_adsr(temp_buffer, buffer_length,
												   buffer_length - 100, // attack length
												   50, 0.5, // decay length, target amplification after decay
												   50); // release length (counted back from end*/


     add_waveforms(current_buffer, temp_buffer, buffer_length);
//				}

    apply_adsr(current_buffer, buffer_length,
												   attack_length, // attack length
												   decay_length, 0.3, // decay length, target amplification after decay
												   release_length); // release length (counted back from end

    amplify(current_buffer, buffer_length, 0.2);
     scale_sample [flip] [instrument] [i] = finish_sample(current_buffer, buffer_length); // finish_sample exits on failure to create sample

	 }
	 break;


  case INSTRUMENT_TYPE_SQUARE_HARM1:
  case INSTRUMENT_TYPE_SQUARE_HARM2:
	 case INSTRUMENT_TYPE_SQUARE:

// base_freq = 125 * pow(1.059463, tone_offset); // these instruments sound terrible at very high pitch


	  for (i = 0; i < SCALE_TONES; i ++)
	  {

    current_buffer = scale_sample_data [flip] [instrument] [i];
    init_waveform(current_buffer, buffer_length);

    add_waveform_square(current_buffer, buffer_length, base_freq * scale_ratio [scale_type] [i], instrument_type - INSTRUMENT_TYPE_SQUARE, 0, 0.5);

//    if (i % 3 == 0)
//				{


    apply_adsr(current_buffer, buffer_length,
												   attack_length, // attack length
												   decay_length, 0.3, // decay length, target amplification after decay
												   release_length); // release length (counted back from end

    amplify(current_buffer, buffer_length, 0.2);

    scale_sample [flip] [instrument] [i] = finish_sample(current_buffer, buffer_length); // finish_sample exits on failure to create sample

	 }
	 break;
 }

/*
 scale_note_type = 0;

	for (i = 0; i < SCALE_TONES; i ++)
	{

  current_buffer = scale_sample_data [flip] [scale_note_type] [i];
  init_waveform(current_buffer, buffer_length);

  add_waveform_sine(current_buffer, buffer_length, base_freq * scale_ratio [0] [i], 0, 1);

  apply_adsr(current_buffer, buffer_length,
												 50, // attack length
												 200, 0.5, // decay length, target amplification after decay
												 3500); // release length (counted back from end

 amplify(current_buffer, buffer_length, 0.1);
  scale_sample [scale_note_type] [i] = finish_sample(current_buffer, buffer_length); // finish_sample exits on failure to create sample

	}


 scale_note_type = 1;

	for (i = 0; i < SCALE_TONES; i ++)
	{

  current_buffer = scale_sample_data [flip] [scale_note_type] [i];
  init_waveform(current_buffer, buffer_length);

  add_waveform_sine(current_buffer, buffer_length, base_freq * scale_ratio [0] [i] * 2, 0, 1);


/ *
   init_waveform(temp_buffer, buffer_length);
   add_waveform_sine(temp_buffer, buffer_length, base_freq * scale_ratio [0] [i] * 4, 0, 0.4);

   add_waveforms(current_buffer, temp_buffer, buffer_length);
* /

   init_waveform(temp_buffer, buffer_length);
   add_waveform_sine(temp_buffer, buffer_length, base_freq * scale_ratio [0] [i], 0, 0.4);

   add_waveforms(current_buffer, temp_buffer, buffer_length);



  apply_adsr(current_buffer, buffer_length,
												 50, // attack length
												 200, 0.5, // decay length, target amplification after decay
												 3500); // release length (counted back from end

 amplify(current_buffer, buffer_length, 0.1);
  scale_sample [scale_note_type] [i] = finish_sample(current_buffer, buffer_length); // finish_sample exits on failure to create sample

	}

*/

}


static void sthread_play_scale_sample(int instrument_flip, int instrument, int play_note, int vol, int pan)
{
//fpr("\n play sti %i note %i", scale_type_index, play_note);
 al_play_sample(scale_sample [instrument_flip] [instrument] [play_note], vol * 0.01 * sound_config.music_volume, pan * 0.01, 1.0, ALLEGRO_PLAYMODE_ONCE, NULL);

}



#ifdef OLD_MUSIC

//#define MBOX_W 11
//#define MBOX_H 11

// TRACKS is the number of simultaneous tracks:
#define TRACKS 4


struct mustatestruct
{
 int piece; // which piece is being played
 int piece_pos; // position in the piece command list
 int beat; // time position in bar
 int track_playing [TRACKS]; // which bar the track is playing
 int track_pos [TRACKS]; // data position in bar

// scales not currently used
 int track_scale [TRACKS]; // this is which scale the track is using
 int base_track_scale [TRACKS]; // this is which scale the track will return to after track is finished

 int track_note_offset [TRACKS];

 int note_list [TRACKS] [NOTES]; // this relates a note value to an index in the msample array.

 int track_length; // length of each track

};

#define PIECE_LENGTH 64

enum
{
PIECE_BASIC,
PIECES
};

struct piece_elementstruct
{
 int piece_command;
 int value1;
 int value2;
 int value3;
};

// A piece is a list of commands:
enum
{
PC_END,
PC_PLAY_TRACK, // values: track index, bar index
PC_MUTE, // values: track index
PC_WAIT, // values: bars to wait (not yet implemented)
PC_SCALE, // values: track index, new scale
PC_BASE_SCALE, // values: track index, new base scale
PC_BAR_LENGTH, // values: new bar length
PC_SET_NOTE, // values: track index, NOTE index, msample index
PC_SET_TRACK_NOTE_OFFSET, // values: track_index, offset
};

#define BAR_VALUES 4
#define BAR_LENGTH 16

//A bar is also a list of commands
enum
{
BC_END,
BC_PLAY, // plays sample value [0], tone value [1], time value [2], volume offset value [3]
};

struct barstruct
{
 int bar_command;
 int value [BAR_VALUES];
};

enum
{
BAR_EMPTY,
BAR_BASIC_BACK,
BAR_BASIC_UP,
BAR_BASIC_UP2,
BAR_BASIC_UP3,
BAR_BASIC_UP4,
BAR_BASIC_UP5,
BAR_BASIC_DOWN,

BARS
};

struct barstruct bar [BARS] [BAR_LENGTH] =
{
 {
  {BC_END},
 }, // BAR_EMPTY
 {
  {BC_PLAY, {NOTE_1, 1, 0}},
  {BC_END},
 }, // BAR_BASIC_BACK
 {
  {BC_PLAY, {NOTE_0, TONE_2C, 0, 3}},
  {BC_PLAY, {NOTE_0, TONE_1G, 2}},
  {BC_PLAY, {NOTE_0, TONE_1F, 3, 3}},
  {BC_PLAY, {NOTE_0, TONE_2C, 5}},
  {BC_PLAY, {NOTE_0, TONE_1E, 7, 3}},
  {BC_PLAY, {NOTE_0, TONE_2E, 9}},
  {BC_PLAY, {NOTE_0, TONE_1DS, 11, 3}},
  {BC_PLAY, {NOTE_0, TONE_1A, 13}},
  {BC_END},
 }, // BAR_BASIC_UP
 {
  {BC_PLAY, {NOTE_0, TONE_2C, 0, 3}},
  {BC_PLAY, {NOTE_0, TONE_1G, 2}},
  {BC_PLAY, {NOTE_0, TONE_1F, 3, 3}},
  {BC_PLAY, {NOTE_0, TONE_2C, 5}},
  {BC_PLAY, {NOTE_0, TONE_1E, 7, 3}},
  {BC_PLAY, {NOTE_0, TONE_2E, 9}},
  {BC_PLAY, {NOTE_0, TONE_2DS, 11, 3}},
  {BC_PLAY, {NOTE_0, TONE_2D, 13}},
  {BC_END},
 }, // BAR_BASIC_UP2
 {
  {BC_PLAY, {NOTE_0, TONE_2C, 0, 3}},
  {BC_PLAY, {NOTE_0, TONE_1G, 2}},
  {BC_PLAY, {NOTE_0, TONE_1DS, 3, 3}},
  {BC_PLAY, {NOTE_0, TONE_2C, 5}},
  {BC_PLAY, {NOTE_0, TONE_1E, 7, 3}},
  {BC_PLAY, {NOTE_0, TONE_2E, 9}},
  {BC_PLAY, {NOTE_0, TONE_1DS, 11, 3}},
  {BC_PLAY, {NOTE_0, TONE_1AS, 13}},
  {BC_END},
 }, // BAR_BASIC_UP3
 {
  {BC_PLAY, {NOTE_0, TONE_3C, 0}},
  {BC_PLAY, {NOTE_0, TONE_3DS, 2}},
  {BC_PLAY, {NOTE_0, TONE_3F, 3}},
  {BC_PLAY, {NOTE_0, TONE_3G, 5}},
  {BC_END},
 }, // BAR_BASIC_UP4
 {
  {BC_PLAY, {NOTE_0, TONE_3C, 0}},
  {BC_PLAY, {NOTE_0, TONE_2A, 2}},
  {BC_PLAY, {NOTE_0, TONE_2G, 3}},
  {BC_PLAY, {NOTE_0, TONE_3G, 5}},
  {BC_END},
 }, // BAR_BASIC_UP5
 {
  {BC_PLAY, {NOTE_1, TONE_4C, 0}},
  {BC_PLAY, {NOTE_1, TONE_3C, 2}},
  {BC_PLAY, {NOTE_1, TONE_3A, 3}},
  {BC_PLAY, {NOTE_1, TONE_3F, 5}},
  {BC_END},
 }, // BAR_BASIC_DOWN


};

struct piece_elementstruct piece [PIECES] [PIECE_LENGTH] =
{

// PIECE_BASIC
{
// {PC_PLAY_TRACK, 0, BAR_BASIC_BACK},
 {PC_BAR_LENGTH, 1},
 {PC_SET_NOTE, 0, NOTE_0, MSAMPLE_NOTE},
 {PC_SET_NOTE, 0, NOTE_1, MSAMPLE_NOTE},
 {PC_SET_NOTE, 1, NOTE_0, MSAMPLE_NOTE},
 {PC_SET_NOTE, 1, NOTE_1, MSAMPLE_NOTE},
 {PC_WAIT}, // always need to start with wait
 {PC_BAR_LENGTH, 15},
 {PC_SET_TRACK_NOTE_OFFSET, 0, 15},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP2},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP3},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP2},
 {PC_WAIT},
/* {PC_PLAY_TRACK, 0, BAR_BASIC_UP2},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP3},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP4},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP2},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP5},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_PLAY_TRACK, 1, BAR_BASIC_DOWN},
 {PC_WAIT},
 {PC_PLAY_TRACK, 0, BAR_BASIC_UP},
 {PC_PLAY_TRACK, 1, BAR_BASIC_DOWN},
 {PC_WAIT},*/
 {PC_END},
}, // end PIECE_BASIC

};

struct mustatestruct mustate;

enum
{
SCALE_BASIC,
SCALE_MINOR,
SCALE_MAJOR,
SCALE_TYPES
};

#define SCALE_SIZE 11

int scale [SCALE_TYPES] [SCALE_SIZE] =
{

 { // SCALE_BASIC (pentatonic)
  TONE_2C,
  TONE_2D,
  TONE_2E,
  TONE_2G,
  TONE_2A,
  TONE_3C,
  TONE_3D,
  TONE_3E,
  TONE_3G,
  TONE_2C,
  TONE_2D
 },

//C, D, E, G, and A.
 { // SCALE_MINOR
  TONE_2C,
  TONE_2DS,
  TONE_2F,
  TONE_2G,
  TONE_2AS,
  TONE_3C,
  TONE_3DS,
  TONE_3F,
  TONE_3G,
  TONE_1AS,
  TONE_2C
 },
  { // SCALE_MAJOR
  TONE_2C,
  TONE_2D,
  TONE_2E,
  TONE_2G,
  TONE_2A,
  TONE_3C,
  TONE_3D,
  TONE_3E,
  TONE_3G,
  TONE_3A,
  TONE_1A
 }
};




// This function is called whenever a new piece is started.
void sthread_init_mustate(int play_piece)
{
 int i, j;

 mustate.piece = play_piece;
 mustate.piece_pos = 0;
 mustate.beat = 0;
 mustate.track_length = BAR_LENGTH; // should be reset by piece
 for (i = 0; i < TRACKS; i ++)
 {
  mustate.track_playing [i] = BAR_EMPTY;
  mustate.track_playing [i] = 0;
  mustate.track_pos [i] = 0;
  mustate.track_scale [i] = SCALE_BASIC;
  mustate.base_track_scale [i] = SCALE_BASIC;
  mustate.track_note_offset [i] = 0;
  for (j = 0; j < NOTES; j ++)
		{
			mustate.note_list [i] [j] = MSAMPLE_NOTE;
		}
 }


}

// returns 1 if piece still playing, or 0 if it's finished
int sthread_play_current_piece(void)
{

// Not currently used because I haven't written any music for it yet. Will be returned to service at some point.
 // should be static

 int i;

// First we go through the tracks:
 for (i = 0; i < TRACKS; i ++)
 {
  while(TRUE)
  {

   switch(bar [mustate.track_playing [i]] [mustate.track_pos [i]].bar_command)
   {
    case BC_PLAY:
     if (mustate.beat >= bar [mustate.track_playing [i]] [mustate.track_pos [i]].value [2])
     {
      sthread_play_msample(mustate.note_list [i] [bar [mustate.track_playing [i]] [mustate.track_pos [i]].value [0]], // Note type
																							    bar [mustate.track_playing [i]] [mustate.track_pos [i]].value [1] + mustate.track_note_offset [i], // pitch
																							    bar [mustate.track_playing [i]] [mustate.track_pos [i]].value [3], 0);
//                        scale [mustate.track_scale [i]] [bar [mustate.track_playing [i]] [mustate.track_pos [i]].value [1]],
//                        1);
      break; // will increment mustate.track_pos [i] (after this loop) then continue the loop
     }
     goto finished_track; // not ready to play yet
    case BC_END:
     goto finished_track;
// other BC types might just break instead of exiting the loop.
   }
   mustate.track_pos [i]++;
  };
  finished_track:
   continue;
 }


 mustate.beat++;

// now we check the piece commands:
// int value1, value2;

 while(TRUE)
 {
// these continue until they reach PC_WAIT or PC_END
  switch(piece [mustate.piece] [mustate.piece_pos].piece_command)
  {
   case PC_WAIT:
    if (mustate.beat >= mustate.track_length)
    {
     mustate.beat = 0;
     mustate.piece_pos++;
     continue;
    }
    return 1;
   case PC_PLAY_TRACK: // values: track index, bar index
    mustate.track_playing [piece [mustate.piece] [mustate.piece_pos].value1] = piece [mustate.piece] [mustate.piece_pos].value2;
    mustate.track_pos [piece [mustate.piece] [mustate.piece_pos].value1] = 0;
    mustate.track_scale [piece [mustate.piece] [mustate.piece_pos].value1] = mustate.base_track_scale [piece [mustate.piece] [mustate.piece_pos].value1];
    mustate.piece_pos++;
    break;
   case PC_MUTE:  // values: track index
    mustate.track_playing [piece [mustate.piece] [mustate.piece_pos].value1] = BAR_EMPTY;
    mustate.piece_pos++;
    break;
   case PC_BAR_LENGTH: // values: bar length
    mustate.track_length = piece [mustate.piece] [mustate.piece_pos].value1;
    mustate.piece_pos++;
    break;
   case PC_SCALE: // values: track index, scale index
    mustate.track_scale [piece [mustate.piece] [mustate.piece_pos].value1] = piece [mustate.piece] [mustate.piece_pos].value2;
    mustate.piece_pos++;
    break;
   case PC_BASE_SCALE: // values: track index, scale index
    mustate.base_track_scale [piece [mustate.piece] [mustate.piece_pos].value1] = piece [mustate.piece] [mustate.piece_pos].value2;
    mustate.piece_pos++;
    break;
   case PC_SET_NOTE: // values: track index, NOTE index, msample index
				mustate.note_list [piece [mustate.piece] [mustate.piece_pos].value1] [piece [mustate.piece] [mustate.piece_pos].value2] = piece [mustate.piece] [mustate.piece_pos].value3;
    mustate.piece_pos++;
				break;
			case PC_SET_TRACK_NOTE_OFFSET:
    mustate.track_note_offset [piece [mustate.piece] [mustate.piece_pos].value1] = piece [mustate.piece] [mustate.piece_pos].value2;
    mustate.piece_pos++;
				break;
   case PC_END:
    return 0;
// remember mustate.piece_pos++; if needed!
  }
 };

  return 0; // should never reach here

}


static void sthread_play_msample(int msample_index, int play_tone, int vol, int pan)
{

 al_play_sample(msample [msample_index], vol * 0.01, pan * 0.01, tone [play_tone + 8], ALLEGRO_PLAYMODE_ONCE, NULL);
//			fpr(" vol %f pan %f", vol * 0.01, pan * 0.01);


// al_play_sample(msample [msample_index], 0.5 + ((float) vol * 0.1), 0, tone [play_tone], ALLEGRO_PLAYMODE_ONCE, NULL);

// sthread_add_mbox_echo(mb, note_type, play_tone);
// could pan to left/right depending on x position?

}

#endif

