/* 
 *  VAR.C - part of the EGG system.
 *
 *  Variable getting/setting routines.
 *
 *  By Shawn Hargreaves.
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "egg.h"



/* list of default settings */
typedef struct DEFAULT_VARIABLE
{
   char *name;
   double value;
} DEFAULT_VARIABLE;



static DEFAULT_VARIABLE default_variable[] =
{
   { "x",      0.0   },
   { "y",      0.0   },
   { "z",      0.0   },
   { "size",   1.0   },
   { "focus",  1.0   },
   { "aa",     0.0   },
   { "r",      255.0 },
   { "g",      255.0 },
   { "b",      255.0 },
   { "a",      255.0 },
   { "add",    1.0   },
   { "mul",    0.0   },
   { "sub",    0.0   },
   { "wrapx",  0.0   },
   { "wrapy",  0.0   },
   { "_scale", 1.0   },
   { "_dist",  100.0 },
   { "_fov",   1.0   }
};



/* find_variable:
 *  Finds a variable in the variable tree.
 */
static EGG_VARIABLE *find_variable(EGG_VARIABLE *node, char *name)
{
   char *ptr = name;

   while (*ptr) {
      if (!node) break;
      node = node->characters[tolower(*ptr)-'_'];
      ++ptr;
   }
   return node;
}



/* insert_variable
 *  Inserts a new variable in the variable tree.
 */
static void insert_egg_variable(EGG_VARIABLE **node, char *name, double value)
{
   char *ptr = name;
   int n;

   while (1) {
      if (!*node) {
         *node = (EGG_VARIABLE*)malloc(sizeof(EGG_VARIABLE));
         for (n=0; n<sizeof((*node)->characters)/sizeof((*node)->characters[0]); ++n) {
            (*node)->characters[n] = NULL;
         }
         (*node)->val = 0.0;
      }
      if (!*ptr) break;
      node = &(*node)->characters[tolower(*ptr)-'_'];
      ++ptr;
   }

   (*node)->val = value;
}



/* seed_egg_variables:
 *  Seeds a variable tree with the default variables
 */
void seed_egg_variables(EGG_VARIABLE **var)
{
   int i;
   for (i=0; i<sizeof(default_variable)/sizeof(DEFAULT_VARIABLE); i++) {
      insert_egg_variable(var, default_variable[i].name, default_variable[i].value);
   }
}



/* free_egg_variables:
 *  Frees a variable tree.
 */
void free_egg_variables(EGG_VARIABLE *node)
{
   int n;
   if (!node) return;
   for (n=0; n<sizeof(node->characters)/sizeof(node->characters[0]); ++n) {
      free_egg_variables(node->characters[n]);
   }
   free(node);
}



/* get_egg_variable:
 *  Accesses a particle variable.
 */
double get_egg_variable(EGG_PARTICLE *part, EGG_PARTICLE *other, EGG *egg, char *name)
{
   EGG_VARIABLE *var;

   if (name[0] == '_') {
      if (other) {
	 var = other->var;
	 name++;
      }
      else if (egg)
	 var = egg->var;
      else
	 var = NULL;
   }
   else {
      if (part)
	 var = part->var;
      else
	 var = NULL;
   }

   var = find_variable(var, name);
   if (var) return var->val;

   if ((part) && (strcmp(name, "id") == 0))
      return part->num;

   if ((part) && (strcmp(name, "age") == 0))
      return part->age;

   if ((egg) && (strcmp(name, "_frame") == 0))
      return egg->frame;

   if ((egg) && (strcmp(name, "_count") == 0))
      return egg->part_count;

   return 0.0;
}



/* set_egg_variable:
 *  Sets a particle variable.
 */
void set_egg_variable(EGG_PARTICLE *part, EGG_PARTICLE *other, EGG *egg, char *name, double value)
{
   EGG_VARIABLE **pvar;

   if (name[0] == '_') {
      if (other) {
	 pvar = &other->var;
	 name++;
      }
      else {
	 if (!egg)
	    return;

	 pvar = &egg->var; 
      }
   }
   else {
      if (!part)
	 return;

      pvar = &part->var;
   }

   insert_egg_variable(pvar, name, value);
}
