/* 
 *  RENDER.C - part of the EGG system.
 *
 *  Renders the current animation frame into an Allegro bitmap.
 *
 *  By Shawn Hargreaves.
 */


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <allegro.h>

#include "egg.h"



/* put_add_pixel:
 *  Draws a pixel onto the output buffer using additive blending.
 */
static void put_add_pixel(double *image, double r, double g, double b, double a)
{
   /* additive color */
   image[0] += r*a;
   image[1] += g*a;
   image[2] += b*a;
   image[3] += 256.0*a;
}



/* put_mul_pixel:
 *  Draws a pixel onto the output buffer using multiplicative blending.
 */
static void put_mul_pixel(double *image, double r, double g, double b, double a)
{
   /* multiplicative color */
   image[0] *= r*a;
   image[1] *= g*a;
   image[2] *= b*a;
   image[3] *= 256.0*a;
}



/* put_sub_pixel:
 *  Draws a pixel onto the output buffer using substractive blending.
 */
static void put_sub_pixel(double *image, double r, double g, double b, double a)
{
   /* subtractive color */
   image[0] -= r*a;
   image[1] -= g*a;
   image[2] -= b*a;
   image[3] -= 256.0*a;
}



/* put_int_pixel:
 *  Draws a pixel onto the output buffer using alpha blending.
 */
static void put_int_pixel(double *image, double r, double g, double b, double a)
{
   /* interpolated color blending */
   image[0] = image[0]*(1.0-a) + r*a;
   image[1] = image[1]*(1.0-a) + g*a;
   image[2] = image[2]*(1.0-a) + b*a;
   image[3] = image[3] + 256.0*a;
}



/* draw_particle:
 *  This is where it all happens...
 */
static void draw_particle(EGG_PARTICLE *part, EGG *egg, double *image, int w, int h, int *min_y, int *max_y)
{
   double x = get_egg_variable(part, NULL, egg, "x");
   double y = get_egg_variable(part, NULL, egg, "y");
   double z = get_egg_variable(part, NULL, egg, "z");

   double size = get_egg_variable(part, NULL, egg, "size");
   double focus = get_egg_variable(part, NULL, egg, "focus");
   double aa = get_egg_variable(part, NULL, egg, "aa");

   double wrapx = get_egg_variable(part, NULL, egg, "wrapx");
   double wrapy = get_egg_variable(part, NULL, egg, "wrapy");

   double r = get_egg_variable(part, NULL, egg, "r");
   double g = get_egg_variable(part, NULL, egg, "g");
   double b = get_egg_variable(part, NULL, egg, "b");
   double a = get_egg_variable(part, NULL, egg, "a");

   double scale = get_egg_variable(part, NULL, egg, "_scale");
   double dist = get_egg_variable(part, NULL, egg, "_dist");
   double fov = get_egg_variable(part, NULL, egg, "_fov");

   int add = get_egg_variable(part, NULL, egg, "add");
   int mul = get_egg_variable(part, NULL, egg, "mul");
   int sub = get_egg_variable(part, NULL, egg, "sub");

   int drawn = FALSE;

   double dx, dy, d, cvg, proj_scale, inv2size;
   double xx, yy;
   int x0, y0, x1, y1;
   int xint, yint;
   int width_and = 0, height_and = 0;
   void (*drawer)(double*, double, double, double, double);

   /* 3d projection */
   z = (z + dist) / 100.0 * fov;
   if (z < 0.0001)
      return;
   proj_scale = scale / z;

   x = (x * proj_scale) + w/2;
   y = (y * proj_scale) + h/2;

   size = size * proj_scale;
   inv2size = 2.0 / size;

   a /= 255.0;

   if (focus)
      focus = 1.0 / focus;

   /* choose the drawing function to use */
   if (add) {
      drawer = &put_add_pixel;
   }
   else if (mul) {
      drawer = &put_mul_pixel;
   }
   else if (sub) {
      drawer = &put_sub_pixel;
   }
   else {
      drawer = &put_int_pixel;
   }

   /* find out the limits of the drawing area */
   x0 = x-size/2.0; if (x0<0 && !wrapx) x0 = 0;
   y0 = y-size/2.0; if (y0<0 && !wrapy) y0 = 0;
   x1 = x+size/2.0; if (x1>=w && !wrapx) x1 = w-1;
   y1 = y+size/2.0; if (y1>=h && !wrapy) y1 = h-1;

   /* division is expensive in the inner loop, try to avoid it */
   /* the testing version is three times faster than the unconditional one */
   if (!(w&(w-1))) width_and = w - 1;
   if (!(h&(h-1))) height_and = h - 1;

   /* scan through the blob */
   for (yint = y0, yy = y0; yint <= y1; yy += 1.0, ++yint) {
      for (xint = x0, xx = x0; xint <= x1; xx += 1.0, ++xint) {
         dx = fabs(xx-x);
         dy = fabs(yy-y);

         /* approximate a circle as an octagon */
         if (dx > dy)
            d = dy/2.0 + dx;
         else
            d = dx/2.0 + dy;

         /* coverage calculation */
         cvg = 1.0-d * inv2size;

         /* plot the pixel */
         if (cvg > 0) {
            int x_image, y_image;

            if (!aa)
               cvg = a;
            else if (focus != 1.0)
               cvg = pow(cvg*a, focus);
            else
               cvg *= a;

            x_image = xint+256*w;
            y_image = yint+256*h;
            if (width_and) x_image &= width_and; else x_image %= w;
            if (height_and) y_image &= height_and; else y_image %= h;
            (*drawer)(image+(x_image+y_image*w)*4, r, g, b, cvg);
            if (y_image < *min_y) *min_y = y_image;
            if (y_image > *max_y) *max_y = y_image;

            drawn = TRUE;
         }
      }
   }

   if ((!drawn) && (!aa)) {
      xx = (int)(x+0.5);
      yy = (int)(y+0.5);

      if ((xx >= 0) && (xx < w) && (yy >= 0) && (yy < h)) {
         xint=((int)(xx+256*w))%w;
         yint=((int)(yy+256*h))%h;
         (*drawer)(image+(xint+yint*w)*4, r, g, b, a);
            if (yint < *min_y) *min_y = yint;
            if (yint > *max_y) *max_y = yint;
      }
   }
}



/* part_cmp:
 *  qsort() callback for comparing particle depths.
 */
static int part_cmp(const void *e1, const void *e2)
{
   EGG_PARTICLE **p1 = (EGG_PARTICLE **)e1;
   EGG_PARTICLE **p2 = (EGG_PARTICLE **)e2;

   double d1 = get_egg_variable(*p1, NULL, NULL, "z");
   double d2 = get_egg_variable(*p2, NULL, NULL, "z");

   if (d1 > d2)
      return -1;
   else if (d1 < d2)
      return 1;
   else
      return 0;
}



#define makeacol8(r, g, b, a) makecol8(r, g, b)
#define makeacol15(r, g, b, a) makecol15(r, g, b)
#define makeacol16(r, g, b, a) makecol16(r, g, b)
#define makeacol24(r, g, b, a) makecol24(r, g, b)

 

/* create_bitmap_from_image:
 */
static void create_bitmap_from_image(BITMAP *bitmap, double *image, int y1, int y2)
{
   int x, y;
   int w = bitmap->w, h = bitmap->h;

   image += y1 * w * 4;

   rectfill(bitmap, 0, 0, w, y1-1, bitmap_mask_color(bitmap));
   rectfill(bitmap, 0, y2+1, w, h-1, bitmap_mask_color(bitmap));

#define CONVERT(bpp, size, alpha_component_code)            \
   bmp_select(bitmap);                                      \
   for (y=y1; y<=y2; ++y) {                                 \
      int line = bmp_write_line(bitmap, y);                 \
      for (x=0; x<w; ++x) {                                 \
         int r = 255, g = 0, b = 255, a = 0;                \
         double alpha = image[3];                           \
         if (alpha > 0.0) {                                 \
            /* conversions and clamping take time */        \
            r = image[0]; r = MID(0, r, 255);               \
            g = image[1]; g = MID(0, g, 255);               \
            b = image[2]; b = MID(0, b, 255);               \
            alpha_component_code;                           \
         };                                                 \
                                                            \
         bmp_write##bpp(line, makeacol##bpp(r, g, b, a));   \
         line += size;                                      \
                                                            \
         image += 4;                                        \
      }                                                     \
   }                                                        \
   bmp_unwrite_line(bitmap);

   switch (bitmap_color_depth(bitmap)) {
     case 8: CONVERT(8, 1, (void)a;); break;
     case 15: CONVERT(15, 2, (void)a;); break;
     case 16: CONVERT(16, 2, (void)a;); break;
     case 24: CONVERT(24, 3, (void)a;); break;
     case 32: CONVERT(32, 4, a = alpha; a = MID(0, a, 255)); break;
   }
}



/* lay_egg:
 *  Graphical output routine.
 */
void lay_egg(EGG *egg, BITMAP **bmp, int nplanes, double *planes)
{
   lay_egg_buffer(egg, bmp, nplanes, planes, NULL);
}



/* lay_egg_buffer:
 *  Graphical output routine.
 */
void lay_egg_buffer(EGG *egg, BITMAP **bmp, int nplanes, double *planes, double *buffer)
{
   EGG_PARTICLE *part, **part_list=NULL;
   int i;
   int plane;
   static int min_y = -1;
   static int max_y = -1;
   static double* image = NULL;

   image = buffer;
   if (!buffer || image != buffer || min_y == -1) min_y = 0;
   if (!buffer || image != buffer || max_y == -1) max_y = bmp[0]->h-1;
   if (!buffer) image = malloc(sizeof(double) * bmp[0]->w * bmp[0]->h * 4);

   if (egg->part_count > 0) {
      part_list = malloc(egg->part_count * sizeof(EGG_PARTICLE *));

      part = egg->part;
      for (i=0; i<egg->part_count; i++) {
         part_list[i] = part;
         part = part->next;
      }

      if (egg->part_count > 1)
         qsort(part_list, egg->part_count, sizeof(EGG_PARTICLE *), part_cmp);
   }

   for (plane=0; plane<nplanes-1; plane++) {
      int image_size;
      image_size = bmp[plane]->w * (max_y+1) * 4;
      for (i=min_y*bmp[plane]->w*4; i<image_size; i+=4) {
         image[i] = 0;
         image[i+1] = 0;
         image[i+2] = 0;
         image[i+3] = 0;
      }

      min_y = bmp[0]->h-1;
      max_y = 0;

      if (egg->part_count > 0) {
         for (i=0; i<egg->part_count; i++) {
            double z = get_egg_variable(part_list[i], NULL, egg, "z");
            if (planes[plane]<=z && z<planes[plane+1]) {
               draw_particle(part_list[i], egg, image,
                             bmp[plane]->w, bmp[plane]->h, &min_y, &max_y);
            }
         }
      }

      solid_mode();

      create_bitmap_from_image(bmp[plane], image, min_y, max_y);
   }

   if (egg->part_count > 0) free(part_list);
   if (!buffer) free(image);
}


