/*--- Filename: "ca_vbar.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The vertical bar color attractor.
  
*/

#include "ca_vbar.hpp"

#if !defined(__DB_ca_vbar_BODY_INCLUDED)
#define __DB_ca_vbar_BODY_INCLUDED

// default ctor just sets default values
CA_VBAR::CA_VBAR()
{
  EXP = 1.0;
  X = 0.5;
  RANGE = 0.5;
  TYPE = CAT_VBAR; // remember to override in derived types
}

// other ctor sets user values
CA_VBAR::CA_VBAR(CA_MODE mode, int r, int g, int b, double colint, double exp,
                 double x, double range)
{
  CA_GLOBAL::CA_GLOBAL(mode,r,g,b,colint);
  EXP = exp;
  X = x;
  RANGE = range;
  TYPE = CAT_VBAR; // remember to override in derived types
}

CA_VBAR::~CA_VBAR()
{
  CA_GLOBAL::delete_line_buf();
}

// calculate distance to x in pixels
// (is protected virtual)
double CA_VBAR::distance(int x, int y)
{
  return fabs(_pix_x - x); // y is ignored for vertical bar
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_VBAR::update_absolutes(int width, int height,
                               lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;

  _pix_x = width * X;
  _pix_range = width * RANGE;

  return true;
}

// The *only* function ever called by the "render_gradient" method from..
// ..GRADIENT_GEN class (directly modifies "crgb"; applies CA's influence)
// (is virtual)
void CA_VBAR::modify_crgb(int x, int y, customRGB<double>& crgb)
{
  double d = distance(x,y);

  if(d<=_pix_range)
  {
    double calc = (1.0 - pow(d / _pix_range,EXP))*COLINT; 
    switch(MODE)
    {
      case CAM_ATTRACT:
        crgb.add_cr(calc*R);
        crgb.add_cg(calc*G);
        crgb.add_cb(calc*B);
      break;
      case CAM_ABSORB:
        crgb.add_cr(-calc*R);
        crgb.add_cg(-calc*G);
        crgb.add_cb(-calc*B);
      break;
      default:;// to prevent ".. not handled in switch" warning
    }
  }
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_VBAR::update_vars_from_line()
{
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  _render_var_changed = true;

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_VBAR::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP
          << ", " << X
          << ", " << RANGE;
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_VBAR::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << ",0)";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_vbar_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
