/*--- Filename: "gradient_gen.hpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The gradient generator class.

  Porting notes: (this is the ALLEGRO specific version)
  ----------------------------------------------------------------------------
  This should be easily portable to systems 
  where an 'int' has -> at least <- 32bits
  and you should only need to change the class BMP_T.
  ----------------------------------------------------------------------------
  The used "bitmap type"(see 'class BMP_T' in files "bmp_t.hpp, bmp_t.cpp")
  has to have a "get_pixel32(x,y)" method that returns the color as an integer
  in the format "0x00RRGGBB". And a "put_pixel32(x,y,int)" method that will
  write the color(same format) to the bitmap.
  The bitmap type also has to supply a "get_width()" and a "get_height()" 
  method and an "is_valid()" method that returns true, if the bitmap is 
  correctly initialized and usable.
  
  
  KNOWN BUGS(beta):
  -----------------
  (closed, ELLIPSEP promoted to ELLIPSE)
      Setting the RANGE attribute of the ELLIPSE (not ELLIPSEP) attractor
      did not have any effect.
  (open, tolerated) Polynomials sometimes display wrong
*/

#include "bmp_t.hpp"     // the used bitmap type
#include "shared.hpp"
#include "utf_eight.hpp" // helper functions for encoding,decoding UTF-8

// Attractor Types
#include "ca_global.hpp"
#include "ca_point.hpp"
#include "ca_pointe.hpp"
#include "ca_hbar.hpp"
#include "ca_vbar.hpp"
#include "ca_bar.hpp"
#include "ca_polynomial.hpp"
#include "ca_circle.hpp"
#include "ca_circlef.hpp"
#include "ca_ellipse.hpp" // only left in from beta for compatibility
#include "ca_ellipsep.hpp" // truly this will be seen as type ELLIPSE
// end of Attractor Types

#include<stdexcept>
using namespace std;
#include <vector>
using namespace std;
#include <iterator>
using namespace std;
#include <ctime>

#if !defined(__DB_gradient_gen_HEAD_INCLUDED)
#define __DB_gradient_gen_HEAD_INCLUDED

class GRADIENT_GEN
{
  private:
    wstring AUTHOR;
    wstring TITLE;
    int DEF_W; // specify width and height, this gradient was meant for
    int DEF_H;

    vector<CA_GLOBAL *> CA_VEC;  // holds the color attractors
    double last_render_time; // holds the most recent render time in seconds

    void reset();

  public:
    GRADIENT_GEN();
    ~GRADIENT_GEN();

  // Set and Get Methods
    void set_author(wstring author);
    wstring get_author();
    void set_title(wstring title);
    wstring get_title();

    void set_w(int w);
    int get_w();

    void set_h(int h);
    int get_h();

    double get_last_render_time();
    int get_attractor_count(); // -1 if empty, number of attractors otherwise
  
  // Color Attractor Mangagement
    int add_ca(int before, string params); 
      // add color attractor by string description before position "before"
      // (before: 0 to front append, -1 to rear append)
    int add_ca(string params);
             // add color attractor by string description (always rear append)
    int rem_ca(int which); // remove a certain color attractor
    string get_as_text(int which); // get string description of an attractor

    int set_from_text(int which, string params); // changes an attractor
    /* (Use this only, if you want to change the type of an attractor,
        because it will remove "which" and insert a new attractor
        to that position. This is unnecessary, if you just want to change
        some other property than the type.) */

    CA_GLOBAL* get_pca(int which); // get the pointer to an attractor
    /* (use and recast this to change any property of an attractor,
        except the type(compare: set_from_text)) */

    int switch_ca(int a, int b); // attempts to switch the attractors
    /* will do nothing and return -1 if either attractor number a or b
       does not exist */

    void clear(); // delete all color attractors


  // Render Methods
   /* the parameter 'target_int' specifies the intensity in which the original
       color on the render target will be mixed in (can also be negative) */
    
    /* render to a rectangular area on the target starting at (x,y) with
       width w and height h */
    int render(BMP_T &target, int x, int y, int w, int h, double target_int);
    /* same but with a callback function that is called every 'call_skip'th
       millisecond and it is passed a parameter 'pixels_to_go', describing
       how many pixels still have to be processed.
       (the callback function can return false, to signal that rendering
        should be cancelled and must return true otherwise)
       (lcall is another callback function that is passed on to the attractors
        before rendering, see "ca_global.hpp") */
    int render(BMP_T &target, int x, int y, int w, int h, double target_int,
               bool (*render_callback)(unsigned long pixels_to_go), 
               double call_skip, lut_callback lcall, double lcall_skip);
    
    // render to full target area
    int render(BMP_T &target, double target_int);
    // same with callback function, as described above
    int render(BMP_T &target, double target_int,
               bool (*render_callback)(unsigned long pixels_to_go), 
               double call_skip, lut_callback lcall, double lcall_skip);

  // Saving and Loading
    int save(string filename); // will overwrite existing files
    int load(string filename);

  // Export As Code
    int export_code(string instancename, string filename);
};


#endif // #if !defined(__DB_gradient_gen_HEAD_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
