/*--- Filename: "dycogrgen.cpp" ---
  
  --- Projectname: DB's Dynamic Color Gradient Generator GUI ---
  (Targetsystem: Crossplatform(powered by Allegro))
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content: MAIN PROGRAMME (Version 'beta2')
  The Allegro-Dependant GUI application to my gradient generator.
  (Its all "C style", because Allegro is a C library.)

  SPECIAL FEATURES:
  -----------------
    * scrollable GUI
    * internationalization
        by using custom language files(UTF8 encoded Unicode)
    * high portability (ALLEGRO IS JUST MARVELLOUS!)

  DEPENDENCIES:
  -------------
  Allegro Library >= 4.20 (older versions will NOT work correctly)
  (http://alleg.sourceforge.net)

  GREETINGS:
  ----------
  Greetings fly out to my family, my friends and last but not least
    to all members of the Allegro Community. (http://www.allegro.cc)
*/

#include "dycogrgen.hpp"

// system specific WINDOWS code
#ifdef _WIN32
bool win_mouse_in_window()
{
  POINT mp;
  RECT wr;
  GetCursorPos(&mp);
  GetWindowRect(win_get_window(),&wr);
  if( (mp.x<wr.left)||(mp.x>wr.right)||(mp.y<wr.top)||(mp.y>wr.bottom) )
    return false;
  else
    return true;
}
#endif

// Vars to control external programme resources
#define __DBDCGGV__ "v.beta2 (18/07/2007)"
const string G_LANGUAGE_FILE_VERSION = "beta2";
const bool g_show_beta_warning = false;
const string G_DEFAULT_LANGUAGE_FILE = "./default.dbdcgg_lang";
//const string G_DEFAULT_LANGUAGE_FILE = "./deutsch.dbdcgg_lang";
const char G_DATAFILE_NAME[] = "./dcggdata.dat";
const char G_PROGRAMME_NAME[] = "Dynamic Color Gradient Generator (beta2)";
const char __BMP_EXT__[] = "bmp";
const char __PCX_EXT__[] = "pcx";
const char __TGA_EXT__[] = "tga";
const char __DCGG_EXT__[] = "dcgg";
const char __DCGG_LANG_EXT__[] = "dbdcgg_lang";
const char __CPP_EXT__[] = "cpp";

// Vars to control programme flow
int g_runlevel = RL_NOTHING_OK; // for controlled init,deinit
const double g_cb_msec = 250.0; // milliseconds between calling render callbacks

// Vars for the GFX Subsystem
int g_card = GFX_AUTODETECT_WINDOWED;
int g_w = 800;
int g_h = 600;
const int G_SAFE_W = 320; // do not change!
const int G_SAFE_H = 200; // do not change!
int g_bpp = 32;
volatile bool g_os_close_button_disabled = false;
volatile bool g_os_close_button_pressed = false;
bool g_os_close_button_ignored = false;
static PALETTE g_m_pal;//for 8bit mode / try to find hidden grayscale mode;-)
RGB_MAP g_rgb_map; // for color conversion in 8bit mode
BITMAP *g_back = NULL; // for the fixed size backbuffer to run the dialog on
BITMAP *g_msafe = NULL; // for saving the backbuffer content under the mouse
BITMAP *g_gradient_preview = NULL; // for the gradient preview bitmap
BITMAP *g_gradient_export = NULL; // used for exporting graphics files
GRADIENT_GEN g_work_gradient; // the gradient generator being edited
BMP_T g_bmp_preview; // the gradient will be rendered on the..
                      // ..allegro bitmap attached to it
int G_BACK_W = 800; // do not make smaller than 800
int G_BACK_H = 600; // do not make smaller than 600
const int G_BACK_BPP = 32;  // do not change!
#define G_SCROLL_PIX 20 // amount of pixels to scroll
#define G_SCROLL_AREA 20 // border in which the screen will scroll
#define G_SCROLL_MSEC 50 // scroll only every so many milliseconds
#define G_SCROLL_RIGHT 0x01 // some scroll direction bitflags
#define G_SCROLL_DOWN 0x02
#define G_SCROLL_LEFT 0x04
#define G_SCROLL_UP 0x08
int g_scroll_dir = 0; // bitmask with direction(s) to scroll to
volatile int g_scroll_x=0; // front buffer scroll position x
volatile int g_scroll_y=0; // front buffer scroll position y

// Vars for the GUI
#define G_REST_MSEC 16 // increase to give more cpu time to the OS
dbdcgg_strings g_all_text; // all standard strings go in here
dbdcgg_ext g_all_exts; // all extensions go in here
DATAFILE *g_all_data = NULL; // icons, bitmap buttons, etc. go in here
// Next four strings will be taken as addresses from the datafile
// (they are properties of the BITMAP_DBLOGO item)
edit_contents g_edit_field; // buffers reserved for input fields
int g_request_sub_dialog = 0; // controls which sub dialog are started
int g_usc_alert_caller = 0; // controls sub dialog after unsaved change alert
bool g_auto_recalculate = true; // controls recalculation of the preview
bool g_want_quit = false; // set when user requests quit
bool g_unsaved_changes = false; // is set when changes have occured
int g_gradient_changed = 0; // is set for max. one loop cycle to
                            // notify changes in attractor settings
int g_gradient_change_on_lostfocus = 0; // used to check in edit fields
          // if gradient needs change before the field loses focus
FONT *g_default_font = NULL; // used to save the default allegro font on init

// vars to be able to tell mouse movement and position(even if outside window)
/* (removed, because they rely on a portion of Allegro that isn't sys.indep.)
int g_abs_mx = 0;
int g_abs_my = 0;
int g_mickx = 0;
int g_micky = 0;
bool g_mouse_inside = false;
*/

#define G_FNAME_BUFSIZE 1024
char g_lang_fname[G_FNAME_BUFSIZE]; // some buffers to store file names
char g_bmp_fname[G_FNAME_BUFSIZE];
char g_pcx_fname[G_FNAME_BUFSIZE];
char g_tga_fname[G_FNAME_BUFSIZE];
char g_dcgg_fname[G_FNAME_BUFSIZE];
char g_dcgg_lang_fname[G_FNAME_BUFSIZE];
char g_cpp_fname[G_FNAME_BUFSIZE];
char g_prog_title[60*G_MAX_BYTE_PER_CHAR]; // buffer for programme name
char g_programme_auth[60*G_MAX_BYTE_PER_CHAR];
char g_programme_home[60*G_MAX_BYTE_PER_CHAR];
char g_programme_mail[60*G_MAX_BYTE_PER_CHAR];
char g_allegro_home[60*G_MAX_BYTE_PER_CHAR];

// buffer for edit field in the SUB_DIALOG_EDIT_COEFFICIENTS
// for the coefficients of the currently selected polynomial
double g_current_coefficients[G_MAX_COEFFICIENTS];

// Custom Messages
const int MSG_SCROLL_UP = MSG_USER;
const int MSG_SCROLL_DOWN = MSG_USER+1;
const int MSG_SCROLL_LEFT = MSG_USER+2;
const int MSG_SCROLL_RIGHT = MSG_USER+3;
const int MSG_UPDATE_FIELDS = MSG_USER+4;
const int MSG_GET_ATTRACTOR = MSG_USER+5;
const int MSG_MOVE_UP = MSG_USER+6; // for attractor moving in list
const int MSG_MOVE_DOWN = MSG_USER+7; // for attractor moving in list

char *g_ttxt = ".\0\0\0\0\0\0\0\0\0\0\0\0"; // dummy placeholder string
char *g_xtxt = "X\0\0\0\0\0\0\0\0\0\0\0\0"; // dummy e(x)it string
char *g_0txt = "0\0\0\0\0\0\0\0\0\0\0\0\0"; // dummy number string

// Changer IDs(adress used as dp3 field in gradient changing dialog objects)
const int G_CH_AUTHOR = MAIN_DIALOG_ID_AUTHOR_EDIT;
const int G_CH_TITLE = MAIN_DIALOG_ID_TITLE_EDIT;
const int G_CH_TYPE = MAIN_DIALOG_ID_TYPE_BUTTON;
const int G_CH_MODE = MAIN_DIALOG_ID_MODE_BUTTON;
const int G_CH_COLOR = MAIN_DIALOG_ID_COLOR_BUTTON;
const int G_CH_INTENSITY = MAIN_DIALOG_ID_INTENSITY_EDIT;
const int G_CH_EXPONENT = MAIN_DIALOG_ID_EXPONENT_EDIT;
const int G_CH_X = MAIN_DIALOG_ID_X_EDIT;
const int G_CH_Y = MAIN_DIALOG_ID_Y_EDIT;
const int G_CH_WIDTH = MAIN_DIALOG_ID_WIDTH_EDIT; // also used for RADIUS
const int G_CH_HEIGHT = MAIN_DIALOG_ID_HEIGHT_EDIT;
const int G_CH_RANGE = MAIN_DIALOG_ID_RANGE_EDIT;
const int G_CH_ANGLE = MAIN_DIALOG_ID_ANGLE_EDIT;
const int G_CH_HIGHEST_EXPONENT = MAIN_DIALOG_ID_HIGHEST_EXPONENT_EDIT;
const int G_CH_COEFFICIENTS = MAIN_DIALOG_ID_COEFFICIENTS_BUTTON;
const int G_CH_DISTANCE_MODE = MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON;
const int G_CH_UNIT = MAIN_DIALOG_ID_UNIT_EDIT;
const int G_CH_ADD = MAIN_DIALOG_ID_ADD_BUTTON;
const int G_CH_DUPLICATE = MAIN_DIALOG_ID_DUPLICATE_BUTTON;
const int G_CH_DELETE = MAIN_DIALOG_ID_DELETE_BUTTON;
const int G_CH_DIMENSIONS = MAIN_DIALOG_ID_DIMENSIONS_BUTTON;
//const int G_CH_ = MAIN_DIALOG_ID_;
//const int G_CH_ = MAIN_DIALOG_ID_;
//const int G_CH_ = MAIN_DIALOG_ID_;
//const int G_CH_ = MAIN_DIALOG_ID_;

// Pulldown Menu File
// (text pointers are set later, after language specific strings are loaded)
MENU g_file_menu[] =
{
  { g_ttxt, &file_new, NULL, 0, NULL },
  { g_ttxt, &file_load, NULL, 0, NULL },
  { g_ttxt, &file_save, NULL, 0, NULL },
  { g_ttxt, &file_save_as, NULL, 0, NULL },
  { g_ttxt, &file_export_bmp, NULL, 0, NULL },
  { g_ttxt, &file_export_pcx, NULL, 0, NULL },
  { g_ttxt, &file_export_tga, NULL, 0, NULL },
  { g_ttxt, &file_export_code, NULL, 0, NULL },
  { g_xtxt, &file_exit, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

// Pulldown Menu Options
// (text pointers are set later, after language specific strings are loaded)
MENU g_options_menu[] =
{
  { g_ttxt, &options_select_language, NULL, 0, NULL },
  { g_ttxt, &options_auto_recalculate, NULL , 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

// Pulldown Menu Help
// (text pointers are set later, after language specific strings are loaded)
MENU g_help_menu[] =
{
  { g_ttxt, &help_manual, NULL, D_DISABLED, NULL },
  { g_ttxt, &help_about, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

// Main Menubar
// (text pointers are set later, after language specific strings are loaded)
MENU g_main_menu[] =
{
  { g_ttxt, NULL, g_file_menu, 0, NULL },
  { g_ttxt, NULL, g_options_menu, 0, NULL },
  { g_ttxt, NULL, g_help_menu, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

#include "dialog_defs.inc"

/* initialization dialog proc that stretches/shrinks the first item of 
   the dialog to a size large enough to contain all other dialog objects
   (this should be the last object of any dialog that wants this behaviour,
    because there might be other self stretching objects inited before it)*/
int d_stretch_me_proc(int msg, DIALOG *obj, int c)
{

  if(msg==MSG_START)
  {
    BITMAP *gui_bmp = gui_get_screen();
    DIALOG *current = active_dialog;
    int i=0;
    current[i].w=0; // reset dimensions of the first object to shrink..
    current[i].h=0; // ..when too large
    int sx=current[i].x;
    int sy=current[i].y;
    int ex=sx+current[i].w;
    int ey=sy+current[i].h;
    
    // Adjust clipping
    set_clip_rect(gui_bmp, 0, 0, gui_bmp->w-1, gui_bmp->h-1);
    set_clip_state(gui_bmp, TRUE);

    while(current[i].proc!=NULL)
    {
      if(current[i].proc != d_stretch_me_proc)
      {  
        if(current[i].x < sx)
          sx = current[i].x;
        if(current[i].y < sy)
          sy = current[i].y;
        if(current[i].x + current[i].w > ex)
          ex = current[i].x + current[i].w;
        if(current[i].y + current[i].h > ey)
          ey = current[i].y + current[i].h;
      }
      i++;
    }
    // Add some small extra border to make it look nicer
    sx -= text_height(font);
    sy -= text_height(font);
    ex += text_height(font);
    ey += text_height(font);  
    // Apply the stretched dimensions to the first object
    current[0].x = sx;
    current[0].y = sy;
    current[0].w = ex - sx;
    current[0].h = ey - sy;

    // Move dialog up, if part of it is below the current gui bitmap
    if(current[0].y + current[0].h > gui_bmp->h)
      position_dialog(current,current[0].x,gui_bmp->h - 1 - current[0].h);

    // Move dialog down, if part of it is above the current gui bitmap
    if(current[0].y < 0)
      position_dialog(current,current[0].x,0);

    // Move dialog right, if part of it is left of the current gui bitmap
    if(current[0].x < 0)
      position_dialog(current,0,current[0].y);

    // Move dialog left, if part of it is right of the current gui bitmap
    if(current[0].x + current[0].w > gui_bmp->w)
      position_dialog(current,gui_bmp->w - 1 - current[0].w, current[0].y);
  }
  if(msg==MSG_END)
  {
    DIALOG *current = active_dialog;
    // Remove extra border(prevent dialog from growing on subsequent calls)
    current[0].x += text_height(font);
    current[0].y += text_height(font);
    current[0].w -= 2*text_height(font);
    current[0].h -= 2*text_height(font);
  }

  return D_O_K;
}

// customized edit box, setting a global var to notify a change to gradient
int d_cedit_proc(int msg, DIALOG *obj, int c)
{
  static bool ignore_next_uchar = false;
  char low = (char)c;
  char high = (char)((int)(c >> 8));

  if(msg==MSG_START)
  {
    obj->h = text_height(font);
  }

  if((msg==MSG_CHAR)||((msg==MSG_UCHAR)&&(!ignore_next_uchar)))
  {
    // Disallow TAB,UP,DOWN,PAGE UP, PAGE DOWN
    if((high==KEY_UP)||(high==KEY_DOWN)
       ||(high==KEY_PGUP)||(high==KEY_PGDN)||(high==KEY_TAB))
    {
      ignore_next_uchar = true;
      return D_O_K;
    }
    else ignore_next_uchar = false;

    // content must have been altered at this point
    if(obj->dp3)
      g_gradient_change_on_lostfocus = *(int*)(obj->dp3);

    // on hitting ENTER
    // signal changes to gradient by setting a global var
    /*if((obj->dp3)
       &&(high!=KEY_LEFT)&&(high!=KEY_RIGHT)
       &&(high!=KEY_HOME)&&(high!=KEY_END)&&(high!=KEY_ENTER))*/
    if((obj->dp3)&&((high==KEY_ENTER)||(low==13)))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0;
    }
  }
  
  // on losing focus, also signal changes to gradient by setting global var
  if((msg==MSG_LOSTFOCUS)&&(obj->dp3))
    if(g_gradient_change_on_lostfocus==*(int*)(obj->dp3))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0; 
    } 
  
  return d_edit_proc(msg,obj,c);
}

// customized list object, setting a global var to notify a change to gradient
// dp3 can be used as a callback function of the prototype 
//     void function(int index);
//     that will be called if the selected item changed

int d_clist_proc(int msg, DIALOG *obj, int c)
{
  int prev_d1 = obj->d1;
  int ret;

  // make use of standard list proc to handle everything else before..
  ret = d_list_proc(msg,obj,c);

  // 

  // HACK to make the list work on backbuffer in sub dialogs
  // , if mouse is hold down on drag
  if(msg==MSG_DRAW)
  {
    if((gui_get_screen() != screen)
       &&(active_dialog != (DIALOG *)&g_main_dialog))
    {
      int pm_x;
      int pm_y;
      DIALOG *cbox = active_dialog;

      acquire_screen();
      pm_x=gui_mouse_x_offset();
      pm_y=gui_mouse_y_offset();
      blit(gui_get_screen(),g_msafe,pm_x,pm_y,0,0,g_msafe->w,g_msafe->h);
      draw_sprite(gui_get_screen(),mouse_sprite,pm_x,pm_y);
      blit(gui_get_screen(), screen, cbox->x, cbox->y,
           cbox->x - g_scroll_x, cbox->y - g_scroll_y,
           cbox->w+mouse_sprite->w, cbox->h+mouse_sprite->h);
      blit(g_msafe,gui_get_screen(),0,0,pm_x,pm_y,g_msafe->w,g_msafe->h);
      release_screen();
    }
  } //(END OF HACK) 

  // ..checking again for callback function
  // ( void(*)(int) is a pointer type to a function taking an int argument,
  //   returning nothing. )
  if(((obj->d1 != prev_d1)||((msg==MSG_START)&&(obj->d1>=0)))&&(obj->dp3))
  {
    (*(void(*)(int))obj->dp3)(obj->d1);
    return ret |= D_REDRAW;
  }

  return ret;
}

// customized button, setting a global var to notify a change to gradient
int d_cbutton_proc(int msg, DIALOG *obj, int c)
{
  if(((msg==MSG_KEY)||(msg==MSG_CLICK))&&(obj->dp3))
    g_gradient_changed = *(int*)(obj->dp3);

  return d_button_proc(msg,obj,c);
}

// customized button, same as above and also auto-stretching to fit text
// also moves the immediately following object(in d2) to align right of this
int d_cstretch_move_nextx_button_proc(int msg, DIALOG *obj, int c)
{
  //int high = (char)((int)(c >> 8));

  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font)+8;
        obj->w = new_w;
        obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->x = obj->x + obj->w + 8;
    }
  }

  if(((msg==MSG_KEY)||(msg==MSG_CLICK)) && (obj->dp3))
    g_gradient_changed = *(int*)(obj->dp3);

  return d_button_proc(msg,obj,c);
}

// customized button, same as above but moving following object(in d2)
// to align below of this
int d_cstretch_move_nexty_button_proc(int msg, DIALOG *obj, int c)
{
  //int high = (char)((int)(c >> 8));

  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font)+8;
        obj->w = new_w;
        obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->y = obj->y + obj->h + 8;
    }
  }

  if(((msg==MSG_KEY)||(msg==MSG_CLICK)) && (obj->dp3))
    g_gradient_changed = *(int*)(obj->dp3);

  return d_button_proc(msg,obj,c);
}

// same as "..move_nextx" but followings' y is set to this' y
int d_cstretch_move_nextxy_button_proc(int msg, DIALOG *obj, int c)
{
  //int high = (char)((int)(c >> 8));

  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font)+8;
        obj->w = new_w;
        obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->x = obj->x + obj->w + 8;
      next->y = obj->y;
    }
  }

  if(((msg==MSG_KEY)||(msg==MSG_CLICK)) && (obj->dp3))
    g_gradient_changed = *(int*)(obj->dp3);

  return d_button_proc(msg,obj,c);
}

// customized slider object, just moving the following object
// (address in dp3 and NOT index in d2, like it is with the other move procs)
// to align below of this on MSG_START
int d_slider_move_nexty_proc(int msg, DIALOG *obj, int c)
{
  int ret;

  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font)+8;
        obj->w = new_w;
        obj->h = new_h;
    }
    // move the following object
    if(obj->dp3)
    {
      DIALOG *next = (DIALOG*)obj->dp3;
        next->y = obj->y + obj->h + 8;
    }
  }

  ret = d_slider_proc(msg,obj,c);

  // HACK to make the slider work on backbuffer, if mouse is hold down on drag
  if(msg==MSG_DRAW)
  {
    if((gui_get_screen() != screen)
       &&(active_dialog == (DIALOG *)&g_color_dialog))
    {
      int pm_x;
      int pm_y;
      DIALOG *cbox = (DIALOG *)&g_color_dialog[COLOR_DIALOG_ID_SBOX];
      acquire_screen();

      pm_x=gui_mouse_x_offset();
      pm_y=gui_mouse_y_offset();
      blit(gui_get_screen(),g_msafe,pm_x,pm_y,0,0,g_msafe->w,g_msafe->h);
      draw_sprite(gui_get_screen(),mouse_sprite,pm_x,pm_y);
     
      object_message((DIALOG *)&g_color_dialog[COLOR_DIALOG_ID_BOX],
                     MSG_DRAW,0);
      blit(gui_get_screen(), screen, cbox->x, cbox->y,
           cbox->x - g_scroll_x, cbox->y - g_scroll_y,
           cbox->w+mouse_sprite->w, cbox->h+mouse_sprite->h);
      blit(g_msafe,gui_get_screen(),0,0,pm_x,pm_y,g_msafe->w,g_msafe->h);
      release_screen();
    }
  } //(END OF HACK)

  return ret;
}

// customized text box, autostretching to fit text
// also moves the immediately following object(in d2) to align right of this
int d_stretch_move_nextx_text_proc(int msg, DIALOG *obj, int c)
{
  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font);
      obj->w = new_w;
      obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->x = obj->x + obj->w + 8;
    }
  }
  return d_text_proc(msg,obj,c);
}

// same as above but also aligns the nexts y position to this ones
int d_stretch_move_nextx_aligny_text_proc(int msg, DIALOG *obj, int c)
{
  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font);
      obj->w = new_w;
      obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->y = obj->y;
      next->x = obj->x + obj->w + 8;
    }
  }
  return d_text_proc(msg,obj,c);
}


// customized text box, autostretching to fit text
// also moves the immediately following object(in d2) to align below of this
// also adds the amount of pixels given in d1 to the y offset
int d_stretch_move_nexty_text_proc(int msg, DIALOG *obj, int c)
{
  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font);
      obj->w = new_w;
      obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->y = obj->y + obj->h + 2 + obj->d1;
    }
  }
  return d_text_proc(msg,obj,c);
}

// same as above, also setting nexts' x to this' x
int d_stretch_move_nextxy_text_proc(int msg, DIALOG *obj, int c)
{
  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = gui_strlen((char*)obj->dp)+8;
      int new_h = text_height(font);
      obj->w = new_w;
      obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->x = obj->x;
      next->y = obj->y + obj->h + 2;
    }
  }
  return d_text_proc(msg,obj,c);
}

// customized bitmap object, moving the following object to align at the right
// top of this (following object id in d2)
int d_bitmap_ralign_nextxy_proc(int msg, DIALOG *obj, int c)
{
  if(msg==MSG_START)
  {
    if(obj->dp)
    {
      int new_w = ((BITMAP*)obj->dp)->w;
      int new_h = ((BITMAP*)obj->dp)->h;
      obj->w = new_w;
      obj->h = new_h;
    }
    // move the following object(as specified in d2)
    if(obj->d2)
    {
      DIALOG *next = active_dialog + obj->d2;
      next->x = obj->x + obj->w + 8;
      next->y = obj->y;
    }
  }
  return d_bitmap_proc(msg,obj,c);
}

// customized edit box, allowing only integer numbers to be typed
// dp2 can be used as a callback function of the prototype void function();
//     that will be called on MSG_CHAR, MSG_UCHAR(changed to MSG_LOSTFOCUS)
// dp3 is used to point to an integer number that will set a global var
//     to signal changes to make to the gradient
int d_iedit_proc(int msg, DIALOG *obj, int c)
{
  char low = (char)c;
  char high = (char)((int)(c >> 8));
  int ret;

  if(msg==MSG_START)
  {
    obj->h = text_height(font);
  }

  // catch invalid character input for integer
  if((msg==MSG_CHAR)||(msg==MSG_UCHAR))
  {
    if((obj->d2==0)&&(low==48)) // disallow 0 (ascii 48) as first digit
      return D_O_K;

    // disallow anything but 0-9 (ascii[48,57]) 
    // and ESC,ENTER,BACKSPACE(ascii 27,13,8)
    // and DEL,LEFT,RIGHT
    if(((low<48)||(low>57))&&(low!=27)&&(low!=13)&&(low!=8)
       &&(high!=KEY_DEL)&&(high!=KEY_LEFT)&&(high!=KEY_RIGHT)
       &&(high!=KEY_ENTER))
      return D_O_K;

    // content must have been altered at this point
    if(obj->dp3)
      g_gradient_change_on_lostfocus = *(int*)(obj->dp3);

    // on hitting ENTER
    // signal changes to gradient by setting a global var
    /*
    if((obj->dp3)&&
       (high!=KEY_LEFT)&&(high!=KEY_RIGHT)&&(high!=KEY_UP)&&(high!=KEY_DOWN)
        &&(high!=KEY_HOME)&&(high!=KEY_END)&&(high!=KEY_ENTER))*/
    if((obj->dp3)&&((high==KEY_ENTER)||(low==13)))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0;
    }
  }
  
  // on losing focus, also signal changes to gradient by setting global var
  if((msg==MSG_LOSTFOCUS)&&(obj->dp3))
    if(g_gradient_change_on_lostfocus==*(int*)(obj->dp3))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0;
    } 
  
  // make use of standard edit proc to handle everything else before..
  ret = d_edit_proc(msg,obj,c);

  // ..checking again for callback function
  // ( void(*)() is a pointer type to a function taking no arguments,
  //   returning nothing. )
  if(/*((msg==MSG_CHAR)||(msg==MSG_UCHAR))*/(msg==MSG_LOSTFOCUS)&&(obj->dp2))
  {
    (*(void(*)())obj->dp2)();
    return ret |= D_REDRAW;
  }

  return ret;
}

// customized edit box, allowing only floating point numbers to be typed
// dp2 can be used as a callback function of the prototype void function();
//     that will be called on MSG_CHAR, MSG_UCHAR(changed to MSG_LOSTFOCUS)
// dp3 is used to point to an integer number that will set a global var
//     to signal changes to make to the gradient
int d_fedit_proc(int msg, DIALOG *obj, int c)
{
  char low = (char)c;
  char high = (char)((int)(c >> 8));
  int ret;

  if(msg==MSG_START)
  {
    obj->h = text_height(font);
  }

  // catch invalid character input for floating point
  if((msg==MSG_CHAR)||(msg==MSG_UCHAR))
  {
    if((obj->d2>0)&&(low==45)) // disallow -(ascii 45) anywhere but at pos 0
      return D_O_K;

    // disallow anything BUT 0-9,-,. (ascii[48,57],45,46) 
    // and ESC,ENTER,BACKSPACE(ascii 27,13,8)
    // and DEL,LEFT,RIGHT
    if(((low<48)||(low>57))&&(low!=45)&&(low!=46)
       &&(low!=27)&&(low!=13)&&(low!=8)
       &&(high!=KEY_DEL)&&(high!=KEY_LEFT)&&(high!=KEY_RIGHT)
       &&(high!=KEY_ENTER))
      return D_O_K;
      
    // content must have been altered at this point
    if(obj->dp3)
      g_gradient_change_on_lostfocus = *(int*)(obj->dp3);
      
    // on hitting ENTER
    // signal changes to gradient by setting a global var
    /*if((obj->dp3)&&
       (high!=KEY_LEFT)&&(high!=KEY_RIGHT)&&(high!=KEY_UP)&&(high!=KEY_DOWN)
        &&(high!=KEY_HOME)&&(high!=KEY_END)&&(high!=KEY_ENTER))*/
    if((obj->dp3)&&((high==KEY_ENTER)||(low==13)))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0;
    }
  }
  
  // on losing focus, also signal changes to gradient by setting global var
  if((msg==MSG_LOSTFOCUS)&&(obj->dp3))
    if(g_gradient_change_on_lostfocus==*(int*)(obj->dp3))
    {
      g_gradient_changed = *(int*)(obj->dp3);
      g_gradient_change_on_lostfocus = 0; 
    }  
  
  // make use of standard edit proc to handle everything else before..
  ret = d_edit_proc(msg,obj,c);

  // ..checking again for callback function
  // ( void(*)() is a pointer type to a function taking no arguments,
  //   returning nothing. )
  if(/*((msg==MSG_CHAR)||(msg==MSG_UCHAR))*/(msg==MSG_LOSTFOCUS)&&(obj->dp2))
  {
    (*(void(*)())obj->dp2)();
    return ret |= D_REDRAW;
  }

  return ret;
}

// ALLEGRO 4.20rc2 hotfix
// customized d_clear_proc
/*(at the time of writing this, the latest Allegro Version 4.20rc2 had a bug
in the d_clear_proc, so i use this customised one, where that bug is fixed)*/
/* Now that Allegro4.20 is out, where that BUG is fixed, this is unneeded.
int d_c_clear_proc(int msg, DIALOG *obj, int c)
{
  if (msg == MSG_DRAW) 
  {
    BITMAP *gui_bmp = gui_get_screen();
    set_clip_rect(gui_bmp, 0, 0, gui_bmp->w-1, gui_bmp->h-1); // fixed.
    // BUG in Allegro 4.20rc2 was:
       // set_clip_rect(gui_bmp, 0, 0, SCREEN_W-1, SCREEN_H-1);
    set_clip_state(gui_bmp, TRUE);
    clear_to_color(gui_bmp, obj->bg);
  }

  return D_O_K;
}
*/

// called by d_scroll_bmp_proc on MSG_LRELEASE
// cx,cy receives the click position relative to the bitmaps origin(0,0)
// current_ca receives the currently selected attractor number
bool set_position_by_click(int cx, int cy, int current_ca)
{
  double new_x = double(cx)/double(g_work_gradient.get_w());
  double new_y = double(cy)/double(g_work_gradient.get_h());
  CA_GLOBAL *ca = g_work_gradient.get_pca(current_ca);
  bool changed = false;
  
  if(ca)
  {
    CA_TYPE type = ca->get_type();
    switch(type)
    {
      case CAT_GLOBAL:
      case CAT_HBAR:
      break;
      default:
        uszprintf((char*)&g_edit_field.x,sizeof(g_edit_field.x),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),new_x);
        dt_set_x();
        changed = true;
      break;
    }
    switch(type)
    {
      case CAT_GLOBAL:
      case CAT_VBAR:
      break;
      default:
        uszprintf((char*)&g_edit_field.y,sizeof(g_edit_field.y),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),new_y);
        dt_set_y();
        changed = true;
      break;
    }
  }
  
  if(changed&&g_auto_recalculate)
    button_recalculate();
  
  return changed;
}

// called by d_scroll_bmp_proc on MSG_RRELEASE or KEY_R
// cx,cy receives the click position relative to the bitmaps origin(0,0)
// current_ca receives the currently selected attractor number
bool set_range_by_rclick(int cx, int cy, int current_ca)
{
  double rel_x = double(cx)/double(g_work_gradient.get_w());
  double rel_y = double(cy)/double(g_work_gradient.get_h());
  double rel_range = 0.0;
  CA_GLOBAL *ca = g_work_gradient.get_pca(current_ca);
  bool changed = true;
  
  if(ca)
  {
    CA_TYPE type = ca->get_type();
    
    // first adjust rel_x, rel_y to be in the rotated system of certain
    // attractors, if necessary
    if((type==CAT_BAR)||(type==CAT_POLYNOMIAL)
       ||(type==CAT_POINTE)||(type==CAT_ELLIPSE))
    {
      double angle = 0.0;
      double vx = rel_x - ((CA_POINT*)ca)->get_x();
      double vy = rel_y - ((CA_POINT*)ca)->get_y();
      double vangle = 0.0;
      double length = sqrt(vx*vx + vy*vy);
      if(type==CAT_BAR)
        angle = ((CA_BAR*)ca)->get_angle();
      if(type==CAT_POLYNOMIAL)
        angle = ((CA_POLYNOMIAL*)ca)->get_angle();
      if((type==CAT_POINTE)||(type==CAT_ELLIPSE))
        angle = ((CA_POINTE*)ca)->get_angle();
      
      if(length!=0.0)
      {
        if(vy>=0.0)
          vangle = acos(vx/length);
        if(vy<0.0)
          vangle = _2pi - acos(vx/length);
      
        rel_x = length * cos(vangle - make_rad(angle));
        rel_y = length * sin(vangle - make_rad(angle));
      }
      else
      {
        rel_x = 0.0;
        rel_y = 0.0;
      }
      
      if(type==CAT_ELLIPSE)
      {
        double ex = (((CA_ELLIPSE*)ca)->get_width()/2.0) * cos(vangle - make_rad(angle));
        double ey = (((CA_ELLIPSE*)ca)->get_height()/2.0) * sin(vangle - make_rad(angle));
        vx = rel_x - ex;
        vy = rel_y - ey;
        rel_y = sqrt(vx*vx + vy*vy);
      }
    }   
    
    
    switch(type)
    {
      case CAT_GLOBAL:
      case CAT_POINTE:
        changed = false;
      break;
      
      case CAT_HBAR:
        rel_range = fabs(rel_y - ((CA_BAR*)ca)->get_y());
      break;
      case CAT_POLYNOMIAL:
      case CAT_BAR:
        rel_range = fabs(rel_y); // ugly hack (rel_y is abused as sth. else)
      break;
      case CAT_VBAR:
        rel_range = fabs(rel_x - ((CA_VBAR*)ca)->get_x());
      break;
      case CAT_POINT:
        rel_range = sqrt( (rel_x-((CA_POINT*)ca)->get_x())*(rel_x-((CA_POINT*)ca)->get_x())
                         +(rel_y-((CA_POINT*)ca)->get_y())*(rel_y-((CA_POINT*)ca)->get_y()));
      break;
      case CAT_ELLIPSE:
        rel_range = rel_y; // ugly hack (rel_y is abused as sth. else)
      break;
      case CAT_CIRCLE:
      case CAT_CIRCLEF:
        rel_range = sqrt( (rel_x-((CA_POINT*)ca)->get_x())*(rel_x-((CA_POINT*)ca)->get_x())
                         +(rel_y-((CA_POINT*)ca)->get_y())*(rel_y-((CA_POINT*)ca)->get_y()));
        rel_range = fabs(rel_range - ((CA_CIRCLE*)ca)->get_radius());
      break;
      default:
        changed = false;
      break;
    }
    if(changed)
    {
      uszprintf((char*)&g_edit_field.range,sizeof(g_edit_field.range),
                uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),rel_range);
      dt_set_range();
    }
  }
  
  if(changed&&g_auto_recalculate)
    button_recalculate();
  
  return changed;
}

// called by d_scroll_bmp_proc on MSG_MRELEASE or KEY_D
// cx,cy receives the click position relative to the bitmaps origin(0,0)
// current_ca receives the currently selected attractor number
bool set_dim_by_mclick(int cx, int cy, int current_ca)
{
  double rel_x = double(cx)/double(g_work_gradient.get_w());
  double rel_y = double(cy)/double(g_work_gradient.get_h());
  double rel_w = 0.0;
  double rel_h = 0.0;
  CA_GLOBAL *ca = g_work_gradient.get_pca(current_ca);
  bool changed = false;
  
  if(ca)
  {
    CA_TYPE type = ca->get_type();
    switch(type)
    {
      case CAT_CIRCLE:
      case CAT_CIRCLEF:
        rel_w = sqrt((rel_x - ((CA_CIRCLE*)ca)->get_x())*(rel_x - ((CA_CIRCLE*)ca)->get_x())
                     +(rel_y - ((CA_CIRCLE*)ca)->get_y())*(rel_y - ((CA_CIRCLE*)ca)->get_y()));
        uszprintf((char*)&g_edit_field.width,sizeof(g_edit_field.width),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),rel_w);
        dt_set_width();
        changed = true;
      break;
      case CAT_POINTE:
      case CAT_ELLIPSE:
        rel_w = fabs(rel_x - ((CA_POINTE*)ca)->get_x())*2.0;
        rel_h = fabs(rel_y - ((CA_POINTE*)ca)->get_y())*2.0;
        uszprintf((char*)&g_edit_field.width,sizeof(g_edit_field.width),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),rel_w);
        uszprintf((char*)&g_edit_field.height,sizeof(g_edit_field.height),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),rel_h);
        dt_set_width();
        dt_set_height();
        changed = true;
      break;
      default:
      break;
    }
  }
  
  if(changed&&g_auto_recalculate)
    button_recalculate();
  
  return changed;
}

// called by d_scroll_bmp_proc on KEY_A
// cx,cy receives the click position relative to the bitmaps origin(0,0)
// current_ca receives the currently selected attractor number
bool set_angle_by_a_key(int cx, int cy, int current_ca)
{
  double rel_x = double(cx)/double(g_work_gradient.get_w());
  double rel_y = double(cy)/double(g_work_gradient.get_h());
  CA_GLOBAL *ca = g_work_gradient.get_pca(current_ca);
  bool changed = false;
  
  if(ca)
  {
    CA_TYPE type = ca->get_type();   
    switch(type)
    {
      case CAT_POINTE:
      case CAT_ELLIPSE:  
      case CAT_POLYNOMIAL:
      case CAT_BAR:
      {
        double cax = ((CA_POINT*)ca)->get_x();
        double cay = ((CA_POINT*)ca)->get_y();
        double vx = rel_x - cax;
        double vy = rel_y - cay;
        double vl = sqrt(vx*vx + vy*vy);
        double angle = 0.0;
    
       if(vy>=0.0)
         if(vl!=0.0)
           angle = acos(vx / vl);
    
       if(vy<0.0)
          if(vl!=0.0)
            angle = _2pi - acos(vx / vl);
    
        if(angle==0.0)
          if(vx<0)
            angle = _pi;
        
        angle = angle*_to_deg;
        uszprintf((char*)&g_edit_field.angle,sizeof(g_edit_field.angle),
                  uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),angle);
        dt_set_angle();
        changed = true;
      }
      break;
      default:
      break;
    }
  }
  
  if(changed&&g_auto_recalculate)
    button_recalculate();
  
  return changed;
}


// customized scrollable bitmap dialog procedure
// d1,d2 hold the scroll position(can also be negative!) dp points to a BMP_T
// scrolling is performed by sending the custom messages MSG_SCROLL_x,
// where x is one of UP,DOWN,LEFT,RIGHT
int d_scroll_bmp_proc(int msg, DIALOG *obj, int c)
{
  //char low = (char)c;
  char high = (char)((int)(c >> 8));

  if(msg==MSG_WANTFOCUS)
    return D_WANTFOCUS;

  if((msg==MSG_CHAR)||(msg==MSG_UCHAR))
  {
    if(high==KEY_UP)
      object_message(obj,MSG_SCROLL_UP,0);
    if(high==KEY_DOWN)
      object_message(obj,MSG_SCROLL_DOWN,0);
    if(high==KEY_LEFT)
      object_message(obj,MSG_SCROLL_LEFT,0);
    if(high==KEY_RIGHT)
      object_message(obj,MSG_SCROLL_RIGHT,0);
    
    if(high==KEY_A)
    {
      int cx = gui_mouse_x() - obj->x + obj->d1;
      int cy = gui_mouse_y() - obj->y + obj->d2;
      int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                  (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
      while(key[KEY_A])
      {
        if(keyboard_needs_poll())
          poll_keyboard();
        rest(10);
      }
      clear_keybuf();
      if(set_angle_by_a_key(cx,cy,current))
        return D_REDRAW;
    }
    
    if(high==KEY_D)
    {
      int cx = gui_mouse_x() - obj->x + obj->d1;
      int cy = gui_mouse_y() - obj->y + obj->d2;
      int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                  (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
      while(key[KEY_D])
      {
        if(keyboard_needs_poll())
          poll_keyboard();
        rest(10);
      }
      clear_keybuf();
      if(set_dim_by_mclick(cx,cy,current))
        return D_REDRAW;
    }
    
    if(high==KEY_R)
    {
      int cx = gui_mouse_x() - obj->x + obj->d1;
      int cy = gui_mouse_y() - obj->y + obj->d2;
      int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                  (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
      while(key[KEY_R])
      {
        if(keyboard_needs_poll())
          poll_keyboard();
        rest(10);
      }
      clear_keybuf();
      if(set_range_by_rclick(cx,cy,current))
        return D_REDRAW;
    }
    
    if((high==KEY_UP)||(high==KEY_DOWN)||(high==KEY_LEFT)||(high==KEY_RIGHT))
      return D_USED_CHAR;
    else 
      return D_O_K;
  }

  if((msg>=MSG_SCROLL_UP)&&(msg<=MSG_SCROLL_RIGHT))
  {
    int sw_pix = obj->w / 8;
    int sh_pix = obj->h / 8;

    switch(msg)
    {
      case MSG_SCROLL_UP:
        obj->d2 -= sh_pix;
      break;
      case MSG_SCROLL_DOWN:
        obj->d2 += sh_pix;
      break;
      case MSG_SCROLL_LEFT:
        obj->d1 -= sw_pix;
      break;
      case MSG_SCROLL_RIGHT:
        obj->d1 += sw_pix;
      break;
    }
    return D_REDRAWME;
  }
  
  if(msg==MSG_LRELEASE)
  {
    int cx = gui_mouse_x() - obj->x + obj->d1;
    int cy = gui_mouse_y() - obj->y + obj->d2;
    int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                  (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
    if(set_position_by_click(cx,cy,current))
      return D_REDRAW;
  }
  
  if(msg==MSG_RRELEASE)
  {
    int cx = gui_mouse_x() - obj->x + obj->d1;
    int cy = gui_mouse_y() - obj->y + obj->d2;
    int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                  (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
    if(set_range_by_rclick(cx,cy,current))
      return D_REDRAW;
  }
  
  if(msg==MSG_MRELEASE)
  {
    int cx = gui_mouse_x() - obj->x + obj->d1;
    int cy = gui_mouse_y() - obj->y + obj->d2;
    int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
    if(set_dim_by_mclick(cx,cy,current))
      return D_REDRAW;
  }

  if(msg==MSG_DRAW) 
  {
    if(obj->dp)
    {
      BITMAP *gui_bmp = gui_get_screen();
      int x,y,x2,y2,w,h;
      get_clip_rect(gui_bmp,&x,&y,&x2,&y2); // remember previous clip area
      w = ((BMP_T*)obj->dp)->get_width();
      h = ((BMP_T*)obj->dp)->get_height();

      // prepare a white canvas with border for the following
      rectfill(gui_bmp,obj->x,obj->y,
               obj->x+obj->w-1,obj->y+obj->h-1,makecol(255,255,255));
      if(obj->flags & D_GOTFOCUS)
        rect(gui_bmp,obj->x,obj->y,
             obj->x+obj->w-1,obj->y+obj->h-1,makecol(128,255,128));
      else
        rect(gui_bmp,obj->x,obj->y,
             obj->x+obj->w-1,obj->y+obj->h-1,makecol(0,0,0));

      // set clipping to dialog objects' inner area
      set_clip_rect(gui_bmp,obj->x+1,obj->y+1,
                    obj->x+1+obj->w-3,obj->y+1+obj->h-3);

      // draw coordinate system guide lines
      hline(gui_bmp,obj->x,obj->y-obj->d2,obj->x+obj->w,makecol(0,0,0));
      vline(gui_bmp,obj->x-obj->d1,obj->y,obj->y+obj->h,makecol(0,0,0));

      // draw unit guide lines
      textprintf_right_ex(gui_bmp,g_default_font,obj->x-obj->d1-2,
                          obj->y-obj->d2-2-text_height(g_default_font),
                          makecol(0,0,0),-1,
                          uconvert("(0.0)",U_ASCII,NULL,U_CURRENT,0));
      if(w)
      {
        vline(gui_bmp,obj->x-obj->d1+w,obj->y-obj->d2-10,obj->y-obj->d2,
              makecol(0,0,0));
        textprintf_right_ex(gui_bmp,g_default_font,obj->x-obj->d1+w,
                              obj->y-obj->d2-4-2*text_height(g_default_font),
                              makecol(0,0,0),-1,
                              uconvert("(%i)",U_ASCII,NULL,U_CURRENT,0),w);
        textprintf_right_ex(gui_bmp,g_default_font,obj->x-obj->d1+w,
                              obj->y-obj->d2-2-text_height(g_default_font),
                              makecol(0,0,0),-1,
                              uconvert("(1.0)",U_ASCII,NULL,U_CURRENT,0));
      }
      if(h)
      {
        hline(gui_bmp,obj->x-obj->d1-10,obj->y-obj->d2+h,obj->x-obj->d1,
              makecol(0,0,0));
        textprintf_right_ex(gui_bmp,g_default_font,obj->x-obj->d1-2,
                            obj->y-obj->d2+h-4-2*text_height(g_default_font),
                            makecol(0,0,0),-1,
                            uconvert("(%i)",U_ASCII,NULL,U_CURRENT,0),h);
        textprintf_right_ex(gui_bmp,g_default_font,obj->x-obj->d1-2,
                            obj->y-obj->d2-2-text_height(g_default_font)+h,
                            makecol(0,0,0),-1,
                            uconvert("(1.0)",U_ASCII,NULL,U_CURRENT,0));
      }

      // draw the bitmap into the coordinate system
      if(((BMP_T*)obj->dp)->get_al_bmp())
        blit(((BMP_T*)obj->dp)->get_al_bmp(),gui_bmp,0,0,
             obj->x-obj->d1,obj->y-obj->d2,w,h);

      // draw additional info for the selected attractor (if focus is set)
      if(obj->flags & D_GOTFOCUS)
      {
        int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                      (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
        CA_GLOBAL *ca = g_work_gradient.get_pca(current);
        if(ca)
        {
          double angle = 0.0;
          bool draw_arc = false;
          bool draw_center = true;
          CA_TYPE t=ca->get_type();
          switch(t)
          {
            case CAT_GLOBAL:
              draw_center = false;
            break;
            case CAT_POINTE:
            case CAT_ELLIPSE:
              angle = ((CA_POINTE*)ca)->get_angle();
              draw_arc = true;
            break;
            case CAT_BAR:
              angle = ((CA_BAR*)ca)->get_angle();
              draw_arc = true;
            break;
            case CAT_POLYNOMIAL:
              angle = ((CA_POLYNOMIAL*)ca)->get_angle();
              draw_arc = true;
            break;
            default:
            break;
          }
          if(draw_arc)
          {
            int cx = obj->x - obj->d1 + int(((CA_POINT*)ca)->get_x()*g_work_gradient.get_w());
            int cy = obj->y - obj->d2 + int(((CA_POINT*)ca)->get_y()*g_work_gradient.get_h());
            //int cx2 = gui_mouse_x() - obj->x + obj->d1;
            //int cy2 = gui_mouse_y() - obj->y + obj->d2;
            int cx3 = cx + int(20.0*cos(_to_rad*angle));
            int cy3 = cy + int(20.0*sin(_to_rad*angle));
            line(gui_bmp,cx,cy,cx+20,cy,makecol(255,0,255));
            line(gui_bmp,cx,cy,cx3,cy3,makecol(255,0,255));
            arc(gui_bmp,cx,cy,itofix(int((360.0-angle/360.0)*256)),itofix(0),20,makecol(255,0,255));
          }
          if(draw_center)
          {
            int cx = obj->x - obj->d1 + int(((CA_POINT*)ca)->get_x()*g_work_gradient.get_w());
            int cy = obj->y - obj->d2 + int(((CA_POINT*)ca)->get_y()*g_work_gradient.get_h());
            vline(gui_bmp,cx,cy-5,cy+5,makecol(255,0,255));
            hline(gui_bmp,cx-5,cy,cx+5,makecol(255,0,255));
          }
        }
      }

      // reset previous clip rectangle
      set_clip_rect(gui_bmp,x,y,x2,y2);
    }
    return D_O_K;
  }

  return D_O_K;
}

// customized message button dialog proc
// on clicking, it sends the message stored in d2, to the object number d1
// in the active dialog
// optional: dp  points to the "normal" button image
// optional: dp2 points to the "got focus" button image
// if dp or dp2 is NULL, a filled rectangle(color bg)
// with a border(color fg) is
// drawn "normally" and a rectangle with swapped colors "on mouse over"
int d_message_button_proc(int msg, DIALOG *obj, int c)
{
  //char low = (char)c;
  char high = (char)((int)(c >> 8));

  if(msg==MSG_START)
  {
    if(obj->dp) // adjust size according to bitmap size(if any)
    {
      obj->w = ((BITMAP*)obj->dp)->w;
      obj->h = ((BITMAP*)obj->dp)->h;
    }
  }

  if(msg==MSG_WANTFOCUS)
    return D_WANTFOCUS;

  if(msg==MSG_DRAW)
  {
    BITMAP *me = (BITMAP*)obj->dp2;
    BITMAP *out = gui_get_screen();

    if(obj->flags & D_GOTFOCUS) // gotfocus draw
    {
      if(me)
        blit(me,out,0,0,obj->x,obj->y,me->w,me->h);
      else // fall back to color swapped colored bordered rectangle
      {
        rectfill(out,obj->x,obj->y,obj->x+obj->w,obj->y+obj->h,obj->fg);
        rect(out,obj->x,obj->y,obj->x+obj->w,obj->y+obj->h,obj->bg);
      }
    }
    else // normal draw
    {
      me = (BITMAP*)obj->dp;
      if(me)
        blit(me,out,0,0,obj->x,obj->y,me->w,me->h);
      else // fall back to colored bordered rectangle
      {
        rectfill(out,obj->x,obj->y,obj->x+obj->w,obj->y+obj->h,obj->bg);
        rect(out,obj->x,obj->y,obj->x+obj->w,obj->y+obj->h,obj->fg);
      }
    }
    return D_O_K;
  }

  if((msg==MSG_CHAR)||(msg==MSG_UCHAR))
  {
    if((high==KEY_SPACE)||(high==KEY_ENTER))
    {
      // send your friend object a message ;-)
      object_message((DIALOG*)&active_dialog[obj->d1],obj->d2,0); 
      return D_USED_CHAR;
    }
    else
      return D_O_K;
  }

  if(msg==MSG_CLICK)
  {
    // send your friend object a message ;-)
    object_message((DIALOG*)&active_dialog[obj->d1],obj->d2,0);
    return D_O_K;
  }

  return D_O_K;
}

// customized scroll list dialog proc for showing the attractors list of my
// gradient generator
// d1 (-1 if list is empty, selected attractor index otherwise)
// d2 (-1 if list is empty, topmost shown attractor index otherwise)
// dp points to the instance of GRADIENT_GEN to use as source for attractors
int d_attractor_list_proc(int msg, DIALOG *obj, int c)
{
  //char low = (char)c;
  char high = (char)((int)(c >> 8));
  int attractors = -1;
  int i = 0;
  int to_draw = 0;

  if(obj->dp)
    attractors = ((GRADIENT_GEN*)obj->dp)->get_attractor_count();

  // always update selected attractor information
  if( attractors == -1)
  {
    obj->d1 = -1; // select none
    obj->d2 = -1;
  }
  else if(attractors < obj->d1)
  {
    obj->d1 = attractors-1; // select last
    obj->d2 = clamp_val<int>(0,obj->d1,attractors-1);
  }
  else
  {
    // select the one that was selected last time if it's still availabe
    obj->d1 = clamp_val<int>(0,obj->d1, attractors-1);
    obj->d2 = clamp_val<int>(0,obj->d2,attractors-1);
  }

  if(msg==MSG_GET_ATTRACTOR) // don't use with "object_message", it will
    return obj->d1;          // NOT work correctly. hack around: calling this
                             // proc directly on a specific object

  if(msg==MSG_WANTFOCUS)
    return D_WANTFOCUS;

  if((msg==MSG_START)||(msg==MSG_UPDATE_FIELDS))
  {  
    dt_get_all(); // updates visible dialog items and their contents
    return D_REDRAW;
  }

  if(msg==MSG_LRELEASE) // select attractor entry under mouse
  {
    obj->d1 = clamp_val<int>(obj->d2,
                             obj->d2 + (gui_mouse_y() - obj->y - 1)/25,
                             attractors-1);
    return object_message(obj,MSG_UPDATE_FIELDS,0);
  }

  if((msg==MSG_CHAR)||(msg==MSG_UCHAR))
  {
    if(high==KEY_UP)
      object_message(obj,MSG_SCROLL_UP,0);
    if(high==KEY_DOWN)
      object_message(obj,MSG_SCROLL_DOWN,0);
    if((high==KEY_SPACE)||(high==KEY_ENTER))
    {
      // select attractor at scroll pos
      obj->d1 = clamp_val<int>(0,obj->d2,attractors-1);
      object_message(obj,MSG_UPDATE_FIELDS,0);
      broadcast_dialog_message(MSG_DRAW,0);
    }

    if((high==KEY_UP)||(high==KEY_DOWN)||(high==KEY_SPACE)||(high==KEY_ENTER))
      return D_USED_CHAR;
    else 
      return D_O_K;
  }
 
  if(msg==MSG_WHEEL) // scroll attractor list by wheel
  {
    obj->d2 = clamp_val<int>(0,
                             obj->d2 - c,
                             attractors-1);
    return D_REDRAWME;
  }

  if(msg==MSG_SCROLL_UP) // scroll attractor list by custom message
  {
    obj->d2 = clamp_val<int>(0,
                             obj->d2 - 1,
                             attractors-1);
    return D_REDRAWME;
  }
  if(msg==MSG_SCROLL_DOWN) // scroll attractor list by custom message
  {
    obj->d2 = clamp_val<int>(0,
                             obj->d2 + 1,
                             attractors-1);
    return D_REDRAWME;
  }
  
  if(msg==MSG_MOVE_UP) // move attractor up by custom message
  {
    if(((GRADIENT_GEN*)obj->dp)->switch_ca(obj->d1,obj->d1-1)!=-1)
    {
      obj->d1 = obj->d1-1; // (keep moved attractor selected)
      if(obj->d1 < obj->d2) // eventually scroll to follow moved attr.
        obj->d2 = clamp_val<int>(0,
                                 obj->d2 - 1,
                                 attractors-1);
      return D_REDRAWME;
    }
  }
  
  if(msg==MSG_MOVE_DOWN) // move attractor down by custom message
  {
    if(((GRADIENT_GEN*)obj->dp)->switch_ca(obj->d1,obj->d1+1)!=-1)
    {
      obj->d1 = obj->d1+1; // (keep moved attractor selected)
      if(obj->d1 >= obj->d2 + obj->h/25) // eventually scroll to follow moved attr.
        obj->d2 = clamp_val<int>(0,
                                 obj->d2 + 1,
                                 attractors-1);
      return D_REDRAWME;
    }
  }

  if(msg==MSG_DRAW)
  {
    BITMAP *gui_bmp = gui_get_screen();
    BITMAP *icon = NULL;
    int x,y,x2,y2;

    get_clip_rect(gui_bmp,&x,&y,&x2,&y2); // remember previous clip area

    // prepare a white canvas with border for the following
    rectfill(gui_bmp,obj->x,obj->y,
             obj->x+obj->w-1,obj->y+obj->h-1,makecol(255,255,255));
      
    if(obj->flags & D_GOTFOCUS)
      rect(gui_bmp,obj->x,obj->y,
           obj->x+obj->w-1,obj->y+obj->h-1,makecol(128,255,128));
    else
      rect(gui_bmp,obj->x,obj->y,
           obj->x+obj->w-1,obj->y+obj->h-1,makecol(0,0,0));

    // set clipping to dialog objects' inner area
    set_clip_rect(gui_bmp,obj->x+1,obj->y+1,
                  obj->x+1+obj->w-3,obj->y+1+obj->h-3);

    // draw color attractor info entries
    for(i=0; i<attractors; i++)
    {
      CA_GLOBAL *td = NULL;
      if(1+i*25 > obj->w) // out of draw area?
        break;

      to_draw=obj->d2+i;
      if(to_draw >= attractors)
        break;
      if(to_draw==obj->d1) // currently selected attractor?
        rectfill(gui_bmp,obj->x+1,obj->y+2+i*25,
                 obj->x+obj->w,obj->y+2+i*25+23, makecol(255,196,0));

      // blit in icons, describing the attractor
      td = ((GRADIENT_GEN*)obj->dp)->get_pca(to_draw);

      // color icon
      rectfill(gui_bmp,obj->x+3,obj->y+3+i*25,obj->x+3+22,obj->y+3+i*25+21,
               makecol(td->get_r(),td->get_g(),td->get_b()));
      rect(gui_bmp,obj->x+2,obj->y+2+i*25,obj->x+2+23,obj->y+2+i*25+23,
           makecol(0,0,0));

      // attractor type icon
      icon = (BITMAP*)g_all_data[ICO_ATT_T_000+td->get_type()].dat;
      blit(icon,gui_bmp,0,0,obj->x+28,obj->y+2+i*25,icon->w,icon->h);

      // attractor mode icon
      icon = (BITMAP*)g_all_data[ICO_MODE_000+td->get_mode()].dat;
      blit(icon,gui_bmp,0,0,obj->x+54,obj->y+2+i*25,icon->w,icon->h);

      // some text info, returned by the attractor itself
      textout_ex(gui_bmp,font,uconvert(td->gui_quick_info().c_str(),U_ASCII,
                                       NULL,U_CURRENT,0),
                 obj->x+80,obj->y+2+i*25+12-text_height(font)/2,
                 makecol(0,0,0),-1);

    }

    // reset previous clip rectangle
    set_clip_rect(gui_bmp,x,y,x2,y2);

    return D_O_K;
  }

  return D_O_K;
}

// callback function to fill the type list
char *fill_type_list(int index, int *list_size)
{
  if(index < 0)
  {
    *list_size = CAT_NUM_TYPES;
    return NULL;
  }
  else
    return g_all_text.s[DBDCGG_TXT_ID_GLOBAL+index];
}


// callback function to fill the mode list
char *fill_mode_list(int index, int *list_size)
{
  if(index < 0)
  {
    *list_size = CAM_NUM_MODES;
    return NULL;
  }
  else
    return g_all_text.s[DBDCGG_TXT_ID_ATTRACT+index];
}

// callback function to fill the distance mode list
char *fill_dmode_list(int index, int *list_size)
{
  if(index < 0)
  {
    *list_size = DM_NUM_MODES;
    return NULL;
  }
  else
    return g_all_text.s[DBDCGG_TXT_ID_TO_FUNCVALUE+index];
}

// callback function to change the bg color of the box
int color_change(void *dp3, int d2)
{
  g_color_dialog[COLOR_DIALOG_ID_BOX].bg 
    = makecol(g_color_dialog[COLOR_DIALOG_ID_SLIDER_R].d2,
              g_color_dialog[COLOR_DIALOG_ID_SLIDER_G].d2,
              g_color_dialog[COLOR_DIALOG_ID_SLIDER_B].d2);
  return D_REDRAW;
}

// callback function to fill the coefficients list
char *fill_coefficient_list(int index, int *list_size)
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);

  if(index < 0)
  {
    if(ca)
    {
      if(ca->get_type() == CAT_POLYNOMIAL)
        *list_size = ((CA_POLYNOMIAL*)ca)->get_hep()+1;
      else
        *list_size = 0;
    }
    else
      *list_size = 0;
    return NULL;
  }
  else
  {
    unsigned int hep = 0;

    if(ca)
      if(ca->get_type() == CAT_POLYNOMIAL)
        hep = ((CA_POLYNOMIAL*)ca)->get_hep();
      else
        return g_ttxt;
 
    // print the value into the list text entry field
    // (note: list entry field 0 gets the highest coefficient)
    uszprintf(g_edit_field.coefficients[hep-index],
              sizeof(g_edit_field.coefficients[hep-index]),
              uconvert("%.8f*(x^%i)+",U_ASCII,NULL,U_CURRENT,0),
              g_current_coefficients[hep-index],hep-index);

    return g_edit_field.coefficients[hep-index];
  }
}

// callback function to change the currently edited coefficient in the list
void coefficient_update_list()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  int index = g_coefficients_dialog[COEFFICIENTS_DIALOG_ID_LIST].d1;
  unsigned int hep = 0;

  if(ca)
    if(ca->get_type() == CAT_POLYNOMIAL)
      hep = ((CA_POLYNOMIAL*)ca)->get_hep();
    else  // not of type POLYNOMIAL
      return;

  // 1) set the value by converting the text from the fedit field
  g_current_coefficients[hep-index] 
    = ustrtod(g_edit_field.current_coeff,NULL);

  // 2) also set the list text entry by printing the just set value into it
  uszprintf(g_edit_field.coefficients[hep-index],
            sizeof(g_edit_field.coefficients[hep-index]),
            uconvert("%.8f*(x^%i)+",U_ASCII,NULL,U_CURRENT,0),
            g_current_coefficients[hep-index],hep-index);
}

// callback function to put the currently selected coefficient to the edit box
void coefficient_update_fedit(int index)
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  double def_coeff = 0.0;
  unsigned int hep = 0;

  memset(g_edit_field.current_coeff,0,sizeof(g_edit_field.current_coeff));
  if(ca)
    if(ca->get_type() == CAT_POLYNOMIAL)
      hep = ((CA_POLYNOMIAL*)ca)->get_hep();
    else // not of type POLYNOMIAL
    {
      // set the edit field text by printing in a default value
      uszprintf(g_edit_field.current_coeff,sizeof(g_edit_field.current_coeff),
                uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),
                def_coeff);
      return;
    }

  // print the selected value into the edit field
  uszprintf(g_edit_field.current_coeff,sizeof(g_edit_field.current_coeff),
            uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),
            g_current_coefficients[hep-index]);
}

// callback functions to prevent the edit fields from getting insane values
void dimensions_w_edit_clamp()
{
  int w = (int)ustrtol(g_edit_field.current_pixw,NULL,10);
  w = clamp_val<int>(1,w,4096);
  uszprintf(g_edit_field.current_pixw,sizeof(g_edit_field.current_pixw),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),w);
}

void dimensions_h_edit_clamp()
{
  int h = (int)ustrtol(g_edit_field.current_pixh,NULL,10);
  h = clamp_val<int>(1,h,4096);
  uszprintf(g_edit_field.current_pixh,sizeof(g_edit_field.current_pixh),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),h);
}

// callback functions to prevent the edit fields from getting insane values
void exp_dimensions_w_edit_clamp()
{
  int w = (int)ustrtol(g_edit_field.current_expixw,NULL,10);
  w = clamp_val<int>(1,w,4096);
  uszprintf(g_edit_field.current_expixw,sizeof(g_edit_field.current_expixw),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),w);
}
void exp_dimensions_h_edit_clamp()
{
  int h = (int)ustrtol(g_edit_field.current_expixh,NULL,10);
  h = clamp_val<int>(1,h,4096);
  uszprintf(g_edit_field.current_expixh,sizeof(g_edit_field.current_expixh),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),h);
}

// File Menu Items Control Functions
int file_new()
{
  if(g_unsaved_changes)
  {
    g_usc_alert_caller = USC_ALERT_CALLER_NEW;
    g_request_sub_dialog = SUB_DIALOG_CHANGE_ALERT;
    return D_CLOSE;
  }
  else
  {
    g_work_gradient.clear();
    g_work_gradient.set_w(320);
    g_work_gradient.set_h(240);
    reset_preview_bitmap();
    reset_edit_fields();
    memset(g_dcgg_fname,0,G_FNAME_BUFSIZE); // clear previous filename
    uszprintf(g_edit_field.current_expixw,sizeof(g_edit_field.current_expixw),
              uconvert("%i",U_ASCII,NULL,U_CURRENT,0),320);
    uszprintf(g_edit_field.current_expixh,sizeof(g_edit_field.current_expixh),
              uconvert("%i",U_ASCII,NULL,U_CURRENT,0),240);
    if(active_dialog == (DIALOG*)&g_main_dialog)
    {
      object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],
                     MSG_UPDATE_FIELDS,0);
      return D_REDRAW;
    }
    
    return D_O_K;
  }
}

int file_load()
{
  if(g_unsaved_changes)
  {
    g_usc_alert_caller = USC_ALERT_CALLER_LOAD;
    g_request_sub_dialog = SUB_DIALOG_CHANGE_ALERT;
    return D_CLOSE;
  }
  else
  {
    int objnum = 0;
    
    // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
    acquire_screen();
    dialog_message(g_main_dialog,D_REDRAW,0,&objnum);
    blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
    release_screen();
    
    // Set standard GUI behaviour
    gui_behave_std();
    g_os_close_button_disabled = true;

    if(file_select_ex(g_all_text.s[DBDCGG_TXT_ID_SELECT_GRADIENT_FILE],
                      g_dcgg_fname,
                      g_all_exts.s[F_EXT_DCGG],G_FNAME_BUFSIZE,0,0)!=0)
    { // OK pressed
      if(g_work_gradient.load(uconvert(g_dcgg_fname,U_CURRENT,
                                       NULL,U_ASCII,0))!=0) // load error?
      {
        g_work_gradient.clear();
        show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_LOAD_FAILED],
                       g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC);
      }
      else // load success
      {
        g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d1 = 0;
        g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d2 = 0;
        reset_preview_bitmap();
        reset_edit_fields();
        uszprintf(g_edit_field.current_expixw,
                  sizeof(g_edit_field.current_expixw),
                  uconvert("%i",U_ASCII,NULL,U_CURRENT,0),
                  g_work_gradient.get_w());
        uszprintf(g_edit_field.current_expixh,
                  sizeof(g_edit_field.current_expixh),
                  uconvert("%i",U_ASCII,NULL,U_CURRENT,0),
                  g_work_gradient.get_h());
        button_recalculate();
      }
    }//(OK pressed)

    // Back to my custom GUI behaviour
    g_os_close_button_disabled = false;
    gui_behave_custom();

    if(active_dialog == (DIALOG*)&g_main_dialog)
      broadcast_dialog_message(MSG_START,0);
    return D_REDRAW;
  }
}

int file_save()
{
  if(ustricmp(g_all_exts.s[F_EXT_DCGG],
              get_extension(g_dcgg_fname)) != 0) // wrong extension?
    return file_save_as(); // fall back to save_as
  else
  if(file_exists(g_dcgg_fname,FA_ARCH | FA_HIDDEN,NULL))
  {
    // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
    acquire_screen();
    broadcast_dialog_message(D_REDRAW,0);
    blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
    release_screen();
  
    // Set standard GUI behaviour
    gui_behave_std();
    g_os_close_button_disabled = true;
    gui_fg_color = makecol(255,0,0);
    gui_bg_color = makecol(255,255,200); // set alert colors
    if(alert(g_all_text.s[DBDCGG_TXT_ID_QUESTION],
             g_all_text.s[DBDCGG_TXT_ID_OVERWRITE_QUESTION],
             g_dcgg_fname,
             g_all_text.s[DBDCGG_TXT_ID_YES],
             g_all_text.s[DBDCGG_TXT_ID_NO], KEY_ENTER, KEY_ESC) == 1)
    { // overwrite question answered YES
      if(g_work_gradient.save(uconvert(g_dcgg_fname,U_CURRENT,
                                       NULL,U_ASCII,0))==0) // save success?
        g_unsaved_changes = false;
      gui_fg_color = makecol(0,0,0);
      gui_bg_color = makecol(255,255,255); // reset gui colors
    }
    else // overwrite question answered NO
    {
      gui_fg_color = makecol(0,0,0);
      gui_bg_color = makecol(255,255,255); // reset gui colors

      // Back to my custom GUI behaviour
      g_os_close_button_disabled = false;
      gui_behave_custom();
      return D_O_K;
    }
  }
    
  if(g_unsaved_changes) // saving failed, fall back to main dialog
  {
    show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                   g_all_text.s[DBDCGG_TXT_ID_SAVE_FAILED],
                   g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                   g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC);
    g_request_sub_dialog = SUB_DIALOG_MAIN;
  }

  // Back to my custom GUI behaviour
  g_os_close_button_disabled = false;
  gui_behave_custom();
  return D_O_K;
}

int file_save_as()
{
  bool cancelled = false;
  bool wrong_ext = false;

  // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
  acquire_screen();
  broadcast_dialog_message(D_REDRAW,0);
  blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
  release_screen();

  // Set standard GUI behaviour
  gui_behave_std();
  g_os_close_button_disabled = true;

  if(file_select_ex(g_all_text.s[DBDCGG_TXT_ID_SELECT_GRADIENT_FILE],
                    g_dcgg_fname,
                    g_all_exts.s[F_EXT_DCGG],G_FNAME_BUFSIZE,0,0)!=0)
  { // OK pressed
    if(ustricmp(g_all_exts.s[F_EXT_DCGG],
                get_extension(g_dcgg_fname)) == 0) // correct extension?
    {
      if(file_exists(g_dcgg_fname,FA_ARCH | FA_HIDDEN,NULL)) 
      {
        gui_fg_color = makecol(255,0,0);
        gui_bg_color = makecol(255,255,200); // set alert colors
        if(alert(g_all_text.s[DBDCGG_TXT_ID_QUESTION],
           g_all_text.s[DBDCGG_TXT_ID_OVERWRITE_QUESTION],
           g_dcgg_fname,
           g_all_text.s[DBDCGG_TXT_ID_YES],
           g_all_text.s[DBDCGG_TXT_ID_NO], KEY_ENTER, KEY_ESC) == 1)
        { // overwrite question answered YES
          if(g_work_gradient.save(uconvert(g_dcgg_fname,U_CURRENT,
                                       NULL,U_ASCII,0))==0) // save success?
          g_unsaved_changes = false;

          gui_fg_color = makecol(0,0,0);
          gui_bg_color = makecol(255,255,255); // reset gui colors
        }
        else // overwrite question answered NO
        {
          gui_fg_color = makecol(0,0,0);
          gui_bg_color = makecol(255,255,255); // reset gui colors

          // Back to my custom GUI behaviour
          g_os_close_button_disabled = false;
          gui_behave_custom();
          return D_O_K;
        }
      }
      else // file does not exist already, so just save it
      if(g_work_gradient.save(uconvert(g_dcgg_fname,U_CURRENT,
                                       NULL,U_ASCII,0))==0) // save success?
        g_unsaved_changes = false;
    }
    else // wrong file extension
      wrong_ext = true;
  }
  else // CANCEL pressed
    cancelled = true;

  // Back to my custom GUI behaviour
  g_os_close_button_disabled = false;
  gui_behave_custom();

  // if saving failed, fall back to main dialog
  if(((g_unsaved_changes)&&(!cancelled))||(wrong_ext))
  {
    if(wrong_ext)
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_SAVE_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_EXTENSION_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC);
    else
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_SAVE_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC);
    g_request_sub_dialog = SUB_DIALOG_MAIN;
  }

  return D_O_K;
}

// generic export bitmap helper function, taking a pointer to the desired
// extension and to the target filename
// returns -1 on not enough memory, 0 on succuesful rendering
// NOTE: calling functions are responsible to free the "g_gradient_export"
//       bitmap afterwards, if return value is 0
int file_export_bitmap(char *extension, char *fname)
{
  int ret = 0;
  int objnum = 0;
  // Set standard GUI behaviour
  gui_behave_std();
  g_os_close_button_disabled = true;

  position_dialog(g_exp_dimensions_dialog,
                  2*text_height(font),2*text_height(font));

  if(do_dialog(g_exp_dimensions_dialog,
               EXP_DIMENSIONS_DIALOG_ID_W_EDIT)
     == EXP_DIMENSIONS_DIALOG_ID_OK) // export dimensions dialog said OK?
  {
    // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
    acquire_screen();
    gui_behave_custom();
    dialog_message(g_main_dialog,D_REDRAW,0,&objnum);
    blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
    gui_behave_std();
    release_screen();
  
    BMP_T export_bmp;
    int w = (int)ustrtol(g_edit_field.current_expixw,NULL,10);
    int h = (int)ustrtol(g_edit_field.current_expixh,NULL,10);
    w = clamp_val<int>(1,w,4096);
    h = clamp_val<int>(1,h,4096);

    // kill previous export bitmap
    if(g_gradient_export)
    {
      destroy_bitmap(g_gradient_export);
      g_gradient_export = NULL;
    }

    // create new export bitmap
    g_gradient_export = create_bitmap(w,h);
    if(g_gradient_export == NULL) // creation failed?
    {
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_MEMORY_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC);
      // Back to my custom GUI behaviour
      g_os_close_button_disabled = false;
      gui_behave_custom();
      return -1; // not enough memory
    }

    // perform the rendering
    export_bmp.attach_al_bmp(g_gradient_export);
    if((g_work_gradient.render(export_bmp,0.0,render_callback,g_cb_msec,
                             lookup_callback,g_cb_msec)!=0)
       ||(g_work_gradient.get_attractor_count()<=0))
    { // rendering cancelled
      draw_place_holder(g_gradient_export);
    }

    // now ask for the filename
    if(file_select_ex(g_all_text.s[DBDCGG_TXT_ID_SELECT_BITMAP_FILE],
                      fname,
                      extension,G_FNAME_BUFSIZE,0,0)!=0)
    { // OK pressed
      if(ustricmp(extension,
                  get_extension(fname)) == 0) // correct extension?
      {
        if(file_exists(fname,FA_ARCH | FA_HIDDEN,NULL)) 
        {
          gui_fg_color = makecol(255,0,0);
          gui_bg_color = makecol(255,255,200); // set alert colors
          if(alert(g_all_text.s[DBDCGG_TXT_ID_QUESTION],
             g_all_text.s[DBDCGG_TXT_ID_OVERWRITE_QUESTION],
             fname,
             g_all_text.s[DBDCGG_TXT_ID_YES],
             g_all_text.s[DBDCGG_TXT_ID_NO], KEY_ENTER, KEY_ESC) == 1)
          { // overwrite question answered YES
            gui_fg_color = makecol(0,0,0);
            gui_bg_color = makecol(255,255,255); // reset gui colors
            ret = 0;
          }
          else // overwrite question answered NO
          {
            gui_fg_color = makecol(0,0,0);
            gui_bg_color = makecol(255,255,255); // reset gui colors

            // destroy export bitmap
            destroy_bitmap(g_gradient_export);
            g_gradient_export = NULL;
            ret = -2;
          }
        }//END of (file_exists)
        else // file does not exist already, so just save it
          ret = 0;
      }//END of (correct extension?)
      else // wrong extension
      {
        show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                       g_all_text.s[DBDCGG_TXT_ID_EXTENSION_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
        // destroy export bitmap
        destroy_bitmap(g_gradient_export);
        g_gradient_export = NULL;                       
        ret = -2;
      }
    }
    else // cancelled save bitmap file selection
    {
      // destroy export bitmap
      destroy_bitmap(g_gradient_export);
      g_gradient_export = NULL;
      ret = -2;
    }
  }
  else // export dimensions dialog said CANCEL
    ret = -2;

  // Back to my custom GUI behaviour
  g_os_close_button_disabled = false;
  gui_behave_custom();
  return ret;
}

int file_export_bmp()
{

  if(file_export_bitmap(g_all_exts.s[F_EXT_BMP],g_bmp_fname)==0)
  {
    if(save_bmp(g_bmp_fname,g_gradient_export,g_m_pal)!=0) // save bmp error
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
    destroy_bitmap(g_gradient_export);
    g_gradient_export = NULL;
  }
  return D_REDRAW;
}

int file_export_pcx()
{
  if(file_export_bitmap(g_all_exts.s[F_EXT_PCX],g_pcx_fname)==0)
  {
    if(save_pcx(g_pcx_fname,g_gradient_export,g_m_pal)!=0) // save pcx error
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
    destroy_bitmap(g_gradient_export);
    g_gradient_export = NULL;
  }
  return D_REDRAW;
}

int file_export_tga()
{
  if(file_export_bitmap(g_all_exts.s[F_EXT_TGA],g_tga_fname)==0)
  {
    if(save_tga(g_tga_fname,g_gradient_export,g_m_pal)!=0) // save bmp error
      show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                     g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                     g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
    destroy_bitmap(g_gradient_export);
    g_gradient_export = NULL;
  }
  return D_REDRAW;
}

// small function as helper for next function
void exp_code(string instancename)
{
  string filename = string(uconvert(g_cpp_fname,
                                    U_CURRENT,NULL,U_ASCII,0));

  if(g_work_gradient.export_code(instancename,filename)!=0)
    show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                   g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                   g_all_text.s[DBDCGG_TXT_ID_FILE_ERROR],
                   g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
}
int file_export_code()
{
  int objnum = 0;

  // Set standard GUI behaviour
  gui_behave_std();
  g_os_close_button_disabled = true;

  position_dialog(g_exp_code_dialog,
                  2*text_height(font),2*text_height(font));

  if(do_dialog(g_exp_code_dialog,
               EXP_CODE_DIALOG_ID_INSTANCENAME_EDIT)
     == EXP_CODE_DIALOG_ID_OK) // export code dialog said OK?
  {
    string exp_instname = string(uconvert(g_edit_field.exc_instancename,
                                 U_CURRENT,NULL,U_ASCII,0));
  
    // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
    acquire_screen();
    gui_behave_custom();
    dialog_message(g_main_dialog,D_REDRAW,0,&objnum);
    blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
    gui_behave_std();
    release_screen();
  
    if(file_select_ex(g_all_text.s[DBDCGG_TXT_ID_SELECT_CPP_FILE],
                      g_cpp_fname,g_all_exts.s[F_EXT_CPP],
                      G_FNAME_BUFSIZE,0,0)!=0)
    { // file selected and OK'ed
      if(ustricmp(g_all_exts.s[F_EXT_CPP],
                  get_extension(g_cpp_fname)) == 0) // correct extension?
      {
        if(file_exists(g_cpp_fname,FA_ARCH | FA_HIDDEN,NULL)) 
        {
          gui_fg_color = makecol(255,0,0);
          gui_bg_color = makecol(255,255,200); // set alert colors
          if(alert(g_all_text.s[DBDCGG_TXT_ID_QUESTION],
             g_all_text.s[DBDCGG_TXT_ID_OVERWRITE_QUESTION],
             g_cpp_fname,
             g_all_text.s[DBDCGG_TXT_ID_YES],
             g_all_text.s[DBDCGG_TXT_ID_NO], KEY_ENTER, KEY_ESC) == 1)
          { // overwrite question answered YES
            gui_fg_color = makecol(0,0,0);
            gui_bg_color = makecol(255,255,255); // reset gui colors
            exp_code(exp_instname); // export code
            // Back to my custom GUI behaviour
            g_os_close_button_disabled = false;
            gui_behave_custom();
            return D_REDRAW;
          }
          else // overwrite question answered NO
          {
            gui_fg_color = makecol(0,0,0);
            gui_bg_color = makecol(255,255,255); // reset gui colors   
          }
        }//END of (file_exists)
        
        // file does not exist already, so just export code to it
        exp_code(exp_instname);
        
      }//END of "if correct extension"
      else // wrong extension
        show_red_alert(g_all_text.s[DBDCGG_TXT_ID_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_EXPORT_FAILED],
                       g_all_text.s[DBDCGG_TXT_ID_EXTENSION_ERROR],
                       g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],KEY_ESC); 
      
    }//END of if(file_select_ex..)
    
  }//END of if(do_dialog..)
  
  // Back to my custom GUI behaviour
  g_os_close_button_disabled = false;
  gui_behave_custom();
  return D_REDRAW;
}

int file_exit()
{
  if(g_unsaved_changes)
  {
    g_usc_alert_caller = USC_ALERT_CALLER_EXIT;
    g_request_sub_dialog = SUB_DIALOG_CHANGE_ALERT;
    return D_CLOSE;
  }
  else
  {
    g_request_sub_dialog = SUB_DIALOG_NONE;
    return D_CLOSE;
  }
}

// Options Menu Items Control Functions
int options_select_language()
{
  bool new_language_ok = false;
 
  // restore backbuffer to avoid "graphical glitch"(purely visual cosmetic)
  acquire_screen();
  broadcast_dialog_message(D_REDRAW,0);
  blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
  release_screen();
 
  // Set standard GUI behaviour
  gui_behave_std();
  g_os_close_button_disabled = true;

  if(file_select_ex(g_all_text.s[DBDCGG_TXT_ID_SELECT_LANGUAGE_FILE],
                    g_dcgg_lang_fname,
                    g_all_exts.s[F_EXT_DCGG_LANG],G_FNAME_BUFSIZE,0,0)!=0)
  { // OK was pressed
    dbdcgg_strings g_new_text;
    int lang_load_ret = 0;
    null_strings(&g_new_text); // init new strings to null
    // load new strings
    lang_load_ret = reload_strings(uconvert(g_dcgg_lang_fname,
                                   U_CURRENT,NULL,U_ASCII,0),&g_new_text);
    if(lang_load_ret!=0) // load failure?
    {
      switch(lang_load_ret)
      {
        case -1:
        show_error_message("Error! Languagefile not found!","T&oo bad.", 'o');
        break;
        case -2:
        show_error_message("Error! Wrong languagefile version!",
                           "T&oo bad!",'o');
        break;
        case -3:
        show_error_message("Error! Languagefile is incomplete!",
                           "T&oo bad!",'o');
        break;
        case -4:
        show_error_message("Error! Not enough memory!",
                           "T&oo bad!",'o');
        break;
      }
    }
    else // loading new language successfull
    {
      // release old strings
      kill_strings(&g_all_text);
      // replace string pointers with newly loaded string pointers 
      g_all_text = g_new_text;
      attach_all_strings(); // sets GUI object string pointers
      // reset ok,cancel strings
      change_ok_cancel();
      new_language_ok = true;
    }
  }//(ok pressed)

  // Back to my custom GUI behaviour
  g_os_close_button_disabled = false;
  gui_behave_custom();
  
  if(new_language_ok)
  {
    broadcast_dialog_message(MSG_START,0);
    return D_REDRAW;
  }
  else
    return D_O_K;
}

int options_auto_recalculate()
{
  g_auto_recalculate = !g_auto_recalculate;
  if(g_auto_recalculate)
    g_options_menu[OPTIONS_MENU_ID_AUTO_RECALCULATE].flags=D_SELECTED;
  else
    g_options_menu[OPTIONS_MENU_ID_AUTO_RECALCULATE].flags=0;
  return D_O_K;
}

// Help Menu Items Control Functions
int help_manual()
{
  return D_O_K;
}

int help_about()
{
  g_request_sub_dialog = SUB_DIALOG_ABOUT;
  return D_CLOSE;
}

// certain button pressed Control Fucntions

void button_add()
{
  // Put a new default attractor to the end of the list
  g_work_gradient.add_ca("POINT,ATTRACT, 255,255,255, 1.0,1.0, 0.5,0.5, 0.1");
  // Select the newly added attractor
  g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d1
    = g_work_gradient.get_attractor_count()-1;
  // Set the attractor list scroll position, so that it is visible
  g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d2
    = g_work_gradient.get_attractor_count()-1
      - g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].h / 26;
}

void button_duplicate()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  
  // get the selected attractors description and add a new one based on that
  if(ca)
  {
    g_work_gradient.add_ca(g_work_gradient.get_as_text(current));
    // Select the newly added attractor
    g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d1
      = g_work_gradient.get_attractor_count()-1;
    // Set the attractor list scroll position, so that it is visible
    g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].d2
      = g_work_gradient.get_attractor_count()-1
        - g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].h / 26;
  }
}

void button_delete()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  g_work_gradient.rem_ca(current);
}

int button_recalculate()
{
  g_os_close_button_disabled = true;
  gui_behave_std();
  
  if((g_work_gradient.render(g_bmp_preview,0.0,render_callback,g_cb_msec,
                             lookup_callback,g_cb_msec)!=0)
     ||(g_work_gradient.get_attractor_count()<=0))
    draw_place_holder(g_gradient_preview);
  gui_behave_custom();
  g_os_close_button_disabled = false;
  return 0;
}
   
// callback function to show lookup table generation progress to user
//(so he does not think we crashed, in case we're taking our sweet time ;-) )
bool lookup_callback(int values_to_go, CA_TYPE type, int n)
{
  if(keyboard_needs_poll())
    poll_keyboard();
  if(mouse_needs_poll())
    poll_mouse();

  // check for keypress or mouseclick
  if((key[KEY_ESC])||
     ((mouse_b & 1)&&
     (mouse_y > g_h/2-text_height(font)*2-5)&&
     (mouse_y <= g_h/2+text_height(font)*2+5)))
  {
    if(key[KEY_ESC]) 
      while(key[KEY_ESC])
        if(keyboard_needs_poll())
          poll_keyboard();
    if(mouse_b & 1)
      while(mouse_b &1)
        if(mouse_needs_poll())
          poll_mouse();
    clear_keybuf();
    return false; 
  }

  // show the "values to process" status bar
  rectfill(g_back,0,g_h/2-text_height(font)*2-5,
                  g_w,g_h/2+text_height(font)*3+5,makecol(255,255,200));
  hline(g_back,0,g_h/2-text_height(font)*2-5,g_w,makecol(255,0,0));
  hline(g_back,0,g_h/2-text_height(font)-2,g_w,makecol(255,0,0));
  hline(g_back,0,g_h/2+text_height(font)*3+5,g_w,makecol(255,0,0));
  textout_centre_ex(g_back,font,g_all_text.s[DBDCGG_TXT_ID_CALCULATING],
                    g_w/2,g_h/2-text_height(font)*2-3,makecol(255,0,0),-1);
  textprintf_centre_ex(g_back,font,g_w/2,(g_h-text_height(font))/2,
                       makecol(255,0,0),-1,
                       uconvert("%s (%i)%s",U_ASCII,NULL,U_CURRENT,0),
                       g_all_text.s[DBDCGG_TXT_ID_LOOKUP_TABLE_FOR],
                       n,
                       g_all_text.s[DBDCGG_TXT_ID_GLOBAL+(int)type]);
  textprintf_centre_ex(g_back,font,g_w/2,g_h/2+text_height(font)+3,
                       makecol(255,0,0),-1,
                       uconvert("%s %12i",U_ASCII,NULL,U_CURRENT,0),
                       g_all_text.s[DBDCGG_TXT_ID_VALUES_TO_GO],
                       values_to_go);
  textprintf_centre_ex(g_back,font,g_w/2,g_h/2+2*text_height(font)+3,
                       makecol(255,0,0),-1,
                       uconvert("%s = %s",U_ASCII,NULL,U_CURRENT,0),
                       g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],
                       g_all_text.s[DBDCGG_TXT_ID_CANCEL]);
  acquire_screen();
  blit(g_back,screen,0,g_h/2-text_height(font)*2-5,
                     0,g_h/2-text_height(font)*2-5,
                     g_w,text_height(font)*5+11);
  release_screen();
  return true;  
}

// callback function to show render progress to user
//(so he does not think we crashed, in case we're taking our sweet time ;-) )
bool render_callback(unsigned long pixels_to_go)
{
  if(keyboard_needs_poll())
    poll_keyboard();
  if(mouse_needs_poll())
    poll_mouse();

  // check for keypress or mouseclick
  if((key[KEY_ESC])||
     ((mouse_b & 1)&&
     (mouse_y > g_h/2-text_height(font)*2-5)&&
     (mouse_y <= g_h/2+text_height(font)*2+5)))
  {
    if(key[KEY_ESC]) 
      while(key[KEY_ESC])
        if(keyboard_needs_poll())
          poll_keyboard();
    if(mouse_b & 1)
      while(mouse_b &1)
        if(mouse_needs_poll())
          poll_mouse();
    clear_keybuf();
    return false; 
  }

  // show the "pixels to process" status bar
  rectfill(g_back,0,g_h/2-text_height(font)*2-5,
                  g_w,g_h/2+text_height(font)*3+5,makecol(255,255,200));
  hline(g_back,0,g_h/2-text_height(font)*2-5,g_w,makecol(255,0,0)); 
  hline(g_back,0,g_h/2-text_height(font)-2,g_w,makecol(255,0,0));
  hline(g_back,0,g_h/2+text_height(font)*3+5,g_w,makecol(255,0,0));
  textout_centre_ex(g_back,font,g_all_text.s[DBDCGG_TXT_ID_CALCULATING],
                    g_w/2,g_h/2-text_height(font)*2-3,makecol(255,0,0),-1);
  textprintf_centre_ex(g_back,font,g_w/2,(g_h-text_height(font))/2,
                       makecol(255,0,0),-1,
                       uconvert("%s %12u",U_ASCII,NULL,U_CURRENT,0),
                       g_all_text.s[DBDCGG_TXT_ID_PIXELS_TO_PROCESS],
                       pixels_to_go);
  textprintf_centre_ex(g_back,font,g_w/2,g_h/2+2*text_height(font)+3,
                       makecol(255,0,0),-1,
                       uconvert("%s = %s",U_ASCII,NULL,U_CURRENT,0),
                       g_all_text.s[DBDCGG_TXT_ID_ESC_OR_CLICK],
                       g_all_text.s[DBDCGG_TXT_ID_CANCEL]);
  acquire_screen();
  blit(g_back,screen,0,g_h/2-text_height(font)*2-5,
                     0,g_h/2-text_height(font)*2-5,
                     g_w,text_height(font)*5+11);
  release_screen();
  return true;               
}

// draws a border rectangle and a cross to the whole size of the bitmap
void draw_place_holder(BITMAP *where)
{
  if(!where)
    return;
  clear_to_color(where,makecol(255,255,255));
  rect(where,0,0,where->w-1,where->h-1,makecol(0,0,0));
  line(where,0,0,where->w-1,where->h-1,makecol(0,0,0));
  line(where,0,where->h-1,where->w-1,0,makecol(0,0,0));
}

// destroy, recreates and reattaches the preview bitmap
// returns -1 on error, 0 on success
int reset_preview_bitmap()
{
  // destroy old preview bitmap
  if(g_gradient_preview)
  {
    destroy_bitmap(g_gradient_preview);
    g_gradient_preview = NULL;
    g_bmp_preview.attach_al_bmp(NULL);
  }
  
  // create new one
  g_gradient_preview = create_bitmap(g_work_gradient.get_w(),
                                     g_work_gradient.get_h());

  if(g_gradient_preview==NULL)
    return -1; // error

  // draw a black cross onto it
  draw_place_holder(g_gradient_preview);

  // re-attach it
  g_bmp_preview.attach_al_bmp(g_gradient_preview);

  // also reset the scroll position
  g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].d1 = -60; 
  g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].d2 = -40;
  return 0;
}

// zeroes all edit fields, set all coefficients to 0.0
int reset_edit_fields()
{
  int i=0;
  zero_edit_contents(&g_edit_field);
  for(i=0; i<G_MAX_COEFFICIENTS; i++)
    g_current_coefficients[i] = 0.0;
  return 0;
}

// calls all the "get_"s to update the interface
void dt_get_all()
{
  int i=0;
  // hide most input fields by default(other "dt_get_"s will re-activate some)
  for(i=MAIN_DIALOG_ID_SETTINGS+1; i<=MAIN_DIALOG_ID_UNIT_EDIT; i++)
    g_main_dialog[i].flags |= D_HIDDEN;

  dt_get_author();
  dt_get_title();
  dt_get_type();
  dt_get_mode();
  dt_get_color();
  dt_get_intensity();
  dt_get_exponent();
  dt_get_x();
  dt_get_y();
  dt_get_width();
  dt_get_height();
  dt_get_range();
  dt_get_angle();
  dt_get_highest_exp();
  dt_get_coefficients();
  dt_get_distance_mode();
  dt_get_unit();
  dt_get_dimensions();
}

// calls "dt_set_"s and other 
// to communicate changes made by any dialog item to the selected attractor
void dt_apply_changes(int ch_which)
{
  switch(ch_which) 
  {
    case G_CH_AUTHOR:
      dt_set_author();
    break;
    case G_CH_TITLE:
      dt_set_title();
    break;
    case G_CH_TYPE:
      dt_set_type();
    break;
    case G_CH_MODE:
      dt_set_mode();
    break;
    case G_CH_COLOR:
      dt_set_color();
    break;
    case G_CH_INTENSITY:
      dt_set_intensity();
    break;
    case G_CH_EXPONENT:
      dt_set_exponent();
    break;
    case G_CH_X:
      dt_set_x();
    break;
    case G_CH_Y:
      dt_set_y();
    break;
    case G_CH_WIDTH: // also for "radius"
      dt_set_width();
    break;
    case G_CH_HEIGHT:
      dt_set_height();
    break;
    case G_CH_RANGE:
      dt_set_range();
    break;
    case G_CH_ANGLE:
      dt_set_angle();
    break;
    case G_CH_HIGHEST_EXPONENT:
      dt_set_highest_exp();
    break;
    case G_CH_COEFFICIENTS:
      dt_set_coefficients();
    break;
    case G_CH_DISTANCE_MODE:
      dt_set_distance_mode();
    break;
    case G_CH_UNIT:
      dt_set_unit();
    break;
    case G_CH_DIMENSIONS:
      dt_set_dimensions();
    break;
    case G_CH_ADD:
      button_add();
    break;
    case G_CH_DUPLICATE:
      button_duplicate();
    break;
    case G_CH_DELETE:
      button_delete();
    break;
    default:
      ascii_red_alert("ERROR!","unknown CHANGE ID received",
                      "in dt_apply_changes",
                      "(click or press esc)",KEY_ESC);
    break;
  }
  if((g_auto_recalculate)&& 
     !((ch_which==G_CH_AUTHOR)||(ch_which==G_CH_TITLE))) 
  {
    button_recalculate();
    broadcast_dialog_message(MSG_DRAW,0);
  }
}

void dt_get_author()
{
  do_uconvert((char*)g_work_gradient.get_author().c_str(),U_UNICODE
              ,(char*)&g_edit_field.author,
              U_CURRENT,sizeof(g_edit_field.author));
}

void dt_set_author()
{
  wstring::value_type *conv 
    = (wstring::value_type *)uconvert(g_edit_field.author,U_CURRENT,NULL,
                                      U_UNICODE,0);
  g_work_gradient.set_author(conv);
}

void dt_get_title()
{
  do_uconvert((char*)g_work_gradient.get_title().c_str(),U_UNICODE
              ,(char*)&g_edit_field.title,
              U_CURRENT,sizeof(g_edit_field.title));
}

void dt_set_title()
{
  wstring::value_type *conv
    = (wstring::value_type *)uconvert(g_edit_field.title,U_CURRENT,NULL,
                                      U_UNICODE,0);
  g_work_gradient.set_title(conv);
}

void dt_get_type()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(!ca)
  {
    g_main_dialog[MAIN_DIALOG_ID_TYPE_BUTTON].dp = g_ttxt;
  }
  else
  {
    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_TYPE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_TYPE_BUTTON].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_TYPE_BUTTON].dp
      = g_all_text.s[DBDCGG_TXT_ID_GLOBAL+(int)ca->get_type()];

    g_type_dialog[TYPE_DIALOG_ID_LIST].d1 = (int)ca->get_type();
  }
}

void dt_set_type()
{
  CA_TYPE new_type = (CA_TYPE)g_type_dialog[TYPE_DIALOG_ID_LIST].d1;
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  unsigned int highest_exp 
    = ustrtol((char*)&g_edit_field.highest_exponent,NULL,10);
  unsigned int i;
  ostringstream rline;

  if(!ca)
    return;

  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);

    // write type text identifier to the string
    rline << get_ca_type_text(new_type) << " ,";
    // write parameters that every type uses to the string
    // (copying the values already stored in the attractor)
    rline << get_ca_mode_text(ca->get_mode()) << " ,"
          << ca->get_r() << " ," << ca->get_g() << " ,"
          << ca->get_b() << " ," << ca->get_colint() << " ,";

    // now write inidividual attractor type parameters to the string
    // (converting the values from the contents of the input fields)
    switch(new_type)
    {
      case CAT_GLOBAL:
        // do nothing
      break;      

      case CAT_POINT:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      case CAT_POINTE:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.width,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.height,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.angle,NULL) << "\n";
      break;

      case CAT_HBAR:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      case CAT_VBAR:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      case CAT_BAR:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.angle,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      case CAT_POLYNOMIAL:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.unit,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.angle,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << " ,"
              << get_d_mode_text(DM_TOFUNCTIONVALUE) << " ,"
              << highest_exp;
        for(i=0; i<=highest_exp; i++)
          rline << " ," << g_current_coefficients[highest_exp-i];
        rline << "\n";
      break;

      case CAT_CIRCLE:
      case CAT_CIRCLEF:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.width,NULL) << " ," /*radius*/
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      case CAT_ELLIPSE:
        rline << ustrtod((char*)&g_edit_field.exponent,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.x,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.y,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.width,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.height,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.angle,NULL) << " ,"
              << ustrtod((char*)&g_edit_field.range,NULL) << "\n";
      break;

      default:
        ascii_red_alert("ERROR!","unknown TYPE ID received","in dt_set_type",
                        "(click or press esc)",KEY_ESC);
        return; // abort on unknown type
      break;
    }

    rline.clear();
  }
  catch(ios_base::failure)
  {
    return; 
  }

  // set the new type with the string prepared above
  g_work_gradient.set_from_text(current,rline.str());
}

void dt_get_mode()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(!ca)
  {
    g_main_dialog[MAIN_DIALOG_ID_MODE_BUTTON].dp = g_ttxt;
  }
  else
  {
    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_MODE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_MODE_BUTTON].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_MODE_BUTTON].dp
      = g_all_text.s[DBDCGG_TXT_ID_ATTRACT+(int)ca->get_mode()];

    g_mode_dialog[MODE_DIALOG_ID_LIST].d1 = (int)ca->get_mode();
  }
}

void dt_set_mode()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_MODE new_mode = (CA_MODE)g_mode_dialog[MODE_DIALOG_ID_LIST].d1;
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    ca->set_mode(new_mode);
  }
}

void dt_get_color()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  float brightness;// arbitrary "brightness" value, used to determine, whether
                   // the caption on the color button should be black or white

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(!ca)
  {
    g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].dp = g_ttxt;
    g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].bg = makecol(0,0,0);
  }
  else
  {
    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_COLOR_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].dp
      = g_all_text.s[DBDCGG_TXT_ID_EDIT];
    g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].bg
      = makecol(ca->get_r(),ca->get_g(),ca->get_b());
    brightness = (float(ca->get_r()) / 255.0) +
                 1.5*(float(ca->get_g()) / 255.0) +
                 (float(ca->get_b()) / 255.0);
    if((ca->get_r()==ca->get_g()) && (ca->get_r()==ca->get_b()))
      brightness = 2.9 * (float(ca->get_r()) / 255.0);
    if(brightness < 1.5)
      g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].fg = makecol(255,255,255);
    else
      g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].fg = makecol(0,0,0);

    g_color_dialog[COLOR_DIALOG_ID_SLIDER_R].d2 = ca->get_r();
    g_color_dialog[COLOR_DIALOG_ID_SLIDER_G].d2 = ca->get_g();
    g_color_dialog[COLOR_DIALOG_ID_SLIDER_B].d2 = ca->get_b();
    g_color_dialog[COLOR_DIALOG_ID_BOX].bg 
      = makecol(ca->get_r(),ca->get_g(),ca->get_b());
  }
}

void dt_set_color()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    ca->set_rgb(g_color_dialog[COLOR_DIALOG_ID_SLIDER_R].d2,
                g_color_dialog[COLOR_DIALOG_ID_SLIDER_G].d2,
                g_color_dialog[COLOR_DIALOG_ID_SLIDER_B].d2);
  }
}

void dt_get_intensity()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    double intensity = ca->get_colint();
    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_INTENSITY_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_INTENSITY_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.intensity,sizeof(g_edit_field.intensity),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),intensity);
  }
}

void dt_set_intensity()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    // set new value from converted edit field text
    double new_intensity = ustrtod((char*)&g_edit_field.intensity,NULL);
    ca->set_colint(new_intensity);

    // reload the just set value to the interface
    dt_get_intensity();
    g_main_dialog[MAIN_DIALOG_ID_INTENSITY_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_INTENSITY_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_INTENSITY_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_exponent()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double exponent = 1.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
        ok = false;
      break;

      default:
        exponent = ((CA_POINT*)ca)->get_exp();
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_EXPONENT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_EXPONENT_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.exponent,sizeof(g_edit_field.exponent),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),exponent);
  }
}

void dt_set_exponent()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_exponent = ustrtod((char*)&g_edit_field.exponent,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
        ok = false;
      break;

      default:
        ((CA_POINT*)ca)->set_exp(new_exponent);
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_exponent();
    g_main_dialog[MAIN_DIALOG_ID_EXPONENT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_EXPONENT_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_EXPONENT_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_x()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double x = 1.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_HBAR:
        ok = false;
      break;

      default:
        x = ((CA_POINT*)ca)->get_x();
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_X_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_X_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.x,sizeof(g_edit_field.x),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),x);
  }
}

void dt_set_x()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_x = ustrtod((char*)&g_edit_field.x,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_HBAR:
        ok = false;
      break;

      default:
        ((CA_POINT*)ca)->set_x(new_x);
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_x();
    g_main_dialog[MAIN_DIALOG_ID_X_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_X_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_X_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_y()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double y = 1.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_VBAR:
        ok = false;
      break;

      default:
        y = ((CA_POINT*)ca)->get_y();
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_Y_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_Y_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.y,sizeof(g_edit_field.y),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),y);
  }
}

void dt_set_y()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_y = ustrtod((char*)&g_edit_field.y,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_VBAR:
        ok = false;
      break;

      default:
        ((CA_POINT*)ca)->set_y(new_y);
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_y();
    g_main_dialog[MAIN_DIALOG_ID_Y_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_Y_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_Y_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_width() // "radius" for CA_CIRCLE and CA_CIRCLEF
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double width = 1.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_CIRCLE:
      case CAT_CIRCLEF: // "radius" will be put in the "width" edit field
        width = ((CA_CIRCLE*)ca)->get_radius();
        g_main_dialog[MAIN_DIALOG_ID_WIDTH_DESC].dp
          = g_all_text.s[DBDCGG_TXT_ID_RADIUS];
      break;

      case CAT_POINTE:
      case CAT_ELLIPSE:
        width = ((CA_POINTE*)ca)->get_width();
        g_main_dialog[MAIN_DIALOG_ID_WIDTH_DESC].dp
          = g_all_text.s[DBDCGG_TXT_ID_WIDTH];
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_WIDTH_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_WIDTH_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.width,sizeof(g_edit_field.width),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),width);
  }
}

void dt_set_width()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_width = ustrtod((char*)&g_edit_field.width,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_CIRCLE:
      case CAT_CIRCLEF: // "radius" will be put in the "width" edit field
        ((CA_CIRCLE*)ca)->set_radius(new_width);
        
      break;

      case CAT_POINTE:
      case CAT_ELLIPSE:
        ((CA_POINTE*)ca)->set_width(new_width);
        
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;
    
    // reload the just set value to the interface
    dt_get_width();
    g_main_dialog[MAIN_DIALOG_ID_WIDTH_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_WIDTH_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_WIDTH_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_height()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double height = 1.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POINTE:
      case CAT_ELLIPSE:
        height = ((CA_POINTE*)ca)->get_height();
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_HEIGHT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_HEIGHT_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.height,sizeof(g_edit_field.height),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),height);
  }
}

void dt_set_height()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_height = ustrtod((char*)&g_edit_field.height,NULL); 

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POINTE:
      case CAT_ELLIPSE:
        ((CA_POINTE*)ca)->set_height(new_height);
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_height();
    g_main_dialog[MAIN_DIALOG_ID_HEIGHT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_HEIGHT_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_HEIGHT_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_range()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double range = 0.01;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_POINTE:
        ok = false;
      break;

      default:
        range = ((CA_POINT*)ca)->get_range();
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_RANGE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_RANGE_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.range,sizeof(g_edit_field.range),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),range);
  }
}

void dt_set_range()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_range = ustrtod((char*)&g_edit_field.range,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_GLOBAL:
      case CAT_POINTE:
        ok = false;
      break;

      default:
        ((CA_POINT*)ca)->set_range(new_range);
      break;
    }

    if(!ok) // current attractor did not have the field
      return;
    
    // reload the just set value to the interface
    dt_get_range();
    g_main_dialog[MAIN_DIALOG_ID_RANGE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_RANGE_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_RANGE_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_angle()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double angle = 0.0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_BAR:
        angle = ((CA_BAR*)ca)->get_angle();
      break;
      
      case CAT_POLYNOMIAL:
        angle = ((CA_POLYNOMIAL*)ca)->get_angle();
      break;

      case CAT_POINTE:
      case CAT_ELLIPSE:
        angle = ((CA_POINTE*)ca)->get_angle();
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_ANGLE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_ANGLE_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.angle,sizeof(g_edit_field.angle),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),angle);
  }
}

void dt_set_angle()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_angle = ustrtod((char*)&g_edit_field.angle,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_BAR:
        ((CA_BAR*)ca)->set_angle(new_angle);
      break;
      
      case CAT_POLYNOMIAL:
        ((CA_POLYNOMIAL*)ca)->set_angle(new_angle);
      break;

      case CAT_POINTE:
      case CAT_ELLIPSE:
        ((CA_POINTE*)ca)->set_angle(new_angle);
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_angle();
    g_main_dialog[MAIN_DIALOG_ID_ANGLE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_ANGLE_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ANGLE_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_highest_exp()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    unsigned int hep = 0;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        hep = ((CA_POLYNOMIAL*)ca)->get_hep();
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_HIGHEST_EXPONENT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_HIGHEST_EXPONENT_EDIT].flags ^= D_HIDDEN;
    if(hep > 0)
      uszprintf((char*)&g_edit_field.highest_exponent,
                sizeof(g_edit_field.highest_exponent),
                uconvert("%u",U_ASCII,NULL,U_CURRENT,0),hep);
    else
      memset((void *)&g_edit_field.highest_exponent,
             0,sizeof(g_edit_field.highest_exponent));
  }
}

void dt_set_highest_exp()
{
  // remember you have an arbitrary maximum value here: G_MAX_COEFFICIENTS-1
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  int i;

  if(ca)
  {
    if(ca->get_type() == CAT_POLYNOMIAL)
    {
      unsigned int new_hep = (unsigned int)
                              ustrtol((char*)&g_edit_field.highest_exponent,
                              NULL,10);/**/

      ((CA_POLYNOMIAL*)ca)->set_hep(clamp_val<unsigned int>(0,
                                    new_hep,G_MAX_COEFFICIENTS-1));
      // reload all currently stored coefficients into new attractor
      for(i=0; i<G_MAX_COEFFICIENTS; i++)
          ((CA_POLYNOMIAL*)ca)->set_co(i,g_current_coefficients[i]);
      
 
      // reload the just set value to the interface
      dt_get_highest_exp();
      g_main_dialog[MAIN_DIALOG_ID_HIGHEST_EXPONENT_DESC].flags ^= D_HIDDEN;
      g_main_dialog[MAIN_DIALOG_ID_HIGHEST_EXPONENT_EDIT].flags ^= D_HIDDEN;
      object_message(&g_main_dialog[MAIN_DIALOG_ID_HIGHEST_EXPONENT_EDIT],
                     MSG_DRAW,0);
      object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],
                     MSG_DRAW,0);
    }
  }
}

void dt_get_coefficients()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    unsigned int i = 0;
    bool ok = true;
    
    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        for(i=0; i<G_MAX_COEFFICIENTS; i++)
          g_current_coefficients[i] = ((CA_POLYNOMIAL*)ca)->get_co(i);
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_BUTTON].flags ^= D_HIDDEN;
  }
}

void dt_set_coefficients()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    unsigned int i = 0;
    bool ok = true;
    
    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        for(i=0; i<G_MAX_COEFFICIENTS; i++)
          ((CA_POLYNOMIAL*)ca)->set_co(i,g_current_coefficients[i]);
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_coefficients();
    g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_BUTTON].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);    
  }
}

void dt_get_distance_mode()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    D_MODE distance_mode = DM_TOFUNCTIONVALUE;
    bool ok = true;
    
    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        distance_mode = ((CA_POLYNOMIAL*)ca)->get_dmode();
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_DISTANCE_MODE_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON].dp
      = g_all_text.s[DBDCGG_TXT_ID_TO_FUNCVALUE+(int)distance_mode];

    g_dmode_dialog[DMODE_DIALOG_ID_LIST].d1 = (int)distance_mode;
  }
}

void dt_set_distance_mode()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);
  D_MODE new_dmode = (D_MODE)g_dmode_dialog[DMODE_DIALOG_ID_LIST].d1;
  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    if(ca->get_type() == CAT_POLYNOMIAL)
      ((CA_POLYNOMIAL*)ca)->set_dmode(new_dmode);
  }
}

void dt_get_unit()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double unit = 0.5;

    // check and get the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        unit = ((CA_POLYNOMIAL*)ca)->get_unit();
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // activate dialog objects and set content
    g_main_dialog[MAIN_DIALOG_ID_UNIT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_UNIT_EDIT].flags ^= D_HIDDEN;
    uszprintf((char*)&g_edit_field.unit,
              sizeof(g_edit_field.unit),
              uconvert("%.8f",U_ASCII,NULL,U_CURRENT,0),unit);
  }
}

void dt_set_unit()
{
  int current = d_attractor_list_proc(MSG_GET_ATTRACTOR,
                (DIALOG*)&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],0);

  CA_GLOBAL *ca = g_work_gradient.get_pca(current);
  if(ca)
  {
    bool ok = true;
    double new_unit = ustrtod((char*)&g_edit_field.unit,NULL);

    // check and set the field, if current attractor has it
    switch(ca->get_type())
    {
      case CAT_POLYNOMIAL:
        ((CA_POLYNOMIAL*)ca)->set_unit(new_unit);
      break;

      default:
        ok = false;
      break;
    }

    if(!ok) // current attractor did not have the field
      return;

    // reload the just set value to the interface
    dt_get_unit();
    g_main_dialog[MAIN_DIALOG_ID_UNIT_DESC].flags ^= D_HIDDEN;
    g_main_dialog[MAIN_DIALOG_ID_UNIT_EDIT].flags ^= D_HIDDEN;
    object_message(&g_main_dialog[MAIN_DIALOG_ID_UNIT_EDIT],MSG_DRAW,0);
    object_message(&g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST],MSG_DRAW,0);
  }
}

void dt_get_dimensions()
{
  uszprintf(g_edit_field.current_pixw,sizeof(g_edit_field.current_pixw),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),g_work_gradient.get_w());
  uszprintf(g_edit_field.current_pixh,sizeof(g_edit_field.current_pixh),
            uconvert("%i",U_ASCII,NULL,U_CURRENT,0),g_work_gradient.get_h());
}

void dt_set_dimensions()
{
  g_work_gradient.set_w((int)ustrtol(g_edit_field.current_pixw,NULL,10));
  g_work_gradient.set_h((int)ustrtol(g_edit_field.current_pixh,NULL,10));
  reset_preview_bitmap();
}

// Graphical Error Message (one line, one button, one key shortcut)
// (used to report errors at startup sequence)
void show_error_message(char *ascii_msg, char *ascii_btn,int key_code)
{
  int a_size=(strlen(ascii_msg)+1)*uwidth_max(U_CURRENT);
  int b_size=(strlen(ascii_btn)+1)*uwidth_max(U_CURRENT);
  char *err_msg=NULL;
  char *ok_btn=NULL;
  err_msg = new(nothrow) char[a_size];
  if(!err_msg)
    return; // error not enough mem for temporary string
  ok_btn = new(nothrow)char[b_size];
  if(!ok_btn)
  {
    delete []err_msg;
    return; // error not enough mem for temporary string
  }
  memset(err_msg,0,a_size);
  memset(ok_btn,0,b_size);
  do_uconvert(ascii_msg,U_ASCII,err_msg,U_CURRENT,a_size);
  do_uconvert(ascii_btn,U_ASCII,ok_btn,U_CURRENT,b_size);

  // Set to default GUI behaviour
  gui_behave_std();

  set_window_title(uconvert("Error!", U_ASCII, NULL, U_CURRENT,0));
  gui_fg_color = makecol(255,0,0);
  gui_bg_color = makecol(255,255,200);
  alert(err_msg,0,0,ok_btn,0,key_code,0);
  gui_fg_color = makecol(0,0,0);
  gui_bg_color = makecol(255,255,255);
  set_window_title(uconvert("DB's Dynamic Color Gradient Generator "
                            "("__DBDCGGV__")", U_ASCII, NULL, U_CURRENT,0));

  // Set back to my custom GUI behaviour
  gui_behave_custom();

  delete []err_msg;
  delete []ok_btn;
}

// THE RED ALERT ERROR MESSAGE (centered on screen to report critical errors)
// caption plus three lines of text and a key to cancel
void show_red_alert(char *uc_caption, char *uc1, char *uc2, char *uc3,
                    int key_code)
{
  BITMAP *bmp_gui = gui_get_screen(); // save current gui screen
  int th = text_height(g_default_font);
  int rs = SCREEN_H/2 - th*3;
  int re = SCREEN_H/2 + th*4;

  set_window_title(uconvert("CRITICAL ERROR!", U_ASCII, NULL, U_CURRENT,0));

  show_mouse(NULL);
  acquire_screen();
  rectfill(screen,0,rs, SCREEN_W-1, rs + th/2,makecol(255,0,0));
  rectfill(screen,0,rs + th/2 +1, SCREEN_W-1, re - th/2 - 1, makecol(0,0,0));
  rectfill(screen,0,rs + th/2 + 4 + th, SCREEN_W-1,
                    rs + 2*th + 4,makecol(255,0,0));
  rectfill(screen,0,re - th/2, SCREEN_W-1, re, makecol(255,0,0));

  if(uc_caption)
    textout_centre_ex(screen, g_default_font,
                      uc_caption, SCREEN_W/2, rs+th/2+2,
                      makecol(255,0,0), -1);
  if(uc1)
    textout_centre_ex(screen, g_default_font,
                      uc1, SCREEN_W/2, rs+th/2+4+th*2,
                      makecol(255,0,0), -1);
  if(uc2)
    textout_centre_ex(screen, g_default_font,
                      uc2, SCREEN_W/2, rs+th/2+4+th*3,
                      makecol(255,0,0), -1);
  if(uc3)
    textout_centre_ex(screen, g_default_font,
                      uc3, SCREEN_W/2, rs+th/2+4+th*4,
                      makecol(255,0,0), -1);
  release_screen();

  // Make sure everything is shown on screen
  show_mouse(screen);
  gui_set_screen(screen);

  // wait for keypress or mouseclick
  while(!key[key_code])
  {
    if(keyboard_needs_poll())
      poll_keyboard();
    if(mouse_needs_poll())
      poll_mouse();
    if((mouse_b & 1)&&(mouse_y > rs)&&(mouse_y <= re+1))
    {
      while(mouse_b & 1) // wait until button is released again
      {
        if(mouse_needs_poll())
          poll_mouse();
        rest(0);
      }
      break;
    }
    // Play nice with multitasking OS, yield some time slices to the CPU
    rest(0);
  }
  if(key[key_code]) while(key[key_code])// wait until key is released
  {
    if(keyboard_needs_poll())
      poll_keyboard();
    rest(0);
  }

  set_window_title(uconvert("DB's Dynamic Color Gradient Generator "
                            "("__DBDCGGV__")", U_ASCII, NULL, U_CURRENT,0));  

  // reset previous gui screen
  show_mouse(NULL);
  gui_set_screen(bmp_gui);
}

// Same as above but takes ascii strings as params
//(used only in beta version)
void ascii_red_alert(char *uc_caption, char *uc1, char *uc2, char *uc3,
                     int key_code)
{
  char cap[512];
  char c1[512];
  char c2[512];
  char c3[512];
  memset(cap,0,512);memset(c1,0,512);memset(c2,0,512);memset(c3,0,512);
  if(uc_caption) 
    do_uconvert(uc_caption,U_ASCII,cap,U_CURRENT,512);
  if(uc1)
    do_uconvert(uc1,U_ASCII,c1,U_CURRENT,512);
  if(uc2)
    do_uconvert(uc2,U_ASCII,c2,U_CURRENT,512);
  if(uc3)
    do_uconvert(uc3,U_ASCII,c3,U_CURRENT,512);
  show_red_alert(cap,c1,c2,c3,key_code);
}

// Used to scroll the GUI only every "G_SCROLL_MSEC" milliseconds
void scroll_gui_callback()
{
  if(!g_scroll_dir)
    return;
  if(g_scroll_dir & G_SCROLL_RIGHT)
    g_scroll_x = clamp_val<int>(0,g_scroll_x+G_SCROLL_PIX,G_BACK_W-g_w);
  if(g_scroll_dir & G_SCROLL_DOWN)
    g_scroll_y = clamp_val<int>(0,g_scroll_y+G_SCROLL_PIX,G_BACK_H-g_h);
  if(g_scroll_dir & G_SCROLL_LEFT)
    g_scroll_x = clamp_val<int>(0,g_scroll_x-G_SCROLL_PIX,G_BACK_W-g_w);
  if(g_scroll_dir & G_SCROLL_UP)
    g_scroll_y = clamp_val<int>(0,g_scroll_y-G_SCROLL_PIX,G_BACK_H-g_h);
}
END_OF_FUNCTION(scroll_gui_callback);

// Used by Allegro's close button re-mapping
void close_button_callback()
{
  if(!g_os_close_button_disabled)
    g_os_close_button_pressed = true;
}
END_OF_FUNCTION(close_button_callback);

// Writes 0 to every character in the specified edit_contents structure
void zero_edit_contents(edit_contents *which)
{
  #define _zero_member(x) memset(which->x,0,sizeof(which->x))
  _zero_member(author);
  _zero_member(title);
  _zero_member(intensity);
  _zero_member(exponent);
  _zero_member(x);
  _zero_member(y);
  _zero_member(width);
  _zero_member(height);
  _zero_member(range);
  _zero_member(angle);
  _zero_member(highest_exponent);
  _zero_member(unit);
  _zero_member(current_coeff);
  _zero_member(current_pixw);
  _zero_member(current_pixh);
  _zero_member(current_expixw);
  _zero_member(current_expixh);
  _zero_member(coefficients);
  _zero_member(exc_instancename);
  #undef _zero_me
}

// Helper function to disallow any modes bigger than the needed backbuffer
// and also to disallow paletted mode
int mode_filter(int card, int w, int h, int c_depth)
{
  //if((w>G_BACK_W)||(h>G_BACK_H)||(c_depth<8)||(w<G_SAFE_W)||(h<G_SAFE_H))
  if((c_depth<8)||(h<G_SAFE_H)||(w<G_SAFE_W))
    return -1;
  else return 0;
}

// Helper function to reset OK,CANCEL language specific text
void change_ok_cancel()
{
  char config_override[1024];// should be large enough to set ok,cancel texts
  // set OK and CANCEL button texts for allegro standard file select dialog
  memset(config_override,0,sizeof(config_override));
  uszprintf(config_override,sizeof(config_override),
            uconvert("[language]\nOK=%s\nCancel=%s\n",U_ASCII,
                     NULL,U_CURRENT,0),
            g_all_text.s[DBDCGG_TXT_ID_OK],
            g_all_text.s[DBDCGG_TXT_ID_CANCEL]);
  // (the following function makes a copy of config_override and applys the
  //  information found in there to the Allegro standard dialogs)
  set_config_data(config_override,sizeof(config_override));
}

/* (following two function are removed, because they rely on a portion of
    Allegro that isn't fully sys.indep. (the mickeys to be precise)
// replacement function for position_mouse to help with initializing and
// maintaining a more flexible mouse_position determination
void position_mouse_override(int x, int y)
{
  position_mouse(x,y);
  g_abs_mx = x;
  g_abs_my = y;
  g_mickx = 0;
  g_micky = 0;
  if((g_abs_mx<0)||(g_abs_mx>=SCREEN_W)||(g_abs_my<0)||(g_abs_my>=SCREEN_H))
    g_mouse_inside = false;
  else
    g_mouse_inside = true;
}

// updates the internal mouse postion according to the mickeys reported
// sets a global variable report the "mouse is inside the window" status
// (can get negative or larger than the dimensions of allegros "screen")
// (in fullscreen mode g_mouse_inside is not trustable, so you have to
//  use the usual mouse_x, mouse_y from allegro in fullscreen mode)
void update_abs_mousepos()
{
  if(mouse_needs_poll())
    poll_mouse();
    
  get_mouse_mickeys(&g_mickx,&g_micky);
  g_abs_mx += g_mickx;
  g_abs_my += g_micky;
  if((g_abs_mx<0)||(g_abs_mx>=SCREEN_W)||(g_abs_my<0)||(g_abs_my>=SCREEN_H))
    g_mouse_inside = false;
  else
    g_mouse_inside = true;
}
*/

// Init and load everything
int initialize()
{
  char *df_name = NULL; // datafile name
  int df_size = 0;
  int lang_load_ret = 0;

  g_runlevel = RL_NOTHING_OK;

  // Set UNICODE Format
  set_uformat(U_UNICODE);
  //set_uformat(U_UTF8);// *** left in from testing different formats

  // init allegro 
  if (allegro_init()!=0)
  {
    return -1;
  }
  g_runlevel = RL_ALLEGRO_OK;
  
  // init keyboard
  if (install_keyboard()!=0) 
  {
    return -2;
  }
  three_finger_flag = false; // disable Allegros CTRL+ALT+END termination
  g_runlevel = RL_KEYBOARD_OK;

  // init mouse
  if (install_mouse()==-1)
  {
    return -3;
  }
  g_runlevel = RL_MOUSE_OK;
  disable_hardware_cursor();

  // init gfx subsystem
  set_color_depth(32); //safe color depth
  // try windowed
  if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,G_SAFE_W,G_SAFE_H,0,0)!=0)
  {
    // try fullscreen, if windowed failed
    if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,G_SAFE_W,G_SAFE_H,0,0)!=0)
    {
      return -4;
    }
  }

  gui_fg_color = makecol(0, 0, 0);
  gui_mg_color = makecol(128, 128, 128);
  gui_bg_color = makecol(255, 255, 255);
  // now let the user choose the gfx mode
  set_window_title(uconvert("Please select GFX Mode:", U_ASCII, 
                            NULL, U_CURRENT,0));
  if(gfx_mode_select_filter(&g_card,&g_w,&g_h,&g_bpp,&mode_filter)!=0)
  {
    set_color_depth(g_bpp);
    if(set_gfx_mode(g_card,g_w,g_h,0,0)!=0) // set user selected mode
    {
      // set safe mode to report error to user
      set_color_depth(8);
      if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,G_SAFE_W,G_SAFE_H,0,0)!=0)
      {
        // try fullscreen, if windowed failed
        if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,G_SAFE_W,G_SAFE_H,0,0)!=0)
        {
          return -4;
        }
      }
      show_error_message("Error! Can not set selected mode.",
                         "T&oo bad.", 'o');
      return -4;
    }
  } 
  else
    return -5; // user pressed cancel on select gfx mode
  g_default_font = font;
  g_runlevel = RL_GFXMODE_OK;

  set_window_title(uconvert("DB's Dynamic Color Gradient Generator "
                            "("__DBDCGGV__")", U_ASCII, NULL, U_CURRENT,0));
  set_display_switch_mode(SWITCH_BACKGROUND);

  // init backbuffer(s)
  // large backbuffer for GUI
  // adjust backbuffer size first, 
  // if resolution is bigger than the default(800x600) backbuffer res.
  if(SCREEN_W>G_BACK_W)
    G_BACK_W=SCREEN_W;
  if(SCREEN_H>G_BACK_H)
    G_BACK_H=SCREEN_H;  
  g_back = create_bitmap(G_BACK_W,G_BACK_H);
  if(!g_back)
  {
    show_error_message("Error! Not enough memory.","T&oo bad.", 'o');
    return -6; 
  }
  // small backbuffer for mouse
  g_msafe = create_bitmap(mouse_sprite->w,mouse_sprite->h);
  if(!g_msafe)
  {
    destroy_bitmap(g_back);
    g_back=NULL;
    show_error_message("Error! Not enough memory.","T&oo bad.", 'o');
    return -6;
  }
  g_runlevel = RL_BACKBUFFER_OK;

  // install close button callback
  g_os_close_button_disabled = false;
  g_os_close_button_pressed = false;
  g_os_close_button_ignored = false;
  LOCK_VARIABLE(g_os_close_button_disabled);
  LOCK_VARIABLE(g_os_close_button_pressed);
  LOCK_FUNCTION(close_button_callback);
  if(set_close_button_callback(close_button_callback)!=0)
  {
    g_os_close_button_ignored = true;
  }

  /* load default strings(*** change later to read language file as specified
                              by a variable in an allegro cfg file ***) */
  null_strings(&g_all_text); // init strings to null
  lang_load_ret = reload_strings(G_DEFAULT_LANGUAGE_FILE,&g_all_text);
  if(lang_load_ret!=0)
  {
    switch(lang_load_ret)
    {
      case -1:
      show_error_message("Error! Languagefile not found!","T&oo bad.", 'o');
      break;
      case -2:
      show_error_message("Error! Wrong languagefile version!",
                         "T&oo bad!",'o');
      break;
      case -3:
      show_error_message("Error! Languagefile is incomplete!",
                         "T&oo bad!",'o');
      break;
      case -4:
      show_error_message("Error! Not enough memory!",
                         "T&oo bad!",'o');
      break;
    }
    return -7;
  }
  attach_all_strings(); // sets GUI object string pointers
  zero_edit_contents(&g_edit_field); // clears all edit box contents

  // set OK and CANCEL button texts for allegro standard file select dialog
  change_ok_cancel();

  // set file extensions strings to current unicode format
  null_ext(&g_all_exts);
  if(convert_exts(&g_all_exts)!=0)
  {
    show_error_message("Error! Not enough memory!",
                       "T&oo bad!",'o');
    kill_strings(&g_all_text);
    return -7;
  }  
  g_runlevel = RL_TEXT_OK;

  // before loading the data file
  // set color fake truecolor palette and lookup table for 8bit mode
  if(g_bpp == 8)
  {
    generate_332_palette(g_m_pal);
    
    // experimental grayscale mode (activated by holding shift on OK click)
    if(keyboard_needs_poll())
      poll_keyboard();
    rest(250);
    if(key[KEY_LSHIFT] || key[KEY_RSHIFT])
    for(int i=0; i<PAL_SIZE-1; i++)
    {
      RGB col;
      col.r = i/4; col.g = i/4; col.b = i/4; col.filler = 0;
      g_m_pal[i] = col;
    }

    set_palette(g_m_pal);
    create_rgb_table(&g_rgb_map,g_m_pal,NULL);    
    rgb_map = &g_rgb_map;
    set_color_conversion(COLORCONV_TOTAL | COLORCONV_DITHER);

    // fix wrong mouse cursor colors by letting allegro recreate its cursors
    remove_mouse();
    install_mouse();
  }

  // load datafile
  g_all_data=NULL;
  df_size = (strlen(G_DATAFILE_NAME)+1)*uwidth_max(U_CURRENT);
  df_name = NULL;
  df_name = new(nothrow)char[df_size];
  if(!df_name)
    return -8; // not enough memory for temporary datafile name string
  memset(df_name,0,df_size);
  do_uconvert(G_DATAFILE_NAME,U_ASCII,df_name,U_CURRENT,df_size);

  g_all_data = load_datafile(df_name);
  delete []df_name;
  if(!g_all_data)
  {
    show_error_message("Error! Datafile not found.","T&oo bad.", 'o');
    return -8;
  }

  // get the three strings from the datafile object "BITMAP_DBLOGO"
  memset(g_programme_auth,0,sizeof(g_programme_auth));
  do_uconvert(get_datafile_property(g_all_data+BITMAP_DBLOGO,
                                    DAT_ID('A','U','T','H')),
              U_CURRENT,g_programme_auth,U_CURRENT, sizeof(g_programme_auth));
  memset(g_programme_mail,0,sizeof(g_programme_mail));
  do_uconvert(get_datafile_property(g_all_data+BITMAP_DBLOGO,
                                    DAT_ID('M','A','I','L')),
              U_CURRENT,g_programme_mail,U_CURRENT, sizeof(g_programme_mail));
  memset(g_programme_home,0,sizeof(g_programme_home));
  do_uconvert(get_datafile_property(g_all_data+BITMAP_DBLOGO,
                                    DAT_ID('H','O','M','E')),
              U_CURRENT,g_programme_home,U_CURRENT, sizeof(g_programme_home));
  // also get the string from the datafile object "BITMAP_ALOGO"
  memset(g_allegro_home,0,sizeof(g_allegro_home));
  do_uconvert(get_datafile_property(g_all_data+BITMAP_ALOGO,
                                    DAT_ID('H','O','M','E')),
              U_CURRENT,g_allegro_home,U_CURRENT, sizeof(g_allegro_home));
  // set programme title for about dialog
  memset(g_prog_title,0,sizeof(g_prog_title));
  do_uconvert(G_PROGRAMME_NAME,U_ASCII,
              g_prog_title,U_CURRENT,sizeof(g_prog_title));

  // set some special button graphics
  g_main_dialog[MAIN_DIALOG_ID_BMP_UP_BUTTON].dp 
    = g_all_data[BUTTON_UP_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_UP_BUTTON].dp2 
    = g_all_data[BUTTON_UP_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_DOWN_BUTTON].dp 
    = g_all_data[BUTTON_DOWN_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_DOWN_BUTTON].dp2 
    = g_all_data[BUTTON_DOWN_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_LEFT_BUTTON].dp 
    = g_all_data[BUTTON_LEFT_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_LEFT_BUTTON].dp2 
    = g_all_data[BUTTON_LEFT_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_RIGHT_BUTTON].dp 
    = g_all_data[BUTTON_RIGHT_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_BMP_RIGHT_BUTTON].dp2 
    = g_all_data[BUTTON_RIGHT_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_UP_BUTTON].dp
    = g_all_data[BUTTON_S_UP_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_UP_BUTTON].dp2
    = g_all_data[BUTTON_S_UP_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_DOWN_BUTTON].dp
    = g_all_data[BUTTON_S_DOWN_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_DOWN_BUTTON].dp2
    = g_all_data[BUTTON_S_DOWN_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEUP_BUTTON].dp
    = g_all_data[BUTTON_M_UP_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEUP_BUTTON].dp2
    = g_all_data[BUTTON_M_UP_1].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEDOWN_BUTTON].dp
    = g_all_data[BUTTON_M_DOWN_0].dat;
  g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEDOWN_BUTTON].dp2
    = g_all_data[BUTTON_M_DOWN_1].dat;

  // set some special bitmap graphics
  g_about_dialog[ABOUT_DIALOG_ID_DB_LOGO].dp = g_all_data[BITMAP_DBLOGO].dat;
  g_about_dialog[ABOUT_DIALOG_ID_ALLEGRO_LOGO].dp 
    = g_all_data[BITMAP_ALOGO].dat;

  g_runlevel = RL_DATA_OK;
  
  //font = (FONT*)g_all_data[FONT_BIG].dat;

  LOCK_FUNCTION(scroll_gui_callback);
  LOCK_FUNCTION(clamp_val<int>);
  LOCK_VARIABLE(g_scroll_x);
  LOCK_VARIABLE(g_scroll_y);
  if(install_int_ex(&scroll_gui_callback,MSEC_TO_TIMER(G_SCROLL_MSEC))!=0)
  {
    show_error_message("Error! Can't install scroll timer.","T&oo bad.", 'o');
    return -9;
  }
  g_runlevel = RL_SCROLL_TIMER_OK;

  return 0;
}


// Free and close everything
void deinitialize()
{
  set_close_button_callback(NULL);

  if(g_runlevel >= RL_SCROLL_TIMER_OK)
  {
    remove_int(&scroll_gui_callback);
  }

  if(g_runlevel >= RL_DATA_OK)
  {
    unload_datafile(g_all_data);
    g_all_data = NULL;
  }
  if(g_runlevel >= RL_TEXT_OK)
  {
    kill_strings(&g_all_text);
    kill_ext(&g_all_exts);
  }
  if(g_runlevel >= RL_BACKBUFFER_OK)
  {
    if(g_gradient_export)
      destroy_bitmap(g_gradient_export);
    g_gradient_export = NULL;
    if(g_gradient_preview)
      destroy_bitmap(g_gradient_preview);
    g_gradient_preview = NULL;
    destroy_bitmap(g_msafe);
    g_msafe = NULL;
    destroy_bitmap(g_back);
    g_back = NULL;
  }
  if(g_runlevel >= RL_GFXMODE_OK)
  {
    set_gfx_mode(GFX_TEXT,0,0,0,0);
  }
  if(g_runlevel >= RL_MOUSE_OK)
  {
    remove_mouse();
  }
  if(g_runlevel >= RL_KEYBOARD_OK)
  {
    remove_keyboard();
  }
  if(g_runlevel >= RL_ALLEGRO_OK)
  {
    allegro_exit();
  }
}


// Function to load all text data from a UTF-8-encoded-Unicode textfile
/* (when this is called for the first time, all strings in the out_table
    MUST BE SET TO NULL) */
int reload_strings(string filename, dbdcgg_strings *out_table)
{
  FILE *in = NULL;
  wstring temp; /* DO NOT CHANGE THE TYPE OF THIS VAR!(if you do, you'll have
                   to change the "do_uconvert" call in this func as well) */
  string version; // used to identify valid language files
  unsigned int tmp_size;
  int i;

  in = fopen(filename.c_str(),"r");
  if(!in)
    return -1; // file error!

  // kill old strings first
  kill_strings(out_table);

  // Skip the first two lines of the language file
  if(readline_utf8(in,&temp)<0){fclose(in);return -1;}
  if(readline_utf8(in,&temp)<0){fclose(in);return -1;}
  
  // read version information
  if(readline_utf8(in,&version)<0){fclose(in);return -1;}
  if(version.compare(G_LANGUAGE_FILE_VERSION)!=0)
  {
    fclose(in);
    return -2; // error language file has incompatible version id
  }

  // Read all text lines into internal strings
  for(i=0; i<DBDCGG_NUM_STRINGS; i++)
  {
    if(readline_utf8(in,&temp)<0) // error language file incomplete
    {
      fclose(in);
      kill_strings(out_table);
      return -3;
    }
    else // line read successfully
    {
      // Allocate memory for Allegro Unicode string
      tmp_size = (temp.length()+1)*uwidth_max(U_CURRENT);
      out_table->s[i] = NULL;
      out_table->s[i] = new(nothrow)char[tmp_size];
      if(!(out_table->s[i])) // error not enough memory for current string
      {
        fclose(in);
        kill_strings(out_table);
        return -4;
      }
      else // now convert to Allegro's format
      {
        memset(out_table->s[i],0,tmp_size);
        do_uconvert((char *)temp.c_str(),U_UNICODE,
                    out_table->s[i],U_CURRENT,tmp_size);
      }
    }
  }

  // All language specific strings are loaded and converted now.

  fclose(in);
  return 0;
}

// same as above but just converts from constants to current unicode format
//(also pointers in out table must be NULL, when this is called first time)
int convert_exts(dbdcgg_ext *out_table)
{
  unsigned int tmp_size = 0;
  bool failed = false;

  // Convert _BMP extension
  tmp_size = (sizeof(__BMP_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_BMP] = NULL;
  out_table->s[F_EXT_BMP] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_BMP])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_BMP],0,tmp_size);
        do_uconvert(__BMP_EXT__,U_ASCII,
                    out_table->s[F_EXT_BMP],U_CURRENT,tmp_size);
  }
 
  // Convert _PCX extension
  tmp_size = (sizeof(__PCX_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_PCX] = NULL;
  out_table->s[F_EXT_PCX] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_PCX])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_PCX],0,tmp_size);
        do_uconvert(__PCX_EXT__,U_ASCII,
                    out_table->s[F_EXT_PCX],U_CURRENT,tmp_size);
  }

  // Convert _TGA extension
  tmp_size = (sizeof(__TGA_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_TGA] = NULL;
  out_table->s[F_EXT_TGA] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_TGA])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_TGA],0,tmp_size);
        do_uconvert(__TGA_EXT__,U_ASCII,
                    out_table->s[F_EXT_TGA],U_CURRENT,tmp_size);
  }

  // Convert _DCGG extension
  tmp_size = (sizeof(__DCGG_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_DCGG] = NULL;
  out_table->s[F_EXT_DCGG] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_DCGG])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_DCGG],0,tmp_size);
        do_uconvert(__DCGG_EXT__,U_ASCII,
                    out_table->s[F_EXT_DCGG],U_CURRENT,tmp_size);
  }

  // Convert _DCGG_LANG extension
  tmp_size = (sizeof(__DCGG_LANG_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_DCGG_LANG] = NULL;
  out_table->s[F_EXT_DCGG_LANG] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_DCGG_LANG])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_DCGG_LANG],0,tmp_size);
        do_uconvert(__DCGG_LANG_EXT__,U_ASCII,
                    out_table->s[F_EXT_DCGG_LANG],U_CURRENT,tmp_size);
  }

  // Convert _CPP extension
  tmp_size = (sizeof(__CPP_EXT__)+1)*uwidth_max(U_CURRENT);
  out_table->s[F_EXT_CPP] = NULL;
  out_table->s[F_EXT_CPP] = new(nothrow)char[tmp_size];
  if(!out_table->s[F_EXT_CPP])
    failed = true;
  else
  {
    memset(out_table->s[F_EXT_CPP],0,tmp_size);
        do_uconvert(__CPP_EXT__,U_ASCII,
                    out_table->s[F_EXT_CPP],U_CURRENT,tmp_size);
  }

  if(failed)
  {
    kill_ext(out_table);
    return -1;
  }
  else
    return 0;
}

// Function to bind the loaded strings to dialog/menu items
void attach_all_strings()
{
  MENU *mm = g_main_menu;
  MENU *fm = g_file_menu;
  MENU *om = g_options_menu;
  MENU *hm = g_help_menu;
  DIALOG *uscd = g_unsaved_dialog;
  DIALOG *st = g_type_dialog;
  DIALOG *sm = g_mode_dialog;
  DIALOG *sc = g_color_dialog;
  DIALOG *sdm = g_dmode_dialog;
  DIALOG *ec = g_coefficients_dialog;
  DIALOG *sd = g_dimensions_dialog;
  DIALOG *sxd = g_exp_dimensions_dialog;
  DIALOG *sxc = g_exp_code_dialog;
  DIALOG *md = g_main_dialog;
  DIALOG *ad = g_about_dialog;
  dbdcgg_strings *at = &g_all_text;

  // Menubar 
  mm[MAIN_MENU_ID_FILE].text = at->s[DBDCGG_TXT_ID_FILE];
  mm[MAIN_MENU_ID_OPTIONS].text = at->s[DBDCGG_TXT_ID_OPTIONS];
  mm[MAIN_MENU_ID_HELP].text = at->s[DBDCGG_TXT_ID_HELP];

  // File menu
  fm[FILE_MENU_ID_NEW].text = at->s[DBDCGG_TXT_ID_NEW];
  fm[FILE_MENU_ID_LOAD].text = at->s[DBDCGG_TXT_ID_LOAD];
  fm[FILE_MENU_ID_SAVE].text = at->s[DBDCGG_TXT_ID_SAVE];
  fm[FILE_MENU_ID_SAVE_AS].text = at->s[DBDCGG_TXT_ID_SAVE_AS];
  fm[FILE_MENU_ID_EXPORT_BMP].text = at->s[DBDCGG_TXT_ID_EXPORT_BMP];
  fm[FILE_MENU_ID_EXPORT_PCX].text = at->s[DBDCGG_TXT_ID_EXPORT_PCX];
  fm[FILE_MENU_ID_EXPORT_TGA].text = at->s[DBDCGG_TXT_ID_EXPORT_TGA];
  fm[FILE_MENU_ID_EXPORT_CODE].text = at->s[DBDCGG_TXT_ID_EXPORT_CODE];
  fm[FILE_MENU_ID_EXIT].text = at->s[DBDCGG_TXT_ID_EXIT];

  // Options menu
  om[OPTIONS_MENU_ID_SELECT_LANGUAGE_FILE].text
    = at->s[DBDCGG_TXT_ID_SELECT_LANGUAGE_FILE];
  om[OPTIONS_MENU_ID_AUTO_RECALCULATE].text
    = at->s[DBDCGG_TXT_ID_AUTO_RECALCULATE];

  // Help menu
  hm[HELP_MENU_ID_MANUAL].text = at->s[DBDCGG_TXT_ID_MANUAL];
  hm[HELP_MENU_ID_ABOUT].text = at->s[DBDCGG_TXT_ID_ABOUT];

  // Unsaved changes dialog
  uscd[UNSAVED_DIALOG_ID_CAPTION].dp = at->s[DBDCGG_TXT_ID_WARNING];
  uscd[UNSAVED_DIALOG_ID_LINE].dp = at->s[DBDCGG_TXT_ID_UNSAVED_CHANGES];
  uscd[UNSAVED_DIALOG_ID_SAVE_BUTTON].dp = at->s[DBDCGG_TXT_ID_SAVE];
  uscd[UNSAVED_DIALOG_ID_SAVE_AS_BUTTON].dp = at->s[DBDCGG_TXT_ID_SAVE_AS];
  uscd[UNSAVED_DIALOG_ID_IGNORE_BUTTON].dp = at->s[DBDCGG_TXT_ID_IGNORE];
  uscd[UNSAVED_DIALOG_ID_CANCEL_BUTTON].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Set Type dialog
  st[TYPE_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  st[TYPE_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Set Mode dialog
  sm[MODE_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sm[MODE_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Set Color dialog
  sc[COLOR_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sc[COLOR_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Set Distance Mode dialog
  sdm[DMODE_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sdm[DMODE_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Edit Coefficients dialog
  ec[COEFFICIENTS_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  ec[COEFFICIENTS_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];

  // Set Dimensions dialog
  sd[DIMENSIONS_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sd[DIMENSIONS_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];
  sd[DIMENSIONS_DIALOG_ID_W_DESC].dp = at->s[DBDCGG_TXT_ID_WIDTH];
  sd[DIMENSIONS_DIALOG_ID_H_DESC].dp = at->s[DBDCGG_TXT_ID_HEIGHT];
 
  // Set Export Dimensions dialog
  sxd[EXP_DIMENSIONS_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sxd[EXP_DIMENSIONS_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];
  sxd[EXP_DIMENSIONS_DIALOG_ID_CAPTION].dp 
    = at->s[DBDCGG_TXT_ID_EXPORT_BITMAP_DIMENSIONS];
  sxd[EXP_DIMENSIONS_DIALOG_ID_W_DESC].dp = at->s[DBDCGG_TXT_ID_WIDTH];
  sxd[EXP_DIMENSIONS_DIALOG_ID_H_DESC].dp = at->s[DBDCGG_TXT_ID_HEIGHT];
  
  // Export Code dialog
  sxc[EXP_CODE_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  sxc[EXP_CODE_DIALOG_ID_CANCEL].dp = at->s[DBDCGG_TXT_ID_CANCEL];
  sxc[EXP_CODE_DIALOG_ID_INSTANCENAME_DESC].dp = at->s[DBDCGG_TXT_ID_INSTANCENAME];

  // Main dialog
  md[MAIN_DIALOG_ID_AUTHOR_DESC].dp = at->s[DBDCGG_TXT_ID_AUTHOR];
  md[MAIN_DIALOG_ID_TITLE_DESC].dp = at->s[DBDCGG_TXT_ID_TITLE];
  md[MAIN_DIALOG_ID_SETTINGS].dp = at->s[DBDCGG_TXT_ID_SETTINGS_ATTRACTOR];
  md[MAIN_DIALOG_ID_TYPE_DESC].dp = at->s[DBDCGG_TXT_ID_TYPE];
  md[MAIN_DIALOG_ID_MODE_DESC].dp = at->s[DBDCGG_TXT_ID_MODE];
  md[MAIN_DIALOG_ID_COLOR_DESC].dp = at->s[DBDCGG_TXT_ID_COLOR];
  md[MAIN_DIALOG_ID_INTENSITY_DESC].dp = at->s[DBDCGG_TXT_ID_INTENSITY];
  md[MAIN_DIALOG_ID_EXPONENT_DESC].dp = at->s[DBDCGG_TXT_ID_EXPONENT];
  md[MAIN_DIALOG_ID_X_DESC].dp = at->s[DBDCGG_TXT_ID_X];
  md[MAIN_DIALOG_ID_Y_DESC].dp = at->s[DBDCGG_TXT_ID_Y];
  md[MAIN_DIALOG_ID_WIDTH_DESC].dp = at->s[DBDCGG_TXT_ID_WIDTH];
  md[MAIN_DIALOG_ID_HEIGHT_DESC].dp = at->s[DBDCGG_TXT_ID_HEIGHT];
  md[MAIN_DIALOG_ID_RANGE_DESC].dp = at->s[DBDCGG_TXT_ID_RANGE];
  md[MAIN_DIALOG_ID_ANGLE_DESC].dp = at->s[DBDCGG_TXT_ID_ANGLE];
  md[MAIN_DIALOG_ID_HIGHEST_EXPONENT_DESC].dp 
  = at->s[DBDCGG_TXT_ID_HIGHEST_EXPONENT];
  md[MAIN_DIALOG_ID_COEFFICIENTS_DESC].dp 
  = at->s[DBDCGG_TXT_ID_COEFFICIENTS];
  md[MAIN_DIALOG_ID_COEFFICIENTS_BUTTON].dp = at->s[DBDCGG_TXT_ID_EDIT];
  md[MAIN_DIALOG_ID_DISTANCE_MODE_DESC].dp 
  = at->s[DBDCGG_TXT_ID_DISTANCE_MODE];
  md[MAIN_DIALOG_ID_UNIT_DESC].dp = at->s[DBDCGG_TXT_ID_UNIT];
  md[MAIN_DIALOG_ID_ADD_BUTTON].dp = at->s[DBDCGG_TXT_ID_ADD];
  md[MAIN_DIALOG_ID_DUPLICATE_BUTTON].dp = at->s[DBDCGG_TXT_ID_DUPLICATE];
  md[MAIN_DIALOG_ID_DELETE_BUTTON].dp = at->s[DBDCGG_TXT_ID_DELETE];
  md[MAIN_DIALOG_ID_ATTRACTORS].dp = at->s[DBDCGG_TXT_ID_ATTRACTORS];
  md[MAIN_DIALOG_ID_GRADIENT].dp = at->s[DBDCGG_TXT_ID_GRADIENT];
  md[MAIN_DIALOG_ID_DIMENSIONS_BUTTON].dp 
  = at->s[DBDCGG_TXT_ID_SET_DIMENSIONS];
  md[MAIN_DIALOG_ID_RECALCULATE_BUTTON].dp 
  = at->s[DBDCGG_TXT_ID_RECALCULATE];
  //  md[MAIN_DIALOG_ID_].dp = at->s[DBDCGG_TXT_ID_];


  // About dialog
  ad[ABOUT_DIALOG_ID_CAPTION].dp = &g_prog_title;
  ad[ABOUT_DIALOG_ID_OK].dp = at->s[DBDCGG_TXT_ID_OK];
  ad[ABOUT_DIALOG_ID_AUTHOR_DESC].dp = at->s[DBDCGG_TXT_ID_AUTHOR];

  ad[ABOUT_DIALOG_ID_AUTHOR_TEXT].dp = g_programme_auth;
  ad[ABOUT_DIALOG_ID_HOME_TEXT].dp = g_programme_home;
  ad[ABOUT_DIALOG_ID_MAIL_TEXT].dp = g_programme_mail;
  ad[ABOUT_DIALOG_ID_POWERED_BY_TEXT].dp = at->s[DBDCGG_TXT_ID_POWERED_BY];
  ad[ABOUT_DIALOG_ID_ALLEGRO_HOME_TEXT].dp = g_allegro_home;
  ad[ABOUT_DIALOG_ID_TRANSLATION_BY_TEXT].dp 
    = at->s[DBDCGG_TXT_ID_TRANSLATED_BY];
  ad[ABOUT_DIALOG_ID_TRANSLATOR_TEXT].dp
    = at->s[DBDCGG_TXT_ID_TRANSLATION_AUTHOR];
}

// Called at programme termination to free occupied memory
/* (when this is called for the first time, all strings in the out_table
    MUST BE SET TO NULL) */
void kill_strings(dbdcgg_strings *out_table)
{
  int i;
  char *tmp = NULL;
  for(i=0; i<DBDCGG_NUM_STRINGS; i++)
  {
    tmp = out_table->s[i];
    if(tmp) delete []tmp;
    tmp = NULL;
  } 
  null_strings(out_table);
}
// same for extensions
void kill_ext(dbdcgg_ext *out_table)
{
  int i;
  char *tmp = NULL;
  for(i=0; i<F_EXT_NUM_EXT; i++)
  {
    tmp = out_table->s[i];
    if(tmp) delete []tmp;
    tmp = NULL;
  } 
  null_ext(out_table);
}

// Helper function to set all char pointers in the string table to NULL
void null_strings(dbdcgg_strings *out_table)
{
  int i;
  for(i=0; i<DBDCGG_NUM_STRINGS; i++)
    out_table->s[i]=NULL;
}

// Helper function to set all char pointer in the ext table to NULL
void null_ext(dbdcgg_ext *out_table)
{
  int i;
  for(i=0; i<F_EXT_NUM_EXT; i++)
    out_table->s[i]=NULL;
}

// Helper Functions to report a correct GUI position
int gui_mouse_x_offset()
{
  if(mouse_needs_poll()) 
    poll_mouse();
  return mouse_x + g_scroll_x;
}
int gui_mouse_y_offset()
{
  if(mouse_needs_poll()) 
    poll_mouse();
  return mouse_y + g_scroll_y;
}
int gui_mouse_x_std()
{
  if(mouse_needs_poll()) 
    poll_mouse();
  return mouse_x;
}
int gui_mouse_y_std()
{
  if(mouse_needs_poll()) 
    poll_mouse();
  return mouse_y;
}

// Set GUI behaviour back to standard
void gui_behave_std()
{
  gui_set_screen(NULL);
  gui_mouse_x = &gui_mouse_x_std;
  gui_mouse_y = &gui_mouse_y_std;
  show_mouse(screen);
}

// Set GUI behaviour to playback on backbuffer
void gui_behave_custom()
{
  gui_set_screen(g_back);
  gui_mouse_x = &gui_mouse_x_offset;
  gui_mouse_y = &gui_mouse_y_offset;
  show_mouse(NULL);
}

// Runs any dialog on my backbuffer(with scrolling supported)
// (returns the object index that caused closing)
int run_backbuffer_dialog(DIALOG *which, int focus)
{
  int pm_x=0;
  int pm_y=0;
  int quit_obj = -1;
  DIALOG_PLAYER *main_dialog_player = NULL; // the currently active dialog
 
  gui_behave_custom();

  main_dialog_player = init_dialog(which,focus);
  while(update_dialog(main_dialog_player))
  {
    if(!g_os_close_button_ignored && g_os_close_button_pressed)
    { 
      g_want_quit = true;
      break;
    }

    // Update input device states, if necessary
    if(mouse_needs_poll())
      poll_mouse();
    if(keyboard_needs_poll())
      poll_keyboard();

    // Check for scrolling and update scroll directions
    if((g_w < G_BACK_W)||(g_h < G_BACK_H))
    {
      g_scroll_dir = 0;
      if((mouse_x < G_SCROLL_AREA)||(key[KEY_HOME]))
        g_scroll_dir |= G_SCROLL_LEFT;
      if((mouse_x > g_w-G_SCROLL_AREA)||(key[KEY_END]))
        g_scroll_dir |= G_SCROLL_RIGHT;
      if((mouse_y < G_SCROLL_AREA)||(key[KEY_PGUP]))
        g_scroll_dir |= G_SCROLL_UP;
      if((mouse_y > g_h-G_SCROLL_AREA)||(key[KEY_PGDN]))
        g_scroll_dir |= G_SCROLL_DOWN;
    }

    // Show the backbuffer(including a software mousecursor) to screen
    acquire_screen();
//update_abs_mousepos(); // helps with determining "g_mouse_inside"
    pm_x=gui_mouse_x_offset();
    pm_y=gui_mouse_y_offset();
    blit(g_back,g_msafe,pm_x,pm_y,0,0,g_msafe->w,g_msafe->h);
    draw_sprite(g_back,mouse_sprite,pm_x,pm_y);
    blit(g_back,screen,g_scroll_x,g_scroll_y,0,0,g_w,g_h);
    blit(g_msafe,g_back,0,0,pm_x,pm_y,g_msafe->w,g_msafe->h);
/*if(!(g_mouse_inside)
     &&(is_windowed_mode())) // don't show when cursor is outside window*/
#ifdef _WIN32
    if(is_windowed_mode()&& !win_mouse_in_window())
    blit(g_back,screen,pm_x,pm_y,
         pm_x-g_scroll_x,pm_y-g_scroll_y,g_msafe->w,g_msafe->h); /**/      
#endif
    release_screen();

    // Play nice with the multitasking-OS
    rest(G_REST_MSEC);

    // Check for changes made by any edit fields
    // (changes made by buttons are processed in the main function)
    if(g_gradient_changed)
    {
      g_unsaved_changes = true;
      // Perform changes (made by EDIT BOXES)
      dt_apply_changes(g_gradient_changed);
      // signal that changes are processed
      g_gradient_changed = 0;
    }
  } // (while update_dialog)(loop only exits if the running dialog closes)

  gui_behave_std();

  // Deinitialize dialog and return index of object that caused closing
  quit_obj = main_dialog_player->obj;
  shutdown_dialog(main_dialog_player);
  return quit_obj;
}

// (Allegro will do its' magic to allow "main" on any OS)
int main(int arg_count, char *argument[])
{
  int init_ret = 0;
  int sub_dialog_ran = 0; // number of most recent ran sub dialog
  DIALOG *next_dialog = NULL;
  int next_focus = -1;
  int last_dialog_return = -1; // return value of most recent ran dialog
  int i=0;
  bool canceled = false;
  bool ignored = false;

  // INITIALIZATION
  init_ret = initialize();
  if(init_ret) // initialization error?
  {
    deinitialize();
    return init_ret; 
  }

  if(g_show_beta_warning)
    ascii_red_alert("-BETA Warning-",
                    "This warning is here to remind you",
                    "that this is a beta version.",
                    "(press ESC or click to continue)",
                    KEY_ESC);

  // Init file name buffers
  memset(g_lang_fname,0,G_FNAME_BUFSIZE);
  memset(g_bmp_fname,0,G_FNAME_BUFSIZE);
  memset(g_pcx_fname,0,G_FNAME_BUFSIZE);
  memset(g_tga_fname,0,G_FNAME_BUFSIZE);
  memset(g_dcgg_fname,0,G_FNAME_BUFSIZE);
  memset(g_dcgg_lang_fname,0,G_FNAME_BUFSIZE);
  memset(g_cpp_fname,0,G_FNAME_BUFSIZE);

  // Set dialog colors
  gui_fg_color = makecol(0, 0, 0);
  gui_mg_color = makecol(128, 128, 128);
  gui_bg_color = makecol(255, 255, 255);
  set_dialog_color(g_main_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_unsaved_dialog,makecol(255,0,0),makecol(255,255,200));
  set_dialog_color(g_type_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_mode_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_color_dialog,gui_fg_color,gui_bg_color);
  g_color_dialog[COLOR_DIALOG_ID_SLIDER_R].fg = makecol(255,0,0);
  g_color_dialog[COLOR_DIALOG_ID_SLIDER_G].fg = makecol(0,255,0);
  g_color_dialog[COLOR_DIALOG_ID_SLIDER_B].fg = makecol(0,0,255);
  set_dialog_color(g_dmode_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_coefficients_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_dimensions_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_exp_dimensions_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_exp_code_dialog,gui_fg_color,gui_bg_color);
  set_dialog_color(g_about_dialog,gui_bg_color,makecol(0,0,72));
  g_main_dialog[MAIN_DIALOG_ID_CLEAR].bg = makecol(192,192,255);

  // Adjust positions of some dialog elements
  g_main_dialog[MAIN_DIALOG_ID_SETTINGS].y -= text_height(font);
  g_main_dialog[MAIN_DIALOG_ID_SETTINGS].h = text_height(font);
  g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS].y -= text_height(font);
  g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS].h = text_height(font);
  g_main_dialog[MAIN_DIALOG_ID_GRADIENT].y -= text_height(font);
  g_main_dialog[MAIN_DIALOG_ID_GRADIENT].h = text_height(font);
  for(i=MAIN_DIALOG_ID_MENU_BAR+1; i<MAIN_DIALOG_ID_YIELD; i++)
  {// adjust all elements' x offset (except menu bar)
    g_main_dialog[i].x += 3;
  }
  
  // Stretch preview window and adjust preview window button positions
  if(G_BACK_W > 800)
  {
    int w_stretch = G_BACK_W - 800;
    int h_stretch = G_BACK_H - 600;
    #define sbx g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].x
    #define sby g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].y
    #define sbw g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].w
    #define sbh g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].h
    #define bw2 g_main_dialog[MAIN_DIALOG_ID_BMP_UP_BUTTON].w/2
    g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].w += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_GRADIENT_SCROLLBMP].h += h_stretch;
    g_main_dialog[MAIN_DIALOG_ID_DIMENSIONS_BUTTON].y += h_stretch;
    g_main_dialog[MAIN_DIALOG_ID_BMP_DOWN_BUTTON].x = sbx + sbw/2 - bw2;
    g_main_dialog[MAIN_DIALOG_ID_BMP_DOWN_BUTTON].y += h_stretch;
    g_main_dialog[MAIN_DIALOG_ID_RECALCULATE_BUTTON].x = sbx + sbw/2 +bw2+2;
    g_main_dialog[MAIN_DIALOG_ID_RECALCULATE_BUTTON].y += h_stretch;
    g_main_dialog[MAIN_DIALOG_ID_BMP_RIGHT_BUTTON].x += w_stretch;
    g_main_dialog[MAIN_DIALOG_ID_BMP_RIGHT_BUTTON].y = sby + sbh/2 - bw2;
    g_main_dialog[MAIN_DIALOG_ID_BMP_LEFT_BUTTON].y = sby + sbh/2 - bw2;
    g_main_dialog[MAIN_DIALOG_ID_BMP_UP_BUTTON].x = sbx + sbw/2 - bw2;
    g_main_dialog[MAIN_DIALOG_ID_ATTRACTORS_LIST].w += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_LIST_UP_BUTTON].x += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_LIST_DOWN_BUTTON].x += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEUP_BUTTON].x += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_LIST_MOVEDOWN_BUTTON].x += w_stretch - 1;
    g_main_dialog[MAIN_DIALOG_ID_WHITE_BAR].w += w_stretch;
    g_main_dialog[MAIN_DIALOG_ID_MENU_BAR].w += w_stretch;
    #undef sbx
    #undef sby
    #undef sbw
    #undef sbh
    #undef bw2
  }
  
  // Adjust selected options
  if(g_auto_recalculate)
    g_options_menu[OPTIONS_MENU_ID_AUTO_RECALCULATE].flags = D_SELECTED;
  else
    g_options_menu[OPTIONS_MENU_ID_AUTO_RECALCULATE].flags = 0;
  
  // END OF INIT

  // MAIN PROG LOOP
  position_mouse(0,0);
  file_new(); // init gradient
  next_dialog = g_main_dialog;
  sub_dialog_ran = SUB_DIALOG_MAIN;
  while(!g_want_quit)
  {
    if(next_dialog)
      last_dialog_return = run_backbuffer_dialog(next_dialog,next_focus);
    else
      g_want_quit = true;

    next_focus = -1; // no object on next dialog shall have focus by default

    // Check for changes made by any buttons here
    // (changes made by edit fields are processed in "run_backbuffer_dialog")
    if(g_gradient_changed)
    {
      g_unsaved_changes = true;
      // Perform changes (made by BUTTONS)
      dt_apply_changes(g_gradient_changed);
      // signal that changes are processed
      g_gradient_changed = 0;
    }

    // Check and process simple button presses
    if((last_dialog_return == MAIN_DIALOG_ID_RECALCULATE_BUTTON)&&
       (next_dialog==(DIALOG*)&g_main_dialog))
    {
      button_recalculate();
      next_focus = last_dialog_return;
    }

    // Check and process eventually ending sub dialogs
    canceled = false;
    ignored = false;
    switch(sub_dialog_ran)
    {
      case SUB_DIALOG_MAIN:
        if(last_dialog_return!=MAIN_DIALOG_ID_MENU_BAR)
          g_request_sub_dialog = SUB_DIALOG_MAIN; // default main dialog again

        if(last_dialog_return==MAIN_DIALOG_ID_TYPE_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_SET_TYPE;

        if(last_dialog_return==MAIN_DIALOG_ID_MODE_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_SET_MODE;

        if(last_dialog_return==MAIN_DIALOG_ID_COLOR_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_SET_COLOR;

        if(last_dialog_return==MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_SET_DISTANCE_MODE;

        if(last_dialog_return==MAIN_DIALOG_ID_COEFFICIENTS_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_EDIT_COEFFICIENTS;

        if(last_dialog_return==MAIN_DIALOG_ID_DIMENSIONS_BUTTON)
          g_request_sub_dialog = SUB_DIALOG_SET_DIMENSIONS;
      break;

      case SUB_DIALOG_CHANGE_ALERT:
        switch(last_dialog_return)
        {
          case UNSAVED_DIALOG_ID_SAVE_BUTTON:  
            file_save();
          break;
          case UNSAVED_DIALOG_ID_SAVE_AS_BUTTON:  
            file_save_as();
          break;
          case UNSAVED_DIALOG_ID_IGNORE_BUTTON:
            ignored = true;
            g_unsaved_changes = false;
          break;
          default: // behave like cancel
          case UNSAVED_DIALOG_ID_CANCEL_BUTTON:
            if(g_os_close_button_pressed)// ((press close twice)==(ignore))
            { 
              ignored = true;
              g_unsaved_changes = false;
            }
            else
            {
              canceled = true;
              g_want_quit = false;
            }
          break;
        }//(switch(last_dialog_return))

        g_request_sub_dialog = SUB_DIALOG_MAIN; // main dialog default f'up

        if((!g_unsaved_changes)||(ignored)) // save succeded or was ignored
          switch(g_usc_alert_caller)
          {
            case USC_ALERT_CALLER_NEW:
              file_new();
            break;
            case USC_ALERT_CALLER_LOAD:
              file_load();
            break;
            case USC_ALERT_CALLER_EXIT:
              g_request_sub_dialog = SUB_DIALOG_NONE;
            break;
            default:
              g_unsaved_changes = false;
            break;
          }//(switch(g_usc_alert_caller))

        g_usc_alert_caller = USC_ALERT_CALLER_NONE;

      break;//(case SUB_DIALOG_CHANGE_ALERT)

      case SUB_DIALOG_SET_TYPE:
      case SUB_DIALOG_SET_MODE:
      case SUB_DIALOG_SET_COLOR:
      case SUB_DIALOG_EDIT_COEFFICIENTS:
      case SUB_DIALOG_SET_DISTANCE_MODE:
      case SUB_DIALOG_SET_DIMENSIONS:
      case SUB_DIALOG_ABOUT:
        g_os_close_button_disabled = false;
        g_request_sub_dialog = SUB_DIALOG_MAIN;
      break;
    }//(switch(sub_dialog_ran))
    
    // Check and process any eventually starting sub dialogs
    /*(sub dialogs are started by setting "sub_dialog_ran" and "next_dialog",
       so that the next iteration of this loop will run it)*/
    sub_dialog_ran = SUB_DIALOG_NONE;

    // make sure the alert also shows also if the window close button
    // was pressed
    if((g_want_quit)&&(g_unsaved_changes))
    {
      g_os_close_button_pressed = false;
      g_usc_alert_caller = USC_ALERT_CALLER_EXIT;
      g_request_sub_dialog = SUB_DIALOG_CHANGE_ALERT;
    }

    // now handle the initialization of sub dialogs
    switch(g_request_sub_dialog)
    {
      case SUB_DIALOG_NONE:
        g_want_quit = true;
        next_dialog = NULL;
      break;

      case SUB_DIALOG_CHANGE_ALERT:
        sub_dialog_ran = SUB_DIALOG_CHANGE_ALERT;
        position_dialog(g_unsaved_dialog,
                        2*text_height(font),
                        2*text_height(font));
        g_scroll_x = 0;
        g_scroll_y = 0;
        next_dialog = g_unsaved_dialog;
        g_want_quit = false;
      break;

      case SUB_DIALOG_SET_TYPE:
        sub_dialog_ran = SUB_DIALOG_SET_TYPE;
        position_dialog(g_type_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_TYPE_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_TYPE_BUTTON].y);
        next_dialog = g_type_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_SET_MODE:
        sub_dialog_ran = SUB_DIALOG_SET_MODE;
        position_dialog(g_mode_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_MODE_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_MODE_BUTTON].y);
        next_dialog = g_mode_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_SET_DISTANCE_MODE:
        sub_dialog_ran = SUB_DIALOG_SET_DISTANCE_MODE;
        position_dialog(g_dmode_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON].y);
        next_dialog = g_dmode_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_SET_COLOR:
        sub_dialog_ran = SUB_DIALOG_SET_COLOR;
        position_dialog(g_color_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_COLOR_BUTTON].y);
        next_dialog = g_color_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_EDIT_COEFFICIENTS:
        sub_dialog_ran = SUB_DIALOG_EDIT_COEFFICIENTS;
        position_dialog(g_coefficients_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_COEFFICIENTS_BUTTON].y);
        next_dialog = g_coefficients_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_SET_DIMENSIONS:
        sub_dialog_ran = SUB_DIALOG_SET_DIMENSIONS;
        position_dialog(g_dimensions_dialog,
                        g_main_dialog[MAIN_DIALOG_ID_DIMENSIONS_BUTTON].x,
                        g_main_dialog[MAIN_DIALOG_ID_DIMENSIONS_BUTTON].y);
        next_dialog = g_dimensions_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;

      case SUB_DIALOG_ABOUT:
        sub_dialog_ran = SUB_DIALOG_ABOUT;
        position_dialog(g_about_dialog, 
                        2*text_height(font),2*text_height(font));
        next_dialog = g_about_dialog;
        g_os_close_button_disabled = true;
        g_want_quit = false;
      break;
 
      default:
        if(!g_want_quit)
        {
          sub_dialog_ran = SUB_DIALOG_MAIN;
          next_dialog = g_main_dialog;
        }
      break;
    }
    // now that any sub dialog is marked for the next iteration, the request
    // can be set as "processed"
    g_request_sub_dialog = SUB_DIALOG_NONE;

  }//(while(!g_really_quit))
  // END OF MAIN PROG LOOP

  // DEINITIALIZATION
  deinitialize();
  return 0;
  // END OF DEINIT
}
END_OF_MAIN()

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
