/*--- Filename: "ca_polynomial.hpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The rotated polynomial color attractor.

*/

#include "shared.hpp"
#include "ca_point.hpp"
#include <ctime>

#if !defined(__DB_ca_polynomial_HEAD_INCLUDED)
#define __DB_ca_polynomial_HEAD_INCLUDED

// for the polynomial lookup table
typedef struct
{
  double func_val;
  double func_low;
  double func_high;
} FVHILO; // (f)unction(v)alue (hi)gh (lo)w

enum D_MODE // DISTANCE MODE
{
  DM_TOFUNCTIONVALUE=0, // use distance to function value
  DM_TOCURVE, // use smallest distance to any point on the curve
  /* insert more modes here */
  DM_NUM_MODES
};
// returns the internal description string to the specified distance mode
string get_d_mode_text(D_MODE which);

class CA_POLYNOMIAL : public CA_POINT
{
  protected:
    bool _mem_error;/*true if there was not enough memory to hold coefficients
                      and lookup table */
    double _ang_rad; // rotation angle in radians
    double _pix_unit; // the unit in pixels
    int _plut_width; // lookup table width
    int _plut_zero;  // the index of x=0 inside the lookup table

  // variables that affect the attractors influence
    double UNIT; // the unit length (relative to render targets width)
    double ANGLE; /* rotation angle (around the origin of the polynomial)
                     in degrees (0-360) */
    D_MODE DMODE; // distance mode
    unsigned int HEP; // highest exponent
    double *CO; /* variable length array for the HEP+1 coefficients,
      p(x)=CO[HEP]*x^(HEP)+CO[HEP-1]*x^(HEP*1)+...+CO[1]*x+CO[0] */
    FVHILO *PLUT; // lookup table to store the polynomial curve

/*!*/virtual double distance(int x, int y); // calculate distance to (x,y)

  // function for loading the vars from a string
/*!*/virtual int update_vars_from_line(); // continues loading from string

    void kill_me(); // frees internal data

  public:
    CA_POLYNOMIAL();
    CA_POLYNOMIAL(CA_MODE mode, int r, int g, int b, double colint=1.0,
             double exp=1.0, double x=0.5, double y=0.5, double unit=0.125,
             double angle=0.0, double range=0.1,
             D_MODE dmode=DM_TOFUNCTIONVALUE,
             unsigned int hep=0, double *co=NULL);
/*!*/virtual ~CA_POLYNOMIAL();//remember to have virtual dtor in derived types

  // Set and Get Methods
     void set_unit(double unit);
     double get_unit();

     void set_angle(double angle);
     double get_angle();

     void set_dmode(D_MODE dmode);
     D_MODE get_dmode();

     void set_hep(unsigned int hep);
     unsigned int get_hep();
 
     void set_co(unsigned int which, double co);
     double get_co(unsigned int which);

  // called by the GRADIENT_GEN prior to rendering
  // (should return true, if rendering should be done, false otherwise)
/*!*/virtual bool update_absolutes(int width, int height,
                                   lut_callback lcall, double lcall_skip, int n);

  // Saving to string (loading from string is handled by "update_vars_..."
/*!*/virtual string get_as_text();

  // Helper function for GUI building that returns some very sparse info
  // meant to be displayed in a list
/*!*/virtual string gui_quick_info();
};

#endif // #if !defined(__DB_ca_polynomial_HEAD_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
