/*--- Filename: "ca_polynomial.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The rotated polynomial color attractor.  

*/

#include "ca_polynomial.hpp"

#if !defined(__DB_ca_polynomial_BODY_INCLUDED)
#define __DB_ca_polynomial_BODY_INCLUDED

string D_MODE_TEXT[] =
{
  string("TOFUNCTIONVALUE"),
  string("TOCURVE")
};

string get_d_mode_text(D_MODE which)
{
  if(which<DM_NUM_MODES)
    return D_MODE_TEXT[which];
  else
    return string("invalid distance mode id!");
}

// default ctor just sets default values
CA_POLYNOMIAL::CA_POLYNOMIAL()
{
  _mem_error = false;
  UNIT = 0.125;
  RANGE = 0.1;
  ANGLE = 0.0;  
  DMODE = DM_TOFUNCTIONVALUE;
  HEP = 0;
  CO = NULL;
  PLUT = NULL;
  TYPE = CAT_POLYNOMIAL; // remember to override in derived types
}

//
void CA_POLYNOMIAL::kill_me()
{
  if(CO) delete[] CO;
  if(PLUT) delete[] PLUT;
  CO = NULL;
  PLUT = NULL;
  HEP = 0;
  _mem_error = false;

  CA_GLOBAL::delete_line_buf();
}

// other ctor sets user values
CA_POLYNOMIAL::CA_POLYNOMIAL(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, double unit,
                   double angle, double range, 
                   D_MODE dmode, unsigned int hep, double *co)
{
  CA_GLOBAL::CA_GLOBAL(mode,r,g,b,colint);
  EXP = exp;
  X = x;
  Y = y;
  UNIT = unit;
  ANGLE = fmod(angle,360.0);
  _ang_rad = make_rad(ANGLE);
  if(_ang_rad<0.0)
    _ang_rad += _2pi;
  RANGE = range;
  DMODE = dmode;

  kill_me(); // frees previous data in CO and PLUT
  
  HEP = hep;

  _mem_error = false;
  // initialize internal coefficient array
  CO = NULL;
  CO = new(nothrow)double[HEP+1];
  if(!CO)
  {
    _mem_error=true;
  }

  // get coefficients from the dynamic length array(if any)
  if(co && CO) for(unsigned int i=0; i<=hep; i++)
    CO[i] = co[hep-i];
  else if(!co && CO)
    for(unsigned int j=0; j<=hep; j++)
      CO[j] = 0;

  PLUT = NULL; // lookup table is calculated elsewhere(in update_absolutes)

  TYPE = CAT_POLYNOMIAL; // remember to override in derived types
}

// default dtor kills everything
// (is virtual)
CA_POLYNOMIAL::~CA_POLYNOMIAL()
{
  kill_me();
}

// Set and Get Mehods

//
void CA_POLYNOMIAL::set_unit(double unit)
{
  UNIT = unit;
  _render_var_changed = true;
}

//
double CA_POLYNOMIAL::get_unit()
{
  return UNIT;
}

//
void CA_POLYNOMIAL::set_angle(double angle)
{
  ANGLE = fmod(angle, 360.0);
  _ang_rad = make_rad(ANGLE);
  if(_ang_rad<0.0)
    _ang_rad += _2pi;
  //_render_var_changed = true; 
  /* the lookup table will be wide enough to
     hold enough values for the "widest" possible rotation */
}

//
double CA_POLYNOMIAL::get_angle()
{
  return ANGLE;
}

//
void CA_POLYNOMIAL::set_dmode(D_MODE dmode)
{
  DMODE = dmode;  
}

//
D_MODE CA_POLYNOMIAL::get_dmode()
{
  return DMODE;
}

//
void CA_POLYNOMIAL::set_hep(unsigned int hep)
{
  _mem_error = false;
  HEP = hep;
  if(CO) delete[] CO; 
  CO = NULL;
  CO = new(nothrow)double[HEP+1];
  if(!CO)
  {
    _mem_error=true;
  }
  if(CO) 
    for(unsigned int i=0; i<=hep; i++)
      CO[i] = 0.0;
  _render_var_changed = true;
}

//
unsigned int CA_POLYNOMIAL::get_hep()
{
  return HEP;
}

//
void CA_POLYNOMIAL::set_co(unsigned int which, double co)
{
  if((CO)&&(which<=HEP))
  {
    CO[which] = co;
    _render_var_changed = true;
  }
}

//
double CA_POLYNOMIAL::get_co(unsigned int which)
{
  if((CO)&&(which<=HEP))
    return CO[which];
  else return 0.0;
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_POLYNOMIAL::distance(int x, int y)
{
  if(_mem_error) 
    return _pix_range + 1.0; //to prevent any unwanted influence
  else
  {
  // get the vector from the polynomials' origin to current position
  // and its' length
    int i;
    double dint,dfrac; // for seperating a double to its int and frac portion
    double vec_x = x - _pix_x;
    double vec_y = y - _pix_y;
    double vec_l = sqrt(vec_x*vec_x + vec_y*vec_y);
    double vec_ang;

  // calculate the vectors angle in screen coordinate system
    if(vec_l != 0.0)
    {
      if(vec_y>=0.0)
        vec_ang = acos(vec_x / vec_l);
      else // vec_y<0
        vec_ang = _2pi - acos(vec_x / vec_l);
    }
    else vec_ang = 0.0;

  // rotate it back into the polynomial coordinate system
    vec_ang -= _ang_rad;
    if(vec_ang<0.0)
      vec_ang += _2pi;

  // now calculate the lookup table position for that new angle
    vec_x = cos(vec_ang)*vec_l;
    vec_y = sin(vec_ang)*vec_l;

    dfrac = modf((double)_plut_zero + vec_x,&dint);
    i = (fabs(dfrac)>=0.5) ? int(dint+1) : int(dint);// rounds to nearest int

    if((i<0)||(i>=_plut_width)) // this should never be true, but its here..
      return _pix_range + 1.0;  // ..to make sure.

  // now perform the distance calculation
    switch(DMODE)
    {
      case DM_TOFUNCTIONVALUE: // simple distance to function value
        return fabs(vec_y - PLUT[i].func_val);
      break;
      case DM_TOCURVE: // shortest distance to curve
        /* A quadratic area (sidelength is 2*half_side) around the found
           lookup table position will be scanned for curve points.
           The shortest distance to any of these found points gets returned.
           (Scanning a circular area would be sufficient,
            but testing points for being inside a circular area would waste
            too much time.) */
        { // START to prevent "jump to case label default crosses init of.."
        // the current position inside the lookup table
        double di = dint+dfrac; // save vec_x in di and..
        double dj = vec_y;// ..vec_y in dj,so vec(_x,_y,_l) can be reused
        // initial distance
        double d_tofunc = fabs(dj - PLUT[i].func_val);

        // bounding vars for scan area
        int half_side=(d_tofunc<_pix_range)? (int)d_tofunc : (int)_pix_range;
        int m_start = i - half_side;
        int m_end = i + half_side;
        int n_start = (int)dj - half_side;
        int n_end = (int)dj + half_side;
        int m,n;

        // handle special case
        // (the high value of a point is actually a smaller pixel position,
        //  than the low value of a point)
        if((vec_y>=PLUT[i].func_high)&&(vec_y<=PLUT[i].func_low))
          return 0.0;

        // scan the rectangular area for points on the curve
        for(m=m_start; m <= m_end; m++) // for each x position
        {
          // Speeding up by adjusting scan area(making it smaller)
          if(PLUT[m].func_val > dj)
          {
            n_start = (int)dj;
            n_end = n_start + half_side;
          }
          else
          {
            n_start = (int)dj - half_side;
            n_end = (int)dj;
          }

          // Speeding up by ignoring irrelevant "columns"
          if(n_end < PLUT[m].func_high) continue;
          if(n_start > PLUT[m].func_low) continue;

          for(n=n_start; n <= n_end; n++) // for each y position
          {
            // Speeding up by ignoring irrelevant "lines"
            if(n < PLUT[m].func_high) continue;
            if(n > PLUT[m].func_low) continue;
            
            /* If an iteration reaches this line then (m,n) is a point of
               the curve. */
            // Calc distance to that point and adjust shortest distance
            vec_x = (double)m - di;
            vec_y = (double)n - dj;
            vec_l = sqrt(vec_x*vec_x + vec_y*vec_y);
            if(vec_l < d_tofunc)
              d_tofunc = vec_l;
          }
        }
        return d_tofunc;
        } // END to prevent "jump to case label default crosses init of.."
      break; // (END OF case DM_TOCURVE)
      default:;// to prevent ".. not handled in switch" warning
    }

    return _pix_range + 1.0; // prevent influence from undefined DMODEs
  }//else(from 3rd line of distance function)
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_POLYNOMIAL::update_absolutes(int width, int height,
                                     lut_callback lcall, 
                                     double lcall_skip, int n)
{
  if(!dim_changed(width,height) || (!CO))
    return true;
  else
  { 
    // some temporary variables used in various calculations
    int values_left = 0;
    clock_t last_cb_clock;
    clock_t current_cb_clock;
    double _tx;
    double _ty;
    double _tl;
    double _cd;
    int i,j;
 
    _pix_x = width * X;
    _pix_y = height * Y;
    _pix_range = height * RANGE;
    _pix_unit = width * UNIT;
    if(_pix_unit == 0.0) _pix_unit = 1.0; //(set 1 pixel as minimum unit)
  
    /* the lookup table should hold enough values for the "widest" possible
       rotation, so it must be of 
       _plut_width =("longest distance to screencorner" + _pix_range + 1) * 2
       _plut_width/2 will be the x==0 index in the lookup table and
       values for all possible rotations will be contained */

  // find longest distance to one of the render targets' corners
    // check upper left corner
    _tx = 0.0 - _pix_x;
    _ty = 0.0 - _pix_y;
    _tl = sqrt(_tx*_tx + _ty*_ty);
    _cd = _tl;
    // check upper right corner
    _tx += width-1;
    _tl = sqrt(_tx*_tx + _ty*_ty);
    if(_tl>=_cd) _cd = _tl;
    // check lower right corner
    _ty += height-1;
    _tl = sqrt(_tx*_tx + _ty*_ty);
    if(_tl>=_cd) _cd = _tl;
    // check lower left corner
    _tx -= (width-1);
    _tl = sqrt(_tx*_tx + _ty*_ty);
    if(_tl>=_cd) _cd = _tl;

  // set dimensions for lookup table
    _plut_width = 2 * ((int)ceil(_cd) + (int)ceil(_pix_range) + 1);
    _plut_zero = _plut_width / 2; // index of x==0 inside lookup table

  // kill previous lookup table (if any)
    if(PLUT) delete[] PLUT;
  // make new lookup table
    _mem_error = false;
    PLUT = NULL;
    PLUT = new(nothrow)FVHILO[_plut_width];
    if(PLUT==NULL)
    {
      _mem_error = true;
      return true; // not enough memory, makes polynomial attractor inactive
    }

  // prepare handling of callback function and call it once before calculating
    values_left = _plut_width*2-1;
    last_cb_clock = clock();
    if(lcall)
      (*lcall)(values_left,TYPE,n);

  // calculate lookup table base function values
    for(i=0; i<_plut_width; i++)
    {
      PLUT[i].func_val = 0.0;

      _tx = (double)(-(_plut_zero - i)) / _pix_unit;

      // using "Horner Scheme" to compute function value
      _ty=CO[HEP]*_tx;
      for(j=(int)HEP-1; j>0; j--)
        _ty = (CO[j] + _ty)*_tx;
      _ty = _ty + CO[0];

      PLUT[i].func_val = _ty * _pix_unit;// add value to table

      /* overwrite garbage memory content(not doing so leads to display..
         ..anomalies whenever the render dimensions change) */
      PLUT[i].func_high=0.0;
      PLUT[i].func_low=0.0; 

      // call callback function (if any)
      values_left--;
      current_cb_clock = clock();
      
      if((lcall)
         && ((double(current_cb_clock-last_cb_clock)/double(CLOCKS_PER_SEC))*1000.0>=lcall_skip))
      {
        if(!((*lcall)(values_left,TYPE,n)))
        {
          _render_var_changed = true; // mark me for recalculation next time
          _mem_error = true;
          return false; // abort rendering
        }
        else
          last_cb_clock = clock();
      }
    }
    
  // calculate lookup lows and highs
    //(exploiting that a polynomial has a continuous curve in its' graph)
    for(i=1; i<_plut_width; i++)
    {
      _cd = (PLUT[i-1].func_val - PLUT[i].func_val) / 2.0;
        // half distance between current value and previous

      if(PLUT[i-1].func_val <= PLUT[i].func_val) // prev <= current
      {
        PLUT[i].func_high = PLUT[i].func_val + _cd;
        PLUT[i-1].func_low = PLUT[i-1].func_val - _cd;
      }
      else // prev > current
      {
        PLUT[i].func_low = PLUT[i].func_val + _cd;
        PLUT[i-1].func_high = PLUT[i-1].func_val - _cd;
      }

      // call callback function (if any)
      values_left--;
      current_cb_clock = clock();
      
      if((lcall)
         && ((double(current_cb_clock-last_cb_clock)/double(CLOCKS_PER_SEC))*1000.0>=lcall_skip))
      {
        if(!((*lcall)(values_left,TYPE,n)))
        {
          _render_var_changed = true; // mark me for recalculation next time
          _mem_error = true;
          return false; // abort rendering
        }
        else
          last_cb_clock = clock();
      }
    }

  /* ALL DONE. Lookup table is ready and filled with enough info for
     every rotation angle. Note: Due to this, the lookup table will be
     bigger the farther the origin of the polynomial is away from the
     render targets' corners, so you should not put the origin too far
     away outside the render target (try to stay within [0.0,1.0]
     for both x and y to save memory) */

  }//else(from the 3rd line of this function)

  return true;
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_POLYNOMIAL::update_vars_from_line()
{
  int i;
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // Y double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&Y);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // UNIT double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&UNIT);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // ANGLE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&ANGLE);
    if((ret==0)||(ret==EOF)) return -1;
    ANGLE = fmod(ANGLE,360.0);
    _ang_rad = make_rad(ANGLE);
    if(_ang_rad<0.0)
      _ang_rad += _2pi;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

                            // D_MODE string description expected
  TOKEN = strtok(NULL,token_seperators);
  if(TOKEN) 
  {
    if((ret==0)||(ret==EOF)) 
      return -1;
    else
    {
      bool dmode_found = false;
      string dmode_desc(TOKEN);

      // compare token to find distance mode
      for(i = DM_TOFUNCTIONVALUE; (i<DM_NUM_MODES) && (!dmode_found); i++)
      {
        if(dmode_desc.compare(get_d_mode_text((D_MODE)i))==0)// have a match?
        {
          DMODE = (D_MODE)i;
          dmode_found = true;
        }
      }// end of find distance mode loop
      
      if(!dmode_found) return -1;

    }// else
  }// if(TOKEN)
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // HEP unsigned int expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%u",&HEP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;  

  // remove previous coefficients(if any)
  _mem_error = false;
  if(CO) delete[] CO;
  CO = NULL;
  CO = new(nothrow)double[HEP+1];
  if(!CO)
  {
    _mem_error = true;
    return -1;
  }
  else // read in the HEP+1 coefficients from the human readable string
    for(i=(int)HEP; i>=0; i--)
    {
      TOKEN = strtok(NULL,token_seperators); // a CO double expected
      if(TOKEN) 
      {
        ret = sscanf(TOKEN,"%lf",CO+i);
        if((ret==0)||(ret==EOF)) return -1;
      }
      else return -1; 
    }

  _render_var_changed = true;  

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_POLYNOMIAL::get_as_text()
{
  int i;
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP
          << ", " << X
          << ", " << Y
          << ", " << UNIT
          << ", " << ANGLE
          << ", " << RANGE
          << ", " << get_d_mode_text(DMODE).c_str()
          << ", " << HEP;
    if(CO)
    {
      for(i=(int)HEP; i>=0; i--) // append coefficients
        rline << ", " << CO[i];
    }
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_POLYNOMIAL::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ") "
          << ",rn " << RANGE << ((_mem_error) ? " (!)" : " .");
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_polynomial_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
