/*--- Filename: "ca_pointe.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The pointe color attractor.(the "e" behind point stands for elliptic,
  this color attractor has a rotated elliptic range around its' center)

*/

#include "ca_pointe.hpp"

#if !defined(__DB_ca_pointe_BODY_INCLUDED)
#define __DB_ca_pointe_BODY_INCLUDED

// default ctor just sets default values
CA_POINTE::CA_POINTE()
{
  EXP = 1.0;
  X = 0.5;
  Y = 0.5;
  WIDTH = 0.5;
  HEIGHT = 0.5;
  ANGLE = 0;
  _ang_rad = 0;
  TYPE = CAT_POINTE; // remember to override in derived types
}

// other ctor sets user values
CA_POINTE::CA_POINTE(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, 
                   double width, double height, double angle)
{
  CA_GLOBAL::CA_GLOBAL(mode,r,g,b,colint);
  EXP = exp;
  X = x;
  Y = y;
  WIDTH = width;
  HEIGHT = height;  
  ANGLE = fmod(angle,360.0);
  _ang_rad = make_rad(ANGLE);
  TYPE = CAT_POINTE; // remember to override in derived types
}

CA_POINTE::~CA_POINTE()
{
  CA_GLOBAL::delete_line_buf();
}

// Set and Get Mehods

//
void CA_POINTE::set_width(double width)
{
  WIDTH = width;
  _render_var_changed = true;
}

//
double CA_POINTE::get_width()
{
  return WIDTH;
}

//
void CA_POINTE::set_height(double height)
{
  HEIGHT = height;
  _render_var_changed = true;
}

//
double CA_POINTE::get_height()
{
  return HEIGHT;
}

//
void CA_POINTE::set_angle(double angle)
{
  ANGLE = fmod(angle,360.0);
  _ang_rad = make_rad(ANGLE);
  if(_ang_rad<0.0)
    _ang_rad += _2pi;
}

//
double CA_POINTE::get_angle()
{
  return ANGLE;
}

// calculate distance to (x,y) in pixels
// (also updates the current angle specific _pix_range for the elliptic range)
// (is protected virtual)
double CA_POINTE::distance(int x, int y)
{
  double term; // used for elliptic range inclusion test

  double vec_x = x - _pix_x; // vector from elliptic ranges center..
  double vec_y = y - _pix_y; // ..to current point
  double vec_l = sqrt(vec_x*vec_x + vec_y*vec_y); // length of it
  double vec_ang;

  /* if "vec_l" is greater than the radius of the bounding circle, then
     the point can not be inside the elliptic range */
  if(vec_l > _bounding_circle_r)
    return _pix_range + 1.0;

  // Handle special case
  if((vec_x==0.0)&&(vec_y==0.0))
    return 0.0;

  // calculate angle(in radians) of vector in the screen's coordinate system
  if(vec_y>=0.0)
    vec_ang=acos(vec_x / vec_l);
  else // (vec_y<0)
    vec_ang=_2pi - acos(vec_x / vec_l);

  // rotate the vector backwards into the ellipse's coordinate system
  vec_ang = vec_ang - _ang_rad; // (_ang_rad is the angle of ellipse)
  if(vec_ang<0.0)
    vec_ang += _2pi;

  /* simplifying the following calculations by norming the vector into
     positive x and y (in the elliptic range coordinate system) */
  vec_x = fabs(vec_l * cos(vec_ang));
  vec_y = fabs(vec_l * sin(vec_ang));
  
  /* (vec_x,vec_y is now in the first quadrant) */

  // Perform elliptic range inclusion test
  term= (vec_x*vec_x) / (_pix_w2*_pix_w2) + (vec_y*vec_y) / (_pix_h2*_pix_h2);
  if(term > 1.0)
    return _pix_range + 1.0; // out of elliptic range

  return term; // is <= 1.0 and _pix_range is always 1.0 ( ;-) )
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_POINTE::update_absolutes(int width, int height,
                                 lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;
 
  _pix_x = width * X;
  _pix_y = height * Y;
  _pix_w2 = (width * WIDTH) / 2.0;
  _pix_h2 = (height * HEIGHT) / 2.0;
  if(_pix_w2==0.0) 
    _pix_w2 = 1.0; // prevent div by zero in other functions
  if(_pix_h2==0.0) 
    _pix_h2 = 1.0; // prevent div by zero in other functions

  if(fabs(_pix_w2)<=fabs(_pix_h2))
    _bounding_circle_r = _pix_h2;
  else
    _bounding_circle_r = _pix_w2;

  _pix_range = 1.0;

  return true;
}

// The *only* function ever called by the "render_gradient" method from..
// ..GRADIENT_GEN class (directly modifies "crgb"; applies CA's influence)
// (is virtual)
void CA_POINTE::modify_crgb(int x, int y, customRGB<double>& crgb)
{
  double d = distance(x,y);

  if(d<=_pix_range)
  {
    double calc = (1.0 - pow(d / _pix_range,EXP))*COLINT; 
    switch(MODE)
    {
      case CAM_ATTRACT:
        crgb.add_cr(calc*R);
        crgb.add_cg(calc*G);
        crgb.add_cb(calc*B);
      break;
      case CAM_ABSORB:
        crgb.add_cr(-calc*R);
        crgb.add_cg(-calc*G);
        crgb.add_cb(-calc*B);
      break;
      default:;// to prevent ".. not handled in switch" warning
    }
  }
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_POINTE::update_vars_from_line()
{
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // Y double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&Y);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // WIDTH double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&WIDTH);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // HEIGHT double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&HEIGHT);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // ANGLE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&ANGLE);
    if((ret==0)||(ret==EOF)) return -1;

    ANGLE = fmod(ANGLE,360.0);
    _ang_rad = make_rad(ANGLE);
    if(_ang_rad<0.0)
      _ang_rad += _2pi;
  }
  else return -1;

  _render_var_changed = true;

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_POINTE::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP
          << ", " << X
          << ", " << Y
          << ", " << WIDTH
          << ", " << HEIGHT
          << ", " << ANGLE;
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_POINTE::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ")";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_point_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
