/*--- Filename: "ca_point.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The point color attractor.
  
*/

#include "ca_point.hpp"

#if !defined(__DB_ca_point_BODY_INCLUDED)
#define __DB_ca_point_BODY_INCLUDED

// default ctor just sets default values
CA_POINT::CA_POINT()
{
  EXP = 1.0;
  X = 0.5;
  Y = 0.5;
  RANGE = 0.5;
  TYPE = CAT_POINT; // remember to override in derived types
}

// other ctor sets user values
CA_POINT::CA_POINT(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, double range)
{
  CA_GLOBAL::CA_GLOBAL(mode,r,g,b,colint);
  EXP = exp;
  X = x;
  Y = y;
  RANGE = range;
  TYPE = CAT_POINT; // remember to override in derived types
}

CA_POINT::~CA_POINT()
{
  CA_GLOBAL::delete_line_buf();
}

// Set and Get Mehods

//
void CA_POINT::set_exp(double exp)
{
  EXP = exp;
}

//
double CA_POINT::get_exp()
{
  return EXP;
}

//
void CA_POINT::set_x(double x)
{
  X = x;
  _render_var_changed = true;
}

//
double CA_POINT::get_x()
{
  return X;
}

//
void CA_POINT::set_y(double y)
{
  Y = y;
  _render_var_changed = true;
}

//
double CA_POINT::get_y()
{
  return Y;
}

//
void CA_POINT::set_range(double range)
{
  RANGE = range;
  _render_var_changed = true;
}

//
double CA_POINT::get_range()
{
  return RANGE;
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_POINT::distance(int x, int y)
{
  double dx = _pix_x - x;
  double dy = _pix_y - y;
  return sqrt(dx*dx+dy*dy);
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_POINT::update_absolutes(int width, int height,
                                lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;

  _pix_x = width * X;
  _pix_y = height * Y;
  _pix_range = width * RANGE;

  return true;
}

// The *only* function ever called by the "render_gradient" method from..
// ..GRADIENT_GEN class (directly modifies "crgb"; applies CA's influence)
// (is virtual)
void CA_POINT::modify_crgb(int x, int y, customRGB<double>& crgb)
{
  double d = distance(x,y);

  if(d<=_pix_range)
  {
    double calc = (1.0 - pow(d / _pix_range,EXP))*COLINT; 
    switch(MODE)
    {
      case CAM_ATTRACT:
        crgb.add_cr(calc*R);
        crgb.add_cg(calc*G);
        crgb.add_cb(calc*B);
      break;
      case CAM_ABSORB:
        crgb.add_cr(-calc*R);
        crgb.add_cg(-calc*G);
        crgb.add_cb(-calc*B);
      break;
      default:;// to prevent ".. not handled in switch" warning
    }
  }
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_POINT::update_vars_from_line()
{
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // Y double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&Y);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  _render_var_changed = true;  

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_POINT::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP
          << ", " << X
          << ", " << Y
          << ", " << RANGE;
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_POINT::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ")";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_point_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
