/*--- Filename: "ca_global.hpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content: 
  The "Color Attractor Base" Class, that all other "CA_" classes are
  derived from.

  Every pixel is directly affected by a global color attractor, regardless
  of its' position.

  Note: "GLOBAL" has got nothing to do with the programming scope here,
        it merely identifies the behaviour of the Color Attractor

*/

#include "shared.hpp"

#if !defined(__DB_ca_global_HEAD_INCLUDED)
#define __DB_ca_global_HEAD_INCLUDED

// callback function type for "update absolutes" function of attractors
// (used by gradient_gen.cpp)
/* (The "values_to_go" parameter receives the number of lookup table values
    that still have to be calculated.
    The function is only called approx. every 'lcall_skip'th millisecond.
    The "type" parameter receives the TYPE id of the calling attractor.
    The "n" parameter receives the number of the attractor in the gradient 
    generators attractor list.)
   (attractors are of course free NOT to call that function at all,
    if they don't need to build any lookup tables))
   (an actual function of this type should return false, if it wants the
    rendering to be cancelled, true otherwise) */
typedef bool(*lut_callback)(int values_to_go, CA_TYPE type, int n);

class CA_GLOBAL
{
  private:
    int _prev_w; // width of previous render area in pixels
    int _prev_h; // height of previous render area in pixels
    
  protected:
    bool dim_changed(int w, int h);
    /* returns true if render dimensions(or some essential variables) have
       changed since last rendering */

    bool _render_var_changed; 
    /* used by some set methods to signal (essential) changes(e.g.position),
       so that they are recognized in "update_absolutes" */
        
  // variables that affect the attractors influence
    CA_TYPE TYPE; // this is *always* set by ctor (used for faster RTTI)
    CA_MODE MODE; // specifies how the attractor will affect colors
    int R,G,B; // the color triplet ((r)ed, (g)reen, (b)lue)
    double COLINT; // "color intensity" the color will be multiplied by this

  // some variables that are not directly related to the attractors behaviour
    char *LINE; // a text line with values in human readable form
    int create_line_buf(unsigned int length);
    void delete_line_buf();
    char *TOKEN; // used for evaluating the values from a string
/*!*/virtual int update_vars_from_line(); /* used by "set_from_text"
                                         (Note to self:CONTINUES TOKENIZING)*/
  public:
  // Construction and Destruction
    CA_GLOBAL();
    CA_GLOBAL(CA_MODE mode, int r, int g, int b, double colint=1.0);
/*!*/virtual ~CA_GLOBAL(); // remember to have dtor virtual in derived types

  // Set and Get Methods
    CA_TYPE get_type();  // (user can NOT set the type)
    void set_mode(CA_MODE mode);
    CA_MODE get_mode();

    void set_col(int color);
    int get_col();

    void set_rgb(int r, int g, int b);
    int get_r();
    int get_g();
    int get_b();

    void set_colint(double colint);
    double get_colint();
  
  // called by the GRADIENT_GEN prior to rendering
  // (should return true, if rendering should be done, false otherwise)
/*!*/virtual bool update_absolutes(int width, int height,
                                   lut_callback lcall, double lcall_skip, int n);

  // The *only* function ever called by the "render_gradient" method from..
  // ..GRADIENT_GEN class (directly modifies "crgb"; applies CA's influence)
/*!*/virtual void modify_crgb(int x, int y, customRGB<double>& crgb);

  // Saving and Loading to/from strings
/*!*/virtual string get_as_text();
     int set_from_text(string params);//Note to self:INITS TOKENIZING

  // Helper function for GUI building that returns some very sparse info
  // meant to be displayed in a list
/*!*/virtual string gui_quick_info();
};

#endif // #if !defined(__DB_ca_global_HEAD_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
