/*--- Filename: "ca_global.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The "Color Attractor Base" Class, that all other "ca_" classes are
  derived from.
  
*/

#include "ca_global.hpp"

#if !defined(__DB_ca_global_BODY_INCLUDED)
#define __DB_ca_global_BODY_INCLUDED

// checks if render dimensions or any essential variables have changed
bool CA_GLOBAL::dim_changed(int w, int h)
{
  if((w!=_prev_w)||(h!=_prev_h)||(_render_var_changed))
  {
    _prev_w = w;
    _prev_h = h;
    _render_var_changed = false;
    return true;
  }
  else
    return false;
}

// default ctor just sets default values
CA_GLOBAL::CA_GLOBAL()
{
  _prev_w=0;
  _prev_h=0;
  _render_var_changed = true;

  TYPE = CAT_GLOBAL; // remember to override in derived types
  MODE = CAM_ATTRACT;
  R = 255;
  G = 255;
  B = 255;
  COLINT = 1.0;

  LINE = NULL;
  TOKEN = NULL;
}

// other ctor sets user values
CA_GLOBAL::CA_GLOBAL(CA_MODE mode, int r,int g,int b, double colint)
{
  _prev_w=0;
  _prev_h=0;
  _render_var_changed = true;

  TYPE = CAT_GLOBAL; // remember to override in derived types
  MODE = mode;
  R = clamp_val<int>(0,r,255);
  G = clamp_val<int>(0,g,255);
  B = clamp_val<int>(0,b,255);
  COLINT = colint;

  LINE = NULL;
  TOKEN = NULL;
}

CA_GLOBAL::~CA_GLOBAL()
{
  delete_line_buf();
}

int CA_GLOBAL::create_line_buf(unsigned int length)
{
  delete_line_buf(); // clear old buffer first
  
  // get new buffer
  LINE = new(nothrow)char[length];
  if(LINE==NULL)
    return -1;
  return 0;
}
void CA_GLOBAL::delete_line_buf()
{
  if(LINE)
  {
    delete[] LINE;
    LINE = NULL;
  }
}

// Set and Get Methods

//
CA_TYPE CA_GLOBAL::get_type()
{
  return TYPE;
}

//
void CA_GLOBAL::set_mode(CA_MODE mode)
{
  MODE = mode;
}

//
CA_MODE CA_GLOBAL::get_mode()
{
  return MODE;
}

//
void CA_GLOBAL::set_col(int color)
{
  /* Description of the following:
     The layout of the 32bit color in the int is interpreted here as
     0x00RRGGBB, that is bits 0to7 are Blue, 8to15 are Green, 16to31 are Red.
     The OR sets unused bits to 1, the XOR eliminates all unused bits then
     and finally the shiftright makes sure that all values are inside 0to255.
  */
  R = ((color|0xFF00FFFF)^(0xFF00FFFF))>>16;
  G = ((color|0xFFFF00FF)^(0xFFFF00FF))>>8;
  B = ((color|0xFFFFFF00)^(0xFFFFFF00));
}

//
int CA_GLOBAL::get_col()
{
  return ((((0+R)<<8)+G)<<8)+B;
}

// 
void CA_GLOBAL::set_rgb(int r, int g, int b)
{
  R = clamp_val<int>(0,r,255);
  G = clamp_val<int>(0,g,255);
  B = clamp_val<int>(0,b,255);
}

//
int CA_GLOBAL::get_r()
{
  return R;
}

//
int CA_GLOBAL::get_g()
{
  return G;
}

//
int CA_GLOBAL::get_b()
{
  return B;
}

//
void CA_GLOBAL::set_colint(double colint)
{
  COLINT = colint;
}

//
double CA_GLOBAL::get_colint()
{
  return COLINT;
}

// called by the GRADIENT_GEN, whenever render dimensions change
// (is virtual)
bool CA_GLOBAL::update_absolutes(int width, int height,
                                 lut_callback lcall, double lcall_skip, int n)
{
  // nothing to do for global attractor type
  return true;
}

// The *only* function ever called by the render_gradient method from..
// ..GRADIENT_GEN class (directly modifies "crgb"; applies CA's influence)
// (is virtual)
void CA_GLOBAL::modify_crgb(int x, int y, customRGB<double>& crgb)
{
  /* x,y width and height are ignored for the GLOBAL attractor type..
     ..they are just here to make the function call the same on all..
     ..the different attractor types */
  switch(MODE)
  {
    case CAM_ATTRACT:
      crgb.add_cr(COLINT*R);
      crgb.add_cg(COLINT*G);
      crgb.add_cb(COLINT*B);
    break;
    case CAM_ABSORB:
      crgb.add_cr(-COLINT*R);
      crgb.add_cg(-COLINT*G);
      crgb.add_cb(-COLINT*B);
    break;
    default:;// to prevent ".. not handled in switch" warning
  }
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_GLOBAL::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << get_ca_mode_text(MODE).c_str()
          << ", " << R
          << ", " << G
          << ", " << B
		  << ", " << COLINT;
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_GLOBAL::update_vars_from_line()
{
  int ret = 0;

  TOKEN = strtok(NULL,token_seperators); // R integer expected
  if(TOKEN)
  {
    ret = sscanf(TOKEN,"%i",&R);    
    if((ret==0)||(ret==EOF)) return -1;
    dclamp_val(0,R,255);
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // G integer expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%i",&G);    
    if((ret==0)||(ret==EOF)) return -1;
    dclamp_val(0,G,255);
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // B integer expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%i",&B);    
    if((ret==0)||(ret==EOF)) return -1;
    dclamp_val(0,B,255);
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // COLINT double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&COLINT);    
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  // Values for GLOBAL color attractor type succesfully read

  return 0;
}

/* Set the variables' values from a human readable string
  (Initializes string tokenizing and MODE then calls "update_vars_from_line()"
  returns 0 on success, -1 on failure */
int CA_GLOBAL::set_from_text(string params)
{
  if(create_line_buf(params.length()+1)!=0)
    return -1; // not enough memory for internal text buffer

  sprintf(LINE,"%s",params.c_str());
  TOKEN = NULL;
  TOKEN = strtok(LINE,token_seperators);
                                         //get first token(MODE text expected)
  if(TOKEN)
  {
    int i;
    bool mode_found = false;
    string mode_desc(TOKEN);

    // compare token to find MODE
    for(i = CAM_ATTRACT; (i<CAM_NUM_MODES) && (!mode_found); i++)
    {
      if(mode_desc.compare(get_ca_mode_text((CA_MODE)i))==0)// have a match?
      {
        MODE = (CA_MODE)i;
        mode_found = true;
      }
    } // end of find MODE loop

    if (mode_found) return update_vars_from_line();
  }
  return -1; // error! insufficient or unexpected information in params
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_GLOBAL::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT;
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_global_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
