/*--- Filename: "ca_ellipsep.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The ellipse (with precise distance lookup table) color attractor.
  

*/

#include "ca_ellipsep.hpp"

#if !defined(__DB_ca_ellipsep_BODY_INCLUDED)
#define __DB_ca_ellipsep_BODY_INCLUDED

// default ctor just sets default values
CA_ELLIPSEP::CA_ELLIPSEP()
{
  _prev_pix_ew2 = 0.0;
  _prev_pix_eh2 = 0.0;
  _prev_pix_rng = 0.0;

  _mem_error = false;
  ELUT = NULL;
  _lut_w = 0;
  _lut_h = 0;

  TYPE = CAT_ELLIPSE; // remember to override in derived types
}

// other ctor sets user values
CA_ELLIPSEP::CA_ELLIPSEP(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, 
                   double width, double height, double angle, double range)
{
  CA_ELLIPSE::CA_ELLIPSE(mode,r,g,b,colint,exp,x,y,width,height,angle);

  _prev_pix_ew2 = 0.0;
  _prev_pix_eh2 = 0.0;
  _prev_pix_rng = 0.0;

  _mem_error = false;
  ELUT = NULL;
  _lut_w = 0;
  _lut_h = 0;

  TYPE = CAT_ELLIPSE; // remember to override in derived types
}

CA_ELLIPSEP::~CA_ELLIPSEP()
{
  if(ELUT)
  {
    delete [] ELUT;
    ELUT = NULL;
    _mem_error = false;
  }
  _lut_w = 0;
  _lut_h = 0;

  CA_GLOBAL::delete_line_buf();
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_ELLIPSEP::distance(int x, int y)
{
  if (_mem_error) 
    return _pix_range + 1.0; // prevent unwanted behaviour

  int i,j;
  double frac_vx,frac_vy,int_vx,int_vy;
  
  double vec_x = x - _pix_x; // vector from ellipses center..
  double vec_y = y - _pix_y; // ..to current point
  double vec_l = sqrt(vec_x*vec_x + vec_y*vec_y); // length of it
  double vec_ang;

  /* if "vec_l" is greater than the radius of the bounding circle, then
     the point can not be inside the elliptic range */
  if(vec_l > _bounding_circle_r)
    return _pix_range + 1.0;

  // calculate angle(in radians) of vector in the screen's coordinate system
  if(vec_y>=0.0)
    vec_ang=acos(vec_x / vec_l);
  else // (vec_y<0)
    vec_ang=_2pi - acos(vec_x / vec_l);

  // rotate the vector backwards into the ellipse's coordinate system
  vec_ang = vec_ang - _ang_rad; // (_ang_rad is the angle of ellipse)
  if(vec_ang<0.0)
    vec_ang += _2pi;

  /* simplifying the following calculations by norming the vector into
     positive x and y (in the ellipses coordinate system), so that
     the seeked distance can be read from the lookup table */
  vec_x = fabs(vec_l * cos(vec_ang));
  vec_y = fabs(vec_l * sin(vec_ang));
  
  /* (vec_x,vec_y is now the position at which i'll find the shortest distance
      inside the lookup table) */

  if(ceil(vec_x) >= (double)_lut_w)
    return _pix_range + 1.0; // out of range
  if(ceil(vec_y) >= (double)_lut_h)
    return _pix_range + 1.0; // out of range

  // Round the position
  frac_vx = modf(vec_x,&int_vx);
  frac_vy = modf(vec_y,&int_vy);

  i = (frac_vx < 0.5) ? int(int_vx) : int(int_vx+1.0);
  j = (frac_vy < 0.5) ? int(int_vy) : int(int_vy+1.0);

  // finally return the shortest distance stored inside the lookup table
  return ELUT[i+j*_lut_w];
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_ELLIPSEP::update_absolutes(int width, int height,
                                   lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;
 
  int a,i,j;
  double *_alut_x = NULL; // lookup tables with x,y for "every" angle in..
  double *_alut_y = NULL; // ..the first quadrant of the ellipse
  int _alut_num = 0; // number of angles in the lookup table
  double ang_search_frac; // the difference(in radians) between two
                          // .."adjacent" angles
  double cur_dist; // used in lookup table creation
  double cur_dist2;// -"-
  double cur_dist3;// -"-
  int a2,a3;       // -"-
  double shortest_dist; // -"-
  double cur_vx; // -"- (distance vector x)
  double cur_vy; // -"- (distance vector y)
  const int sp = 1;
  int search_dir = sp;
  int values_left = 0;
  clock_t last_cb_clock;
  clock_t current_cb_clock;

  _pix_x = width * X;
  _pix_y = height * Y;
  _pix_w2 = (width * WIDTH) / 2.0;
  _pix_h2 = (height * HEIGHT) / 2.0;
  if(_pix_w2==0.0) 
    _pix_w2 = 1.0; // prevent div by zero in other functions
  if(_pix_h2==0.0) 
    _pix_h2 = 1.0; // prevent div by zero in other functions
  _pix_range = width * RANGE;

  if(fabs(_pix_w2)<=fabs(_pix_h2))
    _bounding_circle_r = _pix_h2 + _pix_range;
  else
    _bounding_circle_r = _pix_w2 + _pix_range;

// check if lookup creation is necessary and leave if it is not
  if((_pix_w2 == _prev_pix_ew2)&&(_pix_h2 == _prev_pix_eh2)
     &&(_pix_range == _prev_pix_rng)&&(!_mem_error))
  {
    return true;
  }
  else // new lookup IS necessary
  {
    _prev_pix_ew2 = _pix_w2;
    _prev_pix_eh2 = _pix_h2;
    _prev_pix_rng = _pix_range;
  }

// prepare handling of callback function and call it once before calculating
// (some calculation that were originally done below are now pulled up here,
//  so that i know, how many values there are to calculate in total)
// (the places of the orignal calculation are still there, but as comments)
  ang_search_frac = 2.0/**_pi*/ / (/*_pi**/(_pix_w2 + _pix_h2));
  _alut_num = (int)ceil( _pi / (2.0*ang_search_frac));
  _lut_w = (int)ceil(fabs(_pix_w2)+fabs(_pix_range));
  _lut_h = (int)ceil(fabs(_pix_h2)+fabs(_pix_range));

  values_left = _alut_num + _lut_w*_lut_h;
  last_cb_clock = clock();
  if(lcall)
    (*lcall)(values_left,TYPE,n);

/* Prepare the temporary lookup table, which will be used to speed up the
   creation of the shortest distance lookup table */
  //ang_search_frac = 2.0/**_pi*/ / (/*_pi**/(_pix_w2 + _pix_h2));
  //_alut_num = (int)ceil( _pi / (2.0*ang_search_frac));
    /* (the gird of an ellipse is PI*(w/2 + h/2), so dividing PI by half of
        that gird, should be precision(small angle fraction) enough
        (as in: give enough points on the ellipse, dividing PI/2 by that
         small angle fraction) in the first quadrant) */
  
  _mem_error = false;
  _alut_x = new(double[_alut_num]);
  if(_alut_x)
  {
    _alut_y = new(double[_alut_num]);
    if(!_alut_y)
    {
      delete [] _alut_x;
      _mem_error = true;
    }
  }
  else
  {
    _mem_error = true;
  }
  if(_mem_error) 
    return true;// error, not enough memory for temporary lookup table

// Fill the temporary lookup table with the ellipses points, for "every" angle
  for(a=0; a<_alut_num; a++)
  {
    _alut_x[a] = _pix_w2 * cos(((double)a)*ang_search_frac);
    _alut_y[a] = _pix_h2 * sin(((double)a)*ang_search_frac);

    // call callback function (if any)
    values_left--;
    current_cb_clock = clock();
    
    if((lcall)
       && ((double(current_cb_clock-last_cb_clock)/double(CLOCKS_PER_SEC))*1000.0>=lcall_skip))
    {      
      if(!((*lcall)(values_left,TYPE,n)))
      {
        _render_var_changed = true; // mark me for recalculation next time
        _prev_pix_ew2 = 0.0;
        _prev_pix_eh2 = 0.0;
        _prev_pix_rng = 0.0;
        delete [] _alut_x;
        delete [] _alut_y;
        _mem_error = true;
        return false; // abort rendering
      }
      else
        last_cb_clock = clock();
    }
  }

// Now prepare the real shortest distances lookup table for the first quadrant
  //_lut_w = (int)ceil(fabs(_pix_w2)+fabs(_pix_range));
  //_lut_h = (int)ceil(fabs(_pix_h2)+fabs(_pix_range));

  if(ELUT) // kill previous lookup table, if any
  {
    delete [] ELUT;
    ELUT = NULL;
  }

  ELUT = new(double[_lut_w*_lut_h]);
  if(!ELUT)
  {
    delete [] _alut_x;
    delete [] _alut_y;
    _mem_error = true;
    return true; // error, not enough memory for ellipse distance lookup table
  }

// Fill the ellipse distance lookup table for the first quadrant
  for(j=0; j<_lut_h; j++)
    for(i=0; i<_lut_w; i++)
    {
      /*/ // OLD (SLOWER "check EVERY angle") METHOD
      // init distance and shortest dist
      cur_vx = i-_alut_x[0];
      cur_vy = j-_alut_y[0];
      cur_dist = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);
      shortest_dist = cur_dist;
      
      // Now brute force find the shortest distance to any of the points
      for(a=1; a<_alut_num; a++)
      {
        cur_vx = i-_alut_x[a];
        cur_vy = j-_alut_y[a];
        cur_dist = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);
        if(cur_dist<shortest_dist)
          shortest_dist = cur_dist;
      }
      // add that distance value to the lookup table
      ELUT[i+j*_lut_w] = shortest_dist;
      //*/ // END OF OLD METHOD 

      /**/
      // NEW METHOD (should be at least twice as fast as the old one)
      // Initialize three indices and three distances
      a=0;
      a2=_alut_num/2;
      a3=_alut_num-1;

      cur_vx = i-_alut_x[a];
      cur_vy = j-_alut_y[a];
      cur_dist = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);

      cur_vx = i-_alut_x[a2];
      cur_vy = j-_alut_y[a2];
      cur_dist2 = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);

      cur_vx = i-_alut_x[a3];
      cur_vy = j-_alut_y[a3];
      cur_dist3 = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);

      if(cur_dist < cur_dist2)
        shortest_dist = cur_dist;
      else
        shortest_dist = cur_dist2;
      if(cur_dist3 < shortest_dist)
        shortest_dist = cur_dist3;

      // Select as starting index the one with the smallest distance
      if (shortest_dist == cur_dist)
        a=a;
      if (shortest_dist == cur_dist2)
        a=a2;
      if (shortest_dist == cur_dist3)
        a=a3;
      
      // Initialize the search direction
      if(a==_alut_num-1)
        search_dir = -sp;
      else
      if(a==0)
        search_dir = sp;
      else
      if((a>0)&&(a<_alut_num-1))
      {
        a2=a+sp;
        a3=a-sp;
      
        cur_dist2 = cur_dist;
        cur_dist3 = cur_dist;

        if((a2>=0)&&(a2<_alut_num))
        {
          cur_vx = i-_alut_x[a2];
          cur_vy = j-_alut_y[a2];
          cur_dist2 = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);
        }
        if((a3>=0)&&(a3<_alut_num))
        {
          cur_vx = i-_alut_x[a3];
          cur_vy = j-_alut_y[a3];
          cur_dist3 = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);
        }

        if((cur_dist2 < cur_dist3)&&(cur_dist2 < cur_dist))
          search_dir = sp;
        else
        if((cur_dist2 > cur_dist3)&&(cur_dist3 < cur_dist))
          search_dir = -sp;
        else
          search_dir = 0;
      }
      else search_dir = 0;

      // Go on searching in that direction until a found distance is greater
      while(search_dir)
      {
        a = a + search_dir;
        if((a>=0)&&(a<_alut_num))
        {
          cur_vx = i-_alut_x[a];
          cur_vy = j-_alut_y[a];
          cur_dist = sqrt(cur_vx*cur_vx + cur_vy*cur_vy);
          if (cur_dist < shortest_dist)
            shortest_dist = cur_dist;
          else search_dir = 0;
        }
        else search_dir = 0;
      }

      // add that distance value to the lookup table
      ELUT[i+j*_lut_w] = shortest_dist; // END OF NEW METHOD */

      // call callback function (if any)
      values_left--;
      current_cb_clock = clock();
      
      if((lcall)
         && ((double(current_cb_clock-last_cb_clock)/double(CLOCKS_PER_SEC))*1000.0>=lcall_skip))
      { 
        if(!((*lcall)(values_left,TYPE,n)))
        {
          _render_var_changed = true; // mark me for recalculation next time
          _prev_pix_ew2 = 0.0;
          _prev_pix_eh2 = 0.0;
          _prev_pix_rng = 0.0;
          delete [] _alut_x;
          delete [] _alut_y;
          delete [] ELUT;
          ELUT = NULL;
          _mem_error = true;
          return false; // abort rendering
        }
        else
          last_cb_clock = clock();
      }
    }//END of lookup table creation loop

// remember to delete the temporary lookup table
  delete [] _alut_x;
  delete [] _alut_y;
  return true;
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_ELLIPSEP::update_vars_from_line()
{
  return CA_ELLIPSE::update_vars_from_line();
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_ELLIPSEP::get_as_text()
{
  return CA_ELLIPSE::get_as_text();
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_ELLIPSEP::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ") " << ((_mem_error) ? "(!)" : ".");
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_ellipse_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
