/*--- Filename: "ca_ellipse.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The ellipse color attractor.

  NOTE:
    This is not being used anymore in the final version as as standalone
    attractor.
    Instead the previous ellipsep! has been promoted to be externally
    known as ellipse(without p).
    This is still here, because the ellipsep attractor type inherits
    the standard ellipse attractor type.
*/

#include "ca_ellipse.hpp"

#if !defined(__DB_ca_ellipse_BODY_INCLUDED)
#define __DB_ca_ellipse_BODY_INCLUDED

// default ctor just sets default values
CA_ELLIPSE::CA_ELLIPSE()
{
  RANGE = 0.25;
  TYPE = CAT_ELLIPSE; // remember to override in derived types
}

// other ctor sets user values
CA_ELLIPSE::CA_ELLIPSE(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, 
                   double width, double height, double angle, double range)
{
  CA_POINTE::CA_POINTE(mode,r,g,b,colint,exp,x,y,width,height,angle);
  RANGE = range;
  TYPE = CAT_ELLIPSE; // remember to override in derived types
}

CA_ELLIPSE::~CA_ELLIPSE()
{
  CA_GLOBAL::delete_line_buf();
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_ELLIPSE::distance(int x, int y)
{ 
  double term; // used in distance calculation
  double vec_x = x - _pix_x; // vector from ellipses center..
  double vec_y = y - _pix_y; // ..to current point
  double vec_l = sqrt(vec_x*vec_x + vec_y*vec_y); // length of it
  double vec_ang;

  // Handle special case
  if((vec_x==0.0)&&(vec_y==0.0))
    return _pix_range + 1.0;

  // calculate angle(in radians) of vector in the screen's coordinate system
  if(vec_y>=0.0)
    vec_ang=acos(vec_x / vec_l);
  else // (vec_y<0)
    vec_ang=_2pi - acos(vec_x / vec_l);

  // rotate the vector backwards into the ellipse's coordinate system
  vec_ang = vec_ang - _ang_rad; // (_ang_rad is the angle of ellipse)
  if(vec_ang<0.0)
    vec_ang += _2pi;

  /* simplifying the following calculations by norming the vector into
     positive x and y (in the ellipses coordinate system) */
  //vec_x = fabs(vec_l * cos(vec_ang));
  //vec_y = fabs(vec_l * sin(vec_ang));
  vec_x = vec_l * cos(vec_ang);
  vec_y = vec_l * sin(vec_ang);

  term=(vec_x*vec_x) / (_pix_w2*_pix_w2) + (vec_y*vec_y) / (_pix_h2*_pix_h2);
  return fabs(1.0 - term)*_pix_range; // WRONG! DOES NOT WORK CORRECTLY. 
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_ELLIPSE::update_absolutes(int width, int height,
                                  lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;
 
  _pix_x = width * X;
  _pix_y = height * Y;
  _pix_w2 = (width * WIDTH) / 2.0;
  _pix_h2 = (height * HEIGHT) / 2.0;
  if(_pix_w2==0.0) 
    _pix_w2 = 1.0; // prevent div by zero in other functions
  if(_pix_h2==0.0) 
    _pix_h2 = 1.0; // prevent div by zero in other functions
  _pix_range = width * RANGE;
  if(_pix_range==0.0)
    _pix_range = 1.0;

  return true;
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_ELLIPSE::update_vars_from_line()
{
  int ret = CA_POINTE::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  _render_var_changed = true;

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_ELLIPSE::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << RANGE;
    rline.clear();
    return CA_POINTE::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_ELLIPSE::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ")";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_ellipse_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
