/*--- Filename: "ca_bar.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The rotated bar color attractor.
  
*/

#include "ca_bar.hpp"

#if !defined(__DB_ca_bar_BODY_INCLUDED)
#define __DB_ca_bar_BODY_INCLUDED

// default ctor just sets default values
CA_BAR::CA_BAR()
{
  ANGLE = 0.0;
  _ang_rad = 0.0;
  TYPE = CAT_BAR; // remember to override in derived types
}

// other ctor sets user values
CA_BAR::CA_BAR(CA_MODE mode, int r, int g, int b, 
                   double colint, double exp,
                   double x, double y, double angle, double range)
{
  CA_POINT::CA_POINT(mode,r,g,b,colint,exp,x,y,range);

  ANGLE = fmod(angle,360.0);
  _ang_rad = make_rad(ANGLE);
  if(_ang_rad<0)
    _ang_rad += _2pi;

  TYPE = CAT_BAR; // remember to override in derived types
}

CA_BAR::~CA_BAR()
{
  CA_GLOBAL::delete_line_buf();
}

// Set and Get Mehods

//
void CA_BAR::set_angle(double angle)
{
  ANGLE = fmod(angle,360.0);
  _ang_rad = make_rad(ANGLE);
  if(_ang_rad<0.0)
    _ang_rad += _2pi;
}

//
double CA_BAR::get_angle()
{
  return ANGLE;
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_BAR::distance(int x, int y)
{
  double vec_x = x - _pix_x;
  double vec_y = y - _pix_y;
  double vec_l = sqrt(vec_x*vec_x + vec_y*vec_y);
  double vec_ang;

  // special case fix
  if((vec_x==0.0) && (vec_y==0.0))
    return 0.0;

  // calculate the vectors angle in screens' coordinate system
  if(vec_y>=0)
    vec_ang = acos(vec_x / vec_l);
  else
    vec_ang = _2pi - acos(vec_x / vec_l);

  // rotate the y component back into the bars' coordinate system
  vec_ang -= _ang_rad;
  if(vec_ang<0.0)
    vec_ang += _2pi;
  vec_y = sin(vec_ang)*vec_l;

  return abs(vec_y);
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_BAR::update_vars_from_line()
{
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // Y double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&Y);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // ANGLE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&ANGLE);

    ANGLE = fmod(ANGLE,360.0);
    _ang_rad = make_rad(ANGLE);
    if(_ang_rad<0.0)
      _ang_rad += _2pi;

    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  _render_var_changed = true;  

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_BAR::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP 
          << ", " << X 
          << ", " << Y 
          << ", " << ANGLE 
          << ", " << RANGE;
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_BAR::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ")";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_bar_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
