/*--- Filename: "shared.hpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  Shared functions and types used by other classes.

*/

#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <string>
using namespace std;
#include <sstream>
using namespace std;

#if !defined(__DB_shared_HEAD_INCLUDED)
#define __DB_shared_HEAD_INCLUDED

const double _pi = 3.1415926535897932384626433832795;
const double _2pi = 2 * _pi;
const double _to_rad = _pi / 180.0; // multiply with deg to convert to radians
const double _to_deg = 180.0 / _pi; // multiply with rad to convert to degrees

inline double make_rad(double degrees) { return degrees * _to_rad; }
inline double make_deg(double radians) { return radians * _to_deg; }

const char token_seperators[] = " ,;\t"; // for string(char* ) parsing

// returns the passed "val" or "low" if it's lower, or "high" if it's higher
template<class type> type clamp_val(type low, type val, type high)
{
  if(val<low) 
    return low;
  if(val>high) 
    return high;
  return val;
}

// same as above, except it (d)irectly modifies passed "val"
template<class type> void dclamp_val(type low, type& val, type high)
{
  if(val<low) val = low;
  if(val>high) val = high;
}

// returns and integer of the format 0x00rrggbb, r,g,b will be clamped [0,255]
int make_rgb32(int r, int g, int b);

// return a portion from an 32rgb int as described in the function above
int get_r32(int color);
int get_g32(int color);
int get_b32(int color);

template<class type> class customRGB // meant for a "float or double" color
{
  private:
    type CR,CG,CB; // color triplet

  public:
    customRGB() { CR = 0.0; CG = 0.0; CB = 0.0; }
    customRGB(int color) { CR = clamp_val<type>(0,(type)get_r32(color),255);
                           CG = clamp_val<type>(0,(type)get_g32(color),255);
                           CB = clamp_val<type>(0,(type)get_b32(color),255); }

    customRGB(int r,int g,int b) { CR = clamp_val<type>(0,(type)r,255);
                                   CG = clamp_val<type>(0,(type)g,255);
                                   CB = clamp_val<type>(0,(type)b,255); }
    // Set with clamping
    void set_col(int color) 
                { CR = clamp_val<type>(0,(type)get_r32(color),255);
                  CG = clamp_val<type>(0,(type)get_g32(color),255);
                  CB = clamp_val<type>(0,(type)get_b32(color),255); }
    void set_r(int r) { CR = clamp_val<type>(0,(type)r,255); }
    void set_g(int g) { CG = clamp_val<type>(0,(type)g,255); }
    void set_b(int b) { CB = clamp_val<type>(0,(type)b,255); }

    // Set without clamping
    void set_cr(type cr) { CR = cr; }
    void set_cg(type cg) { CG = cg; }
    void set_cb(type cb) { CB = cb; }

    // Adding to the current triplet
    void add_cr(type cr) { CR +=cr; }
    void add_cg(type cg) { CG +=cg; }
    void add_cb(type cb) { CB +=cb; }

    // Get clamped color
    int get_col() { int r,g,b;
                    r = clamp_val<int>(0,(int)CR,255);
                    g = clamp_val<int>(0,(int)CG,255);
                    b = clamp_val<int>(0,(int)CB,255);
                    return make_rgb32(r,g,b); }

    // Get unclamped components
    type get_cr() { return CR; }
    type get_cg() { return CG; }
    type get_cb() { return CB; }
};

enum CA_TYPE /* COLOR ATTRACTOR TYPE (*not* used by the "CA_.." classes, but
                                      by the COL_GRADIENT class) */
{
  CAT_GLOBAL=0, // a fixed percentage of the color gets mixed in
  CAT_POINT,    // mix color depends on distance to the point
  CAT_POINTE, // point with a rotated elliptic range
  CAT_HBAR,     // mix color depends on distance to the bar
  CAT_VBAR,     // mix color depends on distance to the bar
  CAT_BAR, // mix color depends on distance to rotated bar
  CAT_POLYNOMIAL, // mix color depends on distance to the function value
  CAT_CIRCLE, // mix color depends on distance to the circle line
  CAT_CIRCLEF, // same, except distance is always 0 for points inside circle
  CAT_ELLIPSE, // mix color depends on distance to rotated ellipse line
  // *** insert more types above this line, leave this line in ***
  CAT_NUM_TYPES // number of different types
};
// returns a wstring containing the internal description of the CA_TYPE
string get_ca_type_text(CA_TYPE which);

enum CA_MODE // COLOR ATTRACTOR MODE (used by the "CA_.." classes)
{
  CAM_ATTRACT=0, // ADDs fractions of its' color
  CAM_ABSORB,    // SUBTRACTs fractions of its' color
  /* Note to self: Actually the absorb mode is obsolete because the same
                   effect can be achieved by setting a negative value to
                   the "color intensity" of an attractor.
                   I leave this in for better distinction anyway and because
                   there may come other modes sometime that will have a more
                   unique behaviour.
                   */
  // *** insert more modes above this line, leave this line in ***
  CAM_NUM_MODES
};
// return a wstring containing the internal description of the CA_MODE
string get_ca_mode_text(CA_MODE which);

// returns the newly allocated buffer or NULL on failure
char *make_char_buffer(unsigned int length);

#endif // #if !defined(__DB_shared_HEAD_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
