/*--- Filename: "gradient_gen.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The gradient generator class.  

*/

#include "gradient_gen.hpp"

#if !defined(__DB_gradient_gen_BODY_INCLUDED)
#define __DB_gradient_gen_BODY_INCLUDED

const char no_attr_err[] = "no such attractor!";

// reset
void GRADIENT_GEN::reset()
{
  //AUTHOR.assign(L"default ctor"); // ;D
  //wostringstream address; //doesn't work in MinGW3.4.2
  //address << L"instance at: 0x" << this;
  //TITLE.assign(address.str()); // ;)
  AUTHOR.assign(L"...");
  TITLE.assign(L"...");
  DEF_W = 320;
  DEF_H = 240;
  last_render_time = 0.0;
}

// default ctor
GRADIENT_GEN::GRADIENT_GEN()
{
  reset();
}

// default dtor
GRADIENT_GEN::~GRADIENT_GEN()
{
  clear();
}

// Set And Get Methods

//
void GRADIENT_GEN::set_author(wstring author)
{
  AUTHOR.assign(author);
}

//
wstring GRADIENT_GEN::get_author()
{
  return wstring(AUTHOR);
}

//
void GRADIENT_GEN::set_title(wstring title)
{
  TITLE.assign(title);
}

//
wstring GRADIENT_GEN::get_title()
{
  return wstring(TITLE);
}

//
void GRADIENT_GEN::set_w(int w)
{
  if(w<=0)
   DEF_W=1;
  else DEF_W = w;
  // idea: change this later to support mirroring, using negative w
}

//
int GRADIENT_GEN::get_w()
{
  return DEF_W;
}

//
void GRADIENT_GEN::set_h(int h)
{
  if(h<=0)
   DEF_H=1;
  else DEF_H = h;
  // idea: change this later to support mirroring, using negative h
}

//
int GRADIENT_GEN::get_h()
{
  return DEF_H;
}

//
double GRADIENT_GEN::get_last_render_time()
{
  return last_render_time;
}

// returns -1 if empty, number of attractors otherwise
int GRADIENT_GEN::get_attractor_count()
{
  if(CA_VEC.empty())
    return -1;
  else return (int)CA_VEC.size();
}

// COLOR ATTRACTOR management

// add a color attractor by string description
// (added before attractor number "before")
// (before: 0 to front append, -1 to rear append)
// return 0 on success
int GRADIENT_GEN::add_ca(int before, string params)
{
  char *LINE = make_char_buffer(params.length()+1);
  char *TOKEN = NULL;
  vector<CA_GLOBAL *>::iterator inserted;
  int ret;

  if(LINE==NULL)
    return -1; // not enough memory for LINE buffer!

  sprintf(LINE,"%s",params.c_str());
  TOKEN = strtok(LINE,token_seperators);//get token(TYPE text expected)
  if(TOKEN)
  {
    int i;
    bool type_found = false;
    CA_TYPE TYPE = CAT_GLOBAL;
    string type_desc(TOKEN);

    // compare token to find TYPE
    for(i = CAT_GLOBAL; (i<CAT_NUM_TYPES)&&(!type_found); i++)
    {
      if(type_desc.compare(get_ca_type_text((CA_TYPE)i))==0)// have a match?
      {
        TYPE = (CA_TYPE)i;
        type_found = true;
      }

    } // end of find TYPE loop
    
    // old ELLIPSE has been thrown out from the beta version
    // and ELLIPSE P was promoted to be the new ELLIPSE
    // hence adding from old files that still have ELLIPSE P
    // in them has to be automatically adjusted
    if(type_desc.compare(string("ELLIPSEP"))==0)
    {
      TYPE = CAT_ELLIPSE;
      type_found = true;
    }

    if (type_found) // have a valid color attractor type
    {
      // determine the correct insert position
      if(before == 0) // (push front desired)
      {
        inserted = CA_VEC.begin();
      }
      else if(before < 0) // (push back desired)
      {
        inserted = CA_VEC.end();
      }
      else if(before > 0) // (insert in the middle desired)
      {
        // determine insert iterator position
        CA_GLOBAL *ref = NULL;
        vector<CA_GLOBAL *>::iterator vecit = CA_VEC.begin();
        bool force_push_back = false;

        if (CA_VEC.empty()) // force push back
        {
          force_push_back = true;
        }
        try
        {
          ref = CA_VEC.at(before);
        }
        catch (out_of_range) // force push back
        {
          force_push_back = true;
        }
        if(force_push_back)
          inserted = CA_VEC.end();
        else // desired insert position exists :)
        {
          vecit += before;
          inserted = vecit;
        }
      }
      // insert position is determined now

      switch(TYPE) // add the specific color attractor to the vector
      {
        case CAT_GLOBAL:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_GLOBAL());
        break;
        case CAT_POINT:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_POINT());
        break;
        case CAT_POINTE:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_POINTE());
        break;
        case CAT_HBAR:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_HBAR());
        break;
        case CAT_VBAR:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_VBAR());
        break;
        case CAT_BAR:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_BAR());
        break;
        case CAT_POLYNOMIAL:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_POLYNOMIAL());
        break;
        case CAT_CIRCLE:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_CIRCLE());
        break;
        case CAT_CIRCLEF:
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_CIRCLEF());
        break;
        case CAT_ELLIPSE: // ELLIPSEP got promoted to this since beta version
          inserted = CA_VEC.insert(inserted,new(nothrow)CA_ELLIPSEP());
        break;
        // *** insert new attractor types here ***
        default:;// to prevent ".. not handled in switch" warning
      }

      // initialize the color attractor with the rest of the params string
      TOKEN = strtok(NULL,"\0");
      ret = (*inserted)->set_from_text(string(TOKEN));
      if(ret!=0)// remove newly created attractor on failure
      {
        delete (*inserted);
        CA_VEC.erase(inserted);
      }
      delete[]LINE;
      return ret;
    }
    else // (!type_found)
    { 
      delete[]LINE;
      return -1; 
    }
  }
  else // (TOKEN == NULL)
  { 
    delete[]LINE;
    return -1; 
  }
}

// add color attractor by string description, (back append)
int GRADIENT_GEN::add_ca(string params)
{
  return add_ca(-1,params);
}

// removes a certain attractor from the list
// (counting starts from 0) returns 0 on success
int GRADIENT_GEN::rem_ca(int which)
{
  CA_GLOBAL *to_del = NULL;
  vector<CA_GLOBAL *>::iterator i = CA_VEC.begin();

  if (CA_VEC.empty()) return -1;

  try
  {
    to_del = CA_VEC.at(which);
  }
  catch (out_of_range)
  {
    return -1;
  }

  i += which;

  delete (*i);
  CA_VEC.erase(i);
  return 0;
}

// return a string description of the specified attractor
string GRADIENT_GEN::get_as_text(int which)
{
  CA_GLOBAL *ref = NULL;

  if (CA_VEC.empty()) return string("empty attractor list!");

  try
  {
    ref = CA_VEC.at(which);
  }
  catch (out_of_range)
  {
    return string(no_attr_err);
  }

  return get_ca_type_text(ref->get_type()) + ref->get_as_text();
}

// Change an attractor
// (actually totally removes it and then adds a whole new one)
// returns 0 on success
int GRADIENT_GEN::set_from_text(int which, string params)
{
  int ret = rem_ca(which);  
  if(ret!=0)
    return ret;
  else
    return add_ca(which,params);
}

// get the pointer to an attractor
CA_GLOBAL* GRADIENT_GEN::get_pca(int which)
{
  CA_GLOBAL *ref = NULL;
  vector<CA_GLOBAL *>::iterator i = CA_VEC.begin();

  if (CA_VEC.empty()) return NULL;

  try
  {
    ref = CA_VEC.at(which);
  }
  catch (out_of_range)
  {
    return NULL;
  }

  i += which;

  return *i; // i is a pointer to a pointer of CA_GLOBAL, so dereferencing
             // the pointer will give the pointer to a CA_GLOBAL
}

// attempts to switch the attractors
/* will do nothing and return -1 if either attractor number a or b
   does not exist */
int GRADIENT_GEN::switch_ca(int a, int b)
{
  CA_GLOBAL *refa = NULL;
  CA_GLOBAL *refb = NULL;
  string a_desc;
  string b_desc;
  
  if (CA_VEC.empty()) return -1;

  try
  {
    refa = CA_VEC.at(a);
  }
  catch (out_of_range)
  {
    return -1;
  }
  try
  {
    refb = CA_VEC.at(b);
  }
  catch (out_of_range)
  {
    return -1;
  }
  
  // if the method reaches this point, attractors a and b do exist
  
  // now first retrieve information about each attractor
  a_desc = get_as_text(a);
  b_desc = get_as_text(b);
  
  // then reset each with the respective other ones information
  set_from_text(a,b_desc);
  set_from_text(b,a_desc);
  return 0;
}

// kill all color attractors
void GRADIENT_GEN::clear()
{
  reset();
  while( rem_ca(0)==0 )
    ; // do nothing, as long as there are still attractors to remove
}

// Render Methods
int GRADIENT_GEN::render(BMP_T &target, int x, int y,
                         int w, int h, double target_int)
{
  return render(target, x, y, w, h, target_int, NULL, 0, NULL, 0);
}

// render to a rectangular area on the target bitmap
// with a callback function called every 'call_skip'th pixel rendered
int GRADIENT_GEN::render(BMP_T &target, int x, int y,
                         int w, int h, double target_int,
                         bool (*render_callback)(unsigned long pixels_to_go), 
                         double call_skip,
                         lut_callback lcall, double lcall_skip)
{
  if(!target.is_valid()) return -1; // not a valid render target
  if (CA_VEC.empty()) return -2; // no color attractors

  customRGB<double> current_col;
  vector<CA_GLOBAL *>::iterator CA;
  bool lut_ok = true;
  int i, j, n;
  int col;
  unsigned long pixels_left = 0;
  clock_t start_clock;
  clock_t end_clock;
  clock_t last_cb_clock;
  clock_t current_cb_clock;

  int sx;
  int sy;
  int ex;
  int ey;

  if(x>=target.get_width()) return 1; // right of target
  if(y>=target.get_height()) return 3; // below target

  sx = x;
  sy = y;
  ex = sx+w;
  ey = sy+h;

  if(sx<0)
    sx=0;
  if(sy<0)
    sy=0;

  if(ex>target.get_width()) ex=target.get_width();
  if(ey>target.get_height()) ey=target.get_height();

  if(ex<0) return 2; // left of target
  if(ey<0) return 4; // above target

  // Update the absolute values inside the color attractors
  n=0;
  for(CA = CA_VEC.begin(); CA != CA_VEC.end(); CA++,n++)
  {
    lut_ok = (*CA)->update_absolutes(abs(w),abs(h),lcall,lcall_skip,n);
    if(!lut_ok)
      return 6; // lookup callback cancelled rendering
  }

  // Prepare callback parameter and call callback once before render start
  if(render_callback)
  {
    pixels_left = abs(ex-sx)*abs(ey-sy);
    if(!((*render_callback)(pixels_left)))
      return 5; // callback canceled rendering
  }  

  // For determining the render time
  start_clock = clock();
  // For determining when to call callback
  last_cb_clock = clock();

  // Finally start doing the rendering: :-)
  for(j=sy; j<ey; j++) //for each line
    for(i=sx; i<ex; i++) //for each pixel along the line
    {
      current_col.set_col(0);
      if(target_int != 0.0)  // mix in target bitmap content
      {
        col = target.get_pixel32(i,j);
        current_col.add_cr(target_int*get_r32(col)); // init color r
        current_col.add_cg(target_int*get_g32(col)); // init color g
        current_col.add_cb(target_int*get_b32(col)); // init color b
      }

      for(CA = CA_VEC.begin(); CA != CA_VEC.end(); ++CA) // apply attractors
      {
        (*CA)->modify_crgb(i-x,j-y,current_col);
      }

      target.put_pixel32(i,j,current_col.get_col()); // write color to target

      // Update information for callback function
      pixels_left--;
      current_cb_clock = clock();

      // Call callback function
      if((render_callback)
         && ((double(current_cb_clock-last_cb_clock)/double(CLOCKS_PER_SEC))*1000.0>=call_skip))
      {
        if(!((*render_callback)(pixels_left)))
          return 5; // callback canceled rendering
        else
          last_cb_clock = clock();
      }
    }

  // For determining the render time
  end_clock = clock();
  last_render_time = (double)(end_clock - start_clock) / CLOCKS_PER_SEC;

  // Call callback once after render finished
  if(render_callback)
  {
    pixels_left = 0;
    if(!((*render_callback)(pixels_left)))
      return 5; // callback canceled rendering
  }  

  return 0;
}

// render to full target bitmap
int GRADIENT_GEN::render(BMP_T &target, double target_int)
{
  return render(target,0,0,target.get_width(),target.get_height(),target_int,
                NULL,0,NULL,0);
}

// same as above with callback function
int GRADIENT_GEN::render(BMP_T &target, double target_int,
           bool (*render_callback)(unsigned long pixels_to_go), 
           double call_skip, lut_callback lcall, double lcall_skip)
{
  return render(target,0,0,target.get_width(),target.get_height(),target_int,
                render_callback,call_skip,lcall,lcall_skip);
}

// write a wstring as UTF-16 little endian to a file pointer
int write_utf16_le(FILE *out, wstring what)
{
  int chars_written = 0;
  
  return chars_written;
}

// Saving
int GRADIENT_GEN::save(string filename)
{
  // error handling macros
  #define check_errs(x) ret=x;if(ret<0){fclose(out);return-1;}

  #define reset_sbuf() delete sbuf; sbuf=NULL; sbuf=new(nothrow)ostringstream; \
                        if(sbuf==NULL){fclose(out);return -1;} \
                        (*sbuf).exceptions(ios_base::badbit|  \
                                           ios_base::failbit| \
                                           ios_base::eofbit); \
                        (*sbuf).setf(ios_base::dec,ios_base::basefield);

  #define put_to_sbuf(x) try{(*sbuf) << x; (*sbuf).clear(); } \
                          catch(ios_base::failure){              \
                          delete sbuf; fclose(out); return -1;}

  FILE *out = NULL;
  int i = 0;
  int ret = 0;
  ostringstream *sbuf=NULL;
  string ca_text;

  // Open File for saving
  out = fopen(filename.c_str(),"wb");
  if(!out) 
    return -1; // file error
  
  // Write UTF-8 byte order mark to signal the file as UTF-8 for other editors
  write_bom_utf8(out);

  // Write Author and Title
  if(AUTHOR.empty())
    AUTHOR.assign(L".");
  if(TITLE.empty())
    TITLE.assign(L".");
  check_errs(writeline_utf8(out,AUTHOR));
  check_errs(writeline_utf8(out,TITLE));

  // Write suggested Width and Height
  reset_sbuf();
  put_to_sbuf(DEF_W);
  check_errs( writeline_utf8(out,sbuf->str()) );
  reset_sbuf();
  put_to_sbuf(DEF_H);
  check_errs( writeline_utf8(out,sbuf->str()) );

  // Write all attractors
  i=0;
  ca_text = get_as_text(i);
  if(!CA_VEC.empty())
    while(ca_text.compare(string(no_attr_err))!=0)
    {
      if(writeline_utf8(out,ca_text)<0)
        break;
      i++;
      ca_text = get_as_text(i);
    }

  // Close File
  fclose(out);

  #undef check_errs
  #undef reset_sbuf
  #undef put_to_sbuf

  return 0;
}

// Loading
int GRADIENT_GEN::load(string filename)
{
  // error handling macros
  #define check_errscan(x) ret=x;if(ret!=1){fclose(in);return-1;}
  #define check_errlstr(x) if(x<0){fclose(in);return-1;}

  FILE *in = NULL;
  int ret = 0;
  
  string sbuffer;
  string ca_text;

  // Clear old gradient first
  clear();

  // Open File for loading 
  in = fopen(filename.c_str(),"rb");
  if(!in) 
    return -1; // file error
  fseek(in, 0L, SEEK_SET);

  // Read Author and Title
  check_errlstr( readline_utf8(in,&AUTHOR) );
  check_errlstr( readline_utf8(in,&TITLE) );

  // Read suggested Width and Height
  check_errlstr( readline_utf8(in,&sbuffer) );
  check_errscan( sscanf(sbuffer.c_str(),"%i",&DEF_W) );
  check_errlstr( readline_utf8(in,&sbuffer) );
  check_errscan( sscanf(sbuffer.c_str(),"%i",&DEF_H) );

  // Read lines(attractor descriptions expected) until any error occurs
  while(readline_utf8(in,&sbuffer)==0)
    add_ca(sbuffer);

  // Close File
  fclose(in);

  #undef check_errscan
  #undef check_errlstr

  return 0;
}

// Export As Code
int GRADIENT_GEN::export_code(string instancename, string filename)
{
  // error handling macros
  #define check_errs(x) ret=x;if(ret<0){fclose(out);return-1;}

  #define reset_sbuf() delete sbuf; sbuf=NULL; sbuf=new(nothrow)ostringstream; \
                        if(sbuf==NULL){fclose(out);return-1;} \
                        (*sbuf).exceptions(ios_base::badbit|  \
                                           ios_base::failbit| \
                                           ios_base::eofbit); \
                        (*sbuf).setf(ios_base::dec,ios_base::basefield);

  #define put_to_sbuf(x) try{(*sbuf) << x; (*sbuf).clear(); } \
                          catch(ios_base::failure){              \
                          delete sbuf; fclose(out); return-1;}

  FILE *out = NULL;
  int i = 0;
  int ret = 0;
  ostringstream *sbuf = NULL;
  string nl = "\n";
  string ca_text;

  if(CA_VEC.empty())
    return -1; // nothing to export

  // Check 'instancename' for being a valid identifier
  if(instancename.empty())
    instancename.assign("gradient"); // default id for empty identifiers
  else // modify it to be a valid identifier
  // Note: This does not prevent the identifier from being a language keyword!
  //       it just replaces invalid chars with underscores.
  {
    unsigned int cpos=0;
    string valid_id_chars = "_abcdefghijklmnopqrstuvwxyz"
                            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                            "0123456789";
    string dec_digits = "0123456789";
    
    cpos = instancename.find_first_of(dec_digits,0);
    if(cpos == 0) // first char of identifier is a dec_digit?
    {
      instancename.insert(0,"_"); // front append an underscore
    }

    cpos = 0;
    while(1)
    {
      cpos = instancename.find_first_not_of(valid_id_chars,0);
      if (cpos == instancename.npos) // no more invalid chars?
        break;
      instancename.replace(cpos,1,"_");// replace invalid char with underscore
    }
  }
  // 'instancename' should now contain a valid C++ identifier

  // Open File for exporting code
  out = fopen(filename.c_str(),"w");
  if(!out) 
    return -1; // file error

  // Author and Title are ignored when exporting as code

  // Write intro comment and declaration
  reset_sbuf();
  put_to_sbuf("// START of exported gradient generator code\n");
  check_errs( fputs(sbuf->str().c_str(),out) );
  reset_sbuf();
  put_to_sbuf("GRADIENT_GEN " << instancename << ";\n");
  check_errs( fputs(sbuf->str().c_str(),out) );

  // Export default width and height as code lines
  reset_sbuf();
  put_to_sbuf(instancename << '.' << "set_w(" << DEF_W << ");\n");
  check_errs( fputs(sbuf->str().c_str(),out) );
  reset_sbuf();
  put_to_sbuf(instancename << '.' << "set_h(" << DEF_H << ");\n");
  check_errs( fputs(sbuf->str().c_str(),out) );

  // Export all attractors as code lines
  i=0;
  ca_text = get_as_text(i);
  while(ca_text.compare(string(no_attr_err))!=0)
  {
    reset_sbuf();
    put_to_sbuf(instancename << '.' << "add_ca(\"" 
                             << ca_text << "\");\n");
    check_errs( fputs(sbuf->str().c_str(),out) );
    i++;
    ca_text = get_as_text(i);
  }

  // Write outro comment
  reset_sbuf();
  put_to_sbuf("// END of exported gradient generator code\n");
  check_errs( fputs(sbuf->str().c_str(),out) );

  // Close File
  fclose(out);

  #undef check_errs
  #undef reset_sbuf
  #undef put_to_sbuf

  return 0;
}

#endif // #if !defined(__DB_gradient_gen_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
