/*--- Filename: "dycogrgen.hpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  Includes and Definitions for the Allegro-Dependant GUI
  to my gradient generator.

*/

#include "allegro.h"
#ifdef _WIN32
#include "winalleg.h"
#endif
#include "dcggdata.h"
#include "gradient_gen.hpp"

#if !defined(__DB_dycogrgen_HEAD_INCLUDED)
#define __DB_dycogrgen_HEAD_INCLUDED

/*
  NOTE:
  "one way street"style:
  ----------------------
  MOST functions in here are only working safely, if they are called AFTER
  the programme has succesfully initialized itself by "initialize()".
  And NONE of the functions in here may be called, if the
  programme deinitialized itself by "deinitialize()" or if "initialize()"
  failed.
  In other words: Ripping out some of these functions to use them in
  other programmes is mostly useless, because they might depend
  on specific types or global variables used by this specific application.
*/

// Runlevel IDs
enum runlevels
{
  RL_NOTHING_OK=0,
  RL_ALLEGRO_OK,
  RL_KEYBOARD_OK,
  RL_MOUSE_OK,
  RL_GFXMODE_OK,
  RL_BACKBUFFER_OK,
  RL_TEXT_OK,
  RL_DATA_OK,
  RL_SCROLL_TIMER_OK
};

// Main Menu Array Item Indices
enum MAIN_MENU_ID
{
  MAIN_MENU_ID_FILE=0,
  MAIN_MENU_ID_OPTIONS,
  MAIN_MENU_ID_HELP
};

// File Menu Array Item Indices
enum FILE_MENU_ID
{
  FILE_MENU_ID_NEW=0,
  FILE_MENU_ID_LOAD,
  FILE_MENU_ID_SAVE,
  FILE_MENU_ID_SAVE_AS,
  FILE_MENU_ID_EXPORT_BMP,
  FILE_MENU_ID_EXPORT_PCX,
  FILE_MENU_ID_EXPORT_TGA,
  FILE_MENU_ID_EXPORT_CODE,
  FILE_MENU_ID_EXIT
};

// Options Menu Array Item Indices
enum OPTIONS_MENU_ID
{
  OPTIONS_MENU_ID_SELECT_LANGUAGE_FILE=0,
  OPTIONS_MENU_ID_AUTO_RECALCULATE
};

// Help Menu Array Item Indices
enum HELP_MENU_ID
{
  HELP_MENU_ID_MANUAL=0,
  HELP_MENU_ID_ABOUT
};

// Identifiers for sub dialogs
enum SUB_DIALOG
{
  SUB_DIALOG_NONE = 0,
  SUB_DIALOG_MAIN,
  SUB_DIALOG_CHANGE_ALERT,
  SUB_DIALOG_EXPORT_BMP,
  SUB_DIALOG_EXPORT_PCX,
  SUB_DIALOG_EXPORT_TGA,
  SUB_DIALOG_EXPORT_CODE,
  SUB_DIALOG_MANUAL,
  SUB_DIALOG_ABOUT,
  SUB_DIALOG_SET_TYPE,
  SUB_DIALOG_SET_MODE,
  SUB_DIALOG_SET_COLOR,
  SUB_DIALOG_EDIT_COEFFICIENTS,
  SUB_DIALOG_SET_DISTANCE_MODE,
  SUB_DIALOG_SET_DIMENSIONS,
  
  SUB_DIALOG_NUM_DIALOGS
  /* SAVE, SAVE_AS, LOAD, LANGUAGE_LOAD are not treated as custom sub dialogs,
     they are called with the Allegro standard dialog "file_select_ex",
     which unfortunately can not easily be run on a backbuffer, the way it is
     implemented in the latest Allegro Version(4.20rc2) at the time of writing
     this.(it could be done with timed callback blits, but that would be 
     very hacky and would probably crash on some platforms) */
};

// Identifier for unsaved changes alert caller
enum USC_ALERT_CALLER
{
  USC_ALERT_CALLER_NONE = 0,
  USC_ALERT_CALLER_NEW,
  USC_ALERT_CALLER_LOAD,
  USC_ALERT_CALLER_EXIT,

  USC_ALERT_CALLER_NUM_CALLERS
};

#define SUB_DIALOG_ID_FIRST 0 // the first item in EVERY sub dialog should
                              // be large enough to contain all other objects
                              // to make some backbuffer HACKs work

// About Dialog Array Item Indices
enum ABOUT_DIALOG_ID
{
  ABOUT_DIALOG_ID_SBOX=0,
  ABOUT_DIALOG_ID_CAPTION, // "Dynamic Color Gradient Generator"
  ABOUT_DIALOG_ID_OK,
  ABOUT_DIALOG_ID_DB_LOGO,
  ABOUT_DIALOG_ID_AUTHOR_DESC,
  ABOUT_DIALOG_ID_AUTHOR_TEXT,
  ABOUT_DIALOG_ID_MAIL_TEXT,
  ABOUT_DIALOG_ID_HOME_TEXT,
  ABOUT_DIALOG_ID_TRANSLATION_BY_TEXT,
  ABOUT_DIALOG_ID_TRANSLATOR_TEXT,
  ABOUT_DIALOG_ID_POWERED_BY_TEXT,
  ABOUT_DIALOG_ID_ALLEGRO_HOME_TEXT,
  ABOUT_DIALOG_ID_ALLEGRO_LOGO,
  ABOUT_DIALOG_ID_STRETCHER,
  ABOUT_DIALOG_ID_YIELD
};

// Unsaved Changes Dialog Array Item Indices
enum UNSAVED_DIALOG_ID
{
  UNSAVED_DIALOG_ID_SBOX=0,
  UNSAVED_DIALOG_ID_CAPTION,
  UNSAVED_DIALOG_ID_LINE,
  UNSAVED_DIALOG_ID_SAVE_BUTTON,
  UNSAVED_DIALOG_ID_SAVE_AS_BUTTON,
  UNSAVED_DIALOG_ID_IGNORE_BUTTON,
  UNSAVED_DIALOG_ID_CANCEL_BUTTON,
  UNSAVED_DIALOG_ID_STRETCHER,
  UNSAVED_DIALOG_ID_YIELD
};

// Select Type Dialog Array Item Indices
enum TYPE_DIALOG_ID
{
  TYPE_DIALOG_ID_SBOX=0,
  TYPE_DIALOG_ID_OK,
  TYPE_DIALOG_ID_CANCEL,
  TYPE_DIALOG_ID_LIST,
  TYPE_DIALOG_ID_STRETCHER,
  TYPE_DIALOG_ID_YIELD
};
// callback function to fill the type list
char *fill_type_list(int index, int *list_size);

// Select Mode Dialog Array Item Indices
enum MODE_DIALOG_ID
{
  MODE_DIALOG_ID_SBOX=0,
  MODE_DIALOG_ID_OK,
  MODE_DIALOG_ID_CANCEL,
  MODE_DIALOG_ID_LIST,
  MODE_DIALOG_ID_STRETCHER,
  MODE_DIALOG_ID_YIELD
};
// callback function to the mode list
char *fill_mode_list(int index, int *list_size);

// Select Color Dialog Array Item Indices
enum COLOR_DIALOG_ID
{
  COLOR_DIALOG_ID_SBOX=0,
  COLOR_DIALOG_ID_OK,
  COLOR_DIALOG_ID_CANCEL,
  COLOR_DIALOG_ID_SLIDER_R,
  COLOR_DIALOG_ID_SLIDER_G,
  COLOR_DIALOG_ID_SLIDER_B,
  COLOR_DIALOG_ID_BOX,
  COLOR_DIALOG_ID_STRETCHER,
  COLOR_DIALOG_ID_YIELD
};
// callback function to change the bg color of the box
int color_change(void *dp3, int d2);

// Select Distance Mode Dialog Array Item Indices
enum DMODE_DIALOG_ID
{
  DMODE_DIALOG_ID_SBOX=0,
  DMODE_DIALOG_ID_OK,
  DMODE_DIALOG_ID_CANCEL,
  DMODE_DIALOG_ID_LIST,
  DMODE_DIALOG_ID_STRETCHER,
  DMODE_DIALOG_ID_YIELD
};
// callback function to fill the distance mode list
char *fill_dmode_list(int index, int *list_size);

// Edit Coefficient Dialog Array Item Indices
enum COEFFICIENTS_DIALOG_ID
{
  COEFFICIENTS_DIALOG_ID_SBOX=0,
  COEFFICIENTS_DIALOG_ID_CO_EDIT,
  COEFFICIENTS_DIALOG_ID_OK,
  COEFFICIENTS_DIALOG_ID_CANCEL,
  COEFFICIENTS_DIALOG_ID_LIST,
  COEFFICIENTS_DIALOG_ID_STRETCHER,
  COEFFICIENTS_DIALOG_ID_YIELD
};
// callback function to fill the coefficients list
char *fill_coefficient_list(int index, int *list_size);
// callback function to change the currently edited coefficient in the list
void coefficient_update_list();
// callback function to put the currently selected coefficient to the edit box
void coefficient_update_fedit(int index);

// Set Dimensions Dialog Array Item Indices
enum DIMENSIONS_DIALOG_ID
{
  DIMENSIONS_DIALOG_ID_SBOX=0,
  DIMENSIONS_DIALOG_ID_OK,
  DIMENSIONS_DIALOG_ID_CANCEL,
  DIMENSIONS_DIALOG_ID_W_DESC,
  DIMENSIONS_DIALOG_ID_W_EDIT,
  DIMENSIONS_DIALOG_ID_H_DESC,
  DIMENSIONS_DIALOG_ID_H_EDIT,
  DIMENSIONS_DIALOG_ID_STRETCHER,
  DIMENSIONS_DIALOG_ID_YIELD
};
// callback functions to prevent the edit fields from getting insane values
void dimensions_w_edit_clamp();
void dimensions_h_edit_clamp();

// Export Resolution Dialog Array Item Indices
enum EXP_DIMENSIONS_DIALOG_ID
{
  EXP_DIMENSIONS_DIALOG_ID_SBOX=0,
  EXP_DIMENSIONS_DIALOG_ID_OK,
  EXP_DIMENSIONS_DIALOG_ID_CANCEL,
  EXP_DIMENSIONS_DIALOG_ID_CAPTION,
  EXP_DIMENSIONS_DIALOG_ID_W_DESC,
  EXP_DIMENSIONS_DIALOG_ID_W_EDIT,
  EXP_DIMENSIONS_DIALOG_ID_H_DESC,
  EXP_DIMENSIONS_DIALOG_ID_H_EDIT,
  EXP_DIMENSIONS_DIALOG_ID_STRETCHER,
  EXP_DIMENSIONS_DIALOG_ID_YIELD
};
// callback functions to prevent the edit fields from getting insane values
void exp_dimensions_w_edit_clamp();
void exp_dimensions_h_edit_clamp();

// Export Code Dialog Array Item Indices
enum EXP_CODE_DIALOG_ID
{
  EXP_CODE_DIALOG_ID_SBOX=0,
  EXP_CODE_DIALOG_ID_OK,
  EXP_CODE_DIALOG_ID_CANCEL,
  EXP_CODE_DIALOG_ID_INSTANCENAME_DESC,
  EXP_CODE_DIALOG_ID_INSTANCENAME_EDIT,
  EXP_CODE_DIALOG_ID_STRETCHER,
  EXP_CODE_DIALOG_ID_YIELD
};

// Manual Dialog
enum MANUAL_DIALOG_ID
{
  MANUAL_DIALOG_ID_SBOX=0,
  MANUAL_DIALOG_ID_CAPTION,
  MANUAL_DIALOG_ID_OK,
  MANUAL_DIALOG_ID_TEXTBOX,
  MANUAL_DIALOG_ID_STRETCHER,
  MANUAL_DIALOG_ID_YIELD
};

// Main Dialog Array Item Indices
enum MAIN_DIALOG_ID
{
  MAIN_DIALOG_ID_CLEAR=0,
  MAIN_DIALOG_ID_WHITE_BAR,
  MAIN_DIALOG_ID_MENU_BAR,
  MAIN_DIALOG_ID_AUTHOR_DESC,
  MAIN_DIALOG_ID_AUTHOR_EDIT,
  MAIN_DIALOG_ID_TITLE_DESC,
  MAIN_DIALOG_ID_TITLE_EDIT,
  MAIN_DIALOG_ID_SETTINGS,
  MAIN_DIALOG_ID_TYPE_DESC,
  MAIN_DIALOG_ID_TYPE_BUTTON,
  MAIN_DIALOG_ID_MODE_DESC,
  MAIN_DIALOG_ID_MODE_BUTTON,
  MAIN_DIALOG_ID_COLOR_DESC,
  MAIN_DIALOG_ID_COLOR_BUTTON,
  MAIN_DIALOG_ID_INTENSITY_DESC,
  MAIN_DIALOG_ID_INTENSITY_EDIT,
  MAIN_DIALOG_ID_EXPONENT_DESC,
  MAIN_DIALOG_ID_EXPONENT_EDIT,
  MAIN_DIALOG_ID_X_DESC,
  MAIN_DIALOG_ID_X_EDIT,
  MAIN_DIALOG_ID_Y_DESC,
  MAIN_DIALOG_ID_Y_EDIT,
  MAIN_DIALOG_ID_WIDTH_DESC,
  MAIN_DIALOG_ID_WIDTH_EDIT,
  MAIN_DIALOG_ID_HEIGHT_DESC,
  MAIN_DIALOG_ID_HEIGHT_EDIT,
  MAIN_DIALOG_ID_RANGE_DESC,
  MAIN_DIALOG_ID_RANGE_EDIT,
  MAIN_DIALOG_ID_ANGLE_DESC,
  MAIN_DIALOG_ID_ANGLE_EDIT,
  MAIN_DIALOG_ID_HIGHEST_EXPONENT_DESC,
  MAIN_DIALOG_ID_HIGHEST_EXPONENT_EDIT,
  MAIN_DIALOG_ID_COEFFICIENTS_DESC,
  MAIN_DIALOG_ID_COEFFICIENTS_BUTTON,
  MAIN_DIALOG_ID_DISTANCE_MODE_DESC,
  MAIN_DIALOG_ID_DISTANCE_MODE_BUTTON,
  MAIN_DIALOG_ID_UNIT_DESC,
  MAIN_DIALOG_ID_UNIT_EDIT,
  MAIN_DIALOG_ID_ADD_BUTTON,
  MAIN_DIALOG_ID_DUPLICATE_BUTTON,
  MAIN_DIALOG_ID_DELETE_BUTTON,
  MAIN_DIALOG_ID_ATTRACTORS,
  MAIN_DIALOG_ID_ATTRACTORS_LIST,
  MAIN_DIALOG_ID_LIST_UP_BUTTON,
  MAIN_DIALOG_ID_LIST_MOVEUP_BUTTON,
  MAIN_DIALOG_ID_LIST_BAR,
  MAIN_DIALOG_ID_LIST_MOVEDOWN_BUTTON,
  MAIN_DIALOG_ID_LIST_DOWN_BUTTON,
  MAIN_DIALOG_ID_GRADIENT,
  MAIN_DIALOG_ID_BMP_UP_BUTTON,
  MAIN_DIALOG_ID_BMP_LEFT_BUTTON,
  MAIN_DIALOG_ID_GRADIENT_SCROLLBMP,
  MAIN_DIALOG_ID_BMP_RIGHT_BUTTON,
  MAIN_DIALOG_ID_DIMENSIONS_BUTTON,
  MAIN_DIALOG_ID_BMP_DOWN_BUTTON,
  MAIN_DIALOG_ID_RECALCULATE_BUTTON,
  MAIN_DIALOG_ID_YIELD
};

// String IDs
enum DBDCGG_TXT
{
  DBDCGG_TXT_ID_FILE=0,
  DBDCGG_TXT_ID_NEW,
  DBDCGG_TXT_ID_LOAD,
  DBDCGG_TXT_ID_SAVE,
  DBDCGG_TXT_ID_SAVE_AS,
  DBDCGG_TXT_ID_SELECT_GRADIENT_FILE,
  DBDCGG_TXT_ID_EXPORT_BMP,
  DBDCGG_TXT_ID_EXPORT_PCX,
  DBDCGG_TXT_ID_EXPORT_TGA,
  DBDCGG_TXT_ID_EXPORT_BITMAP_DIMENSIONS,
  DBDCGG_TXT_ID_SELECT_BITMAP_FILE,
  DBDCGG_TXT_ID_EXPORT_CODE,
  DBDCGG_TXT_ID_INSTANCENAME,
  DBDCGG_TXT_ID_SELECT_CPP_FILE,
  DBDCGG_TXT_ID_EXIT,
  DBDCGG_TXT_ID_OPTIONS,
  DBDCGG_TXT_ID_SELECT_LANGUAGE_FILE,
  DBDCGG_TXT_ID_AUTO_RECALCULATE,
  DBDCGG_TXT_ID_HELP,
  DBDCGG_TXT_ID_MANUAL,
  DBDCGG_TXT_ID_ABOUT,
  DBDCGG_TXT_ID_POWERED_BY,
  DBDCGG_TXT_ID_TRANSLATED_BY, // translated to "language" by:
  DBDCGG_TXT_ID_TRANSLATION_AUTHOR, // name of the nice person who translated the strings
  DBDCGG_TXT_ID_OK,
  DBDCGG_TXT_ID_CANCEL,
  DBDCGG_TXT_ID_IGNORE,
  DBDCGG_TXT_ID_YES,
  DBDCGG_TXT_ID_NO,
  DBDCGG_TXT_ID_AUTHOR,
  DBDCGG_TXT_ID_TITLE,
  DBDCGG_TXT_ID_SETTINGS_ATTRACTOR,
  DBDCGG_TXT_ID_TYPE,
  DBDCGG_TXT_ID_MODE,
  DBDCGG_TXT_ID_COLOR,
  DBDCGG_TXT_ID_INTENSITY,
  DBDCGG_TXT_ID_EXPONENT,
  DBDCGG_TXT_ID_X,
  DBDCGG_TXT_ID_Y,
  DBDCGG_TXT_ID_WIDTH,
  DBDCGG_TXT_ID_RADIUS,
  DBDCGG_TXT_ID_HEIGHT,
  DBDCGG_TXT_ID_RANGE,
  DBDCGG_TXT_ID_ANGLE,
  DBDCGG_TXT_ID_HIGHEST_EXPONENT,
  DBDCGG_TXT_ID_COEFFICIENTS,
  DBDCGG_TXT_ID_EDIT,
  DBDCGG_TXT_ID_DISTANCE_MODE,
  DBDCGG_TXT_ID_UNIT,
  DBDCGG_TXT_ID_ADD,
  DBDCGG_TXT_ID_DUPLICATE,
  DBDCGG_TXT_ID_DELETE,
  DBDCGG_TXT_ID_ATTRACTORS,
  DBDCGG_TXT_ID_GRADIENT,
  DBDCGG_TXT_ID_SET_DIMENSIONS,
  DBDCGG_TXT_ID_RECALCULATE,
  DBDCGG_TXT_ID_CALCULATING,
  DBDCGG_TXT_ID_PIXELS_TO_PROCESS,
  DBDCGG_TXT_ID_LOOKUP_TABLE_FOR,
  DBDCGG_TXT_ID_VALUES_TO_GO,

  DBDCGG_TXT_ID_GLOBAL,
  DBDCGG_TXT_ID_POINT,
  DBDCGG_TXT_ID_POINTE,
  DBDCGG_TXT_ID_HBAR,
  DBDCGG_TXT_ID_VBAR,
  DBDCGG_TXT_ID_BAR,
  DBDCGG_TXT_ID_POLYNOMIAL,
  DBDCGG_TXT_ID_CIRCLE,
  DBDCGG_TXT_ID_CIRCLEF,
  DBDCGG_TXT_ID_ELLIPSE,

  DBDCGG_TXT_ID_ATTRACT,
  DBDCGG_TXT_ID_ABSORB,

  DBDCGG_TXT_ID_TO_FUNCVALUE,
  DBDCGG_TXT_ID_TO_CURVE,
  
  DBDCGG_TXT_ID_ERROR,
  DBDCGG_TXT_ID_ESC_OR_CLICK,
  DBDCGG_TXT_ID_LOAD_FAILED,
  DBDCGG_TXT_ID_SAVE_FAILED,
  DBDCGG_TXT_ID_EXPORT_FAILED,
  DBDCGG_TXT_ID_FILE_ERROR,
  DBDCGG_TXT_ID_MEMORY_ERROR,
  DBDCGG_TXT_ID_EXTENSION_ERROR,
  DBDCGG_TXT_ID_WARNING,
  DBDCGG_TXT_ID_UNSAVED_CHANGES,
  DBDCGG_TXT_ID_QUESTION,
  DBDCGG_TXT_ID_OVERWRITE_QUESTION,

  DBDCGG_NUM_STRINGS
};

// Type to hold all text data
typedef char* pchar;
struct dbdcgg_strings
{
  pchar s[DBDCGG_NUM_STRINGS];
};

// File Extensions Array Indices
enum F_EXT
{
  F_EXT_BMP = 0,
  F_EXT_PCX,
  F_EXT_TGA,
  F_EXT_DCGG,
  F_EXT_DCGG_LANG,
  F_EXT_CPP,

  F_EXT_NUM_EXT
};

// Type to hold all extension data
struct dbdcgg_ext
{
  pchar s[F_EXT_NUM_EXT];
};

// Type to hold GUI edit contents
#define G_MAX_BYTE_PER_CHAR 4//(according to Allegro"/src/unicode.c"(utypes[])
#define G_MAX_TEXT_CHARS 41 // maximum characters per text edit field
#define G_MAX_NUMBER_CHARS 21 // maximum characters per number edit field
#define G_CO_SUF_COUNT 10 // maximum length of the suffix in coefficient list
#define G_MAX_COEFFICIENTS 32 // so maximum highest exponent is 31
struct edit_contents
{
  char author[G_MAX_TEXT_CHARS*G_MAX_BYTE_PER_CHAR];
  char title[G_MAX_TEXT_CHARS*G_MAX_BYTE_PER_CHAR];
  char intensity[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char exponent[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char x[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char y[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char width[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char height[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char range[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char angle[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char highest_exponent[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  char unit[G_MAX_NUMBER_CHARS*G_MAX_BYTE_PER_CHAR];
  // for sub dialogs
  char current_coeff[G_MAX_BYTE_PER_CHAR*G_MAX_NUMBER_CHARS];
  char current_pixw[G_MAX_BYTE_PER_CHAR*G_MAX_NUMBER_CHARS];
  char current_pixh[G_MAX_BYTE_PER_CHAR*G_MAX_NUMBER_CHARS];
  char current_expixw[G_MAX_BYTE_PER_CHAR*G_MAX_NUMBER_CHARS];
  char current_expixh[G_MAX_BYTE_PER_CHAR*G_MAX_NUMBER_CHARS];
  char coefficients[G_MAX_COEFFICIENTS]
                   [G_MAX_BYTE_PER_CHAR*(G_MAX_NUMBER_CHARS+G_CO_SUF_COUNT)];
  char exc_instancename[G_MAX_TEXT_CHARS*G_MAX_BYTE_PER_CHAR];
};

// Dialog/Menu Items Control Functions
// for custom GUI objects
int d_stretch_me_proc(int msg, DIALOG *obj, int c);
int d_cedit_proc(int msg, DIALOG *obj, int c);
int d_clist_proc(int msg, DIALOG *obj, int c);
int d_cbutton_proc(int msg, DIALOG *obj, int c);
int d_cstretch_move_nextx_button_proc(int msg, DIALOG *obj, int c);
int d_cstretch_move_nexty_button_proc(int msg, DIALOG *obj, int c);
int d_cstretch_move_nextxy_button_proc(int msg, DIALOG *obj, int c);
int d_slider_move_nexty_proc(int msg, DIALOG *obj, int c);
int d_stretch_move_nextx_text_proc(int msg, DIALOG *obj, int c);
int d_stretch_move_nextx_aligny_text_proc(int msg, DIALOG *obj, int c);
int d_stretch_move_nexty_text_proc(int msg, DIALOG *obj, int c);
int d_stretch_move_nextxy_text_proc(int msg, DIALOG *obj, int c);
int d_bitmap_ralign_nextxy_proc(int msg, DIALOG *obj, int c);
int d_iedit_proc(int msg, DIALOG *obj, int c);
int d_fedit_proc(int msg, DIALOG *obj, int c);
int d_c_clear_proc(int msg, DIALOG *obj, int c);
bool set_position_by_click(int cx, int cy, int current_ca);
bool set_range_by_rclick(int cx, int cy, int current_ca);
bool set_dim_by_mclick(int cx, int cy, int current_ca);
bool set_angle_by_a_key(int cx, int cy, int current_ca);
int d_scroll_bmp_proc(int msg, DIALOG *obj, int c);
int d_message_button_proc(int msg, DIALOG *obj, int c);
int d_attractor_list_proc(int msg, DIALOG *obj, int c);

// for File Menu Entries
int file_new();
int file_load();
int file_save();
int file_save_as();
int file_export_bmp();
int file_export_pcx();
int file_export_tga();
int file_export_code();
int file_exit();

// for Options Menu Entries
int options_select_language();
int options_auto_recalculate();

// for certain Button presses
//( *** to do)
void button_add();
void button_duplicate();
void button_delete();
int button_recalculate();
bool lookup_callback(int values_to_go, CA_TYPE type, int n);
bool render_callback(unsigned long pixels_to_go);

// for Help Manu Entries
int help_manual();
int help_about();

// Functions for managing Interface<->GradientGenerator (d)ata(t)ransfer
// get_ prefix means data is transferred FROM gradient generator TO interface
// set_ prefix means data is transferred TO gradient generator FROM interface
// (they are all void, because they read their inputs/outputs directly from
//  the edit fields or the gradient generator)
// The get_ functions also update visibility of edit fields, according to
// selected attractors type.
void draw_place_holder(BITMAP *where);
int reset_preview_bitmap();
int reset_edit_fields();
void dt_get_all(); // calls all the "get_"s to update the interface
// note: there is no set_all, 
void dt_apply_changes(int ch_which);
// these are used in the "SUB_DIALOG_MAIN"
void dt_get_author();
void dt_set_author();
void dt_get_title();
void dt_set_title();
void dt_get_type();
void dt_set_type();
void dt_get_mode();
void dt_set_mode();
void dt_get_color();
void dt_set_color();
void dt_get_intensity();
void dt_set_intensity();
void dt_get_exponent();
void dt_set_exponent();
void dt_get_x();
void dt_set_x();
void dt_get_y();
void dt_set_y();
void dt_get_width(); // "radius" for CA_CIRCLE and CA_CIRCLEF
void dt_set_width();
void dt_get_height();
void dt_set_height();
void dt_get_range();
void dt_set_range();
void dt_get_angle();
void dt_set_angle();
void dt_get_highest_exp();
void dt_set_highest_exp();
void dt_get_coefficients();
void dt_set_coefficients();
void dt_get_distance_mode();
void dt_set_distance_mode();
void dt_get_unit();
void dt_set_unit();
// these are used in the "SUB_DIALOG_SET_DIMENSIONS"
void dt_get_dimensions();
void dt_set_dimensions();

// Graphical Error Message (one line, one button, one key shortcut)
void show_error_message(char *ascii_msg, char *ascii_btn,int key);
//(one caption, three lines, one key to press, (always shows on screen))
void show_red_alert(char *uc_caption, char *uc1, char *uc2, char *uc3,
                    int key_code);
//(same as above but strings are always in ASCII(not Unicode))
void ascii_red_alert(char *uc_caption, char *uc1, char *uc2, char *uc3,
                     int key_code);

// Functions to init, deinit the programme
void scroll_gui_callback();
void close_button_callback();
void zero_edit_contents(edit_contents *which);
int mode_filter(int card, int w, int h, int c_depth);
void change_ok_cancel();
//void position_mouse_override(int x,int y); // these two were removed, because they rely on a portion
//void update_abs_mousepos(); // of Allegro that isn't truly system independent
int    initialize();
void deinitialize();

// Function to load all text data from a UTF-8 textfile
int reload_strings(string filename, dbdcgg_strings *out_table);
// same but just converts from set constants to current unicode format
int convert_exts(dbdcgg_ext *out_table);
// Function to bind the loaded strings to dialog/menu items
void attach_all_strings();
// Called at programme termination to free occupied memory
void kill_strings(dbdcgg_strings *out_table);
void kill_ext(dbdcgg_ext *out_table);
// Helper function to set all char pointers in the string tables to NULL
void null_strings(dbdcgg_strings *out_table);
void null_ext(dbdcgg_ext *out_table);

// Helper Functions to report a correct GUI position
int gui_mouse_x_offset();
int gui_mouse_y_offset();
int gui_mouse_x_std();
int gui_mouse_y_std();

// Set GUI behaviour back to standard
void gui_behave_std();
// Set GUI behaviour to playback on backbuffer
void gui_behave_custom();

// Runs any dialog on my backbuffer(with scrolling supported)
// (returns the object index that caused closing)
int run_backbuffer_dialog(DIALOG *which,int focus);

//#include "dialog_defs.inc"

#endif // #if !defined(__DB_dycogrgen_HEAD_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
