/*--- Filename: "ca_circle.cpp" ---

  --- Projectname: DB's Dynamic Color Gradient Generator ---
  (Targetsystem: Crossplatform)
  
  Author: Dennis Busch (http://www.dennisbusch.de)

  Content:
  The circle color attractor.
  
*/

#include "ca_circle.hpp"

#if !defined(__DB_ca_circle_BODY_INCLUDED)
#define __DB_ca_circle_BODY_INCLUDED

// default ctor just sets default values
CA_CIRCLE::CA_CIRCLE()
{
  RADIUS = 0.5;
  TYPE = CAT_CIRCLE; // remember to override in derived types
}

// other ctor sets user values
CA_CIRCLE::CA_CIRCLE(CA_MODE mode, int r, int g, int b, 
                     double colint, double exp,
                     double x, double y, double radius, double range)
{
  CA_POINT::CA_POINT(mode,r,g,b,colint,exp,x,y,range);
  RADIUS = radius;
  TYPE = CAT_CIRCLE; // remember to override in derived types
}

CA_CIRCLE::~CA_CIRCLE()
{
  CA_GLOBAL::delete_line_buf();
}

// Set and Get Methods

//
void CA_CIRCLE::set_radius(double radius)
{
  RADIUS = radius;
  _render_var_changed = true;
}

//
double CA_CIRCLE::get_radius()
{
  return RADIUS;
}

// calculate distance to (x,y) in pixels
// (is protected virtual)
double CA_CIRCLE::distance(int x, int y)
{
  return fabs(CA_POINT::distance(x,y) - _pix_r);
}

// called by the GRADIENT_GEN prior to rendering
// (is virtual)
bool CA_CIRCLE::update_absolutes(int width, int height,
                                 lut_callback lcall, double lcall_skip, int n)
{
  if(!dim_changed(width,height)) return true;

  _render_var_changed = true; // following line would do nothing, if false
  CA_POINT::update_absolutes(width,height,lcall,lcall_skip,n);
  _pix_r = RADIUS * width;

  return true;
}

/* continues setting the values from tokenizing the human readable string
   returns 0 on success, -1 on failure */
// (is protected virtual)
int CA_CIRCLE::update_vars_from_line()
{
  int ret = CA_GLOBAL::update_vars_from_line();//read from base type 1st
  if(ret!=0) return ret;

  TOKEN = strtok(NULL,token_seperators); // EXP double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&EXP);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // X double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&X);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // Y double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&Y);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  TOKEN = strtok(NULL,token_seperators); // RADIUS double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RADIUS);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;
  
  TOKEN = strtok(NULL,token_seperators); // RANGE double expected
  if(TOKEN) 
  {
    ret = sscanf(TOKEN,"%lf",&RANGE);
    if((ret==0)||(ret==EOF)) return -1;
  }
  else return -1;

  _render_var_changed = true; 

  return 0;
}

// return the variables(except TYPE) in a human readable string
// (is virtual)
string CA_CIRCLE::get_as_text()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(8);
    rline << ", " << EXP 
          << ", " << X 
          << ", " << Y 
          << ", " << RADIUS 
          << ", " << RANGE;
    rline.clear();
    return CA_GLOBAL::get_as_text() + rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

/* Helper function for GUI building that returns some very sparse info
   meant to be displayed in a list*/
// (is virtual)
string CA_CIRCLE::gui_quick_info()
{
  ostringstream rline;
  rline.exceptions(ios_base::badbit|ios_base::failbit|ios_base::eofbit);
  try
  {
    rline.setf(ios_base::dec,ios_base::basefield);
    rline.setf(ios_base::fixed,ios_base::floatfield);
    rline.precision(2);
    rline << "i " << COLINT
          << ",e " << EXP
          << ",p(" << X << "," << Y << ")";
    rline.clear();
    return rline.str();
  }
  catch(ios_base::failure)
  {
    return string("stream error");
  }
}

#endif // #if !defined(__DB_ca_circle_BODY_INCLUDED)

/*
  Preserving the possibilty to make nicely formatted printouts
  (Format: "Portrait"), the code should be normed to a width of 78 chars.
123456789012345678901234567890123456789012345678901234567890123456789012345678
---------10--------20--------30--------40--------50--------60--------70-----78
*/
