/*  This file is part of the program:

    nV@derz! (nvaderz)
    Copyright (C) 2006-2013  Eric Pellegrin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    Eric Pellegrin can be reached by email at pellea72(at)yahoo(dot)com
    (Note: email address modified to evade unsolicited email search bots.
        Change (at) to @ and (dot) to . to read name@company.com )
*/

#include "nvaderz.h"
#include "highscores.h"
#include "bonus.h"

/* enable reading from a relative Resource directory path */
char bitmapPath[256];
int setBitmapPath(char *filename); /* pass just the file name */
int setupSound(void);

void setdefaults(void)
{
    screen_mode = mode; /* mode was either set by command-line argument
				or set to macro MODE */
    screen_width = WIDTH;
    screen_height = HEIGHT;
    screen_baseline = BASELINE;
    screen_topline = TOPLINE;

    /* Currently the game supports 1 or 2 players
    ** which can be chosen in the Options section of
    ** the game menu.
    **
    ** main() sets numberOfPlayers to -1 so that we
    ** know to set an initial value here before the
    ** player sees the game menu for the first time.
    */
    if(numberOfPlayers < 0)
	    numberOfPlayers = 1;

    player_x_speed = 6;
    player_height = PLAYER_H;
    player_width = PLAYER_W;
    player_space_top = PLAYER_SPACE_H;
    player_topline = PLAYER_LINE;
    player_bounds_offsetx1 = 2;
    player_bounds_offsetx2 = -2;
    player_bounds_offsety1 = 2;
    player_bounds_offsety2 = -4;

    /* set hud defaults */
    score_digit_width = SCORE_DIGIT_WIDTH;
    score_digit_height = SCORE_DIGIT_HEIGHT;
    score_digit_hspace = SCORE_DIGIT_HSPACE;
    score_left_limit = SCORE_LEFT_LIMIT;
    score_right_limit = SCORE_RIGHT_LIMIT;
    score_topline = SCORE_TOPLINE;
    num_score_digits = NUM_SCORE_DIGITS;

    lives_unit_width = LIVES_UNIT_WIDTH;
    lives_unit_height = LIVES_UNIT_HEIGHT;
    lives_unit_hspace = LIVES_UNIT_HSPACE;
    lives_left_limit = LIVES_LEFT_LIMIT;
    lives_right_limit = LIVES_RIGHT_LIMIT;
    lives_topline = LIVES_TOPLINE;
    num_lives_units = NUM_LIVES_UNITS;
    lives_ship1_x = LIVES_SHIP1_X;
    lives_ship2_x = LIVES_SHIP2_X;
    lives_ship_y = LIVES_SHIP_Y;
    lives_ship_width = LIVES_SHIP_WIDTH;
    lives_ship_height = LIVES_SHIP_HEIGHT;

    /* game area boundaries */
    left_boundary = LEFT_BOUNDARY;
    right_boundary = RIGHT_BOUNDARY;

    invader_height = INVADER_H;
    invader_width = INVADER_W;
    invader_space_horizontal = INV_SPACE_W;
    invader_space_vertical = INV_SPACE_H;
    invader_columns = INV_COLUMNS;
    invader_rows = INV_ROWS;
    invader_number = NUMBER_OF_INVADERS;

    horde_x_speed = 1;
    horde_start_y = HORDESTART_Y;

    double_invader_start_level = DOUBLE_INVADER_START_LEVEL;
    triple_invader_start_level = TRIPLE_INVADER_START_LEVEL;
    double_invader_odds = DOUBLE_INVADER_ODDS;
    triple_invader_odds = TRIPLE_INVADER_ODDS;
    double_invader_odds_change = DOUBLE_INVADER_ODDS_CHANGE;
    triple_invader_odds_change = TRIPLE_INVADER_ODDS_CHANGE;

    laser_y_speed = -2;
    laser_height = LASER_H;
    laser_width = LASER_W;

    bomb_y_speed = 2;
    bomb_height = BOMB_H;
    bomb_width = BOMB_W; 
    dropFactor = 1000;
    dropOdds = 10;

    /* sound defaults */
    pan = 128;
    pitch = 1000;
    volume = 128;

    /* shield defaults */
    shield_x = 0;
    shield_y = 330;
    shield_width = 640;
    shield_height = 30;
    shieldSegmentWidth = 10;
    shieldSegmentHeight = 5;
    shieldSpace = 5;

    extraLifePoints = 5000;

    /* set whether we want joystick support */
    joysticksAllowed = 1;

}

void setup(void)
{
	int i, j;
	
	srand(time(NULL));

        /* set up screen buffer */
        screen_buffer = create_bitmap(SCREEN_W, SCREEN_H);


	/* display splash screen at start of game */
	if(splashFlag)
		displaysplash(1);

	setdefaults();

	if(setupSound())
		soundon = 0;
	else
		soundon = 1;

	/* set up the capsules */
	setupBonuses();

	/* load the capsule bitmap and create an array
	 * of 16 rotated capsule bitmaps
	 */
	setBitmapPath("GameObject/CapsuleIcon.bmp");
	if(capsule_bmp[0] == NULL)
	{
	    capsule_bmp[0] = load_bitmap(bitmapPath, NULL);

	    for(i = 1; i < 16; i++)
	    {
	      capsule_bmp[i] = create_bitmap(capsule_bmp[0]->w,
	        capsule_bmp[0]->h);

	      /* clear with a transparent background */
	      clear_to_color(capsule_bmp[i], makecol(255, 0, 255));

	      /* rotate the image based on Allegro's 256 degrees */
	      rotate_sprite(capsule_bmp[i], capsule_bmp[0],
	        0, 0, itofix(i * 16) );
	    }
	
	}

	/* load the missile bitmap and rotate it 90 degrees to the
	 * right, so that it is in standard form for trigonometric
	 * functions
	 */
	if(missile_bmp == NULL)
	{
		BITMAP *temp_bmp;
		/* setBitmapPath("GameObject/MissileIcon.bmp"); */
		setBitmapPath("GameObject/Missile-New-Blackground.bmp");
		temp_bmp = load_bitmap(bitmapPath, NULL);
		if(temp_bmp == NULL)
		{
			printf("Error: Couldn't load %s\n", bitmapPath);
			exit(1);
		}
		else
		{
			missile_bmp = create_bitmap(temp_bmp->w,
				temp_bmp->h);

			clear_to_color(missile_bmp, makecol(255,0,255));
			
			rotate_sprite(missile_bmp, temp_bmp, 0, 0,
				itofix(64));

			destroy_bitmap(temp_bmp);
		}

		/* initialize the rotated bitmap */
		rotatedMissile_bmp = create_bitmap(missile_bmp->w,
			missile_bmp->h);
	}

	/* default to level 1, but settings file could change this */
	level = 1;

	setUpHud();

	/* set up high scores */
	getScoresFromFile();

	/** get settings from settings.file if file exists,
	  * else just use defaults and print message to console
	  */
	if(getfilesettings())
	    printf("Using default game settings.\n");

	/* relative settings variables may need to be reset */
	player_topline = screen_baseline - player_height - player_space_top;

	/* invader_number won't be modified until */
	/* related arrays can be set dynamically */
	/*                                       */
	/* invader_number = invader_columns * invader_rows; */

	if(horde_start_y < screen_topline + invader_space_vertical)
	    horde_start_y = screen_topline + invader_space_vertical;
	
	/* use PlayerIcon1.bmp (player 1) for player bitmap */
	setBitmapPath("GameObject/PlayerIcon1.bmp");
	if(player_bmp[0] == NULL)
	    player_bmp[0] = load_bitmap(bitmapPath, NULL);

	/* use PlayerIcon2.bmp (player 2) for player bitmap */
	setBitmapPath("GameObject/PlayerIcon2.bmp");
	if(player_bmp[1] == NULL)
	    player_bmp[1] = load_bitmap(bitmapPath, NULL);

	/* set up the player sprites */
	for(i = 0; i < 2; i++) {
		/* player = &theplayer; */
		player[i] = &theplayer[i];
		player[i]->width = player_width;
		player[i]->height = player_height;
		player[i]->alive = 0;
		player[i]->moving = 0;

		player[i]->blinking = 0;
		player[i]->visible = 0;
		player[i]->blinkDelay = 5;

		/* set direction to 0 'up' for now... 1 is 'right', 3 is 'left' */
		player[i]->dir = 0;
		player[i]->xspeed = player_x_speed;

		/* place the two players on the screen by thirds horizontally */
		player[i]->x = (screen_width / 3 - PLAYER_W / 2) * (i + 1);
		player[i]->y = player_topline;

		/* set up player sprite bounds */
		player[i]->bounds.x1 = player[i]->x + player_bounds_offsetx1;
		player[i]->bounds.x2 = player[i]->x + player[i]->width
			+ player_bounds_offsetx2;
		player[i]->bounds.y1 = player[i]->y + player_bounds_offsety1;
		player[i]->bounds.y2 = player[i]->y + player[i]->height
			+ player_bounds_offsety2;
	}

	/* support loading of multiple invader bitmaps */
	setBitmapPath("GameObject/BladesIcon.bmp");
	if(invaders_bmp[0][0] == NULL)
	    invaders_bmp[0][0] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/BladesIcon2.bmp");
	if(invaders_bmp[1][0] == NULL)
	    invaders_bmp[1][0] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/BladesIcon3.bmp");
	if(invaders_bmp[2][0] == NULL)
            invaders_bmp[2][0] = load_bitmap(bitmapPath, NULL);
	setBitmapPath("GameObject/StalksIcon.bmp");
	if(invaders_bmp[0][1] == NULL)
	    invaders_bmp[0][1] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/StalksIcon2.bmp");
        if(invaders_bmp[1][1] == NULL)
            invaders_bmp[1][1] = load_bitmap(bitmapPath, NULL);
	setBitmapPath("GameObject/StalksIcon3.bmp");
	if(invaders_bmp[2][1] == NULL)
            invaders_bmp[2][1] = load_bitmap(bitmapPath, NULL);
	setBitmapPath("GameObject/PudgyIcon.bmp");
	if(invaders_bmp[0][2] == NULL)
	    invaders_bmp[0][2] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/PudgyIcon2.bmp");
        if(invaders_bmp[1][2] == NULL)
            invaders_bmp[1][2] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/PudgyIcon3.bmp");
        if(invaders_bmp[2][2] == NULL)
            invaders_bmp[2][2] = load_bitmap(bitmapPath, NULL);
	setBitmapPath("GameObject/CurlyIcon.bmp");
	if(invaders_bmp[0][3] == NULL)
	    invaders_bmp[0][3] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/CurlyIcon2.bmp");
        if(invaders_bmp[1][3] == NULL)
            invaders_bmp[1][3] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/CurlyIcon3.bmp");
        if(invaders_bmp[2][3] == NULL)
            invaders_bmp[2][3] = load_bitmap(bitmapPath, NULL);
	setBitmapPath("GameObject/BigBoyIcon.bmp");
	if(invaders_bmp[0][4] == NULL)
	    invaders_bmp[0][4] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/BigBoyIcon2.bmp");
        if(invaders_bmp[1][4] == NULL)
            invaders_bmp[1][4] = load_bitmap(bitmapPath, NULL);
        setBitmapPath("GameObject/BigBoyIcon3.bmp");
        if(invaders_bmp[2][4] == NULL)
            invaders_bmp[2][4] = load_bitmap(bitmapPath, NULL);


	/* set up the horde sprite */
	horde = &thehorde;
	horde->y = horde_start_y;
	horde->width = invader_columns * invader_width + (invader_columns - 1) * invader_space_horizontal;
	horde->height = invader_rows * invader_height + (invader_rows - 1) * invader_space_vertical;
	horde->x = screen_width / 2 - horde->width / 2;
	horde->alive = 1;
	horde->dir = 1; /* moving right */
	horde->xspeed = horde_x_speed;

	/* fill the invaders sprite array */
	for(i = 0; i < invader_rows; i++)
		for(j = 0; j < invader_columns; j++)
		{
			/* set sprite values based on horde sprite values */
			invaders[i][j].alive = calcInvaderLife();
			invaders[i][j].x = horde->x + (j * invader_width) +
				(j * invader_space_horizontal);
			invaders[i][j].y = horde->y + (i * invader_height) +
				(i * invader_space_vertical);
			invaders[i][j].width = invader_width;
			invaders[i][j].height = invader_height;

			/* define the invader's bounds */
			invaders[i][j].bounds.x1 = invaders[i][j].x;
			invaders[i][j].bounds.x2 = invaders[i][j].x
				+ invaders[i][j].width;
			invaders[i][j].bounds.y1 = invaders[i][j].y;
			invaders[i][j].bounds.y2 = invaders[i][j].y
				+ invaders[i][j].height;
	       	} 
	
	/* load the laser bitmap */
	setBitmapPath("GameObject/LaserIcon.bmp");
	if(laser_bmp == NULL)
	    laser_bmp = load_bitmap(bitmapPath, NULL);

	/* set up the laser sprites */
	for(i = 0; i < 2; i++)
		for(j = 0; j < 3; j++)
		{
		  laser[i][j] = &thelaser[i][j];
		  laser[i][j]->alive = 0; /* initially turned off */
		  laser[i][j]->width = laser_width;
		  laser[i][j]->height = laser_height;
		  laser[i][j]->xspeed = 0;
		  laser[i][j]->yspeed = laser_y_speed;
		}

	/** setting up the laser bounds here is pointless
	 ** since it changes on movement
	 */

	/* load the bomb bitmap */
	setBitmapPath("GameObject/BombIcon.bmp");
	if(bomb_bmp == NULL)
	    bomb_bmp = load_bitmap(bitmapPath, NULL);

	/* set up bombs sprite array */
	for(i = 0; i < invader_columns; i++)
	{
		bombs[i].alive = 0;
		bombs[i].yspeed = bomb_y_speed;
		bombs[i].width = bomb_width;
		bombs[i].height = bomb_height;

		/** setting up bombs bounds here is pointless
		 ** since it changes when the horde moves
		 */
	}

	/* set up the shield */
	shieldTicks = 0;
	shield = &theshield;
	shield->x = shield_x;
	shield->y = shield_y;
	shield->width = shield_width;
	shield->height = shield_height;
	shield->alive = -1;
	shield_bmp = create_bitmap(shieldSegmentWidth,
		shieldSegmentHeight);
	for(i = 0; i < 3; i++)
		for(j = 0; j < (int)(shield_width / shieldSegmentWidth); j++)
		{
			shieldArray[i][j] = 1;
		}
	if(shield_bmp == NULL)
	{
		printf("Error: Could not create shield_bmp.");
		exit(1);
	}
	rectfill(shield_bmp, 0, 0, shieldSegmentWidth - 1,
		shieldSegmentHeight - 1, GREEN);

	/* set up shield bounds */
	shield->bounds.x1 = shield->x;
	shield->bounds.x2 = shield->x + shield->width;
	shield->bounds.y1 = shield->y;
	shield->bounds.y2 = shield->y + shield->height;

	/* call displaysplash again, but pass 0 to loadingFlag */
	if(splashFlag)
		displaysplash(0);

	/* set up game for one or two players */
	gameover = 0;

	for(i = 0; i < 2; i++)
		score[i] = 0;
	for(i = 0; i < 2; i++)
		lives[i] = -1;
	for(i = 0; i < 2; i++)
		for(j = 0; j < 3; j++)
			firing[i][j] = 0;
	for(i = 0; i < numberOfPlayers; i++) {
		lives[i] = 2;
		player[i]->alive = 1;
		player[i]->visible = 1;
	}
	totalinvaders = invader_columns * invader_rows;
	setPause(0);

	updatehorde();

}

void cleanup(void)
{
	remove_sound();
	allegro_exit();
}

void applyflags(int args, char *strargs[])
{
/** This is commented out pending decision of what to allow
  * for command-line arguments.

	int i;

	for(i = 1; i < args; i++)
	{
		if(!strcmp(strargs[i], "-pxspeed"))
		{
			if(args > i+1)
			{
				player->xspeed = atoi(strargs[i+1]);
			}
		}	
		else if(!strcmp(strargs[i], "-hxspeed"))
		{
			if(args > i+1)
			{
				horde->xspeed = atoi(strargs[i+1]);
				updatehorde();
			}
		}	
	}	
	return;
*/
}

void initializegame(void)
{
	allegro_init();
	install_keyboard();
	install_joystick(JOY_TYPE_AUTODETECT);

	/* set up timer */
	gameSpeedInMilliseconds = GAME_SPEED_IN_MILLISECONDS;
	setUpTimer();

	/* set screen mode to mode set in main() */
	screen_mode = mode;

	set_color_depth(16);
	if(set_gfx_mode(screen_mode, WIDTH, HEIGHT, 0, 0))
	{
		allegro_message(allegro_error);
		cleanup();
		exit(1);
	}

	/* set up the sound system, if possible */
	if(install_sound(DIGI_AUTODETECT, MIDI_NONE, ""))
	{
		puts("======================================");
		puts("Unable to initialize the sound system.");
		soundon = 0;
		/* print error message from allegro */
		allegro_message(allegro_error);
		puts("======================================\n");
	}
	else
		soundon = 1;
}

int setBitmapPath(char *filename) /* pass just the file name */
{
    strcpy(bitmapPath, resourceDirectory);
    strcat(bitmapPath, filename);

    return 0; /* to do: return another value if there was a problem */
}

int calcInvaderLife(void)
{
	/** return the value of the starting life of an invader
	  * based on the odds of whether it will be
	  * a 2- or 3-kill Super Invader, or whether it
	  * will just be a normal 1-kill invader.
	  *
	  */
	if(level >= double_invader_start_level &&
		(unsigned)(rand() % 100) <= double_invader_odds)
	{
		if(level >= triple_invader_start_level &&
			(unsigned)(rand() % 100) <= triple_invader_odds)
				return 3;
		else
			return 2;
	}
	else
		return 1;
}

void updateInvaderOdds(void)
{
	if(level >= double_invader_start_level)
		double_invader_odds += double_invader_odds_change;
	if(level >= triple_invader_start_level)
		triple_invader_odds += triple_invader_odds_change;
}

int setupSound(void)
{
	/** Note: We're using setBitmapPath() as a general resource path.
 * 	 ** The name of the function should eventually be changed to reflect
 * 	 ** this.
 * 	 **/
	int soundsNotLoaded = 0; /* for counting sound that fail to load */
	setBitmapPath("Audio/Fire.wav");
	
	if(fireSample == NULL)
		if((fireSample = load_sample(bitmapPath)) == NULL)
		{
			puts("===============================================");
			printf("Could not load Fire.wav from: \n\t%s%s\n",
				resourceDirectory,
				"Audio");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/InvaderKilled.wav");
	
	if(invaderKilledSample == NULL)
		if((invaderKilledSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load InvaderKilled.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/PlayerKilled.wav");
	
	if(playerKilledSample == NULL)
		if((playerKilledSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load PlayerKilledSample.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/Shield.wav");
	
	if(shieldSample == NULL)
		if((shieldSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load Shield.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/Background.wav");
	
	if(backgroundSample == NULL)
		if((backgroundSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load Background.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/NewGame.wav");
	
	if(newGameSample == NULL)
		if((newGameSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load NewGame.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/GameOver.wav");
	
	if(gameOverSample == NULL)
		if((gameOverSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load GameOver.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/NavigateMenu.wav");
	
	if(navigateMenuSample == NULL)
		if((navigateMenuSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load NavigateMenu.wav");
			soundsNotLoaded++;
		}
	setBitmapPath("Audio/Select.wav");
	
	if(selectSample == NULL)
		if((selectSample = load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load Select.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/NavigateMainMenu.wav");

	if(navigateMainMenuSample == NULL)
		if((navigateMainMenuSample =
			load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load NavigateMainMenu.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/Regeneration.wav");

	if(regenerationSample == NULL)
		if((regenerationSample =
			load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load Regeneration.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/ExtraLifeBonus.wav");

	if(extraLifeBonusSample == NULL)
		if((extraLifeBonusSample =
			load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load ExtraLifeBonus.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/MissileBlast.wav");

	if(missileBlastSample == NULL)
		if((missileBlastSample =
			load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load MissileBlast.wav");
			soundsNotLoaded++;
		}

	setBitmapPath("Audio/MissileAlert.wav");

	if(missileAlertSample == NULL)
		if((missileAlertSample =
			load_sample(bitmapPath)) == NULL)
		{
			puts("Could not load MissileAlert.wav");
			soundsNotLoaded++;
		}

	if(soundsNotLoaded)
	{
		printf("\n%d sound files failed to load.\n",
			soundsNotLoaded);
		puts("Sound will be disabled.");
		puts("=================================================");
		return 1; /* tells caller that sounds failed */
	}

	return 0;
}

int addToScore(int playerNumber, int modValue)
{
	int num;

	static unsigned oldscore;
	/* set the first score to receive an extra life
	 * after which extraLifePoints will be used
	 */
	unsigned minimumScore = 10000;

	/* reset oldscore at start (or restart) of game */
	num = playerNumber;
	if(score[num] == 0)
	  oldscore = 0;

	score[num]+= modValue;

	/* check for extra lives */
	if((score[num] >= minimumScore) &&
		(score[num] - oldscore) >= extraLifePoints)
	{
	  oldscore = score[num];
	  lives[num]++;

	/* play the extra life bonus sample */
	if(soundon)
		play_sample(extraLifeBonusSample,
			volume, pan, pitch, FALSE);
	}

	return score[num];
}

/* pause and blinking functions */
void setPause(int pauseState)
{
	gamePause = pauseState;
}

int getPause(void)
{
	return gamePause;
}

int updateBlinking(void)
{
	/* update all blinking objects and
	 * return the number of objects that are blinking
	 */
	int i, numberOfBlinkingObjects;
	numberOfBlinkingObjects = 0;

	/* loop through the two player sprites */
	for(i = 0; i < numberOfPlayers; i++) {
		if(player[i]->blinking)
		{
			numberOfBlinkingObjects++;

			if(!(player[i]->blinking % player[i]->blinkDelay))
			{
				player[i]->visible = !(player[i]->visible);
			}
		
			if(--player[i]->blinking <= 0)
			{
				numberOfBlinkingObjects--;

				player[i]->blinking = 0;
				setPause(0);
				player[i]->visible = 1;

				/* stop player regeneration sample */
				if(soundon)
					stop_sample(regenerationSample);
			}
		}
	}
	
	return numberOfBlinkingObjects;
}
