/*  This file is part of the program:

    nV@derz! (nvaderz)
    Copyright (C) 2006-2013  Eric Pellegrin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    Eric Pellegrin can be reached by email at pellea72(at)yahoo(dot)com
    (Note: email address modified to evade unsolicited email search bots.
        Change (at) to @ and (dot) to . to read name@company.com )
*/

#include "nvaderz.h"
#include "highscores.h"
#include "bonus.h"
#include "joystick.h"

int moveplayer(int playerNumber, int dir)
{
	int i, num;
	num = playerNumber;

	/* if not a valid player, do nothing */
	if(player[num] == NULL)
		return 1;

	if(dir == 1) /* if right... */
	{
		player[num]->x += player[num]->xspeed;

		/* stop before going off right side */
		if(player[num]->x + player[num]->width > right_boundary)
		{
			player[num]->x = right_boundary - player[num]->width;
		}
	}
	else if(dir == 3) /* if left... */
	{
		player[num]->x -= player[num]->xspeed;

		/* stop before going off left side */
		if(player[num]->x < left_boundary)
		{
			player[num]->x = left_boundary;
		}
	}

	/* if guided projectile are active, and player is steering them,
	 * move any of the player's
	 * lasers (more than one if repeat fire is active) by the
	 * player's x speed
	 */
	if(getGuidedStatus(num) && getGuidedSteeringStatus(num))
	{
	  for(i = 0; i < 3; i++)
	  {
		if(laser[num][i]->alive)
		{
			if(dir == 1) /* if right */
				laser[num][i]->x += player[num]->xspeed;
			else if(dir == 3) /* if left */
				laser[num][i]->x -= player[num]->xspeed;
		}
	  }
	}

	/* check for collision with impact */
	if(player[num]->alive && !(player[num]->blinking) && impact.alive)
	{
		if(testcollision(&player[num]->bounds, &impact.bounds))
		{
			killplayer(num);
		}
	}

	/* update player bounds */
	player[num]->bounds.x1 = player[num]->x + player_bounds_offsetx1;
	player[num]->bounds.x2 = player[num]->x + player[num]->width
		+ player_bounds_offsetx2;
	player[num]->bounds.y1 = player[num]->y + player_bounds_offsety1;
	player[num]->bounds.y2 = player[num]->y + player[num]->height
		+ player_bounds_offsety2;

	/* continue game if any player alive, else end the game */
	for(i = 0; i < numberOfPlayers; i++) {
		if((player[i] != NULL) && player[i]->alive > 0)
			return 0;
	}

	/* ... else, end the game */
	endgame(1); /* gameover = 1 */
	return 1;
}

int movehorde(void)
{
	int i, j, shieldCollisionFlag;
	int offtop, offbottom, offleft, offright;
	SPRITE *temp;

	/* define horde to horde->bounds offsets */
	offtop = horde->bounds.y1 - horde->y; /* not currently used */
	offbottom = (horde->y + horde->height) - horde->bounds.y2;
	offleft = horde->bounds.x1 - horde->x;
	offright = (horde->x + horde->width) - horde->bounds.x2;

	shieldCollisionFlag = 0;

        /* check if invaders have collided with shield */
        for(i = 0; i < invader_rows; i++)
	{
                 for(j = 0; j < invader_columns; j++)
                 {
                         temp = &invaders[i][j];
 
                         if( (getOneWayStatus() && ((temp->alive) &&
				testcollision(&temp->bounds,
				  &shield->bounds)) ) ||
				((shield->alive>0) && temp->alive &&
                                 testcollision(&temp->bounds,
					&shield->bounds)
                                 && shield_testcollision(temp,
					"top")) )
                         {
				 shieldCollisionFlag = 1;

                                 temp->alive--;/*decrease lives*/
                                 if(!temp->alive)
                                 {
					if(soundon)
					  play_sample(
					    invaderKilledSample,
					    volume, pan, pitch, FALSE
					    );

                                         if(--totalinvaders <= 0)
                                                 gonextlevel();
                                 }
                         }
                 }
	/* if any invaders hit one way shield, deactivate it */
	if(getOneWayStatus() && shieldCollisionFlag)
		disableOneWay();

	}

	/* if moving right...*/
	if(horde->dir == 1)
	{
		horde->x += horde->xspeed;

		/* if about to move off screen right */
		if((horde->x + horde->width - offright) > right_boundary)
		{
			horde->x = right_boundary - horde->width + offright;
			
			/* drop down a level */
			horde->y += invader_height + invader_space_vertical;

			/* horde will move in opposite direction */
			horde->dir = 3;
		}
	}
	/* if moving left... */
	else if(horde->dir == 3)
	{
		horde->x -= horde->xspeed;

		/* if about to move off screen left */
		if((horde->x + offleft) < left_boundary)
		{
			horde->x = left_boundary - offleft;
			
			/* drop down a level */
			horde->y += invader_height+ invader_space_vertical;

			/* move horde in opposite direction */
			horde->dir = 1;
		}
	}

	/* if about to drop below player */
	if(horde->y + horde->height - offbottom > player_topline)
	{
		horde->y = screen_baseline - horde->height + offbottom;
		updatehorde();
		endgame(1);
	}

	/* update the individual invader sprites */
	updatehorde();

	return 0;
}

void movelasers(void)
{
	int i, j, k, l, num;
	int tempx, tempy;
	SPRITE *temp;

	/* move the bombs */
	for(i = 0; i < invader_columns; i++)
	{
		if(bombs[i].alive)
		{
			bombs[i].y += bombs[i].yspeed;

			/* update bomb's bounds */
			bombs[i].bounds.x1 = bombs[i].x;
			bombs[i].bounds.x2 = bombs[i].x
				+ bombs[i].width;
			bombs[i].bounds.y1 = bombs[i].y;
			bombs[i].bounds.y2 = bombs[i].y
				+ bombs[i].height;

			/* check if bomb hits player */
			for(l = 0; l < numberOfPlayers; l++) {
				if(!player[l]->alive)
					continue;
				if(testcollision(&bombs[i].bounds,
					&player[l]->bounds))
				{
					killplayer(l);
					bombs[i].alive = 0;
				}
			}
			/* check if bomb hit one way shield */
			if(getOneWayStatus() &&
				testcollision(&bombs[i].bounds,
				  &shield->bounds) )
			{
				bombs[i].alive = 0;
			}
			/* check if bomb hit shield */
			else if((shield->alive>0) && testcollision(
				&bombs[i].bounds,&shield->bounds) &&
				shield_testcollision(&bombs[i],
					"top"))
			{
				bombs[i].alive = 0;
			}
			/* check if going off bottom */
			else if(bombs[i].y + bomb_height > screen_baseline)
			{
				bombs[i].alive = 0;
			}

		}
		else if(bombarray[i] >= 0)
		{
			/* add a random factor to firing */
		        if(rand()%dropFactor < dropOdds)
			{
				bombs[i].alive = 1;

				/* give the bomb starting coords */
				tempx = invaders[bombarray[i]][i].x +
					invader_width / 2;
				tempy = invaders[bombarray[i]][i].y +
					invader_height;
				bombs[i].x = tempx - (bomb_width / 2);
				bombs[i].y = tempy;
			}
		}
	}


	/* move the players' lasers */
	for(num = 0; num < 2; num++)
		for(k = 0; k < 3; k++)
		{
		  if(laser[num][k]->alive)
		  {
			if(!firing[num][k])
			{
				if(soundon)
					play_sample(fireSample,
						volume, pan, pitch, FALSE);
				firing[num][k] = 1;
				laser[num][k]->y = player_topline - laser_height;
				laser[num][k]->x = player[num]->x + (player[num]->width / 2);
			}

			laser[num][k]->y += laser[num][k]->yspeed;

			/* update laser bounds */
			laser[num][k]->bounds.x1 = laser[num][k]->x;
			laser[num][k]->bounds.x2 = laser[num][k]->x + laser[num][k]->width;
			laser[num][k]->bounds.y1 = laser[num][k]->y;
			laser[num][k]->bounds.y2 = laser[num][k]->y + laser[num][k]->height;

			/* check if laser off top of screen */
			if(laser[num][k]->y + laser_height < screen_topline)
			{	
				firing[num][k] = 0;
				laser[num][k]->alive = 0;
				return;
			}

			/* check if laser hit shield */
			if((shield->alive>0) && testcollision(&laser[num][k]->bounds,
				&shield->bounds)
				&& shield_testcollision(laser[num][k],"bottom")) 
			{
				firing[num][k] = 0;
				laser[num][k]->alive = 0;
				return;
			}

			/* check if laser collided with invader */
			for(i = 0; i < invader_rows; i++)
				for(j = 0; j < invader_columns; j++)
				{
					temp = &invaders[i][j];

					if(temp->alive && testcollision(
						&laser[num][k]->bounds,&temp->bounds)) 
					{
						activateReveal(); /* activate temporary
							visibility in case invader
							invisible bonus is active */

						temp->alive--;/*decrease lives*/
						laser[num][k]->alive = 0;
						firing[num][k] = 0;
						laser[num][k]->y = 0 -
						  laser[num][k]->height;
						addToScore(num, (50 - (i * 10))
							* (level * .25));
						if(!temp->alive)
						{
							if(soundon)
							  play_sample(
							  invaderKilledSample,
							  volume, pan, pitch, FALSE);
							if(--totalinvaders <= 0)
								gonextlevel();
						}
						/* exit both for loops */
						i = invader_rows;
						break;
					}
				}

			/* check if laser collided with capsule */
			for(i = 0; i < 6; i++)
			{
				if(capsules[i].alive)
				{
					if(testcollision(&capsules[i].bounds,
					  &laser[num][k]->bounds))
					{
					  addToScore(num, 100);
					  capsules[i].alive = 0;
					  laser[num][k]->alive = 0;
					  firing[num][k] = 0;
					  laser[num][k]->y = 0 - laser[num][k]->height;

					  activateBonus(num);
					  
					  break;
					}
				}
			}
			
			/* check if laser collided with missile */
			if(missile.alive)
			{
				if(testcollision(&missile.bounds,
					&laser[num][k]->bounds) )
				{
					missile.alive = 0;
					laser[num][k]->alive = 0;
					laser[num][k]->y = 0 - laser[num][k]->height;
					firing[num][k] = 0;

					score[num] += 100;
				}
			}
		  }
		}
}

void displaystatus(void)
{
	/* print status line to top of screen */
	textprintf_ex(screen_buffer, font, 0, 0, WHITE, -1,
			"nV@derz!  (ESC to quit)   lives[0]: %d   lives[1]: %d     level: %d     score[0]: %d    score[1]: %d",
			lives[0], lives[1], level, score[0], score[1]);

	/* for testing purposes, display topline (top limit) */
	hline(screen_buffer, 0, screen_topline, screen_width, WHITE);
}

void displaysplash(int loadingFlag)
{
	int centerx, centery;

	centerx = screen->w / 2;
	centery = screen->h / 2;

	rectfill(screen_buffer, 0, 0, screen->w, screen->h,
		BLACK);

	rect(screen_buffer, 2, 2, screen->w - 2, screen->h - 2,
		GREEN);

	rect(screen_buffer, centerx - 250, centery - 200,
		centerx + 250, centery + 200, GREEN);

	textout_centre_ex(screen_buffer, font,
		"nV@derz!", centerx, centery - 150,
		GREEN, -1);

	textout_centre_ex(screen_buffer, font,
		"______________________________",
		centerx, centery - 130,
		GREEN, -1);

	textout_centre_ex(screen_buffer, font,
		"(c)2006-2012 Eric Pellegrin",
		centerx, centery - 115,
		GREEN, -1);
	textout_centre_ex(screen_buffer, font,
		VERSION_STRING,
		centerx, centery - 100,
		GREEN, -1);

	/* display sound credits */
	textout_centre_ex(screen_buffer, font,
		"Sounds not available in CVS checkout",
		centerx, centery - 60,
		GREEN, -1);
	textout_centre_ex(screen_buffer, font,
		"They can be downloaded in a separate Resource Pack",
		centerx, centery - 45,
		GREEN, -1);
	textout_centre_ex(screen_buffer, font,
		"at: sourceforge.net/projects/nvaderz",
		centerx, centery -30,
		GREEN, -1);

	textout_centre_ex(screen_buffer, font,
		"Joysticks:",
		centerx, centery + 15,
		WHITE, -1);
	textout_centre_ex(screen_buffer, font,
		"Control with buttons 1, 2, 3, 4",
		centerx, centery + 30,
		WHITE, -1);

	textout_centre_ex(screen_buffer, font,
		"Shoot with button 7",
		centerx, centery + 45,
		WHITE, -1);

	textout_centre_ex(screen_buffer, font,
		"Keyboard:",
		centerx, centery + 80,
		WHITE, -1);
	textout_centre_ex(screen_buffer, font,
		"Player 1: Left, right, up, down arrows",
		centerx, centery + 95,
		WHITE, -1);
	textout_centre_ex(screen_buffer, font,
		"Shoot with spacebar",
		centerx, centery + 110,
		WHITE, -1);
	textout_centre_ex(screen_buffer, font,
		"Player 2: A W S D keys",
		centerx, centery + 135,
		WHITE, -1);
	textout_centre_ex(screen_buffer, font,
		"Shoot with left ALT key",
		centerx, centery + 150,
		WHITE, -1);

	if(loadingFlag)
	{
		textout_centre_ex(screen_buffer, font,
			"[ Loading Game... ]",
			centerx, centery + 175,
			GREEN, -1);

		blit(screen_buffer, screen, 0, 0, 0, 0,
			screen->w, screen->h);
	}
	else
	{
		/* set splashFlag so splash won't be called again */
		splashFlag = 0;

		rest(1000);

		textout_centre_ex(screen_buffer, font,
			"[ Press any key to continue ]",
			centerx, centery + 175,
			GREEN, -1);

		blit(screen_buffer, screen, 0, 0, 0, 0,
			screen->w, screen->h);

		while(!keypressed()) {
			updateJoystickInfo();
			if(joyFire_p1)
				break;
		}
	}
}

void updatehorde(void)
{
	int i, j;
	SPRITE *temp;

	horde->bounds.x1 = screen_width;
	horde->bounds.y1 = screen_height;
	horde->bounds.x2 = 0;
	horde->bounds.y2 = 0;

	/* reset bombarray to -1 */
	for(i = 0; i < invader_columns; i++)
		bombarray[i] = -1;

        /* fill the invaders sprite array */
	for(i = 0; i < invader_rows; i++)
		for(j = 0; j < invader_columns; j++)
		{
			/* set sprite values based on horde values */
			invaders[i][j].x = horde->x + (j * invader_width) +
				(j * invader_space_horizontal);
			invaders[i][j].y = horde->y + (i * invader_height) +
				(i * invader_space_vertical);

			/* update invader's bounds */
			invaders[i][j].bounds.x1 = invaders[i][j].x;
			invaders[i][j].bounds.x2 = invaders[i][j].x
				+ invaders[i][j].width;
			invaders[i][j].bounds.y1 = invaders[i][j].y;
			invaders[i][j].bounds.y2 = invaders[i][j].y
				+ invaders[i][j].height;

			/* if alive, set horde bounds and bomb array */
			if(invaders[i][j].alive)
			{
				temp = &invaders[i][j];
				
				if(temp->x < horde->bounds.x1)
					horde->bounds.x1 = temp->x;
				if(temp->y < horde->bounds.y1)
					horde->bounds.y1 = temp->y;
				if(temp->x + temp->width > horde->bounds.x2)
					horde->bounds.x2 = temp->x + temp->width;
				if(temp->y + temp->height > horde->bounds.y2)
					horde->bounds.y2 = temp->y + temp->height;

				/** if invader lowest in its column so far,
				    turn this column on in the bomb array
				    so it can drop bombs */
				if(i > bombarray[j])
					bombarray[j] = i;
			}
		}
		
}

void displaygameover(void)
{
	int escPressed, spacePressed;

	/* stop the background music */
	if(soundon)
	{
		stop_sample(backgroundSample);
		play_sample(gameOverSample, volume, pan, pitch,
			FALSE);
	}

	drawbackground();
	drawcapsules();
	drawmissile();
	drawhorde();
	drawshield();
	drawimpact();
	drawHud();

	rectfill(screen_buffer, screen_width / 2 - 150, screen_height / 2 - 50,
			screen_width / 2 + 150, screen_height / 2 + 50,
			BLACK);

	rect(screen_buffer, (screen_width / 2 - 150) + 2,
		(screen_height / 2 - 50) + 2,
		(screen_width / 2 + 150) - 2,
		(screen_height / 2 + 50) - 2,
		GREEN);

	textout_centre_ex(screen_buffer, font, "Game Over", screen_width / 2,
			screen_height / 2, GREEN, -1);

	textout_centre_ex(screen_buffer, font, "<FIRE> to continue", screen_width / 2,
			screen_height / 2 + 20, GREEN, -1);
	

        drawscreen();

	/* if still registering key presses from before */

	if(key[KEY_ESC])
		escPressed = 1;
	else
		escPressed = 0;
	if(key[KEY_SPACE] || key[KEY_ALT])
		spacePressed = 1;
	else
		spacePressed = 0;

	/* make sure no longer receiving joystick button presses */
	if(joysticksAllowed) {
		while(1) {
			updateJoystickInfo();
			if(!joyFire_p1 && !joyFire_p2 && !joyReset_p1 && !joyReset_p2) break;
		}
	}

	while(1)
	{
		updateJoystickInfo();

		if(((joyReset_p1 || joyReset_p2 || key[KEY_ESC]) && !escPressed)
			|| ((joyFire_p1 || joyFire_p2 || key[KEY_SPACE] || key[KEY_ALT]) && !spacePressed))
		{
			break;
		}

		if(!key[KEY_ESC] && (!key[KEY_SPACE] || key[KEY_ALT]))
		{
			escPressed = 0;
			spacePressed = 0;

			/* make sure no longer receiving joystick button presses */
			if(joysticksAllowed) {
				while(1) {
					updateJoystickInfo();
					if(!joyFire_p1 && !joyFire_p2 && !joyReset_p1 && !joyReset_p2) break;
				}
			}
		}

	}

}


void fire(int playerNumber)
{
	int i, num;
	num = playerNumber;

	/* if the player is not alive, do nothing */
	if((player[num] != NULL) && player[num]->alive <= 0)
		return;

	if(getThreeShotStatus(num))
	{
		/* turn on the next available shot */
		for(i = 0; i < 3; i++)
		{
		  if(!(laser[num][i]->alive))
		  {
		    laser[num][i]->alive = 1;
		    break;
		  }
		}
	}
	else
		laser[num][0]->alive = 1;
}

int testcollision(BOUNDS *first, BOUNDS *second)
{
	int mx1, mx2, my1, my2;
	int tx1, tx2, ty1, ty2;
	int firstArea, secondArea;

	BOUNDS *projectile, *target;

	/* make sure projectile is the smaller bounds object */
	firstArea = (first->x2 - first->x1)*(first->y2 - first->y1);
	secondArea = (second->x2 - second->x1)
		* (second->y2 - second->y1);
	if(firstArea > secondArea)
	{
		projectile = second;
		target = first;
	}
	else
	{
		projectile = first;
		target = second;
	}

	/* set projectile variables */
	mx1 = projectile->x1;
	my1 = projectile->y1;
	mx2 = projectile->x2;
	my2 = projectile->y2;

	/* set target variables */
	tx1 = target->x1;
	ty1 = target->y1;
	tx2 = target->x2;
	ty2 = target->y2;

	/* compare projectile corners for intersection with target */
	/* top left */
	if((mx1 >= tx1 && mx1 <= tx2) && (my1 >= ty1 && my1 <= ty2))
		return 1;
	/*top right */
	else if((mx2 >= tx1 && mx2 <= tx2) && (my1 >= ty1 && my1 <= ty2))
		return 1;
	/* bottom left */
	else if((mx1 >= tx1 && mx1 <= tx2) && (my2 >= ty1 && my2 <= ty2))
		return 1;
	else if((mx2 >= tx1 && mx2 <= tx2) && (my2 >= ty1 && my2 <= ty2))
		return 1;

	return 0;

}

void killplayer(int playerNumber)
{
	int num;
	num = playerNumber;
	
	/* return if called on an invalid player */
	if(player[num] == NULL) {
		printf("killplayer() called on invalid player # %d\n", num);
		return;
	}

	if(soundon)
	{
		play_sample( playerKilledSample,
			volume, pan, pitch, FALSE
			);
	}

	player[num]->blinking = 100;
	if(--lives[num] < 0)
	{
		int i, numberOfPlayersAlive;

		player[num]->alive = 0;

		/* if any players alive, keep playing... */
		numberOfPlayersAlive = 0;
		for(i = 0; i < numberOfPlayers; i++) {
			if(player[i]->alive > 0) {
				numberOfPlayersAlive++;
			}
		}
		
		/* ...else, end the game */
		if(! numberOfPlayersAlive) {
			endgame(1);
		}
	}
	else
	{
		setPause(2); /* set an in-game pause */
		
		/* while paused, play sound of player regenerating */
		if(soundon)
		{
			play_sample( regenerationSample,
				volume, pan, pitch, FALSE
				);
		}
	}
}

void endgame(int endCode)
{
	/** If endCode is nonzero, assume the game ended normally.
	 ** 	- show game over screen
	 **	- if a new high score reached, display keypad
	 **	- reset the game and return to main menu
	 **
	 ** If endCode is zero, assume game ended early (Escape key).
	 **	- reset and return to main menu
	 */

	int num;

	if(endCode)
	{
		int gameOverDelay = 1000; /* delay after game ends */
		gameover = 1;
		displaygameover();

		rest(gameOverDelay);

		for(num = 0; num < 2; num++)
			updateHighScores(num, score[num]);
		writeScoresToFile();
		reset();
		displaymenu();
	}
	else
	{
		gameover = 1;
		reset();
		displaymenu();
	}
}

void reset(void)
{
	/* commented out pending removal
	** This code is now in setup()
	int i, num;
	

	gameover = 0;
	for(num = 0; num < 2; num++)
		score[num] = 0;
	for(i = 0; i < numberOfPlayers; i++)
		lives[i] = 2;
	for(num = 0; num < 2; num++)
		for(i = 0; i < 3; i++)
			firing[num][i] = 0;
	level = 1;
	totalinvaders = invader_columns * invader_rows;
	*/
	setup();
	updatehorde();
}

void gonextlevel(void)
{
	int i, j, num;
	gameover = 0;
	for(num = 0; num < 2; num++)
		for(i = 0; i < 3; i++)
			firing[num][i] = 0;
	totalinvaders = invader_columns * invader_rows;

	/* stop and restart the background music */
	if(soundon)
	{
		stop_sample(backgroundSample);
		play_sample(backgroundSample, volume, pan, pitch,
			TRUE);
	}

	level++;

	/* adjust Super Invader odds */
	updateInvaderOdds();

	/* reset the horde */
	horde->y = horde_start_y;
	horde->x = screen_width / 2 - horde->width / 2;
	for(i = 0; i < invader_rows; i++)
	    for(j = 0; j < invader_columns; j++)
		invaders[i][j].alive = calcInvaderLife();

	/* reset the player position */
	/* commented out: do we need to return players to start position? */
	/* player[0]->x = screen_width / 2 - player_width / 2; */

	/* turn off bombs and player laser */
	for(num = 0; num < 2; num++)
		for(i = 0; i < 3; i++)
			laser[num][i]->alive = 0;

	for(i = 0; i < invader_columns; i++)
	{
	    bombs[i].alive = 0;
	}

	updatehorde();

	/* apply level adjustments */

	/* every so many levels, start the horde lower */
	if(!(level%5))
	{
	    int nextHordeBottom = horde->bounds.y2 + INVADER_H +
		INV_SPACE_H;

	    /* drop horde only if not past player */
	    if(!(nextHordeBottom > player_topline))
	    {
	    	horde_start_y += invader_height + invader_space_vertical;
		horde->y = horde_start_y;
	    }
	}

	/* increase horde speed fractionally each level */
	if(horde->xspeed < 5)
	    horde->xspeed+= 0.10;

	/* uncomment to gradually increase player speed by level */
	/* if(player->xspeed < 15)
	    player->xspeed += 0.10;
	*/

	/* increase player firing speed by level */
	for(num = 0; num < 2; num++)
		for(i = 0; i < 3; i++)
		{
			if(abs(laser[num][i]->yspeed) <= (invader_height - 1))
			    laser[num][i]->yspeed -= 0.10;
		}

	/* increase invader firing speed by level */
	if(abs( bomb_y_speed) <= (20))
	    bomb_y_speed += 0.10;
	    
	/* increase odds a particular invader will fire */
	if(dropOdds < 40)
	    dropOdds += 0.333;

	updatehorde();

	/* update capsule data */
	incrementAvailableCapsules();
	resetCapsuleTicks();
}

int shield_testcollision(SPRITE *projectile, char *direction)
{
	/* shield collision testing has to accomodate two situations:
	 *   1. catching something inside the shield when it is first
	 *      toggled on by the player, and
	 *
	 *   2. not allowing things to go through the shield, or parts
	 *      of it, as if it were not there.
	 *
	 * To solve this, an activated shield has two states:
	 *   newly active (first game cycle shield is active), and
	 *   continuously active (active for one or more cycles)
	 *
	 * When newly active, the shield tests for collisions with any
	 * shield segment that comes into contact with the object in
	 * the object's current position. This allows for situation 1.
	 *
	 * When continuously active, an objects prior position will
	 * be determined, and collision testing will be performed
	 * on the outermost shield beam that has active segments
	 * between the object's prior position and current position.
	 * This involves passing a direction string to the function
	 * telling it which side of the shield is being hit:
	 * "top" - straight down from above, "bottom" - straight up
	 * from below, and "t_top" a trajectory object (e.g. a capsule)
	 * hitting the top of the shield.
	 * This satisfies situation 2.
	 */
	int i, j;
	unsigned contactFlag;
	unsigned oldTrajPos, currTrajPos;
	unsigned x1, y1, x2, y2;
	float slope;
	unsigned leftLimit, rightLimit;
	unsigned currBeamY;
	BOUNDS tempBounds, segmentBounds;

	contactFlag = 0;

	/* if shield has been active and missile is not "neutral"... */
	if(shieldTicks && strcmp(direction, "neutral") )
	{
	  if(!(strcmp(direction, "t_top")))
	  {
		if(!(strcmp(projectile->type, "capsule")) )
		{
			currTrajPos = getCapsulePosition(projectile->alive - 1);
			if(currTrajPos > 0)
			  oldTrajPos = currTrajPos - 1;
			else
			{
			  /* there must be a problem; return 0 for now */
			  return 0;
			}
			
			x1 = getCapsuleXValue(projectile->alive - 1, currTrajPos);
			y1 = getCapsuleYValue(projectile->alive - 1, currTrajPos);
			x2 = getCapsuleXValue(projectile->alive - 1, oldTrajPos);
			y2 = getCapsuleYValue(projectile->alive - 1, oldTrajPos);
		}
		else if(!(strcmp(projectile->type, "missile")) )
		{
			currTrajPos = getMissilePosition();
			if(currTrajPos > 0)
			  oldTrajPos = currTrajPos - 1;
			else
			{
			  /* there must be a problem; return 0 for now */
			  return 0;
			}
			
			x1 = getMissileXValue(currTrajPos);
			y1 = getMissileYValue(currTrajPos);
			x2 = getMissileXValue(oldTrajPos);
			y2 = getMissileYValue(oldTrajPos);

		}
		else /* neither capsule nor missile */
			return 0;

		/* calculate slope of line between current and
		 * former position
		 *
		 * make sure slope not 0, to avoid division by 0
		 */
		if((x2 - x1))
			slope = (y2 - y1)/(x2 - x1);
		else
			slope = 0.001;

		/* loop through the shield beams */
		for(i=0; i < 3; i++)
		{
			currBeamY = shield->y +
			  ((shieldSpace + shieldSegmentHeight) * i);

			if((y2-(projectile->height / 2) > currBeamY))
			{
			  /* object's prior position was not outside
			   * this beam, so skip to next shield beam
			   */
			  continue;
			}
			else
			{
			  y2 = currBeamY;
			  /* use modified point-slope formula to
			   * find the object's x value if it were
			   * placed at the y position of the current
			   * beam.
			   */
			  x2 = x1 + ((y1 - y2)/ slope);

			  /* calculate a left and right limit of
			   * of shieldArray[] indices within which
			   * the object would fall, so that the
			   * entire array does not have to be tested
			   */
			  leftLimit = (x2 - (projectile->width / 2)) /
			    shieldSegmentWidth;
			  if((int)leftLimit < 0)
			    leftLimit = 0;
			  rightLimit = (x2 + (projectile->width / 2)) /
			    shieldSegmentWidth;
			  if(rightLimit > screen->w /shieldSegmentWidth)
			    rightLimit = screen->w / shieldSegmentWidth;

			  tempBounds.x1 = x2 - (projectile->width / 2);
			  tempBounds.x2 = x2 + (projectile->width / 2);
			  tempBounds.y1 = currBeamY - projectile->height;
			  tempBounds.y2 = currBeamY;

			  for(j = (int)leftLimit; j <= (int)rightLimit; j++)
			  {
			    if(shieldArray[i][j])
			    {
				    segmentBounds.x1 = shield->x +
				      (j * shieldSegmentWidth);
				    segmentBounds.x2 = shield->x +
				      (j * shieldSegmentWidth) +
				      shieldSegmentWidth;
				    segmentBounds.y1 = currBeamY;
				    segmentBounds.y2 = currBeamY +
				      shieldSegmentHeight;

				    if(testcollision(&segmentBounds,
				      &tempBounds))
				    {
					shieldArray[i][j] = 0;
					contactFlag = 1;
				    }
			    }
			}

			/* break if contact was made */
			if(contactFlag)
			  break;
		}
	    }
		
	  }
	  else if(!(strcmp(direction, "top")))
	  {
	    /* loop through the beams from top to bottom */
	    for(i = 0; i < 3; i++)
	    {
	      currBeamY = shield->y + ((shieldSegmentHeight +
	        shieldSpace) * i);

	      if((projectile->bounds.y1 - projectile->yspeed) > currBeamY)
	      {
	        /* object's last position was not outside
		 * current beam
		 */
	        continue;
	      }

	      x1 = projectile->bounds.x1;
	      x2 = projectile->bounds.x2;
	      /* calculate a left and right limit of
	       * of shieldArray[] indices within which
	       * the object would fall, so that the
	       * entire array does not have to be tested
	       */

	      leftLimit = x1 / shieldSegmentWidth;
	      if((int)leftLimit < 0)
	        leftLimit = 0;
	      rightLimit = x2 / shieldSegmentWidth;
	      if(rightLimit > screen->w / shieldSegmentWidth)
	        rightLimit = screen->w / shieldSegmentWidth;

	      tempBounds.x1 = x1;
	      tempBounds.x2 = x2;
	      /* make sure object will collide with y-values of
	       * shield segments, if they are active
	       */
	      tempBounds.y1 = currBeamY - projectile->height;
	      tempBounds.y2 = currBeamY;

	      /* loop through a subset of the shield segments */
	      for(j = (int)leftLimit; j <= (int)rightLimit; j++)
	      {
	        if(shieldArray[i][j])
	        {
	          segmentBounds.x1 = shield->x +
	            (j * shieldSegmentWidth);
	          segmentBounds.x2 = shield->x +
	            (j * shieldSegmentWidth) + shieldSegmentWidth;
	          segmentBounds.y1 = currBeamY;
	          segmentBounds.y2 = currBeamY + shieldSegmentHeight;

	            if(testcollision(&tempBounds, &segmentBounds))
	            {
	              shieldArray[i][j] = 0;
	              contactFlag = 1;
	            }
	        }
	      }

	      if(contactFlag)
	        break;
	    }
	  }
	  else if(!(strcmp(direction, "bottom")))
	  {
	    /* loop through the beams from top to bottom */
	    for(i = 2; i >= 0; i--)
	    {
	      currBeamY = shield->y + ((shieldSegmentHeight +
	        shieldSpace) * i);

	      if((projectile->bounds.y2 - projectile->yspeed) < currBeamY)
	      {
	        /* object's last position was not outside
		 * current beam
		 */
	        continue;
	      }

	      x1 = projectile->bounds.x1;
	      x2 = projectile->bounds.x2;

	      /* calculate a left and right limit of
	       * of shieldArray[] indices within which
	       * the object would fall, so that the
	       * entire array does not have to be tested
	       */
	      leftLimit = x1 / shieldSegmentWidth;
	      if((int)leftLimit < 0)
	        leftLimit = 0;
	      rightLimit = x2 / shieldSegmentWidth;
	      if(rightLimit > screen->w / shieldSegmentWidth)
	        rightLimit = screen->w / shieldSegmentWidth;

	      tempBounds.x1 = x1;
	      tempBounds.x2 = x2;
	      /* make sure object will collide with y-values of
	       * shield segments, if they are active
	       */
	      tempBounds.y1 = currBeamY + projectile->height;
	      tempBounds.y2 = currBeamY;

	      /* loop through a subset of the shield segments */
	      for(j = leftLimit; j <= (int)rightLimit; j++)
	      {
	        if(shieldArray[i][j])
	        {
	          segmentBounds.x1 = shield->x +
	            (j * shieldSegmentWidth);
	          segmentBounds.x2 = shield->x +
	            (j * shieldSegmentWidth) + shieldSegmentWidth;
	          segmentBounds.y1 = currBeamY;
	          segmentBounds.y2 = currBeamY + shieldSegmentHeight;

	            if(testcollision(&tempBounds, &segmentBounds))
	            {
	              shieldArray[i][j] = 0;
	              contactFlag = 1;
	            }
	        }
	      }

	      if(contactFlag)
	        break;
	    }
	  }
	}

	/* shield newly active or direction is "neutral",
	 * test collision on current bounds
	 */
	else
	{
	  x1 = projectile->bounds.x1;
	  x2 = projectile->bounds.x2;

          /* calculate a left and right limit of
           * of shieldArray[] indices within which
           * the object would fall, so that the
           * entire array does not have to be tested
           */
	  leftLimit = x1 / shieldSegmentWidth;
	  if((int)leftLimit < 0)
	    leftLimit = 0;
	  rightLimit = x2 / shieldSegmentWidth;
	  if(rightLimit > screen->w / shieldSegmentWidth)
	    rightLimit = screen->w / shieldSegmentWidth;

	  /* loop through a subset of the shield segments */
	  for(i = 0; i < 3; i++)
	  {
	    for(j = leftLimit; j <= (int)rightLimit; j++)
	    {
	      if(shieldArray[i][j])
	      {
	        currBeamY = shield->y +
	          ((shieldSpace + shieldSegmentHeight) * i);

	        segmentBounds.x1 = shield->x +
	          (j * shieldSegmentWidth);
	        segmentBounds.x2 = shield->x +
	          (j * shieldSegmentWidth) + shieldSegmentWidth;
	        segmentBounds.y1 = currBeamY;
	        segmentBounds.y2 = currBeamY + shieldSegmentHeight;

	        if(testcollision(&segmentBounds, &projectile->bounds))
	        {
	          shieldArray[i][j] = 0;
	          contactFlag = 1;
	        }
	      }
	    }
	  }
	}

	return contactFlag;
	
}

void movecapsules(void)
{
	int i;
	static int capsuleMoveTicks, capsuleMoveDelay;

	capsuleMoveDelay = 5;

	for(i = 0; i < 6; i++)
	{
		if(capsules[i].alive)
		{
			if( getOneWayStatus() &&
			 (testcollision(&capsules[i].bounds,
			 	&shield->bounds)) )
			{
				capsules[i].alive = 0;
			}
			else if((shield->alive>0) && 
			 testcollision(&capsules[i].bounds,
				&shield->bounds)
			 && shield_testcollision(&capsules[i],
				"t_top"))
			{
				capsules[i].alive = 0;
			}
			else if(impact.alive &&
				testcollision(&impact.bounds,
					&capsules[i].bounds) )
			{
				capsules[i].alive = 0;
			}
			else if(!(capsuleMoveTicks % capsuleMoveDelay))
				incrementCapsulePosition(i);
		}
	}

	capsuleMoveTicks++;
	updateCapsules();
}

void movemissile(void)
{
	int num;

	if(missile.alive)
	{
	  /* if missile hits one-way shield */
	  if( getOneWayStatus() &&
		(testcollision(&missile.bounds,
			&shield->bounds)) )
	  {
		missile.alive = 0;
		activateInvaderBonus();
	  }
	  /* if missile hits regular shield */
	  else if((shield->alive > 0) &&
	    testcollision(&missile.bounds,
	    &shield->bounds)
	    && shield_testcollision(&missile,
	    "t_top") ) 
	  {
		missile.alive = 0;
		activateInvaderBonus();
	  }
	  /* if missile hits player */
	  else {
		for(num = 0; num < numberOfPlayers; num++) {
			if(testcollision(&player[num]->bounds, &missile.bounds) ) {
				missile.alive = 0;
				killplayer(num);
				activateInvaderBonus();
			}
		}
	  }
	}

	updateMissile();
}

void updateShield(void)
{
	/* test for collision with impact bonus */
	if(impact.alive && (shield->alive > 0) && !getOneWayStatus()
		&& testcollision(&shield->bounds, &impact.bounds) )
	{
		shield_testcollision(&impact, "neutral");	
	}

	if(shield->alive > 0)
		shieldTicks++;
	else
	{
		shieldTicks = 0;
	}
}
