/*  This file is part of the program:

    nV@derz! (nvaderz)
    Copyright (C) 2006-2013  Eric Pellegrin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

    Eric Pellegrin can be reached by email at pellea72(at)yahoo(dot)com
    (Note: email address modified to evade unsolicited email search bots.
        Change (at) to @ and (dot) to . to read name@company.com )
*/

#include <math.h>
#include "nvaderz.h"
#include "bonus.h"

static int capsuleTrajectoryX[6][200];
static int capsuleTrajectoryY[6][200];
static int capsuleOdds;
static int capsulePosition[6];
static float numAvailableCapsules;
static unsigned long createCapsulesDelay;
static unsigned long createCapsulesTicks;

static int missileTrajectoryX[200];
static int missileTrajectoryY[200];
static int currentMissilePosition;
static int missileOdds;

static int impactTicks;
static int impactRadius;

int threeShotActive[2];
int guidedActive[2];
int guidedSteering[2];
int oneWayActive;

/* invisible invaders bonus variables */
int hordeVisible;
int invisibleActive;

void setCapsuleTrajectory(int capsuleNumber);
void setMissileTrajectory(void);

void setupBonuses(void)
{
	int i, num;

	/* set initial values of variables */
	capsuleOdds = 20; /* 20 */
	numAvailableCapsules = 0.8; /* 0.8 */

	/* reset the bonus flags */
	for(num = 0; num < 2; num++) {
		threeShotActive[num] = 0;
		guidedActive[num] = 0;
		guidedSteering[num] = 0;
	}
	oneWayActive = 0;

	/* these values determine how many ticks
	 * updateCapsules() must wait before
	 * trying to generate random capsules
	 */
	createCapsulesDelay = 1000; /* 1000 */
	createCapsulesTicks = 1; /* 1 */

	/* reset capsules */
	for(i = 0; i < 6; i++)
	{
		capsules[i].type = "capsule";
		capsules[i].alive = 0;
		capsulePosition[i] = 0;
	}

	missileOdds = 5; /* 5 out of 10000 */
	missile.type = "missile";
	missile.alive = 0;
	missile.width = 20;
	missile.height = 20;
	missile.bounds.x1 = missile.x;
	missile.bounds.y2 = missile.y;
	missile.bounds.x2 = missile.x + missile.width;
	missile.bounds.y2 = missile.y + missile.height;

	impactRadius = 50;
	impact.width = impactRadius;
	impact.height = impactRadius;
	impact.type = "impact";
	impact.alive = 0;
	impactTicks = 0;

	if(impact_bmp == NULL)
	  impact_bmp = create_bitmap(impact.width, impact.height);
	clear_to_color(impact_bmp, makecol(255, 0, 255));
	
	circle(impact_bmp, impact.width / 2, impact.height / 2,
		impactRadius, makecol(255, 255, 0) );
	circle(impact_bmp, impact.width / 2, impact.height / 2,
		(impactRadius / 3.0) * 2, makecol(255, 255, 0) );
	circle(impact_bmp, impact.width / 2, impact.height / 2,
		impactRadius / 2.0, makecol(255, 255, 0) );
	circle(impact_bmp, impact.width / 2, impact.height / 2,
		(impactRadius / 3.0), makecol(255, 255, 0) );

	/* set up invisible invader bonus */
	hordeVisible = 1;	/* game begins with horde visible */
	invisibleActive = 0;
}

void setCapsuleTrajectory(int capsuleNumber)
{
	int x1, x2, y1, y2, cx1, cx2, cy1, cy2;
	int pts[8];

	x1 = rand() % 640;
	y1 = 0;
	x2 = rand() % 540;
	x2 += 50;
	y2 = screen_baseline;

	if(x2 < x1)
	{
		cx1 = x1 - (rand() % 100);
		cx2 = x2 - (rand() % 100);
		cy1 = screen_baseline / 4;
		cy2 = (screen_baseline / 2) + cy1;
	}
	else
	{
		cx1 = x1 + (rand() % 100);
		cx2 = x2 + (rand() % 100);
		cy1 = 120;
		cy2 = 360;
	}

	pts[0] = x1;
	pts[1] = y1;
	pts[2] = cx1;
	pts[3] = cy1;
	pts[4] = cx2;
	pts[5] = cy2;
	pts[6] = x2;
	pts[7] = y2;

	calc_spline(pts, 200,
		capsuleTrajectoryX[capsuleNumber],
		capsuleTrajectoryY[capsuleNumber]
		);
}

int getCapsuleOdds(void)
{
	return capsuleOdds;
}

void setCapsuleOdds(int newOdds)
{
	capsuleOdds = newOdds;
}

void incrementAvailableCapsules(void)
{
	if((numAvailableCapsules += 0.2) > 6)
		numAvailableCapsules = 6;
}

void incrementCapsulePosition(int capsuleNumber)
{
	if((capsuleNumber >=0) && (capsuleNumber<=5))
	{
		if(++capsulePosition[capsuleNumber] > 199)
		{
			capsulePosition[capsuleNumber] = 0;
			capsules[capsuleNumber].alive = 0;
		}
	}
}

void updateCapsules(void)
{
	int numCaps, i;
	SPRITE *currCap;

	/* determine the number of active capsules */
	numCaps = 0;
	for(i = 0; i < 6; i++)
	{
		if(capsules[i].alive)
			numCaps++;
	}

	for(i = 0; i < 6; i++)
	{
		/* update the capsule */
		if(capsules[i].alive)
		{
			currCap = &capsules[i];
			currCap->x =
			  capsuleTrajectoryX[i][capsulePosition[i]] -
			  (currCap->width / 2);
			currCap->y =
			  capsuleTrajectoryY[i][capsulePosition[i]] -
			  (currCap->height / 2);
			currCap->bounds.x1 = currCap->x - 1;
			currCap->bounds.x2 = currCap->x + currCap->width + 1;
			/* currCap->bounds.y1 = currCap->y - 4; */
			currCap->bounds.y1 = currCap->y - 1;
			currCap->bounds.y2 = currCap->y + currCap->height + 1;
		}
		else
		{
			/* create capsule if odds permit */
			unsigned long delayFactor = createCapsulesDelay
			  - (level * 75);
			if(delayFactor < 1)
			  delayFactor = 1;
			if( ((rand() % 100) <= capsuleOdds) &&
				(numCaps < (int)numAvailableCapsules)&&
				!(createCapsulesTicks % delayFactor) )
			{
				/* reset the ticks */
				resetCapsuleTicks();
				numCaps++;
				currCap = &capsules[i];
				setCapsuleTrajectory(i);
				/* set alive value to current position
				 * + 1, so that shield collision test
				 * can use that value to determine
				 * which capsule it is working with
				 */
				currCap->alive = i+1;
				currCap->width = 20;
				currCap->height = 20;
				currCap->x =
				  capsuleTrajectoryX[i][0] -
				  (currCap->width / 2);
				currCap->y =
				  capsuleTrajectoryY[i][0] -
				  (currCap->height / 2);
				capsulePosition[i] = 0;
				currCap->bounds.x1 = currCap->x;
				currCap->bounds.x2 = currCap->x +
				  currCap->width;
				currCap->bounds.y1 = currCap->y;
				currCap->bounds.y2 = currCap->y +
				  currCap->height;
			}
		}
	}
	/* increment capsule ticks each time updateCapsules() called */
	createCapsulesTicks++;
}

int getCapsulePosition(int capsuleNumber)
{
	if((capsuleNumber>=0)&&(capsuleNumber<=5))
	  return capsulePosition[capsuleNumber];
	else
	  return -1;
}

int getCapsuleXValue(int capsuleNumber, int position)
{
	if((capsuleNumber>=0)&&(capsuleNumber<=5)&&
	  (position>=0)&&(position<=199))
	  return capsuleTrajectoryX[capsuleNumber][position];
	else
	  return -1;
}

int getCapsuleYValue(int capsuleNumber, int position)
{
	if((capsuleNumber>=0)&&(capsuleNumber<=5)&&
	  (position>=0)&&(position<=199))
	  return capsuleTrajectoryY[capsuleNumber][position];
	else
	  return -1;
}

void resetCapsuleTicks(void)
{
	createCapsulesTicks = 1;
}

void setMissileTrajectory(void)
{
	int i;
	int x1, x2, y1, y2;
	float slope, yfactor;
	float angle, pi;

	BITMAP *temp_bmp;

	x1 = rand() % 640;
	y1 = 0;
	x2 = rand() % 540;
	x2 += 50;
	y2 = screen_baseline;

	pi = 3.14;

	slope = ((float)y2 - (float)y1) / ((float)x2 - (float)x1);
	/* avoid division by zero */
	if(slope == 0.0000000)
		slope = 0.001;
	yfactor = ((float)y2 - (float)y1) / 200;

	/* calculate the x,y points between (x1,y1) and (x2,y2) */
	for(i = 0; i < 200; i++)
	{
		missileTrajectoryY[i] = (y2 = (i * yfactor));
		missileTrajectoryX[i] = ( ((y1 - y2)/slope) - x1) * -1;
	}

	/* rotate the missile bitmap to current angle */
	if(slope < 0)
		angle = atan2(y2-y1,x2-x1);
	if(slope > 0)
		angle = atan2(y2-y1,x2-x1);

	angle = angle * (180/pi);

	angle *= (256.0/360.0);

	temp_bmp = create_bitmap(missile_bmp->w, missile_bmp->h);

	clear_to_color(temp_bmp, makecol(255, 0, 255));

	rotate_sprite(temp_bmp, missile_bmp, 0, 0,
		itofix(angle)
		);

	clear_to_color(rotatedMissile_bmp, makecol(255, 0, 255));

	draw_sprite(rotatedMissile_bmp, temp_bmp, 0, 0);

	destroy_bitmap(temp_bmp);
}

void activateMissile(void)
{
	/** bring invader missile to life 
	 **
	 ** This will reset a missile if it is already active.
	 */
	missile.alive = 1;
	setMissileTrajectory();
	currentMissilePosition = 0;

	/* play the missile alert sample */
	if(soundon)
		play_sample(missileAlertSample,
			volume, pan, pitch, FALSE
			);

}

void updateMissile(void)
{
	static int missileTicks = 0;
	static int missileDelay = 5;

	if(!missile.alive)
	{
		if( (rand() % 10000) < missileOdds )
		{
			activateMissile();
		}
		else
			return;
	}
	missile.x = missileTrajectoryX[currentMissilePosition] -
		(missile.width / 2);
	missile.y = missileTrajectoryY[currentMissilePosition] -
		(missile.height / 2);

	/* set missile bounds */
	missile.bounds.x1 = missile.x;
	missile.bounds.y1 = missile.y;
	missile.bounds.x2 = missile.x + missile.width;
	missile.bounds.y2 = missile.y + missile.height;

	if(!(missileTicks % missileDelay))
	{
	  if(++currentMissilePosition > 199)
	  {
		missile.alive = 0;
		currentMissilePosition = 199;

		/* missile hit ground; activate bonus */
		activateInvaderBonus();
	  }
	}

	missileTicks++;
}

int activateThreeShot(int playerNumber)
{
	int num;
	num = playerNumber;

	threeShotActive[num] = 1000;

	return 0;
}

int updateThreeShot(void)
{
	int num;

	for(num = 0; num < numberOfPlayers; num++) {
		if(--threeShotActive[num] < 0)
		  threeShotActive[num] = 0;
	}

	return 0;
}

int activateGuided(int playerNumber)
{
	guidedActive[playerNumber] = 1000;

	return 0;
}

int updateGuided(void)
{
	int num;

	for(num = 0; num < numberOfPlayers; num++) {
		if(--guidedActive[num] < 0)
		  guidedActive[num] = 0;
	}

	return 0;
}

int setGuidedSteering(int playerNumber, int value)
{
	guidedSteering[playerNumber] = value;

	return guidedSteering[playerNumber];
}

int activateBonus(int playerNumber)
{
	int num;
	num = playerNumber;

	switch(rand() % 3)
	{
		case 0:
		  activateThreeShot(num);
		  break;
		case 1:
		  activateGuided(num);
		  break;
		case 2:
		  activateOneWay();
		  break;
	}

	return 0;
}

int activateInvaderBonus(void)
{
	switch(rand() % 2)
	{
		case 0:
		  activateImpact();
		  break;
		case 1:
		  activateInvisible();
		  break;
	}

	return 0;
}

int activateImpact(void)
{
	impactTicks = 100;
	impact.alive = 1;

	impact.x = getMissileXValue(getMissilePosition()) -
		(impact.width / 2);
	impact.y = getMissileYValue(getMissilePosition()) -
		(impact.height / 2);

	impact.bounds.x1 = impact.x;
	impact.bounds.y1 = impact.y;
	impact.bounds.x2 = impact.bounds.x1 + impact.width;
	impact.bounds.y2 = impact.bounds.y1 + impact.height;

	/* play impact blast sample */
	if(soundon)
		play_sample(missileBlastSample,
			volume, pan, pitch, FALSE);

	return 0;
}

int updateImpact(void)
{
	if(impact.alive)
	{
		if(--impactTicks <=0)
		  impact.alive = 0;
	}
	
	return 0;
}

int activateOneWay(void)
{
	oneWayActive = 1000;
	return 0;
}

int disableOneWay(void)
{
	oneWayActive = 0;
	return 0;
}

int updateOneWay(void)
{
	if(--oneWayActive < 0)
	  oneWayActive = 0;

	return oneWayActive;
}

int getMissilePosition(void)
{
	return currentMissilePosition;
}

int getMissileXValue(int position)
{
	return missileTrajectoryX[position];
}

int getMissileYValue(int position)
{
	return missileTrajectoryY[position];
}

/* invisible invader bonus functionality */
int activateInvisible(void)
{
	hordeVisible = 0;
	invisibleActive = 1000;	/* hardcoded for now */
	return 0;
}

int updateInvisible(void)
{
	/* tick down until bonus ends */
	invisibleActive--;
	if(invisibleActive <= 0)
	{
		invisibleActive = 0;
		hordeVisible = 1;
	}
	else
	{
		/* since invisible is active, tick down temporary visibility */
		hordeVisible--;
		if(hordeVisible < 0)
			hordeVisible = 0;
	}
	
	/* return state of invisible bonus in case it is needed */
	return invisibleActive;
}

int activateReveal(void)
{
	/* Start temporary visibility countdown if invisible bonus active.
	**
	** Since this function checks for the status of the invisible bonus,
	** there is no need to check the status before calling this function.
	*/
	if(invisibleActive)
		hordeVisible = 10;	/* hardcoded for now */	

	return hordeVisible;	/* return state of visiblity in case needed */
}

int getThreeShotStatus(playerNumber)
{
	return threeShotActive[playerNumber];
}

int getGuidedStatus(int playerNumber) {
	return guidedActive[playerNumber];
}

int getGuidedSteeringStatus(int playerNumber) {
	return guidedSteering[playerNumber];
}

int getOneWayStatus(void) {
	return oneWayActive;
}
