#include "../include/main.h"

/* global vars */
BITMAP *backbuffer = NULL;

int fps = 0;
static volatile int ticks = 0;
static volatile int game_time = 0;

enum GAME_STATE game_state;

int qtd_enemies = 0;
BITMAP **bmp_enemies_images[NUM_ENEMIES_TYPE];

// coordenates / point to go
POINT positions[PST_MAX_POSITIONS];

// y offset in pixels 
int yoffset = BOTTOM;
int stop_scroller = FALSE;
int endOfLevel = FALSE;

// general vars
static int map_width, map_height;		/* map width and height in blocks */

// bitmaps and sprites
BITMAP *bmp_title_screen_image;
BITMAP *bmp_game_over_image;
BITMAP *bmp_end_game_image;

BITMAP *explosion_images[6];
SPRITE *explosions[MAX_EXPLOSIONS];

BITMAP *big_explosion_images[7];
SPRITE *big_explosion;

BITMAP *bmp_player_images[3];
BITMAP *bmp_bullets_images[NUM_BULLETS_TYPE];
BITMAP *progress, *bar;

/* upgrades bonus items */
BITMAP *bonus_shot_image;
SPRITE *bonus_shot;

BITMAP *bonus_speed_image;
SPRITE *bonus_speed;

BITMAP *bonus_shot_power_image;
SPRITE *bonus_shot_power;

BITMAP *bonus_shot_double_image;
SPRITE *bonus_shot_double;

BITMAP *bonus_shot_triple_image;
SPRITE *bonus_shot_triple;

/* sounds vars */
MIDI *mid_sounds[SND_MAX_MIDIS];
SAMPLE *wav_sounds[SND_MAX_WAVES];

/* functions */
static int closed = FALSE;
static void closeHook()
{
	closed = TRUE;
}
END_OF_FUNCTION(closeHook);

void ticker()
{
	ticks++;
}
END_OF_FUNCTION(ticker);

void gameTimeTicker()
{
	game_time++;
}
END_OF_FUNCTION(gameTimeTicker);


/******************************
-- Inicio do Programa
*******************************/
int main(int argc, char **argv)
{   
	set_uformat(U_ASCII);

	if (allegro_init() != 0) { exit(1); }
	install_keyboard();
	install_joystick(JOY_TYPE_AUTODETECT);    
	install_timer();

	if (ENABLE_MOUSE_SUPPORT) { install_mouse(); }

	set_color_depth(G_COL_DEPTH);
	if (set_gfx_mode(G_MODE, G_WIDTH, G_HEIGHT, 0, 0) != 0)
	{
		fatalError("NAO FOI POSSIVEL INICIAR O MODO GRAFICO\n");
	}

	if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != 0)
	{
		fatalError("NAO FOI POSSIVEL INICIAR O SOM\n");
	}

	set_window_title(G_GAME_TITLE);
	srand((unsigned)time(NULL));

	backbuffer = create_bitmap(G_WIDTH, G_HEIGHT);
	clear_to_color(backbuffer, BLACK);

	/* setup the game */
	if (!setupGame())
	{
		fatalError("O jogo nao pode iniciar. Nao eh possivel continuar!");
	}

	runGame();
	shutdownGame();
	exitGame();

	return 0;
}
END_OF_MAIN();

/* GS_MAIN_MENU */
void updateMainMenu(void)
{   
	int option_exit = main_menu_item_count;

	if (keyboard_needs_poll())
		poll_keyboard();

	if (num_joysticks > 0)
		poll_joystick();

	if (count_press_key) count_press_key--;

	if (!count_press_key)
	{
		if (keypressed())
		{
			int c = readkey();
			clear_keybuf();

			switch (c >> 8)
			{
			case KEY_ESC:
				if (menu_choice == option_exit)
					game_state = GS_EXIT;
				else
					menu_choice = option_exit;
				count_press_key = 20;
				break;		    
			case KEY_UP:		
				menu_choice--;
				if (menu_choice < 1) menu_choice = option_exit;
				count_press_key = 20;
				play_sample(wav_sounds[SND_WAV_CURSOR_SELECTION], 255, 128, 1000, FALSE);
				break;
			case KEY_DOWN:		
				menu_choice++;
				if (menu_choice > option_exit) menu_choice = 1;
				count_press_key = 20;
				play_sample(wav_sounds[SND_WAV_CURSOR_SELECTION], 255, 128, 1000, FALSE);
				break;
			case KEY_SPACE:
			case KEY_ENTER:
			case KEY_ENTER_PAD:
				switch (menu_choice)
				{
				case 1:		                    
					newGame(1);
					play_sample(wav_sounds[SND_WAV_MENU_SELECT], 255, 128, 1000, FALSE);
					break;
				case 2:	                
					game_state = GS_CREDITS;
					play_sample(wav_sounds[SND_WAV_MENU_SELECT], 255, 128, 1000, FALSE);
					break;
				case 3:
					game_state = GS_EXIT;
					play_sample(wav_sounds[SND_WAV_MENU_SELECT], 255, 128, 1000, FALSE);
					break;
				}
				break;
			}		
		}
	}

	if (!key[KEY_SPACE] && !key[KEY_ENTER] && !key[KEY_ESC] && !key[KEY_UP] && !key[KEY_DOWN])
		count_press_key = 0;	
}

void drawMainMenu(BITMAP *bmp)
{
	int i;
	int y1 = 320;
	int row_height = 30;
	int xCenter = bmp->w / 2;

	draw_sprite(bmp, bmp_title_screen_image, 0, 0);    
	drawRectangleDefault(bmp, xCenter - 100, y1 - 20, xCenter + 100, y1 + (main_menu_item_count * row_height), 3);

	for (i = 0; i < main_menu_item_count; i++)
	{	
		if (menu_choice == (i + 1))
			textout_centre_ex(bmp, font, main_menu_options[i], xCenter, y1 + (i * row_height), YELLOW, -1);
		else
			textout_centre_ex(bmp, font, main_menu_options[i], xCenter, y1 + (i * row_height), WHITE, -1);
	}
}

/* GS_CREDITS */
void updateCredits(void)
{   
	if (keyboard_needs_poll())
		poll_keyboard();

	if (num_joysticks > 0)
		poll_joystick();

	if (keypressed())
	{
		int c = readkey();
		clear_keybuf();

		switch (c >> 8)
		{
		case KEY_ESC:	            
			mainMenu();
			break;		        
		}
	}
}

void drawCredits(BITMAP *bmp)
{
	int y1 = 330;
	int row_height = 30;
	int xCenter = bmp->w / 2;

	draw_sprite(bmp, bmp_title_screen_image, 0, 0);
	drawRectangleDefault(bmp, xCenter - 200, y1 - 20, xCenter + 200, y1 + (row_height * 3), 3);

	textout_centre_ex(bmp, font, "Created by: Leandro A. Dos Anjos", bmp->w / 2, y1, WHITE, -1);	
	textout_centre_ex(bmp, font, "SI - Faccamp (2012)", bmp->w / 2, y1 + row_height, WHITE, -1);
	textout_centre_ex(bmp, font, "Press [ESC] to back Main Menu", bmp->w / 2, y1 + (row_height * 2), WHITE, -1);	
}

/* GS_NEW_GAME */
void updateNewGame(void)
{
	/* close the game */
	if (keypressed())
	{
		int c = readkey();
		clear_keybuf();

		switch (c >> 8)
		{
		case KEY_ESC:
			pausedGame();            
			break;		        
		}		
	}

	getInput();
	getJoystick();
	updateScroller(TRUE);
	updatePlayer();
	updateEnemies();
	updateBullets();            
	updateExplosions();
	updateBonuses();
}

void drawNewGame(BITMAP *bmp)
{   
	drawScroller(bmp);	
	drawEnemies(bmp);

	drawPlayer(bmp);
	drawBullets(bmp);
	drawExplosions(bmp);
	drawBonuses(bmp);

	drawPlayerInfo(bmp, player.health);

#ifdef _DEBUG
	drawDebugInfo(bmp);
#endif
}

/* GS_PAUSED_GAME */
void updatePausedGame(void)
{   
	if (keypressed())
	{
		int c = readkey();
		clear_keybuf();

		switch (c >> 8)
		{
		case KEY_ESC:
			mainMenu();
			break;
		case KEY_SPACE:
			newGame(FALSE);
			break;
		}		
	}
}

void drawPausedGame(BITMAP *bmp)
{
	int y1 = 300;
	int row_height = 30;
	int xCenter = bmp->w / 2;
	
	draw_sprite(bmp, bmp_title_screen_image, 0, 0);
	drawRectangleDefault(bmp, xCenter - 200, y1 - 20, xCenter + 200, y1 + (row_height * 3), 3);
	
	textout_centre_ex(bmp, font, "GAME PAUSED", bmp->w / 2, y1, WHITE, -1);
	textout_centre_ex(bmp, font, "Press [SPACE] to continue", bmp->w / 2, y1 + row_height, WHITE, -1);
	textout_centre_ex(bmp, font, "Press [ESC] to go Main Menu", bmp->w / 2, y1 + (row_height * 2), WHITE, -1);	
}

/* GS_GAME_OVER */
void updateGameOver(void)
{
	/* close the game */
	if (keypressed())
	{
		int c = readkey();
		clear_keybuf();

		switch (c >> 8)
		{
		case KEY_ESC:	            
			mainMenu();		            
			break;		        
		}		
	}
}

void drawGameOver(BITMAP *bmp)
{
	int y1 = 400;
	int row_height = 30;
	int xCenter = bmp->w / 2;

	draw_sprite(bmp, bmp_game_over_image, 0, 0);
	drawRectangleDefault(bmp, xCenter - 200, y1 - 20, xCenter + 200, y1 + (row_height * 2), 3);

	textprintf_centre_ex(bmp, font, bmp->w / 2, y1, WHITE, -1, "SCORE: %d", player.score);    
	textout_centre_ex(bmp, font, "Press [ESC] to continue", bmp->w / 2, y1 + row_height, WHITE, -1);	
}


/* GS_END_GAME */
void updateEndGame(void)
{   
	if (keypressed())
	{
		int c = readkey();
		clear_keybuf();

		switch (c >> 8)
		{
		case KEY_ESC:	            
			mainMenu();		            
			break;		        
		}		
	}
}

void drawEndGame(BITMAP *bmp)
{
	int y1 = 400;
	int row_height = 30;
	int xCenter = bmp->w / 2;

	draw_sprite(bmp, bmp_end_game_image, 0, 0);
	drawRectangleDefault(bmp, xCenter - 200, y1 - 20, xCenter + 200, y1 + (row_height * 2), 3);

	textprintf_centre_ex(bmp, font, bmp->w / 2, y1, WHITE, -1, "SCORE: %d", player.score);    
	textout_centre_ex(bmp, font, "Press [ESC] to continue", bmp->w / 2, y1 + row_height, WHITE, -1);	
}

/* General Functions */

// starts the mainMenu
void mainMenu()
{
	game_state = GS_MAIN_MENU;

	stop_midi();	
	play_midi(mid_sounds[SND_MID_TITLE], TRUE);
}

// starts a new game
void newGame(int reset_data)
{
	game_state = GS_NEW_GAME;
	
	if (reset_data) {
		// y offset in pixels 
		yoffset = BOTTOM;        
		endOfLevel = FALSE;
		stop_scroller = FALSE;    

		initPlayer();
		initEnemies();

		initPositions();	
		initBullets();
		initExplosion();
		initBonuses();

		stop_midi();
		play_midi(mid_sounds[SND_MID_LEGENDARY_WINGS], TRUE);
	}
}

// starts a game pause
void pausedGame()
{
	game_state = GS_PAUSED_GAME;

	play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
}

// starts the game over
void gameOver()
{
	game_state = GS_GAME_OVER;

	stop_midi();
	play_midi(mid_sounds[SND_MID_GAME_OVER], FALSE);
}

// starts the game end
void endGame()
{
	game_state = GS_END_GAME;

	stop_midi();
	play_midi(mid_sounds[SND_MID_GAME_OVER], FALSE);
}

// initializers
void initPlayer(void)
{
	// initialize player
	player.lives = 2; // do not used
	player.score = 0;
	player.health = 25;
	player.bullet_speed = -3;
	player.move_speed = 3;
	player.fire_count = 0;
	player.fire_delay = 30;
	player.shot_power = 1;
	player.bullet_type = SPR_BULLET_PLAYER_SINGLE;

	player.sprite = (SPRITE *)malloc(sizeof(SPRITE));
	player.sprite->x = 320-32;
	player.sprite->y = 400;
	player.sprite->width = bmp_player_images[0]->w;
	player.sprite->height = bmp_player_images[0]->h;
	player.sprite->xdelay = 1;
	player.sprite->ydelay = 0;
	player.sprite->xcount = 0;
	player.sprite->ycount = 0;
	player.sprite->xspeed = 0;
	player.sprite->yspeed = 0;
	player.sprite->curframe = 0;
	player.sprite->maxframe = 2;
	player.sprite->framecount = 0;
	player.sprite->framedelay = 10;
	player.sprite->animdir = 1;
}

void initEnemies(void)
{
	int i, j;
	int n = 0;
	int enemy_type;

	if (qtd_enemies > 0 && switchLayer(1) == 1)
	{
		for (j = 0; j < map_height; j++)
		{
			for (i = 0; i < map_width; i++)
			{
				enemy_type = getMapSprite(i, j);
				switch (enemy_type)
				{
				case SPR_ENEMY_1:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 130;
						enemies[n].health = 1;
						enemies[n].shot_power = 1;
						enemies[n].bullet_speed = 3;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SPEED;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = 0;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_2:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 180;
						enemies[n].health = 2;
						enemies[n].shot_power = 2;
						enemies[n].bullet_speed = 2;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = 0;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_3:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 100;
						enemies[n].health = 3;
						enemies[n].shot_power = 1;
						enemies[n].bullet_speed = 2;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_POWER;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = 0;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_4:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 100;
						enemies[n].health = 4;
						enemies[n].shot_power = 0;
						enemies[n].bullet_speed = 0;							
						enemies[n].can_fire = FALSE;
						enemies[n].bonus = BI_SHOT_DOUBLE;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = 0;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_BOSS_1:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 110;
						enemies[n].health = 90;
						enemies[n].shot_power = 5;
						enemies[n].bullet_speed = 6;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_TRIPLE;
						enemies[n].is_boss = TRUE;
						enemies[n].move_speed = 3;
						enemies[n].bullet_type = SPR_BULLET_BOSS_SIMPLE;

						enemies[n].point_to_go = &positions[3];

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = enemies[n].move_speed;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_BOSS_2:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 50;
						enemies[n].health = 150;
						enemies[n].shot_power = 2;
						enemies[n].bullet_speed = 7;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_TRIPLE;
						enemies[n].is_boss = TRUE;
						enemies[n].move_speed = 6;
						enemies[n].bullet_type = SPR_BULLET_BOSS_SIMPLE;

						enemies[n].point_to_go = &positions[3];

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = enemies[n].move_speed;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_BOSS_3:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 70;
						enemies[n].health = 300;
						enemies[n].shot_power = 3;
						enemies[n].bullet_speed = 6;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_TRIPLE;
						enemies[n].is_boss = TRUE;
						enemies[n].move_speed = 5;
						enemies[n].bullet_type = SPR_BULLET_BOSS_3_SIMPLE;

						enemies[n].point_to_go = &positions[3];

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = enemies[n].move_speed;
						enemies[n].sprite->yspeed = enemies[n].move_speed;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_SUBMARINE_LEFT:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 80;
						enemies[n].health = 20;
						enemies[n].shot_power = 3;
						enemies[n].bullet_speed = 3;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_DOUBLE;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = enemies[n].move_speed;
						enemies[n].sprite->yspeed = 0;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				case SPR_ENEMY_SUBMARINE_RIGHT:
					if (n < qtd_enemies)
					{
						enemies[n].type = enemy_type;
						enemies[n].fire_count = 0;
						enemies[n].fire_delay = 100;
						enemies[n].health = 15;
						enemies[n].shot_power = 3;
						enemies[n].bullet_speed = 3;							
						enemies[n].can_fire = TRUE;
						enemies[n].bonus = BI_SHOT_DOUBLE;
						enemies[n].is_boss = FALSE;
						enemies[n].move_speed = 1;
						enemies[n].bullet_type = SPR_BULLET_ENEMY_SIMPLE;

						enemies[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));                        
						enemies[n].sprite->alive = TRUE;
						enemies[n].sprite->x = i * 32;
						enemies[n].sprite->y = j * 32;
						enemies[n].sprite->width = bmp_enemies_images[enemy_type - 1][0]->w;
						enemies[n].sprite->height = bmp_enemies_images[enemy_type - 1][0]->h;
						enemies[n].sprite->xdelay = 4;
						enemies[n].sprite->ydelay = 4;
						enemies[n].sprite->xcount = 0;
						enemies[n].sprite->ycount = 0;
						enemies[n].sprite->xspeed = enemies[n].move_speed * (-1);
						enemies[n].sprite->yspeed = 0;
						enemies[n].sprite->curframe = 0;
						enemies[n].sprite->maxframe = 2;
						enemies[n].sprite->framecount = 0;
						enemies[n].sprite->framedelay = 10;
						enemies[n].sprite->animdir = 1;

						n++;
					}
					break;
				}
			}
		}
		// back to background layer
		switchLayer(0);
	}
}

void initBullets(void)
{
	int n;

	// initialize player_bullets
	for (n = 0; n < MAX_PLAYER_BULLETS; n++)
	{
		player_bullets[n].damage = 1;

		player_bullets[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));
		player_bullets[n].sprite->alive = FALSE;
		player_bullets[n].sprite->x = 0;
		player_bullets[n].sprite->y = 0;
		player_bullets[n].sprite->width = bmp_bullets_images[0]->w;
		player_bullets[n].sprite->height = bmp_bullets_images[0]->h;
		player_bullets[n].sprite->xdelay = 0;
		player_bullets[n].sprite->ydelay = 0;
		player_bullets[n].sprite->xcount = 0;
		player_bullets[n].sprite->ycount = 0;
		player_bullets[n].sprite->xspeed = 0;
		player_bullets[n].sprite->yspeed = -player.bullet_speed;
		player_bullets[n].sprite->curframe = 0;
		player_bullets[n].sprite->maxframe = 0;
		player_bullets[n].sprite->framecount = 0;
		player_bullets[n].sprite->framedelay = 0;
		player_bullets[n].sprite->animdir = 0;
	}

	// initialize enemies_bullets
	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
	{
		enemies_bullets[n].damage = 1;

		enemies_bullets[n].sprite = (SPRITE *) malloc(sizeof(SPRITE));
		enemies_bullets[n].sprite->alive = FALSE;
		enemies_bullets[n].sprite->x = 0;
		enemies_bullets[n].sprite->y = 0;
		enemies_bullets[n].sprite->width = bmp_bullets_images[0]->w;
		enemies_bullets[n].sprite->height = bmp_bullets_images[0]->h;
		enemies_bullets[n].sprite->xdelay = 0;
		enemies_bullets[n].sprite->ydelay = 0;
		enemies_bullets[n].sprite->xcount = 0;
		enemies_bullets[n].sprite->ycount = 0;
		enemies_bullets[n].sprite->xspeed = 0;
		enemies_bullets[n].sprite->yspeed = 2;
		enemies_bullets[n].sprite->curframe = 0;
		enemies_bullets[n].sprite->maxframe = 0;
		enemies_bullets[n].sprite->framecount = 0;
		enemies_bullets[n].sprite->framedelay = 0;
		enemies_bullets[n].sprite->animdir = 0;
	}
}

void initExplosion(void)
{
	int n;

	// initialize explosions
	for (n = 0; n < MAX_EXPLOSIONS; n++)
	{
		explosions[n] = (SPRITE *)malloc(sizeof(SPRITE));
		explosions[n]->alive = FALSE;
		explosions[n]->x = 0;
		explosions[n]->y = 0;
		explosions[n]->width = explosion_images[0]->w;
		explosions[n]->height = explosion_images[0]->h;
		explosions[n]->xdelay = 0;
		explosions[n]->ydelay = 8;
		explosions[n]->xcount = 0;
		explosions[n]->ycount = 0;
		explosions[n]->xspeed = 0;
		explosions[n]->yspeed = -1;
		explosions[n]->curframe = 0;
		explosions[n]->maxframe = 5;
		explosions[n]->framecount = 0;
		explosions[n]->framedelay = 15;
		explosions[n]->animdir = 1;
	}

	// initialize big explosions
	big_explosion = (SPRITE *)malloc(sizeof(SPRITE));
	big_explosion->alive = FALSE;
	big_explosion->x = 0;
	big_explosion->y = 0;
	big_explosion->width = big_explosion_images[0]->w;
	big_explosion->height = big_explosion_images[0]->h;
	big_explosion->xdelay = 0;
	big_explosion->ydelay = 8;
	big_explosion->xcount = 0;
	big_explosion->ycount = 0;
	big_explosion->xspeed = 0;
	big_explosion->yspeed = -1;
	big_explosion->curframe = 0;
	big_explosion->maxframe = 6;
	big_explosion->framecount = 0;
	big_explosion->framedelay = 10;
	big_explosion->animdir = 1;
}

void initBonuses(void)
{
	// initialize bonus shot
	bonus_shot = (SPRITE *) malloc(sizeof(SPRITE));
	bonus_shot->alive = FALSE;
	bonus_shot->x = 0;
	bonus_shot->y = 0;
	bonus_shot->width = bonus_shot_image->w;
	bonus_shot->height = bonus_shot_image->h;
	bonus_shot->xdelay = 0;
	bonus_shot->ydelay = 2;
	bonus_shot->xcount = 0;
	bonus_shot->ycount = 0;
	bonus_shot->xspeed = 0;
	bonus_shot->yspeed = 2;
	bonus_shot->curframe = 0;
	bonus_shot->maxframe = 0;
	bonus_shot->framecount = 0;
	bonus_shot->framedelay = 0;

	// initialize bonus speed
	bonus_speed = (SPRITE *) malloc(sizeof(SPRITE));
	bonus_speed->alive = FALSE;
	bonus_speed->x = 0;
	bonus_speed->y = 0;
	bonus_speed->width = bonus_speed_image->w;
	bonus_speed->height = bonus_speed_image->h;
	bonus_speed->xdelay = 0;
	bonus_speed->ydelay = 2;
	bonus_speed->xcount = 0;
	bonus_speed->ycount = 0;
	bonus_speed->xspeed = 0;
	bonus_speed->yspeed = 2;
	bonus_speed->curframe = 0;
	bonus_speed->maxframe = 0;
	bonus_speed->framecount = 0;
	bonus_speed->framedelay = 0;

	// initialize bonus speed
	bonus_shot_power = (SPRITE *) malloc(sizeof(SPRITE));
	bonus_shot_power->alive = FALSE;
	bonus_shot_power->x = 0;
	bonus_shot_power->y = 0;
	bonus_shot_power->width = bonus_shot_power_image->w;
	bonus_shot_power->height = bonus_shot_power_image->h;
	bonus_shot_power->xdelay = 0;
	bonus_shot_power->ydelay = 2;
	bonus_shot_power->xcount = 0;
	bonus_shot_power->ycount = 0;
	bonus_shot_power->xspeed = 0;
	bonus_shot_power->yspeed = 2;
	bonus_shot_power->curframe = 0;
	bonus_shot_power->maxframe = 0;
	bonus_shot_power->framecount = 0;
	bonus_shot_power->framedelay = 0;

	// initialize bonus shot double
	bonus_shot_double = (SPRITE *) malloc(sizeof(SPRITE));
	bonus_shot_double->alive = FALSE;
	bonus_shot_double->x = 0;
	bonus_shot_double->y = 0;
	bonus_shot_double->width = bonus_shot_double_image->w;
	bonus_shot_double->height = bonus_shot_double_image->h;
	bonus_shot_double->xdelay = 0;
	bonus_shot_double->ydelay = 2;
	bonus_shot_double->xcount = 0;
	bonus_shot_double->ycount = 0;
	bonus_shot_double->xspeed = 0;
	bonus_shot_double->yspeed = 5;
	bonus_shot_double->curframe = 0;
	bonus_shot_double->maxframe = 0;
	bonus_shot_double->framecount = 0;
	bonus_shot_double->framedelay = 0;

	// initialize bonus shot triple
	bonus_shot_triple = (SPRITE *) malloc(sizeof(SPRITE));
	bonus_shot_triple->alive = FALSE;
	bonus_shot_triple->x = 0;
	bonus_shot_triple->y = 0;
	bonus_shot_triple->width = bonus_shot_triple_image->w;
	bonus_shot_triple->height = bonus_shot_triple_image->h;
	bonus_shot_triple->xdelay = 0;
	bonus_shot_triple->ydelay = 2;
	bonus_shot_triple->xcount = 0;
	bonus_shot_triple->ycount = 0;
	bonus_shot_triple->xspeed = 0;
	bonus_shot_triple->yspeed = 5;
	bonus_shot_triple->curframe = 0;
	bonus_shot_triple->maxframe = 0;
	bonus_shot_triple->framecount = 0;
	bonus_shot_triple->framedelay = 0;
}

void startExplosion(int x, int y)
{
	int n;	

	for (n = 0; n < MAX_EXPLOSIONS; n++)
	{
		if (!explosions[n]->alive)
		{
			explosions[n]->alive = TRUE;
			explosions[n]->x = x;
			explosions[n]->y = y;
			play_sample(wav_sounds[SND_WAV_EXPLOSION_1], 100, 128, 1000, FALSE);
			break;
		}
	}
}

void updateExplosions(void)
{
	int n;

	for (n = 0; n < MAX_EXPLOSIONS; n++)
	{
		if (explosions[n]->alive)
		{   
			updateSprite(explosions[n]);

			if (explosions[n]->curframe >= explosions[n]->maxframe)
			{
				explosions[n]->curframe = 0;
				explosions[n]->alive = FALSE;
			}
		}
	}

	// update the big "player" explosion if needed
	if (big_explosion->alive)
	{
		updateSprite(big_explosion);

		if (big_explosion->curframe >= big_explosion->maxframe)
		{
			big_explosion->curframe = 0;
			big_explosion->alive = FALSE;
		}
	}
}

void drawExplosions(BITMAP *bmp)
{
	int n;

	for (n = 0; n < MAX_EXPLOSIONS; n++)
	{
		if (explosions[n]->alive)
		{
			draw_sprite(bmp, explosion_images[explosions[n]->curframe], explosions[n]->x, explosions[n]->y);			
		}
	}

	// draw the big "player" explosion if needed
	if (big_explosion->alive)
	{	
		draw_sprite(bmp, big_explosion_images[big_explosion->curframe], big_explosion->x, big_explosion->y);		
	}
}

/* miscelaneas  functions */
void drawRectangleDefault(BITMAP *bmp, int x1, int y1, int x2, int y2, int border)
{
	rectfill(bmp, x1 - border, y1 - border, x2 + border, y2 + border, BLACK);
	rectfill(bmp, x1, y1, x2, y2, COLDGREY);	
}

/* bonus */
void createBonus(int bonus_item, int x, int y)
{
	switch (bonus_item)
	{    
	case BI_SHOT:
		// launch bonus shot if ready
		if (!bonus_shot->alive)
		{
			bonus_shot->alive = TRUE;
			bonus_shot->x = x;
			bonus_shot->y = y;
		}
		break;
	case BI_SPEED:
		// launch bonus speed if ready
		if (!bonus_speed->alive)
		{
			bonus_speed->alive = TRUE;
			bonus_speed->x = x;
			bonus_speed->y = y;
		}
		break;
	case BI_SHOT_POWER:
		// launch bonus shot power if ready
		if (!bonus_shot_power->alive)
		{
			bonus_shot_power->alive = TRUE;
			bonus_shot_power->x = x;
			bonus_shot_power->y = y;
		}
		break;
	case BI_SHOT_DOUBLE:
		// launch bonus shot double if ready
		if (!bonus_shot_double->alive)
		{
			bonus_shot_double->alive = TRUE;
			bonus_shot_double->x = x;
			bonus_shot_double->y = y;
		}
		break;
	case BI_SHOT_TRIPLE:
		// launch bonus shot triple if ready
		if (!bonus_shot_triple->alive)
		{
			bonus_shot_triple->alive = TRUE;
			bonus_shot_triple->x = x;
			bonus_shot_triple->y = y;
		}
		break;
	}
}

void updateBonuses(void)
{
	int x, y, x1, y1, x2, y2;

	// update bonus shot if alive
	if (bonus_shot->alive)
	{
		updateSprite(bonus_shot);

		if (bonus_shot->y > (bonus_shot->height + yoffset + G_HEIGHT))
			bonus_shot->alive = FALSE;
		else
		{
			// see if player got the bonus
			x = bonus_shot->x + bonus_shot->width/2;
			y = (bonus_shot->y - yoffset) + bonus_shot->height/2;
			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			if (inside(x, y, x1, y1, x2, y2))
			{
				// increase firing rate
				if (player.fire_delay > 20)
					player.fire_delay -= 2;

				player.score += 25;

				bonus_shot->alive = FALSE;
				play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
			}
		}
	}

	// update bonus speed if alive
	if (bonus_speed->alive)
	{
		updateSprite(bonus_speed);

		if (bonus_speed->y > (bonus_speed->height + yoffset + G_HEIGHT))
			bonus_speed->alive = FALSE;
		else
		{
			// see if player got the bonus
			x = bonus_speed->x + bonus_speed->width/2;
			y = (bonus_speed->y - yoffset) + bonus_speed->height/2;
			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			if (inside(x, y, x1, y1, x2, y2))
			{
				// increase move speed
				if (player.move_speed < MAX_SPEED_PLAYER)
					player.move_speed++;

				player.score += 25;

				bonus_speed->alive = FALSE;
				play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
			}
		}
	}

	// update bonus shot power if alive
	if (bonus_shot_power->alive)
	{
		updateSprite(bonus_shot_power);

		if (bonus_shot_power->y > (bonus_shot_power->height + yoffset + G_HEIGHT))
			bonus_shot_power->alive = FALSE;
		else
		{
			// see if player got the bonus
			x = bonus_shot_power->x + bonus_shot_power->width/2;
			y = (bonus_shot_power->y - yoffset) + bonus_shot_power->height/2;
			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			if (inside(x, y, x1, y1, x2, y2))
			{
				// increase firing power
				switch (player.bullet_type)
				{
				case SPR_BULLET_PLAYER_SINGLE:
					if (player.shot_power < MAX_SHOT_POWER_PLAYER - 3)
						player.shot_power++;
					break;
				case SPR_BULLET_PLAYER_DOUBLE:
					if (player.shot_power < MAX_SHOT_POWER_PLAYER - 2)
						player.shot_power++;
					break;
				case SPR_BULLET_PLAYER_TRIPLE:
					if (player.shot_power < MAX_SHOT_POWER_PLAYER)
						player.shot_power++;
					break;
				}

				player.score += 25;

				bonus_shot_power->alive = FALSE;
				play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
			}
		}
	}

	// update bonus shot double if alive
	if (bonus_shot_double->alive)
	{
		updateSprite(bonus_shot_double);

		if (bonus_shot_double->y > (bonus_shot_double->height + yoffset + G_HEIGHT)
			|| player.bullet_type == SPR_BULLET_PLAYER_DOUBLE)
			bonus_shot_double->alive = FALSE;
		else
		{
			// see if player got the bonus
			x = bonus_shot_double->x + bonus_shot_double->width/2;
			y = (bonus_shot_double->y - yoffset) + bonus_shot_double->height/2;
			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			if (inside(x, y, x1, y1, x2, y2))
			{
				// increase firing rate
				if (player.bullet_type == SPR_BULLET_PLAYER_SINGLE)
				{
					player.shot_power = MAX_SHOT_POWER_PLAYER - 2;
					player.bullet_type = SPR_BULLET_PLAYER_DOUBLE;
				}

				player.score += 25;

				bonus_shot_double->alive = FALSE;
				play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
			}
		}
	}

	// update bonus shot triple if alive
	if (bonus_shot_triple->alive)
	{
		updateSprite(bonus_shot_triple);

		if (bonus_shot_triple->y > (bonus_shot_triple->height + yoffset + G_HEIGHT)
			|| player.bullet_type == SPR_BULLET_PLAYER_TRIPLE)
			bonus_shot_triple->alive = FALSE;
		else
		{
			// see if player got the bonus
			x = bonus_shot_triple->x + bonus_shot_triple->width/2;
			y = (bonus_shot_triple->y - yoffset) + bonus_shot_triple->height/2;
			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			if (inside(x, y, x1, y1, x2, y2))
			{
				// increase firing rate
				if (player.bullet_type == SPR_BULLET_PLAYER_SINGLE)
				{
					player.shot_power = MAX_SHOT_POWER_PLAYER - 1;
					player.bullet_type = SPR_BULLET_PLAYER_DOUBLE;
				}
				else
				{
					player.shot_power = MAX_SHOT_POWER_PLAYER;
					player.bullet_type = SPR_BULLET_PLAYER_TRIPLE;
				}

				player.score += 25;

				bonus_shot_triple->alive = FALSE;
				play_sample(wav_sounds[SND_WAV_ITEM_COLLECTED], 255, 128, 1000, FALSE);
			}
		}
	}
	// add more bonuses here
}

void drawBonuses(BITMAP *bmp)
{
	// draw bonus shot if alive
	if (bonus_shot->alive)
	{
		if (bonus_shot->y > yoffset - bonus_shot->height && bonus_shot->y < yoffset + G_HEIGHT + bonus_shot->height)
			draw_sprite(bmp, bonus_shot_image, bonus_shot->x, bonus_shot->y - yoffset);
	}

	// draw bonus speed if alive
	if (bonus_speed->alive)
	{
		if (bonus_speed->y > yoffset - bonus_speed->height && bonus_speed->y < yoffset + G_HEIGHT + bonus_speed->height)		
			draw_sprite(bmp, bonus_speed_image, bonus_speed->x, bonus_speed->y - yoffset);
	}

	// draw bonus shot power if alive
	if (bonus_shot_power->alive)
	{
		if (bonus_shot_power->y > yoffset - bonus_shot_power->height && bonus_shot_power->y < yoffset + G_HEIGHT + bonus_shot_power->height)
			draw_sprite(bmp, bonus_shot_power_image, bonus_shot_power->x, bonus_shot_power->y - yoffset);
	}

	// draw bonus shot double if alive
	if (bonus_shot_double->alive)
	{
		if (bonus_shot_double->y > yoffset - bonus_shot_double->height && bonus_shot_double->y < yoffset + G_HEIGHT + bonus_shot_double->height)
			draw_sprite(bmp, bonus_shot_double_image, bonus_shot_double->x, bonus_shot_double->y - yoffset);
	}

	// draw bonus shot triple if alive
	if (bonus_shot_triple->alive)
	{
		if (bonus_shot_triple->y > yoffset - bonus_shot_triple->height && bonus_shot_triple->y < yoffset + G_HEIGHT + bonus_shot_triple->height)
			draw_sprite(bmp, bonus_shot_triple_image, bonus_shot_triple->x, bonus_shot_triple->y - yoffset);
	}
	// add more bonuses here
}

void updatePlayerBullet(BULLET bullet)
{
	int n, x, y;
	int x1, y1, x2, y2;
	int bonus_item;

	// update player
	if (bullet.sprite->alive)
	{
		// move the bullet
		updateSprite(bullet.sprite);

		// check bounds
		if (bullet.sprite->y < 0)
		{
			bullet.sprite->alive = FALSE;
		}
		else
		{
			for (n = 0; n < qtd_enemies; n++)
			{
				if (enemies[n].sprite->alive)
				{
					// find center of bullet
					x = bullet.sprite->x + bullet.sprite->width/2;
					y = bullet.sprite->y + bullet.sprite->height/2;

					// get enemy plane bounding rectangle
					x1 = enemies[n].sprite->x;
					y1 = enemies[n].sprite->y - yoffset;
					x2 = x1 + enemies[n].sprite->width;
					y2 = y1 + enemies[n].sprite->height;

					switch (player.bullet_type)
					{
					case (SPR_BULLET_PLAYER_SINGLE):
						x1 -= 3;
						x2 += 3;
						break;
					case (SPR_BULLET_PLAYER_DOUBLE):
						x1 -= 8;
						x2 += 8;
						break;
					case (SPR_BULLET_PLAYER_TRIPLE):
						x1 -= 10;
						x2 += 10;
						break;
					}

					// check for collisions
					if (inside(x, y, x1, y1, x2, y2))
					{
						// bonus items
						switch (enemies[n].type)
						{   
						case SPR_ENEMY_BOSS_1:			                        
						case SPR_ENEMY_BOSS_2:
						case SPR_ENEMY_BOSS_3:
							bonus_item = (rand() % 7 + 1);
							switch (bonus_item)
							{
							case 1:
								createBonus(BI_SHOT_POWER, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							case 2:
								createBonus(BI_SPEED, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							}
							break;
						}

						// damage to enemy
						if ((enemies[n].health -= bullet.damage) <= 0)
						{   
							enemies[n].sprite->alive = FALSE;

							if (enemies[n].is_boss)
							{
								stop_scroller = FALSE;
							}

							// bonus items
							switch (enemies[n].type)
							{
							case SPR_ENEMY_1:
								player.score += 10;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							case SPR_ENEMY_2:
								player.score += 20;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							case SPR_ENEMY_3:
								player.score += 30;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							case SPR_ENEMY_4:
								player.score += 40;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							case SPR_ENEMY_BOSS_1:
								player.score += 100;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);

								if (player.health < 25) player.health += 1;
								break;
							case SPR_ENEMY_BOSS_2:
								player.score += 150;
								if ((rand() % 2 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);

								if (player.health < 25) player.health += 2;
								break;
							case SPR_ENEMY_BOSS_3:
								player.score += 200;			                        
								createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								if (player.health < 25) player.health += 3;
								break;
							case SPR_ENEMY_SUBMARINE_LEFT:			                        
							case SPR_ENEMY_SUBMARINE_RIGHT:
								player.score += 30;
								if ((rand() % 3 + 1) == 1)
									createBonus(enemies[n].bonus, enemies[n].sprite->x, enemies[n].sprite->y);
								break;
							}
						}

						switch (enemies[n].type)
						{
						case SPR_ENEMY_1:
						case SPR_ENEMY_2:
						case SPR_ENEMY_3:
						case SPR_ENEMY_4:
							player.score += 5;
							break;
						case SPR_ENEMY_BOSS_1:
							player.score += 15;
							break;
						case SPR_ENEMY_BOSS_2:
							player.score += 25;
							break;
						case SPR_ENEMY_BOSS_3:
							player.score += 50;
							break;
						case SPR_ENEMY_SUBMARINE_LEFT:
						case SPR_ENEMY_SUBMARINE_RIGHT:
							player.score += 15;
							break;
						}

						bullet.sprite->alive = FALSE;
						startExplosion(bullet.sprite->x, bullet.sprite->y);

						// cancel the loop, destroy the bullet
						break;
					}
				}
			}
		}
	}
}

void updateEnemyBullet(BULLET bullet)
{
	int n, x, y;
	int x1, y1, x2, y2;

	// update enemy bullet
	if (bullet.sprite->alive)
	{
		// move the bullet
		updateSprite(bullet.sprite);

		// check bounds
		if (bullet.sprite->y > (yoffset + G_HEIGHT + bullet.sprite->height)
			|| bullet.sprite->x < 0 
			|| bullet.sprite->x > G_WIDTH)
		{
			bullet.sprite->alive = FALSE;
		}
		else
		{
			// find center of bullet
			x = bullet.sprite->x + bullet.sprite->width/2;
			y = bullet.sprite->y + bullet.sprite->height/2 - yoffset;

			x1 = player.sprite->x;
			y1 = player.sprite->y;
			x2 = x1 + player.sprite->width;
			y2 = y1 + player.sprite->height;

			// player collision
			if (inside(x, y, x1, y1, x2, y2))
			{
				// damage to player
				if ((player.health -= bullet.damage) <= 0)
					gameOver();

				if (player.shot_power > 3)
					player.shot_power--;

				if (player.move_speed > 3)
					player.move_speed--;

				// destroy bullet
				bullet.sprite->alive = FALSE;

				big_explosion->alive = TRUE;
				big_explosion->x = player.sprite->x;
				big_explosion->y = player.sprite->y;

				play_sample(wav_sounds[SND_WAV_EXPLOSION_2], 100, 128, 1000, FALSE);
			}
		}
	}
}

void updateBullets(void)
{
	int n;

	// update player bullets
	for (n = 0; n < MAX_PLAYER_BULLETS; n++)
	{
		if (player_bullets[n].sprite->alive)
		{
			updatePlayerBullet(player_bullets[n]);			
		}
	}

	// update enemies bullets
	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
	{
		if (enemies_bullets[n].sprite->alive)
		{
			updateEnemyBullet(enemies_bullets[n]);			
		}
	}
}

void drawBullets(BITMAP *bmp)
{
	int n;

	// draw player_bullets
	for (n = 0; n < MAX_PLAYER_BULLETS; n++)
	{
		if (player_bullets[n].sprite->alive)
		{	
			draw_sprite(bmp, bmp_bullets_images[player.bullet_type], player_bullets[n].sprite->x, player_bullets[n].sprite->y);
		}
	}

	// draw enemies bullets
	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
	{
		if (enemies_bullets[n].sprite->alive)
		{	
			// draw enemy bullet
			draw_sprite(bmp, bmp_bullets_images[SPR_BULLET_ENEMY_SIMPLE], enemies_bullets[n].sprite->x, enemies_bullets[n].sprite->y - yoffset);		
		}
	}
}

void warpEnemy(SPRITE *spr)
{
	// bounces x off bounds
	if (spr->x < 0 - spr->width)
	{
		spr->x = 0 - spr->width + 1;
		//spr->xspeed *= -1;
	}
	else if (spr->x > backbuffer->w)
	{
		spr->x = backbuffer->w - spr->xspeed;
		//spr->xspeed *= -1;
	}

	// warps y if plane has passed the player
	if (spr->y > yoffset + backbuffer->h + 50)
	{
		// respawn enemy plane
		spr->y = yoffset - 25 - rand() % 25;
		spr->alive = TRUE;
		spr->x = rand() % backbuffer->w;
	}

	// warps y from bottom to top of level
	if (spr->y < 0)
	{
		spr->y = 0;
	}
	else if (spr->y > MAP_SIZE)
	{
		spr->y = 0;
	}
}

void enemyFireAtPlayer(ENEMY enemy)
{
	int n;

	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
	{
		if (!enemies_bullets[n].sprite->alive)
		{
			enemies_bullets[n].damage = enemy.shot_power;	        
			enemies_bullets[n].sprite->alive = TRUE;
			enemies_bullets[n].sprite->x = enemy.sprite->x + (enemy.sprite->width/2) - (enemies_bullets[n].sprite->width / 2);

			switch (enemy.type)
			{
			case SPR_ENEMY_SUBMARINE_LEFT:
				enemies_bullets[n].sprite->yspeed = 0;
				enemies_bullets[n].sprite->xspeed = enemy.bullet_speed;
				enemies_bullets[n].sprite->y = enemy.sprite->y;
				break;
			case SPR_ENEMY_SUBMARINE_RIGHT:
				enemies_bullets[n].sprite->yspeed = 0;
				enemies_bullets[n].sprite->xspeed = enemy.bullet_speed * (-1);
				enemies_bullets[n].sprite->y = enemy.sprite->y;
				break;
			default:
				enemies_bullets[n].sprite->yspeed = enemy.bullet_speed;
				enemies_bullets[n].sprite->xspeed = 0;
				enemies_bullets[n].sprite->y = enemy.sprite->y + 15;
				break;
			}

			break;
		}
	}
}

void updateMovementsIA(ENEMY *enemy)
{
	// movements IA
	enemy->sprite->xspeed = enemy->move_speed;
	enemy->sprite->yspeed = enemy->move_speed;

	if (enemy->sprite->x < enemy->point_to_go->x)
	{
		if (enemy->sprite->x + enemy->sprite->xspeed > enemy->point_to_go->x)
			enemy->sprite->xspeed = enemy->point_to_go->x - enemy->sprite->x;
		else
			enemy->sprite->xspeed = enemy->move_speed;
	}
	else if (enemy->sprite->x > enemy->point_to_go->x)
	{
		if (enemy->sprite->x - enemy->sprite->xspeed < enemy->point_to_go->x)
			enemy->sprite->xspeed = enemy->point_to_go->x - enemy->sprite->x;
		else
			enemy->sprite->xspeed = enemy->move_speed * (-1);
	}
	else
		enemy->sprite->xspeed = 0;

	if ((enemy->sprite->y - yoffset - enemy->sprite->height) < enemy->point_to_go->y)
	{
		if ((enemy->sprite->y - yoffset - enemy->sprite->height) + enemy->move_speed > enemy->point_to_go->y)
			enemy->sprite->yspeed = enemy->point_to_go->y - (enemy->sprite->y - yoffset - enemy->sprite->height);
		else
			enemy->sprite->yspeed = enemy->move_speed;
	}
	else if ((enemy->sprite->y - yoffset - enemy->sprite->height) > enemy->point_to_go->y)
	{
		if ((enemy->sprite->y - yoffset - enemy->sprite->height) - enemy->move_speed < enemy->point_to_go->y)
			enemy->sprite->yspeed = enemy->point_to_go->y - (enemy->sprite->y - yoffset - enemy->sprite->height);
		else
			enemy->sprite->yspeed = enemy->move_speed * (-1);
	}
	else
		enemy->sprite->yspeed = 0;

	// reach the position? Change point to go
	if (enemy->sprite->x == enemy->point_to_go->x &&
		(enemy->sprite->y - yoffset - enemy->sprite->height) == enemy->point_to_go->y) 
	{   
		do
		{
			int pos = (rand() % PST_MAX_POSITIONS);
			if (enemy->point_to_go->x != positions[pos].x  ||
				enemy->point_to_go->y != positions[pos].y)
			{
				enemy->point_to_go = &positions[pos];					            
				break;
			}
		}
		while (TRUE);
	}
}

void updateEnemies(void)
{
	int n;

	// update enemy planes
	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].sprite->alive)
		{
			if (enemies[n].sprite->y > yoffset - enemies[n].sprite->height &&
				enemies[n].sprite->y < yoffset + G_HEIGHT + enemies[n].sprite->height)
			{   
				if (enemies[n].is_boss)
				{
					stop_scroller = TRUE;

					// movements IA
					updateMovementsIA(&enemies[n]);
				}

				updateSprite(enemies[n].sprite);				
				
				enemies[n].fire_count++;

				if (enemies[n].can_fire)
				{
					if (enemies[n].sprite->y > yoffset - enemies[n].sprite->height &&
						enemies[n].sprite->y < yoffset + G_HEIGHT + enemies[n].sprite->height)
					{			
						if (enemies[n].fire_count > enemies[n].fire_delay)
						{
							enemies[n].fire_count = 0;
							enemyFireAtPlayer(enemies[n]);
						}
					}			        
				}
			}
			// kill the enemy if out of boundies screen
			else if (enemies[n].sprite->y > yoffset + G_HEIGHT + enemies[n].sprite->height ||
				(!enemies[n].is_boss && (enemies[n].sprite->x < 0 || enemies[n].sprite->x > G_WIDTH)))
			{
				enemies[n].sprite->alive = FALSE;
			}
		}
	}    
}

void drawEnemies(BITMAP *bmp)
{
	int n;

	// draw enemy planes
	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].sprite->alive)
		{
			// is plane visible on screen?
			if (enemies[n].sprite->y > yoffset - enemies[n].sprite->height &&
				enemies[n].sprite->y < yoffset + G_HEIGHT + enemies[n].sprite->height)
			{
				//draw enemy plane
				draw_sprite(bmp, bmp_enemies_images[enemies[n].type - 1][enemies[n].sprite->curframe],
					enemies[n].sprite->x, enemies[n].sprite->y - yoffset);				
			}
		}
	}
}

void updateScroller()
{
	int n;
	int count_boss = 0;;

	if (!stop_scroller)
	{
		// make sure it doesn't scroll beyond map edge
		if (yoffset < 5) 
		{
			// level is over
			yoffset = 5;
		}
		if (yoffset > BOTTOM)
			yoffset = BOTTOM;


		// scroll map up 2 pixel
		yoffset -= 2;
	}

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].is_boss && enemies[n].sprite->alive)
			count_boss++;
	}

	// end of game
	if (count_boss == 0)
		endGame();
}

void drawScroller(BITMAP *bmp)
{	
	// draw map with single layer
	MapDrawBG(bmp, 0, yoffset, 0, 0, G_WIDTH-1, G_HEIGHT-1);
}

void updatePlayer(void)
{
	int n, x, y, x1, y1, x2, y2;

	// update player sprite
	updateSprite(player.sprite);

	// update player fire count
	player.fire_count++;

	// check for collision with enemy planes
	x = player.sprite->x + player.sprite->width/2;
	y = player.sprite->y + player.sprite->height/2;

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].sprite->alive)
		{
			x1 = enemies[n].sprite->x;
			y1 = enemies[n].sprite->y - yoffset;
			x2 = x1 + enemies[n].sprite->width;
			y2 = y1 + enemies[n].sprite->height;

			// enemy collision
			if (inside(x, y, x1, y1, x2, y2))
			{
				enemies[n].health -= player.shot_power;
				player.health -= enemies[n].shot_power;
				
				big_explosion->alive = TRUE;
				big_explosion->x = player.sprite->x;
				big_explosion->y = player.sprite->y;
				play_sample(wav_sounds[SND_WAV_EXPLOSION_2], 90, 128, 1000, FALSE);

				// is alive?
				if (enemies[n].health <= 0)
					enemies[n].sprite->alive = FALSE;

				if (player.health <= 0)
					gameOver();
			}
		}
	}
}

void drawPlayer(BITMAP *bmp)
{
	draw_sprite(bmp, bmp_player_images[player.sprite->curframe], player.sprite->x, player.sprite->y);
}

// function for debug propose
void drawDebugInfo(BITMAP *bmp)
{
	int n, i;

	// debug
	int count_bullet = 0;
	int count_enemy = 0;
	int count_enemy_boss = 0;
	int count_enemy_submarine_l = 0;
	int count_enemy_submarine_r = 0;


	// display some status information
	textprintf_ex(bmp,font,10, 430,WHITE,-1, "firing rate %d", player.fire_delay);
	textprintf_ex(bmp,font,10, 440,WHITE,-1, "yoffset %d",yoffset);
	textprintf_ex(bmp,font,10, 450,WHITE,-1, "game timer %d", game_time / 10);
	textprintf_ex(bmp,font,10, 460,WHITE,-1, "fps %d", fps);

	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
	{
		if (enemies_bullets[n].sprite->alive)
			count_bullet++;
	}
	textprintf_ex(bmp, font, 10, 400, RED, -1,  "ENEMY BULLET: %d", count_bullet);

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].sprite->alive)
			count_enemy++;
	}
	textprintf_ex(bmp, font, 10, 390, RED, -1,  "ENEMIES: %d", count_enemy);

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].is_boss && enemies[n].sprite->alive)
			count_enemy_boss++;
	}
	textprintf_ex(bmp, font, 10, 380, YELLOW, -1,  "BOSS: %d", count_enemy_boss);

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].type == SPR_ENEMY_SUBMARINE_LEFT && enemies[n].sprite->alive)
			count_enemy_submarine_l++;
	}
	textprintf_ex(bmp, font, 10, 370, ORANGE, -1,  "SUBMARINE LEFt: %d", count_enemy_submarine_l);

	for (n = 0; n < qtd_enemies; n++)
	{
		if (enemies[n].type == SPR_ENEMY_SUBMARINE_RIGHT && enemies[n].sprite->alive)
			count_enemy_submarine_r++;
	}
	textprintf_ex(bmp, font, 10, 360, ORANGE, -1,  "SUBMARINE RIGHT: %d", count_enemy_submarine_r);
}

void drawPlayerInfo(BITMAP *bmp, int life)
{
	int n;
	int border = 3;
	int panel_height = 46;

	rectfill(bmp, 0, 0, bmp->w, panel_height, BLACK);
	rectfill(bmp, border, border, bmp->w - border, panel_height - border, COLDGREY);

	// display score
	// textprintf_ex(bmp, font, 12, 13, GRAY, -1, "SCORE: %d", player.score);
	textprintf_ex(bmp, font, 20, 12, WHITE, -1,  "SCORE: %d", player.score);

	draw_sprite(bmp, progress, 20, 23);

	for (n = 0; n < life; n++)
		draw_sprite(bmp, bar, 22 + n * 5, 25);

	textprintf_ex(bmp, font, bmp->w - 150, 12, WHITE, -1,  "Shot Power: %d", player.shot_power);
	textprintf_ex(bmp, font, bmp->w - 150, 28, WHITE, -1,  "Speed: %d", player.move_speed);
}

/* mappyal aux functions */
int getMapWidth(void)
{
	return mapwidth;
}

int getMapHeight(void)
{
	return mapheight;
}

int getMapSprite(int xo, int yo)
{
	return (MapGetBlock(xo, yo)->user1);
}

void setTile(int xo, int yo, int tile)
{
	MapSetBlock(xo, yo, tile);
}

int switchLayer(int newlayer)
{   
	return MapChangeLayer(newlayer);
}

/* ####################################### */
/* ####################################### */

/* function implementation */
void fatalError(const char *message)
{
	allegro_message("%s\nError: %s", message, allegro_error);
	allegro_exit();
	exit(1);
}


BITMAP **getEnemyBitmaps(int nm_enemy)
{	
	int n, i, j;
	BITMAP *temp = NULL;
	BITMAP **a_enemies = NULL;

	// load enemy plane sprites
	switch (nm_enemy)
	{
	case SPR_ENEMY_1:
		temp = load_bitmap("data/enemy1_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 32, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_2:            
		temp = load_bitmap("data/enemy2_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 32, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_3:
		temp = load_bitmap("data/enemy3_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 32, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_4:
		temp = load_bitmap("data/enemy4_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 32, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_BOSS_1:
		temp = load_bitmap("data/boss1_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 65, 50, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_BOSS_2:
		temp = load_bitmap("data/boss2_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 65, 55, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_BOSS_3:
		temp = load_bitmap("data/boss3_strip3.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 98, 85, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_SUBMARINE_LEFT:
		temp = load_bitmap("data/submarine_left.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 98, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;
	case SPR_ENEMY_SUBMARINE_RIGHT:
		temp = load_bitmap("data/submarine_right.bmp", NULL);
		a_enemies = (BITMAP **) malloc(3 * sizeof(BITMAP *));

		for (n = 0; n < 3; n++)
			a_enemies[n] = grabFrame(temp, 98, 32, 0, 0, 3, n);
		destroy_bitmap(temp);
		break;

	}
	return a_enemies;
}


void initPositions(void)
{
	// left
	positions[0].x = 5;
	positions[0].y = 50;
	positions[0].is_empty = FALSE;

	// right
	positions[1].x = G_WIDTH - 35;
	positions[1].y = 50;
	positions[1].is_empty = FALSE;

	// center
	positions[2].x = G_WIDTH / 2 - 35;
	positions[2].y = G_HEIGHT / 2;
	positions[2].is_empty = FALSE;

	// center top
	positions[3].x = G_WIDTH / 2 - 35;
	positions[3].y = 100;
	positions[3].is_empty = FALSE;
}

int setupGame(void)
{
	// load game resources and stuff here
	// return false if something failed to load or init
	// BITMAPS, SOUND, FONT

	int n, i, j;
	int enemy_type;
	BITMAP *temp = NULL;

	// load the Mappy file
	if (MapLoad("data/level1.fmp") != 0) return FALSE;

	// set palette
	MapSetPal8();

	map_width = getMapWidth();
	map_height = getMapHeight();

	// load screens sprite
	// note: create array of bmp_screen_images
	bmp_title_screen_image = load_bitmap("data/title.bmp", NULL);
	bmp_game_over_image = load_bitmap("data/game_over.bmp", NULL);
	bmp_end_game_image = load_bitmap("data/end_game.bmp", NULL);

	// load player airplane sprite
	temp = load_bitmap("data/player_plane1_strip3.bmp", NULL);
	for (n = 0; n < 3; n++)
		bmp_player_images[n] = grabFrame(temp, 64, 64, 0, 0, 3, n);
	destroy_bitmap(temp);

	// load progress bar
	temp = load_bitmap("data/progress.bmp", NULL);
	progress = grabFrame(temp, 130, 14, 0, 0, 1, 0);
	bar = grabFrame(temp, 6, 10, 130, 2, 1, 0);
	destroy_bitmap(temp);

	// load bonus shot
	bonus_shot_image = load_bitmap("data/bonus_shot.bmp", NULL);	

	// load bonus speed
	bonus_speed_image = load_bitmap("data/bonus_speed.bmp", NULL);

	// load bonus shot power
	bonus_shot_power_image = load_bitmap("data/bonus_shot_power.bmp", NULL);

	// load bonus shot double
	bonus_shot_double_image = load_bitmap("data/bonus_shot_double.bmp", NULL);	

	// load bonus shot double
	bonus_shot_triple_image = load_bitmap("data/bonus_shot_triple.bmp", NULL);

	// load bullet images
	bmp_bullets_images[SPR_BULLET_PLAYER_SINGLE] = load_bitmap("data/player_bullet_single.bmp", NULL);
	bmp_bullets_images[SPR_BULLET_PLAYER_DOUBLE] = load_bitmap("data/player_bullet_double.bmp", NULL);
	bmp_bullets_images[SPR_BULLET_PLAYER_TRIPLE] = load_bitmap("data/player_bullet_triple.bmp", NULL);
	bmp_bullets_images[SPR_BULLET_ENEMY_SIMPLE] = load_bitmap("data/enemy_bullet.bmp", NULL);
	bmp_bullets_images[SPR_BULLET_BOSS_SIMPLE] = load_bitmap("data/boss_bullet.bmp", NULL);
	bmp_bullets_images[SPR_BULLET_BOSS_3_SIMPLE] = load_bitmap("data/boss3_bullet.bmp", NULL);

	// load explosion sprites
	temp = load_bitmap("data/explosion1_strip6.bmp", NULL);
	for (n = 0; n < 6; n++)
		explosion_images[n] = grabFrame(temp, 32, 32, 0, 0, 6, n);
	destroy_bitmap(temp);

	// load big explosion sprites
	temp = load_bitmap("data/explosion2_strip7.bmp", NULL);
	for (n = 0; n < 8; n++)
		big_explosion_images[n] = grabFrame(temp, 65, 65, 0, 0, 7, n);
	destroy_bitmap(temp);


	/* load wave and midi sounds */
	mid_sounds[SND_MID_TITLE] = load_midi("data/title_dd_title_screen.mid");
	mid_sounds[SND_MID_LEGENDARY_WINGS] = load_midi("data/legendary_wings.mid");
	mid_sounds[SND_MID_GAME_OVER] = load_midi("data/game_over.mid");

	wav_sounds[SND_WAV_EXPLOSION_1] = load_sample("data/snd_explosion1.wav");
	wav_sounds[SND_WAV_EXPLOSION_2] = load_sample("data/snd_explosion2.wav");    
	wav_sounds[SND_WAV_CURSOR_SELECTION] = load_sample("data/snd_cursor_selection.wav");
	wav_sounds[SND_WAV_ITEM_COLLECTED] = load_sample("data/snd_item_collected.wav");
	wav_sounds[SND_WAV_MENU_SELECT] = load_sample("data/snd_menu_select.wav");	

	set_volume(255, 130); // set global volume, wav = 255 / mid = 130 (mid sounds more lower than wav sounds)

	/* Enemies */
	// initialize enemies bitmaps    
	for (n = 0; n < NUM_ENEMIES_TYPE; n++)
		bmp_enemies_images[n] = getEnemyBitmaps(n + 1);

	// map have enemies layer?
	if (switchLayer(1) == 1)
	{
		for (j = 0; j < map_height; j++)
		{
			for (i = 0; i < map_width; i++)
			{
				switch (getMapSprite(i, j))
				{
				case SPR_ENEMY_1:            
				case SPR_ENEMY_2:
				case SPR_ENEMY_3:
				case SPR_ENEMY_4:
				case SPR_ENEMY_BOSS_1:
				case SPR_ENEMY_BOSS_2:
				case SPR_ENEMY_BOSS_3:
				case SPR_ENEMY_SUBMARINE_LEFT:
				case SPR_ENEMY_SUBMARINE_RIGHT:
					qtd_enemies++;                        
					break;
				}
			}
		}
		// create enemies array
		enemies = (ENEMY *) malloc(qtd_enemies * sizeof(ENEMY));

		for (n = 0; n < qtd_enemies; n++)
		{
			ENEMY enemy;
			enemy.sprite = NULL;            
			enemy.fire_count = 0;
			enemy.fire_delay = 10;
			enemy.health = 1;
			enemy.shot_power = 1;

			enemies[n] = enemy;
		}

		// back to background layer
		switchLayer(0);        
	}

	return TRUE;
}

void runGame(void)
{
	int done = FALSE;   
	int frames_done = 0;
	int old_time = 0;

	/* initialize the array to 0 */
	int frames_array[10] = { 0 }; /* an array to store the number of frames we did during the last 10 tenths of a second */
	int frame_index = 0; /* used to store the index of the last updated value in the array */

	/* FPS / TIMER Control */
	LOCK_VARIABLE(ticks);
	LOCK_FUNCTION(ticker);
	install_int_ex(ticker, BPS_TO_TIMER(UPDATES_PER_SECOND));

	LOCK_VARIABLE(game_time);
	LOCK_FUNCTION(gameTimeTicker);
	install_int_ex(gameTimeTicker, BPS_TO_TIMER(10)); /* i.e. game time is in tenths of seconds */ 		
	/* FPS / TIMER Control */

	/* Habilitar boto fechar */
	set_close_button_callback(&closeHook);

	mainMenu();

	while (!done)
	{   	
		while(ticks > 0)
		{
			int old_ticks = ticks;

			/***************************			
			** Do the logic stuff here!
			****************************/			
			if (keyboard_needs_poll())
				poll_keyboard();

			if (num_joysticks > 0)
				poll_joystick();

			if (ENABLE_MOUSE_SUPPORT) { 
				if (mouse_needs_poll())
					poll_mouse();
			}

			// Change FullScreen or Windowed            
			if (key[KEY_1]) {
				while(key[KEY_1]);
				set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0);
			}			
			if (key[KEY_2]) {
				while(key[KEY_2]);
				set_gfx_mode(GFX_AUTODETECT_FULLSCREEN, 640, 480, 0, 0);
			}

			switch(game_state)
			{
			case GS_MAIN_MENU:
				updateMainMenu();
				break;			
			case GS_CREDITS:
				updateCredits();
				break;
			case GS_NEW_GAME:
				updateNewGame();
				break;
			case GS_PAUSED_GAME:
				updatePausedGame();
				break;
			case GS_GAME_OVER:
				updateGameOver();
				break;
			case GS_END_GAME:
				updateEndGame();
				break;

			case GS_EXIT:
				done = TRUE;
				break;
			}

			/***************************			
			** Do the logic stuff here!
			****************************/

			ticks--;
			if(old_ticks <= ticks)
				break;
		}

		if (game_time >= old_time + 1) /* i.e. a 0.1 second has passed since we last counted the frames */
		{
			fps -= frames_array[frame_index]; /* decrement the fps by the frames done a second ago */
			frames_array[frame_index] = frames_done; /* store the number of frames done this 0.1 second */
			fps += frames_done; /* increment the fps by the newly done frames */

			frame_index = (frame_index + 1) % 10; /* increment the frame index and snap it to 10 */

			frames_done = 0;
			old_time += 1;
		}

		/* Draw everything here! */
		clear_bitmap(backbuffer);

		if (ENABLE_MOUSE_SUPPORT)
			show_mouse(backbuffer);

		switch(game_state)
		{
		case GS_MAIN_MENU:
			drawMainMenu(backbuffer);
			break;		
		case GS_CREDITS:
			drawCredits(backbuffer);
			break;
		case GS_NEW_GAME:
			drawNewGame(backbuffer);
			break;
		case GS_PAUSED_GAME:
			drawPausedGame(backbuffer);
			break;
		case GS_GAME_OVER:
			drawGameOver(backbuffer);
			break;
		case GS_END_GAME:
			drawEndGame(backbuffer);
			break;
		}

		// DEBUG FPS
		// textprintf_ex(backbuffer, font, 10, 10, YELLOW, -1, "FPS: %d", fps);

		renderGame(backbuffer);
		/* Draw everything here! */

		done = done || closed;

		frames_done++; /* we drew a frame! */		
		while(ticks == 0)
		{
			rest(1);
		}
	}    
}

void renderGame(BITMAP *bmp)
{
	/* render all graphics on screen */
	acquire_screen();
	blit(bmp, screen, 0, 0, 0, 0, bmp->w, bmp->h);
	release_screen();
}

void shutdownGame(void)
{
	/* deallocate anything you allocated for your game here */
	int n, i, j;	

	// delete bitmaps
	for (n = 0; n < 6; n++)
	{
		destroy_bitmap(explosion_images[n]);
	}

	for (n = 0; n < 7; n++)
	{
		destroy_bitmap(big_explosion_images[n]);
	}

	for (n = 0; n < 3; n++)
	{
		destroy_bitmap(bmp_player_images[n]);	    	    
	}

	for (n = 0; n < NUM_BULLETS_TYPE; n++)
		destroy_bitmap(bmp_bullets_images[n]);

	for (i = 0; i < NUM_ENEMIES_TYPE; i++)
	{   
		for (j = 0; j < 3; j++)
			destroy_bitmap(bmp_enemies_images[i][j]);
	}

	// screen bitmaps
	destroy_bitmap(bmp_title_screen_image);
	destroy_bitmap(bmp_game_over_image);
	destroy_bitmap(bmp_end_game_image);

	destroy_bitmap(progress);
	destroy_bitmap(bar);
	destroy_bitmap(bonus_shot_image);	
	destroy_bitmap(bonus_speed_image);
	destroy_bitmap(bonus_shot_power_image);	
	destroy_bitmap(bonus_shot_double_image);	
	destroy_bitmap(bonus_shot_triple_image);	

	// delete sprites
	free(player.sprite);	
	free(bonus_shot);
	free(bonus_speed);
	free(bonus_shot_power);
	free(bonus_shot_double);
	free(bonus_shot_triple);
	free(big_explosion);

	for (n = 0; n < qtd_enemies; n++)
		if (enemies[n].sprite != NULL)
			free(enemies[n].sprite);		
	free(enemies);	

	for (n = 0; n < MAX_EXPLOSIONS; n++)
		free(explosions[n]);

	for (n = 0; n < MAX_PLAYER_BULLETS; n++)
		free(player_bullets[n].sprite);

	for (n = 0; n < MAX_ENEMIES_BULLETS; n++)
		free(enemies_bullets[n].sprite);

	/* sounds */
	stop_midi();

	for (n = 0; n < SND_MAX_MIDIS; n++)
	{
		if (mid_sounds[n] != NULL) destroy_midi(mid_sounds[n]);
	}

	for (n = 0; n < SND_MAX_WAVES; n++)
	{
		if (wav_sounds[n] != NULL) destroy_sample(wav_sounds[n]);
	}

	// delete the Mappy level
	MapFreeMem();
}

void exitGame(void)
{
	if (backbuffer)
	{
		if (ENABLE_MOUSE_SUPPORT) show_mouse(NULL);
		destroy_bitmap(backbuffer);
	}
	allegro_exit();
}