/*
    Title: PONG by Laamaker
    Author: Leandro A. Dos Anjos
    Date: 2012-09-23
    Language: C
    Email: laamaker@gmail.com / laamaker@hotmail.com
*/

#include <stdio.h>
#include <ctype.h>
#include <allegro.h>

#include "pong.h"

/* FPS Controller functions */
static void my_allegro_close_button_handler(void)
{
    done = TRUE;
}
END_OF_FUNCTION(my_allegro_close_button_handler);

void ticker(void)
{
    ticks++;
}
END_OF_FUNCTION(ticker);

void game_time_ticker(void)
{
    game_time++;
}
END_OF_FUNCTION(game_time_ticker);

/* functions */
void displayDebugMode(BITMAP *bmpTarget, int isVisible, int color);
void showStage(BITMAP *bmpTarget);
void showScore(BITMAP *bmpTarget);
void showHighlight(BITMAP *bmpTarget);
void showWinnerPlayer(BITMAP *bmpTarget);
int collisionBallPlayer(BALL *b, PLAYER *p);
void initGame(void);
void newGame(void);
void moveCPU(PLAYER *cpu, const BALL *b);
void updateGame(void);
void updateNormalGame(void);
void updateVersusGame(void);
void updateWinnerGame(void);
void displayNormalGame(BITMAP *bmpTarget);
void displayVersusGame(BITMAP *bmpTarget);
void displayWinnerGame(BITMAP *bmpTarget);

/* "events" */
void on_normal_game_menu_selected(void);
void on_versus_menu_selected(void);
void on_options_menu_selected(void);
void on_exit_menu_selected(void);

/* MAIN MENU */
void menuItemSelected(void);
void initMenu(void);
void updateMainMenu(void);
void displayMainMenu(BITMAP* bmpTarget);

/* OPTIONS MENU */
void initOptionMenu(void);
void updateOptionMenu(void);
void displayOptionMenu(BITMAP* bmpTarget);

/* MISCELANEA FUNCTIONS */
void blindScreenMode(BITMAP *bmpTarget, struct player *p);
int ballInside(BALL *b, int x1, int y1, int x2, int y2);
void createWormHole(BALL *b);
void displayWormHole(BITMAP *bmpTarget);


/******************************
-- GAME START
*******************************/
int main(int argc, char **argv)
{
    /*
    int c; 

    for (c = 1; c < argc; c++)
    {
        if (stricmp(argv[c], "-debug_mode") == 0)
        {
            debug_mode = 1;
        }

        if (stricmp(argv[c], "-maxlevel") == 0)
        {
            if (argc > c && sscanf(argv[c+1], "%d") == 1)
            {
                max_count_hit = atoi(argv[c+1]);
            }
        }        
    }
    */

    char **data_config;
    int g_width, g_height, g_col_depth, g_mode;
    int count;

    set_uformat(U_ASCII);    
    if (allegro_init() != 0) exit(1);

    /* Obtm os valores do arquivo de config INICIO */
    push_config_state();
    set_config_file("data\\pong_config.ini");

    data_config = get_config_argv("graphics", "mode", &count);
    if (count != 3)
    {
        g_width = G_WIDTH;
        g_height = G_HEIGHT;
        g_col_depth = G_COL_DEPTH;
    }
    else
    {
        g_width = atoi(data_config[0]);
        g_height = atoi(data_config[1]);
        g_col_depth = atoi(data_config[2]);
    }

    g_mode = get_config_int("graphics", "fullscreen", NULL);
    if (g_mode < 0 || g_mode > 1)
        g_mode = G_MODE; /* GFX_AUTODETECT_WINDOWED */
    else if (g_mode == 1)    
        g_mode = GFX_AUTODETECT_FULLSCREEN;
    else
        g_mode = GFX_AUTODETECT_WINDOWED;

    if ((max_count_hit = get_config_int("general", "max_count_hit", NULL)) == 0)
        max_count_hit = 9;

    debug_mode = get_config_int("debugger", "debug_mode", NULL);

    pop_config_state();
    /* Obtm os valores do arquivo de config FIM */

    set_color_depth(g_col_depth);
    if (set_gfx_mode(g_mode, g_width, g_height, 0, 0) != 0)
    {
        errorExit("NAO FOI POSSIVEL INICIAR O MODO GRAFICO\n");
    }

    install_timer();
    install_keyboard();
    if (ENABLE_MOUSE_SUPPORT) { install_mouse(); }

    if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != 0)
    {
        errorExit("NAO FOI POSSIVEL INICIAR O SOM\n");
    }

    set_window_title(G_GAME_TITLE);
    srand((unsigned)time(0));

    backbuffer = create_bitmap(g_width, g_height);
    clear_to_color(backbuffer, BLACK);

    /* setup the game */
    if (!setupGame())
    {
        errorExit("O jogo nao pode iniciar. Nao eh possivel continuar!");
    }

    runGame();
    shutdownGame();
    exitGame();

    return 0;
}
END_OF_MAIN();

/* function implementation */
void displayDebugMode(BITMAP *bmpTarget, int isVisible, int color)
{
    if (isVisible)
    {
        textout(bmpTarget, font, "DEBUG MODE: ON", 10, 10, color);
        textprintf(bmpTarget, font, 10, 20, color, "FPS: %d", fps);
        textprintf(bmpTarget, font, 10, 30, color, "Player 1 Speed: %d", P1.speed);
        textprintf(bmpTarget, font, 10, 40, color, "Player 2 Speed: %d", P2.speed);        
        textprintf(bmpTarget, font, 10, 50, color, "count_hit: %d", count_hit);
        textprintf(bmpTarget, font, 10, 60, color, "Ball Speed: %d", bola.speed_x > 0 ? bola.speed_x : bola.speed_x * (-1));
        textprintf(bmpTarget, font, 10, 70, color, "Max Ball Speed: %d", max_count_hit);
        textprintf(bmpTarget, font, 10, 80, color, "MAX POINT GAME: %d", max_point_game);
        textprintf(bmpTarget, font, 10, 90, color, "Blind Screen Mode: %s", enabled_blind_screen_mode ? "ON" : "OFF");
        textprintf(bmpTarget, font, 10, 100, color, "Game Level: %d", game_level);
        
    }
}

void blindScreenMode(BITMAP *bmpTarget, struct player *p)
{
    int screen_half_x = SCREEN_W / 2;
    int screen_half_y = SCREEN_H / 2;

    if (enabled_blind_screen_mode)
    {
        if ((p->id == 0 || p->id == 1) && count_hit > 3)
        {   
            p->color = BLACK;
            textout_centre_ex(bmpTarget, font, "BLIND SCREEN MODE", screen_half_x - screen_half_x / 2, screen_half_y - 10, WHITE, BLACK);
        }
        if (p->id == 2 && count_hit > 3)
        {   
            p->color = BLACK;
            textout_centre_ex(bmpTarget,font, "BLIND SCREEN MODE", screen_half_x + screen_half_x / 2, screen_half_y - 10, WHITE, BLACK);
        }
    }
}

void showWinnerPlayer(BITMAP *bmpTarget)    
{
    int posX = bmpTarget->w / 2;
    int posY = bmpTarget->h / 2 - 70;

    if (winner_player == 1)
    {
        textout_centre(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, "PLAYER 1 WIN!!!", posX, posY, WHITE);
    }
    else if (winner_player == 2)
    {
        textout_centre(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, "PLAYER 2 WIN!!!", posX, posY, WHITE);
    }
    else
    {
        textout_centre(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, "CPU WIN!!!", posX, posY, WHITE);
    }
    textout_centre(bmpTarget, font, "Pressione [ESC] para continuar...", posX, posY + text_height((FONT *)datFont[AGENCYFB72PX].dat) + 20, WHITE);
}

void showStage(BITMAP *bmpTarget)
{
    int i;
    int rect_h = 60;
    int abc = SCREEN_H;
    rectfill(bmpTarget, SCREEN_W / 2 - 5, 0, SCREEN_W / 2 + 5, rect_h-10, WHITE);
    for (i = rect_h; i <= SCREEN_H; i += rect_h)
        rectfill(bmpTarget, SCREEN_W / 2 - 5, i, SCREEN_W / 2 + 5, i + rect_h - 10, WHITE);
}

void showScore(BITMAP *bmpTarget)
{
    int pos_y = 10;

    if (gameSession == NORMALGAME)
        textprintf_right_ex(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, SCREEN_W/2-50, pos_y, WHITE, -1, "%d", cpuScore);
    else
        textprintf_right_ex(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, SCREEN_W/2-50, pos_y, WHITE, -1, "%d", p1Score);
    
    textprintf_ex(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, SCREEN_W/2+50, pos_y, WHITE, -1, "%d", p2Score);
}

void showHighlight(BITMAP *bmpTarget)
{   
    if (gameSession == NORMALGAME)
    {
        if (cpuScore - p2Score >= 3)
        {
            blindScreenMode(bmpTarget, &CPU);
        }
        if (p2Score - cpuScore >= 3)
        {
            blindScreenMode(bmpTarget, &P2);
        }
    }
    
    if (gameSession == VERSUSGAME)
    {    
        if (p1Score - p2Score >= 3)
        {
            blindScreenMode(bmpTarget, &P1);
        }
        if (p2Score - p1Score >= 3)
        {
            blindScreenMode(bmpTarget, &P2);
        }
    }
    
    if (p2Score > 1 && p2Score % 2 == 1)
    {   
        if (p1Score == p2Score || cpuScore == p2Score && count_hit < 10)
            createWormHole(&bola);
    }
}

int collisionBallPlayer(BALL *b, PLAYER *p)
{
    if (p->id == 0 || p->id == 1)
    {
        if (b->x + b->radius >= p->x && b->x <= p->x + p->width && 
            b->y + b->radius >= p->y && b->y <= p->y + p->height)
        {
            b->x = p->x + p->width + (int)(b->radius / 2); 
            b->speed_x *= (-1);
            
            return 1;
        }
    }
    else if (p->id == 2)
    {
        if (b->x + b->radius >= p->x && b->x <= p->x + p->width && 
            b->y + b->radius >= p->y && b->y <= p->y + p->height)
        {
            b->x = p->x - (int)(b->radius / 2);   
            b->speed_x *= (-1);
            
            return 1;
        }
    }
    return 0;
}

void initGame(void)
{
    P1.id = 1;
    P2.id = 2;       
    CPU.id = 0;
    
    initPlayer(&P1);
    initPlayer(&P2);
    initPlayer(&CPU);
    
    initBall(&bola);
    
    winner_player = -1;
    count_hit = 3;
    
    play_sample((SAMPLE *) datSound[WIN].dat, 255, 128, 1000, FALSE);    
}

void newGame(void)
{   
    initGame();
    p1Score = p2Score = cpuScore = 0;    
}

void updateGame(void)
{   
    switch (gameSession)
    {
        case NORMALGAME:
            if (collisionBallPlayer(&bola, &CPU) || collisionBallPlayer(&bola, &P2))
            {
                play_sample((SAMPLE *) datSound[PING].dat, 255, 128, 1000, FALSE);
                count_hit++;
                CPU.speed = P2.speed = count_hit;
            }
            break;
        case VERSUSGAME:
            if (collisionBallPlayer(&bola, &P1) || collisionBallPlayer(&bola, &P2))
            {
                play_sample((SAMPLE *) datSound[PING].dat, 255, 128, 1000, FALSE);
                count_hit++;
                P1.speed = P2.speed = count_hit;
            }
            break;
    }
    
    if (count_hit <= max_count_hit)
    {            
        bola.speed_x = bola.speed_x < 0 ? -count_hit : count_hit;
        bola.speed_y = bola.speed_y < 0 ? -count_hit : count_hit;

        switch (count_hit)
        {
        case 5:
            bola.color = YELLOW;
            break;            
        case 7:
            bola.color = ORANGE;
            break;
        case 9:
            bola.color = RED;
            break;
        case 12:
            bola.color = GREY;
            bola.radius = 8;
            break;
        case 15:
            bola.color = CYAN;
            bola.radius = 20;
            break;
        }
    }
   
    if (bola.x >= SCREEN_W)
    {
        switch (gameSession)
        {
            case NORMALGAME:
                if (++cpuScore >= max_point_game) {
                    winner_player = 0;
                    gameSession = WINNERGAME;
                }
                else initGame();
                break;
            case VERSUSGAME:
                if (++p1Score >= max_point_game) {
                    winner_player = 1;
                    gameSession = WINNERGAME;
                }
                else initGame();
                break;
        }
    }
    else if (bola.x <= 0)
    {   
        if (++p2Score >= max_point_game) {
            winner_player = 2;
            gameSession = WINNERGAME;
        }
        else initGame();
    }
}

void moveCPU(PLAYER *cpu, const BALL *b)
{
    // Se a bola estiver voltando para a esquerda
    /* IA do CPU paddle */
    int hit_control = rand() % (game_level * 6);
    if (b->speed_x < 0 && hit_control > 1)
    {
        if (cpu->y + (cpu->height / 2) > b->y)
            cpu->y -= cpu->speed;
            
        if (cpu->y + (cpu->height / 2) < b->y)
            cpu->y += cpu->speed;
            
        if (cpu->y < 0) cpu->y = 0;
    
        if (cpu->y + cpu->height >= SCREEN_H)
            cpu->y = SCREEN_H - cpu->height;
    }
}

void updateNormalGame(void)
{
    moveCPU(&CPU, &bola);
    movePlayer(&P2);
    moveBall(&bola);

    updateGame();

    if (key[KEY_ESC]) {
        gameSession = MAINMENUGAME;
        rest(5);
    }
}

void displayNormalGame(BITMAP *bmpTarget)
{
    drawPlayer(bmpTarget, &CPU);
    drawPlayer(bmpTarget, &P2);
    drawBall(bmpTarget, &bola);

    showStage(bmpTarget);
    showScore(bmpTarget);
    showHighlight(bmpTarget);
}

void updateVersusGame(void)
{
    movePlayer(&P1);    
    movePlayer(&P2);
    moveBall(&bola);
        
    updateGame();

    if (key[KEY_ESC]) {
        gameSession = MAINMENUGAME;
        rest(5);
    }
}

void displayVersusGame(BITMAP *bmpTarget)
{
    drawPlayer(bmpTarget, &P1);
    drawPlayer(bmpTarget, &P2);
    drawBall(bmpTarget, &bola);

    showStage(bmpTarget);
    showScore(bmpTarget);
    showHighlight(bmpTarget);
}

void updateWinnerGame(void)
{
    if (key[KEY_ESC]) {
        gameSession = MAINMENUGAME;
    }
}

void displayWinnerGame(BITMAP *bmpTarget)
{
    showWinnerPlayer(bmpTarget);
}

/* MENU */
/* "events" */
void on_normal_game_menu_selected(void)
{   
    gameSession = NORMALGAME;    
    newGame();
}

void on_versus_menu_selected(void)
{
    gameSession = VERSUSGAME;    
    newGame();
}

void on_options_menu_selected(void)
{
    gameSession = OPTIONSGAME;
}

void on_exit_menu_selected(void)
{
    done = TRUE;
}
/* "events" */

void menuItemSelected(void)
{
    static funcptr actions[] =
    {
        &on_normal_game_menu_selected,
        &on_versus_menu_selected,
        &on_options_menu_selected,
        &on_exit_menu_selected
    };
    actions[mainmenu.menuSelection]();
}

void initMenu(void)
{
    mainmenu.menuSelection = 0;
    mainmenu.backgroundColor = GREY;
    mainmenu.menuItemCount = sizeof(menuItems) / sizeof(const char*);
}

void updateMainMenu(void)
{
    static int upKey = FALSE;
    static int downKey = FALSE;
    static int enterKey = FALSE;	

    if (key[KEY_UP])
    {
        if (!upKey)
            upKey = TRUE;
    }
    else
    {
        if (upKey)
        {
            if (--mainmenu.menuSelection < 0)
                mainmenu.menuSelection = mainmenu.menuItemCount - 1;

            upKey = FALSE;
        }
    }

    if (key[KEY_DOWN])
    {
        if (!downKey)
            downKey = TRUE;
    }
    else
    {
        if (downKey)
        {
            if (++mainmenu.menuSelection == mainmenu.menuItemCount)
                mainmenu.menuSelection = 0;

            downKey = FALSE;
        }
    }

    if (key[KEY_ENTER])
    {
        if (!enterKey)
            enterKey = TRUE;
    }
    else
    {
        if (enterKey)
        {
            menuItemSelected();
            enterKey = FALSE;
        }
    }
}

void displayMainMenu(BITMAP* bmpTarget)
{
    int centerX = bmpTarget->w / 2;
    int centerY = bmpTarget->h / 2;
    int menuGap = 32;
    int index;
    int selectedMenuItemColor = YELLOW;
    int unSelectedMenuItemColor = GREY;

    textout_centre(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, "P O N G", centerX, 100, WHITE);
    textprintf_centre(bmpTarget, font, centerX, bmpTarget->h - 40, WHITE, "Version: %s", G_GAME_TITLE);
    textprintf_centre(bmpTarget, font, centerX, bmpTarget->h - 20, WHITE, "(c) 2012 by %s", G_GAME_AUTOR);	

    for (index = 0; index < mainmenu.menuItemCount; index++)
    {
        if (index == mainmenu.menuSelection)
        {
            textout_centre_ex(bmpTarget, font, ">> ",centerX - text_length(font, menuItems[index]) / 2 - 30, centerY + index * menuGap, selectedMenuItemColor, -1);
            textprintf_centre_ex(bmpTarget, font, centerX, centerY + index * menuGap, selectedMenuItemColor, -1, "%s", menuItems[index]);
        }
        else
        {
            textprintf_centre_ex(bmpTarget, font, centerX, centerY + index * menuGap, unSelectedMenuItemColor, -1, "%s", menuItems[index]);
        }		
    }
}

/* OPTION MENU */
void initOptionMenu(void)
{
    optionmenu.optionSelection = 0;
    optionmenu.backgroundColor = GREY;
    optionmenu.optionItemCount = sizeof(optionItems) / sizeof(const char*);
}

void updateOptionMenu(void)
{
    static int upKey = FALSE;
    static int downKey = FALSE;
    static int enterKey = FALSE;
       
    if (key[KEY_UP])
    {
        if (!upKey)
            upKey = TRUE;
    }
    else
    {
        if (upKey)
        {
            if (--optionmenu.optionSelection < 0)
                optionmenu.optionSelection = optionmenu.optionItemCount - 1;

            upKey = FALSE;
        }
    }

    if (key[KEY_DOWN])
    {
        if (!downKey)
            downKey = TRUE;
    }
    else
    {
        if (downKey)
        {
            if (++optionmenu.optionSelection == optionmenu.optionItemCount)
                optionmenu.optionSelection = 0;

            downKey = FALSE;
        }
    }

    if (key[KEY_0] || key[KEY_0_PAD])
    {       
        if (optionmenu.optionSelection == 2) // "BLIND SCREEN MODE"
            enabled_blind_screen_mode = FALSE;
    }

    if (key[KEY_1] || key[KEY_1_PAD])
    {
        if (optionmenu.optionSelection == 0) // "MAX BALL SPEED"        
            max_count_hit = 10;
            
        if (optionmenu.optionSelection == 1) // "MAX POINT GAME"
            max_point_game = 15;
            
        if (optionmenu.optionSelection == 2) // "BLIND SCREEN MODE"
            enabled_blind_screen_mode = TRUE;
            
        if (optionmenu.optionSelection == 3) // "GAME LEVEL"
            game_level = 3;
    }
    
    if (key[KEY_2] || key[KEY_2_PAD])
    {
        if (optionmenu.optionSelection == 0) // "MAX BALL SPEED"        
            max_count_hit = 15;
        
        if (optionmenu.optionSelection == 1) // "MAX POINT GAME"
            max_point_game = 20;
            
        if (optionmenu.optionSelection == 3) // "GAME LEVEL"
            game_level = 4;
    }
    
    if (key[KEY_3] || key[KEY_3_PAD])
    {
        if (optionmenu.optionSelection == 0) // "MAX BALL SPEED"        
            max_count_hit = 18;
            
        if (optionmenu.optionSelection == 1) // "MAX POINT GAME"
            max_point_game = 30;
            
        if (optionmenu.optionSelection == 3) // "GAME LEVEL"
            game_level = 6;
    }
    
    if (key[KEY_ESC])
    {
        gameSession = MAINMENUGAME;
    }
}

void displayOptionMenu(BITMAP* bmpTarget)
{
    int centerX = bmpTarget->w / 2;
    int centerY = bmpTarget->h / 2;
    int optionGap = 45;
    int index;
    int selectedOptionItemColor = YELLOW;
    int unSelectedOptionItemColor = GREY;
    int strAux[256];
       

    textout_centre(bmpTarget, (FONT *)datFont[AGENCYFB72PX].dat, "P O N G", centerX, 100, WHITE);
    textprintf_centre(bmpTarget, font, centerX, bmpTarget->h - 40, WHITE, "Version: %s", G_GAME_TITLE);
    textprintf_centre(bmpTarget, font, centerX, bmpTarget->h - 20, WHITE, "(c) 2012 by %s", G_GAME_AUTOR);	
    
    textout_centre(bmpTarget, font, "OPTIONS", centerX, centerY - 30, WHITE);    

    for (index = 0; index < optionmenu.optionItemCount; index++)
    {
        switch (index)
        {
            case 0: // "MAX BALL SPEED"
                sprintf(strAux, " : %d", max_count_hit);
                break;
            case 1: // "MAX POINT GAME"
                sprintf(strAux, " : %d", max_point_game);
                break;
            case 2: // "BLIND SCREEN MODE"                
                sprintf(strAux, " : %s", enabled_blind_screen_mode ? "ON" : "OFF");
                break;
            case 3: // "GAME LEVEL"
                switch (game_level)
                {
                    case 3:
                        sprintf(strAux, " : %s", "HUMAN");
                        break;
                    case 4:
                        sprintf(strAux, " : %s", "SPARTAN");
                        break;
                    case 6:
                        sprintf(strAux, " : %s", "GOD");
                        break;                    
                }
                break;
            default:
                strcpy(strAux, "");
                break;
        }
        
        if (index == optionmenu.optionSelection)
        {
            textout_centre_ex(bmpTarget, font, ">> ",centerX - (text_length(font, optionItems[index]) + text_length(font, strAux)) / 2 - 30, centerY + index * optionGap, selectedOptionItemColor, -1);
            textprintf_centre_ex(bmpTarget, font, centerX, centerY + index * optionGap, selectedOptionItemColor, -1, "%s%s", optionItems[index], strAux);
            textprintf_centre_ex(bmpTarget, font, centerX, centerY + index * optionGap + 15, WHITE, -1, " %s ", optionItemsDescription[index]);
        }
        else
        {
            textprintf_centre_ex(bmpTarget, font, centerX, centerY + index * optionGap, unSelectedOptionItemColor, -1, "%s%s", optionItems[index], strAux);
        }
    }
}

/* MISCELANEA FUNCTIONS */
int ballInside(BALL *b, int x1, int y1, int x2, int y2)
{
    if (b->x + b->radius > x1 && b->x + b->radius < x2 &&
        b->y + b->radius > y1 && b->y + b->radius < y2)
        return 1;
    else
        return 0;
}

void createWormHole(BALL *b)
{   
    displayWormHole(backbuffer);
    
    if (ballInside(b, 180, 180, 220, 220))
    {
        b->x = SCREEN_W - 270;
        b->y = SCREEN_H - 270;
    }
        
    if (ballInside(b, SCREEN_W - 220, SCREEN_H - 220, SCREEN_W - 180, SCREEN_H - 180))
    {
        b->x = 270;
        b->y = 270;
    }
}

void displayWormHole(BITMAP *bmpTarget)
{
    int n;
    for (n = 0; n < 20; n++)
    {
        circle(backbuffer, 200 - 2 * n, 200 - 2 * n, n * 2, makecol(10 * n, 10 * n, 10 * n));
        circle(backbuffer, (SCREEN_W - 200) - 2 * n, (SCREEN_H - 200) - 2 * n, n * 2, makecol(10 * n, 10 * n, 10 * n));
    }
    
    textout_centre_ex(bmpTarget, font, "WORM HOLE MODE", SCREEN_W / 4, 10, YELLOW, -1);
    textout_centre_ex(bmpTarget, font, "WORM HOLE MODE", SCREEN_W / 2 + SCREEN_W / 4, 10, YELLOW, -1);
}

/* game functions default */
void errorExit(const char *message)
{
    allegro_message("Error: %s\n", message);
    allegro_exit();
    exit(1);
}

int setupGame(void)
{
    // load game resources and stuff here
    // return false if something failed to load or init
    // BITMAPS, SOUND, FONT
    datFont = load_datafile("data\\fonts.dat");
    datSound = load_datafile("data\\sounds.dat");

    
    initMenu();
    initOptionMenu();

    return TRUE;
}

void renderGame(void)
{
    clear_to_color(backbuffer, BLACK);

    // draw stuff here
    switch (gameSession)
    {
    case MAINMENUGAME:			    
        displayMainMenu(backbuffer);
        break;
    case NORMALGAME:
        displayNormalGame(backbuffer);	        
        break;
    case VERSUSGAME:
        displayVersusGame(backbuffer);
        break;
    case OPTIONSGAME:
        displayOptionMenu(backbuffer);
        break;
    case WINNERGAME:
        displayWinnerGame(backbuffer);
        break;
    default:
        done = TRUE;
        break;
    }
    
    displayDebugMode(backbuffer, debug_mode, YELLOW);
}

void runGame(void)
{   
    int frames_done = 0;
    int old_time = 0;

    /* initialize the array to 0 */
    int frames_array[10] = { 0 }; /* an array to store the number of frames we did during the last 10 tenths of a second */
    int frame_index = 0; /* used to store the index of the last updated value in the array */

    /* FPS / TIMER Control */
    LOCK_VARIABLE(ticks);
    LOCK_FUNCTION(ticker);
    install_int_ex(ticker, BPS_TO_TIMER(UPDATES_PER_SECOND));

    LOCK_VARIABLE(game_time);
    LOCK_FUNCTION(game_time_ticker);
    install_int_ex(game_time_ticker, BPS_TO_TIMER(10)); /* i.e. game time is in tenths of seconds */ 		
    /* FPS / TIMER Control */

    /* Habilitar boto fechar */
    set_close_button_callback(my_allegro_close_button_handler);	

    gameSession = MAINMENUGAME;

    while (!done)
    {   
        while(ticks == 0)
        {
            rest(1);
        }

        while(ticks > 0)
        {
            int old_ticks = ticks;

            /* Do the logic stuff here! */            
            /* ensure the keyboard data is current */			
            if (keyboard_needs_poll()) { poll_keyboard(); }

            /* ensure the mosue data is current */
            if (ENABLE_MOUSE_SUPPORT) { 
                if (mouse_needs_poll()) { poll_mouse(); }
            }

            switch (gameSession)
            {
                case MAINMENUGAME:			    
                    updateMainMenu();
                    break;
                case NORMALGAME:
                    updateNormalGame();
                    break;
                case VERSUSGAME:
                    updateVersusGame();
                    break;
                case OPTIONSGAME:
                    updateOptionMenu();
                    break;
                case WINNERGAME:
                    updateWinnerGame();
                    break;
                default:
                    done = TRUE;
                    break;
            }			
            /* Do the logic stuff here! */

            ticks--;
            if(old_ticks <= ticks)
                break;
        }

        if (game_time >= old_time + 1) /* i.e. a 0.1 second has passed since we last counted the frames */
        {
            fps -= frames_array[frame_index];/* decrement the fps by the frames done a second ago */
            frames_array[frame_index] = frames_done; /* store the number of frames done this 0.1 second */
            fps += frames_done; /* increment the fps by the newly done frames */

            frame_index = (frame_index + 1) % 10; /* increment the frame index and snap it to 10 */

            frames_done = 0;
            old_time += 1;
        }

        /* Draw everything here! */
        renderGame();		
        if (ENABLE_MOUSE_SUPPORT) { show_mouse(backbuffer); }
        
        blit(backbuffer, screen, 0, 0, 0, 0, backbuffer->w, backbuffer->h);		
        /* Draw everything here! */

        frames_done++; /* we drew a frame! */
    }
}

void shutdownGame(void)
{
    /* deallocate anything you allocated for your game here */
    if (datSound) unload_datafile(datSound);
    if (datFont) unload_datafile(datFont);
}

void exitGame(void)
{
    if (backbuffer)
    {
        if (ENABLE_MOUSE_SUPPORT) show_mouse(NULL);
        destroy_bitmap(backbuffer);
    }
    allegro_exit();
}

/*
###################################
# TEMPLATE pong_config.ini
###################################

[graphics]
# define width height color_depth
mode = 640 480 32

# define se o jogo rodara em tela cheia
# 0 = off
# 1 = on
fullscreen = 0

[general]
# define a velocidade maxima da bola
max_count_hit = 18

[debugger]
# permite ver o valor de algumas variaveis em modo runtime
# 0 = off
# 1 = on
debug_mode = 0
*/