// Author: Hannes Pabst

#define NOMINMAX

#include "tetris_base.h"
#include "simple_tile.h"
#include "mt_random_number.h"
#include "no_piece_counter.h"
#include "dirty_field.h"
#include "resource.h"
#include <windows.h>
#include <scrnsave.h>
#include <new>

RECT const NULL_RECT = {0, 0, 0, 0};

template <typename T>
T limit(T val, T const min, T const max)
{
	if (val < min)
		return min;
	else if (val > max)
		return max;
	return val;
}

struct Config
{
	static int const ZOOM_IN = 0;
	static int const FIXED_SIZE = 1;
	static int const ONLY_ONE_FIELD = 2;
	static int const MATRIX_LAYOUT_BOUND = ONLY_ONE_FIELD;

	int frameDelay;
	int matrixLayout;
	int playFieldLines;
	int playFieldLinesMax;
	int waitTime;

	Config(int const frameDelay, int const matrixLayout, int const playFieldLines, int const playFieldLinesMax, int const waitTime)
		: frameDelay(frameDelay)
		, matrixLayout(matrixLayout)
		, playFieldLines(playFieldLines)
		, playFieldLinesMax(playFieldLinesMax)
		, waitTime(waitTime)
	{
	}
};

bool operator ==(Config config0, Config config1)
{
	return
		config0.frameDelay == config1.frameDelay &&
		config0.matrixLayout == config1.matrixLayout &&
		config0.playFieldLines == config1.playFieldLines &&
		config0.playFieldLinesMax == config1.playFieldLinesMax &&
		config0.waitTime == config1.waitTime;
}

Config defaultConfig()
{
	return Config(25, Config::ZOOM_IN, 4, 8, 20);
}

Config limitConfig(Config const config)
{
	return Config(
		limit(config.frameDelay, 10, 200),
		limit(config.matrixLayout, 0, Config::MATRIX_LAYOUT_BOUND),
		max(config.playFieldLines, 1),
		max(config.playFieldLinesMax, 2),
		limit(config.waitTime, 5, 300));
}

Config read()
{
	Config config = defaultConfig();
	HKEY hKey;
	if (RegOpenKey(HKEY_CURRENT_USER, "Control Panel\\Screen Saver.TetrisMatrix", &hKey) == ERROR_SUCCESS)
	{
		DWORD data;
		DWORD size = sizeof(data);
		if (RegQueryValueEx(hKey, "FrameDelay", NULL, NULL, (LPBYTE) &data, &size) == ERROR_SUCCESS)
			config.frameDelay = data;
		if (RegQueryValueEx(hKey, "MatrixLayout", NULL, NULL, (LPBYTE) &data, &size) == ERROR_SUCCESS)
			config.matrixLayout = data;
		if (RegQueryValueEx(hKey, "Lines", NULL, NULL, (LPBYTE) &data, &size) == ERROR_SUCCESS)
			config.playFieldLines = data;
		if (RegQueryValueEx(hKey, "LinesMax", NULL, NULL, (LPBYTE) &data, &size) == ERROR_SUCCESS)
			config.playFieldLinesMax = data;
		if (RegQueryValueEx(hKey, "WaitTime", NULL, NULL, (LPBYTE) &data, &size) == ERROR_SUCCESS)
			config.waitTime = data;
		RegCloseKey(hKey);
	}
	return config;
}

void write(Config config)
{
	HKEY hKey;
	if (RegCreateKey(HKEY_CURRENT_USER, "Control Panel\\Screen Saver.TetrisMatrix", &hKey) == ERROR_SUCCESS)
	{
		DWORD data = config.frameDelay;
		RegSetValueEx(hKey, "FrameDelay", 0, REG_DWORD, (BYTE *) &data, sizeof(data));
		data = config.matrixLayout;
		RegSetValueEx(hKey, "MatrixLayout", 0, REG_DWORD, (BYTE *) &data, sizeof(data));
		data = config.playFieldLines;
		RegSetValueEx(hKey, "Lines", 0, REG_DWORD, (BYTE *) &data, sizeof(data));
		data = config.playFieldLinesMax;
		RegSetValueEx(hKey, "LinesMax", 0, REG_DWORD, (BYTE *) &data, sizeof(data));
		data = config.waitTime;
		RegSetValueEx(hKey, "WaitTime", 0, REG_DWORD, (BYTE *) &data, sizeof(data));
		RegCloseKey(hKey);
	}
}

class PlayerBaseImpl
	: public TetrisBase<PlayerBaseImpl, SimpleTile, DirtyField>
{
public:
	typedef NoPieceCounter PieceCounter;
	typedef MtRandomNumber RandomNumber;

	PlayerBaseImpl()
		: TetrisBase<PlayerBaseImpl, SimpleTile, DirtyField>(4)
	{
	}

	RandomNumber &getRandomNumber() { return randomNumber; }
	PieceCounter &getPieceCounter() { return pieceCounter; }
	DirtyState &getDirtyState() { return dirtyState; }

private:
	RandomNumber randomNumber;
	PieceCounter pieceCounter;
	DirtyState dirtyState;
};

class Player
	: private PlayerBaseImpl
{
public:
	typedef PlayerBaseImpl::PlayField PlayField;
	typedef PlayerBaseImpl::NextField NextField;

	using PlayerBaseImpl::getNextField;
	using PlayerBaseImpl::getPlayField;
	using PlayerBaseImpl::getDirtyState;

	Player()
	{
		PlayerBaseImpl::start();
	}

	void execute()
	{
		Message const msg = aiPlayer();
		bool const handled = handle(msg);
		if (!handled || msg == DROP)
		{
			Event event = PlayerBaseImpl::execute();
			if (event == BLOCK_FIXED)
				event = PlayerBaseImpl::execute();
			if (event == GAME_ENDED)
				PlayerBaseImpl::start();
		}
	}
};

class Layout
{
public:
	struct FieldPoints
	{
		int nextX;
		int nextY;
		int playX;
		int playY;

		FieldPoints(int const nextX, int const nextY, int const playX, int const playY)
			: nextX(nextX)
			, nextY(nextY)
			, playX(playX)
			, playY(playY)
		{
		}
	};

	Layout()
		: screenX(0)
		, screenY(0)
		, screenW(0)
		, screenH(0)
		, numX(0)
		, numY(0)
		, tileS(0)
	{
	}

	Layout(int playFieldsLines, int const screenX, int const screenY, int const screenW, int const screenH)
		: screenX(screenX)
		, screenY(screenY)
		, screenW(screenW)
		, screenH(screenH)
		, numX(0)
		, numY(0)
		, tileS(0)
	{
		bool const onlyOneField = playFieldsLines <= 0;
		if (onlyOneField)
			playFieldsLines = 1;
		int const limitX = screenW / (TILE_S_MIN * AREA_W);
		int const limitY = screenH / (TILE_S_MIN * AREA_H);
		if (limitX > 0 && limitY > 0)
		{
			if (limitX < limitY)
			{
				numX = min(playFieldsLines, limitX);
				tileS = screenW / (numX * AREA_W);
				numY = onlyOneField ? 1 : screenH / (tileS * AREA_H);
			}
			else
			{
				numY = min(playFieldsLines, limitY);
				tileS = screenH / (numY * AREA_H);
				numX = onlyOneField ? 1 : screenW / (tileS * AREA_W);
			}
		}
	}

	FieldPoints getFieldPoints(int const i, int const j) const
	{
		int const sectionX = screenW * i / numX;
		int const sectionY = screenH * j / numY;
		int const sectionW = screenW / numX;
		int const sectionH = screenH / numY;
		int const areaX = screenX + sectionX + (sectionW - AREA_W * tileS) / 2;
		int const areaY = screenY + sectionY + (sectionH - AREA_H * tileS) / 2;
		return FieldPoints(
			areaX + tileS * NEXT_FIELD_X,
			areaY + tileS * NEXT_FIELD_Y,
			areaX + tileS * PLAY_FIELD_X,
			areaY + tileS * PLAY_FIELD_Y);
	}

	int getScreenX() const { return screenX; }
	int getScreenY() const { return screenY; }
	int getScreenW() const { return screenW; }
	int getScreenH() const { return screenH; }
	int getNumX() const { return numX; }
	int getNumY() const { return numY; }
	int getTileS() const { return tileS; }

private:
	static int const AREA_W = Player::PlayField::WIDTH + 2;
	static int const AREA_H = Player::PlayField::HEIGHT + Player::NextField::HEIGHT + 3;
	static int const NEXT_FIELD_X = (Player::PlayField::WIDTH - Player::NextField::WIDTH) / 2 + 1;
	static int const NEXT_FIELD_Y = 1;
	static int const PLAY_FIELD_X = 1;
	static int const PLAY_FIELD_Y = Player::NextField::HEIGHT + 2;
	static int const TILE_S_MIN = 3;

	int screenX, screenY, screenW, screenH;
	int numX, numY;
	int tileS;
};

int getPlayFieldLines(Layout const &layout)
{
	if (layout.getNumX() <= 1 && layout.getNumY() <= 1)
		return 0;
	return min(layout.getNumX(), layout.getNumY());
}

int getPlayFieldNum(Layout const &layout)
{
	return layout.getNumX() * layout.getNumY();
}

struct TilePainterIf
{
	virtual void paintTile(HDC hDC, int x, int y, int s, int color, int edge) = 0;
};

COLORREF getColor(int const colorIdx, int const level)
{
	struct Rgb { int r, g, b; };

	static Rgb const colors[Block::BLOCK_ALIGNED_NUM + 1] =
	{
		{39, 39, 39},
		{85, 85, 255},
		{255, 85, 85},
		{255, 85, 255},
		{255, 170, 85},
		{85, 255, 255},
		{255, 255, 85},
		{85, 255, 85},
	};

	Rgb rgb = colors[colorIdx];
	rgb.r = rgb.r * level / 16;
	rgb.g = rgb.g * level / 16;
	rgb.b = rgb.b * level / 16;

	return RGB(rgb.r, rgb.g, rgb.b);
}

struct TilePainter
	: TilePainterIf
{
	void paintTile(HDC hDC, int x, int y, int s, int color, int)
	{
		HGDIOBJ const hOldBrush = SelectObject(hDC, CreateSolidBrush(getColor(color, 16)));
		HGDIOBJ const hOldPen = SelectObject(hDC, CreatePen(PS_INSIDEFRAME, s / 15, getColor(color, 8)));
		Rectangle(hDC, x, y, x + s, y + s);
		DeleteObject(SelectObject(hDC, hOldPen));
		DeleteObject(SelectObject(hDC, hOldBrush));
	}
};

struct Tile3DPainter
	: TilePainterIf
{
	void paintTile(HDC hDC, int x, int y, int s, int color, int edge)
	{
		short const e = (short) max(s / 10, 1);
		short const x0 = (short) x;
		short const y0 = (short) y;
		short const x1 = (short) (x + s);
		short const y1 = (short) (y + s);

		int const topLevel = 13;
		int const leftLevel = 10;
		int const fillLevel = 16;
		int const rightLevel = 7;
		int const bottomLevel = 4;

		HBRUSH const hTopBrush = CreateSolidBrush(getColor(color, topLevel));
		HBRUSH const hLeftBrush = CreateSolidBrush(getColor(color, leftLevel));
		HBRUSH const hFillBrush = CreateSolidBrush(getColor(color, fillLevel));
		HBRUSH const hRightBrush = CreateSolidBrush(getColor(color, rightLevel));
		HBRUSH const hBottomBrush = CreateSolidBrush(getColor(color, bottomLevel));
		HGDIOBJ const hOldBrush = GetCurrentObject(hDC, OBJ_BRUSH);
		HGDIOBJ const hOldPen = SelectObject(hDC, CreatePen(PS_NULL, 0, RGB(0,0,0)));
		{
			RECT const rect = {x0 + e, y0 + e, x1 - e, y1 - e};
			FillRect(hDC, &rect, hFillBrush);
		}
		{
			RECT const rect = {x0 + e, y0, x1 - e, y0 + e};
			FillRect(hDC, &rect, (edge & BlockEdges::EDGE_T) != 0 ? hTopBrush : hFillBrush);
			SelectObject(hDC, (edge & BlockEdges::EDGE_T) != 0 ? hTopBrush : hLeftBrush);
			POINT const pts0[3] = {{x0, y0}, {x0 + e, y0}, {x0 + e, y0 + e}};
			Polygon(hDC, pts0, 3);
			SelectObject(hDC, (edge & BlockEdges::EDGE_T) != 0 ? hTopBrush : hRightBrush);
			POINT const pts1[3] = {{x1, y0}, {x1 - e, y0}, {x1 - e, y0 + e}};
			Polygon(hDC, pts1, 3);
		}
		{
			RECT const rect = {x0, y0 + e, x0 + e, y1 - e};
			FillRect(hDC, &rect, (edge & BlockEdges::EDGE_L) != 0 ? hLeftBrush : hFillBrush);
			SelectObject(hDC, (edge & BlockEdges::EDGE_L) != 0 ? hLeftBrush : hTopBrush);
			POINT const pts0[3] = {{x0, y0}, {x0, y0 + e}, {x0 + e, y0 + e}};
			Polygon(hDC, pts0, 3);
			SelectObject(hDC, (edge & BlockEdges::EDGE_L) != 0 ? hLeftBrush : hBottomBrush);
			POINT const pts1[3] = {{x0, y1}, {x0, y1 - e}, {x0 + e, y1 - e}};
			Polygon(hDC, pts1, 3);
		}
		{
			RECT const rect = {x1 - e, y1, x0 + e, y1 - e};
			FillRect(hDC, &rect, (edge & BlockEdges::EDGE_B) != 0 ? hBottomBrush : hFillBrush);
			SelectObject(hDC, (edge & BlockEdges::EDGE_B) != 0 ? hBottomBrush : hRightBrush);
			POINT const pts0[3] = {{x1, y1}, {x1 - e, y1}, {x1 - e, y1 - e}};
			Polygon(hDC, pts0, 3);
			SelectObject(hDC, (edge & BlockEdges::EDGE_B) != 0 ? hBottomBrush : hLeftBrush);
			POINT const pts1[3] = {{x0, y1}, {x0 + e, y1}, {x0 + e, y1 - e}};
			Polygon(hDC, pts1, 3);
		}
		{
			RECT const rect = {x1, y1 - e, x1 - e, y0 + e};
			FillRect(hDC, &rect, (edge & BlockEdges::EDGE_R) != 0 ? hRightBrush : hFillBrush);
			SelectObject(hDC, (edge & BlockEdges::EDGE_R) != 0 ? hRightBrush : hBottomBrush);
			POINT const pts0[3] = {{x1, y1}, {x1, y1 - e}, {x1 - e, y1 - e}};
			Polygon(hDC, pts0, 3);
			SelectObject(hDC, (edge & BlockEdges::EDGE_R) != 0 ? hRightBrush : hTopBrush);
			POINT const pts1[3] = {{x1, y0}, {x1, y0 + e}, {x1 - e, y0 + e}};
			Polygon(hDC, pts1, 3);
		}
		DeleteObject(SelectObject(hDC, hOldPen));
		SelectObject(hDC, hOldBrush);
		DeleteObject(hTopBrush);
		DeleteObject(hLeftBrush);
		DeleteObject(hFillBrush);
		DeleteObject(hRightBrush);
		DeleteObject(hBottomBrush);
	}
};

class BackBuffer
{
public:
	BackBuffer()
		: hMemDC(NULL)
		, hOldBmp(NULL)
	{
	}

	bool isCreated() const
	{
		return hMemDC != NULL;
	}

	void create(HDC const hDC, int const w, int const h)
	{
		if (hMemDC == NULL)
			hMemDC = CreateCompatibleDC(hDC);
		else if (hOldBmp != NULL)
			DeleteObject(SelectObject(hMemDC, hOldBmp));
		hOldBmp = SelectObject(hMemDC, CreateCompatibleBitmap(hDC, w, h));
	}

	void destroy()
	{
		if (hOldBmp != NULL)
		{
			DeleteObject(SelectObject(hMemDC, hOldBmp));
			hOldBmp = NULL;
		}
		if (hMemDC != NULL)
		{
			DeleteDC(hMemDC);
			hMemDC = NULL;
		}
	}

	HDC getDC() const
	{
		return hMemDC;
	}

private:
	HDC hMemDC;
	HGDIOBJ hOldBmp;

	// no copy
	BackBuffer(BackBuffer const &other);
	BackBuffer &operator=(BackBuffer const &other);
};

template <bool ALL_EDGES>
class BufferedTilePainter
	: public TilePainterIf
{
public:
	BufferedTilePainter(TilePainterIf * const tilePainter)
		: tilePainter(tilePainter)
		, s(-1)
	{
	}

	void create(HDC const hDC, int const s)
	{
		this->s = s;
		blank.create(hDC, s, s);
		tilePainter->paintTile(blank.getDC(), 0, 0, s, 0, BlockEdges::EDGE_L_T_R_B);
		for (int j = 0; j < EDGE_NUM; ++j)
		{
			for (int i = 0; i < Block::BLOCK_ALIGNED_NUM; ++i)
			{
				tiles[j][i].create(hDC, s, s);
				tilePainter->paintTile(tiles[j][i].getDC(), 0, 0, s, i + 1, ALL_EDGES ? j : BlockEdges::EDGE_L_T_R_B);
			}
		}
	}

	void destroy()
	{
		for (int j = 0; j < EDGE_NUM; ++j)
			for (int i = 0; i < Block::BLOCK_ALIGNED_NUM; ++i)
				tiles[j][i].destroy();
		blank.destroy();
		s = -1;
	}

	void paintTile(HDC hDC, int x, int y, int s, int color, int edge)
	{
		if (this->s != s)
			create(hDC, s);
		BitBlt(hDC, x, y, s, s, getDC(color, edge), 0, 0, SRCCOPY);
	}

private:
	static int const EDGE_NUM = ALL_EDGES ? BlockEdges::TOTAL_NUM : 1;

	TilePainterIf *tilePainter;
	int s;
	BackBuffer blank;
	BackBuffer tiles[EDGE_NUM][Block::BLOCK_ALIGNED_NUM];

	HDC getDC(int const color, int const edge) const
	{
		return color == 0 ? blank.getDC() : tiles[ALL_EDGES ? edge : 0][color - 1].getDC();
	}

	// no copy
	BufferedTilePainter(BufferedTilePainter const &);
	BufferedTilePainter &operator=(BufferedTilePainter const &);
};

class Painter
{
public:
	Painter(TilePainterIf * const tilePainter)
		: tilePainter(tilePainter)
	{
	}

	void create(HDC const hDC, int const width, int const height)
	{
		if (!backBuffer.isCreated())
			backBuffer.create(hDC, width, height);
	}

	void destroy()
	{
		backBuffer.destroy();
	}

	void show(HDC hDC, int const x, int const y, int const width, int const height) const
	{
		BitBlt(hDC, x, y, width, height, backBuffer.getDC(), 0, 0, SRCCOPY);
	}

	void clear(int const left, int const top, int const width, int const height) const
	{
		RECT const rect = {left, top, left + width, top + height};
		FillRect(backBuffer.getDC(), &rect, (HBRUSH) GetStockObject(BLACK_BRUSH));
	}

	void paintTile(int const x, int const y, int const s, int const color, int const edge) const
	{
		tilePainter->paintTile(backBuffer.getDC(), x, y, s, color, edge);
	}

private:
	BackBuffer backBuffer;
	TilePainterIf *tilePainter;

	// no copy
	Painter(Painter const &);
	Painter &operator=(Painter const &);
};

class Animation
{
public:
	Animation()
		: players(NULL)
	{
	}

	void create(int const screenX, int const screenY, int const screenW, int const screenH, int const playFieldLinesMax)
	{
		layout = Layout(playFieldLinesMax, screenX, screenY, screenW, screenH);
		this->playFieldLinesMax = getPlayFieldLines(layout);
		playFieldMax = getPlayFieldNum(layout);
		delete [] players;
		players = new (std::nothrow) Player[playFieldMax];
		startIdx = 0;
		backgroundDirty = true;
	}

	void destroy()
	{
		delete [] players;
		players = NULL;
	}

	const Layout &GetLayout() const
	{
		return layout;
	}

	bool changeLayout()
	{
		if (players != NULL)
		{
			if (zoomLayout())
			{
				for (int j = 0; j < layout.getNumY(); ++j)
				{
					for (int i = 0; i < layout.getNumX(); ++i)
					{
						Player &player = players[startIdx + j * layout.getNumX() + i];
						player.getDirtyState().invalidate();
					}
				}
				backgroundDirty = true;
				return true;
			}
		}
		return false;
	}

	void step()
	{
		if (players != NULL)
		{
			for (int j = 0; j < layout.getNumY(); ++j)
				for (int i = 0; i < layout.getNumX(); ++i)
				{
					Player &player = players[startIdx + j * layout.getNumX() + i];
					player.execute();
				}
		}
	}

	void paint(Painter &painter)
	{
		if (players != NULL)
		{
			if (backgroundDirty)
			{
				backgroundDirty = false;
				painter.clear(layout.getScreenX(), layout.getScreenY(), layout.getScreenW(), layout.getScreenH());
			}
			for (int j = 0; j < layout.getNumY(); ++j)
			{
				for (int i = 0; i < layout.getNumX(); ++i)
				{
					Layout::FieldPoints fieldPoints = layout.getFieldPoints(i, j);
					Player &player = players[startIdx + j * layout.getNumX() + i];
					paintField(painter, player.getNextField(), player.getDirtyState().nextField.getDirtyRect(), fieldPoints.nextX, fieldPoints.nextY, layout.getTileS(), false);
					paintField(painter, player.getPlayField(), player.getDirtyState().playField.getDirtyRect(), fieldPoints.playX, fieldPoints.playY, layout.getTileS(), true);
					player.getDirtyState().validate();
				}
			}
		}
	}

private:
	Player *players;
	Layout layout;
	int playFieldLinesMax;
	int playFieldMax;
	int startIdx;
	bool backgroundDirty;

	bool zoomLayout()
	{
		if (playFieldLinesMax > 1)
		{
			int const curPlayFieldLines = getPlayFieldLines(layout);
			int const curPlayFieldNum = getPlayFieldNum(layout);
			int const newPlayFieldLines = curPlayFieldLines > 1 ? (curPlayFieldLines + 1) / 2 : playFieldLinesMax;
			layout = Layout(newPlayFieldLines, layout.getScreenX(), layout.getScreenY(), layout.getScreenW(), layout.getScreenH());
			int const newPlayFieldNum = getPlayFieldNum(layout);
			if (newPlayFieldNum < curPlayFieldNum)
				startIdx += MtRandomNumber::next(curPlayFieldNum - newPlayFieldNum);
			else
				startIdx = MtRandomNumber::next(playFieldMax - newPlayFieldNum);
			return true;
		}
		return false;
	}

	template <typename Field>
		static void paintField(Painter &painter, Field const &field, Rect const dirty, int const fieldX, int const fieldY, int const tileS, bool const background)
	{
		if (dirty.leftTop.x <= dirty.rightBottom.x && dirty.leftTop.y <= dirty.rightBottom.y)
		{
			if (!background)
			{
				int const left = fieldX + dirty.leftTop.x * tileS;
				int const top = fieldY + dirty.leftTop.y * tileS;
				int const width = (dirty.rightBottom.x - dirty.leftTop.x + 1) * tileS;
				int const height = (dirty.rightBottom.y - dirty.leftTop.y + 1) * tileS;
				painter.clear(left, top, width, height);
			}
			for (int y = dirty.leftTop.y; y <= dirty.rightBottom.y; ++y)
			{
				for (int x = dirty.leftTop.x; x <= dirty.rightBottom.x; ++x)
				{
					typename Field::Elem const tile = field.at(x, y);
					if (background || !tile.isBlank())
						painter.paintTile(fieldX + x * tileS, fieldY + y * tileS, tileS, tile.getColor(), tile.getEdge());
				}
			}
		}
	}
};

struct Display
{
public:
	static int const SCREENS_MAX = 64;

	Display()
		: screenNum(0)
	{
	}

	bool addScreen(RECT const screen)
	{
		if (screenNum < SCREENS_MAX)
		{
			screens[screenNum] = screen;
			++screenNum;
			return true;
		}
		return false;
	}

	RECT getScreen(int const idx) const
	{
		if (screenNum >= 0 && screenNum < SCREENS_MAX)
			return screens[idx];
		return NULL_RECT;
	}

	int getScreenNum() const
	{
		return screenNum;
	}

private:
	RECT screens[SCREENS_MAX];
	int screenNum;
};


BOOL CALLBACK queryMonitors(HMONITOR, HDC, LPRECT const lprcMonitor, LPARAM const data)
{
	Display *display = reinterpret_cast<Display *>(data);
	return display->addScreen(*lprcMonitor) ? TRUE : FALSE;
}

long WINAPI ScreenSaverProc(HWND const hWnd, UINT const message, WPARAM const wParam, LPARAM const lParam)
{
	static Config config = defaultConfig();
	static RECT clientRect = NULL_RECT;
	static Animation *animations = NULL;
	static int animationsNum = 0;
	static TilePainter tilePainter;
	static BufferedTilePainter<false> bufferedTilePainter(&tilePainter);
	static Painter painter(&bufferedTilePainter);
	static int timeCount = 0;
	static unsigned int timerID = 1;

	switch(message)
	{
	case WM_CREATE:
		{
#ifdef _DEBUG
			MoveWindow(hWnd, 0, 0, 320, 240, false);
#endif
			config = limitConfig(read());
			GetClientRect(hWnd, &clientRect);
			int const playFieldLinesMax =
				config.matrixLayout == Config::ZOOM_IN ? config.playFieldLinesMax :
				config.matrixLayout == Config::FIXED_SIZE ? config.playFieldLines : 0;
			Display display;
#ifdef _DEBUG
			{
				RECT const screen0 = {0, 0, 320, 240};
				display.addScreen(screen0);
				//RECT const screen1 = {320, 0, 640, 240};
				//display.addScreen(screen1);
				//RECT const screen2 = {0, 240, 640, 480};
				//display.addScreen(screen2);
			}
#else
    		EnumDisplayMonitors(NULL, &clientRect, queryMonitors, reinterpret_cast<LPARAM>(&display));
#endif
			int const screenNum = display.getScreenNum();
			delete [] animations;
			animations = new (std::nothrow) Animation[screenNum];
			animationsNum = animations != NULL ? screenNum : 0;
			for (int i = 0; i < animationsNum; ++i)
			{
				RECT const screen = display.getScreen(i);
				animations[i].create(screen.left, screen.top, screen.right - screen.left, screen.bottom - screen.top, playFieldLinesMax);
			}
			SetTimer(hWnd, timerID, config.frameDelay, NULL);
			timeCount = 0;
			break;
		}
	case WM_TIMER:
		{
			bool changeLayount = false;
			if (config.matrixLayout == Config::ZOOM_IN)
			{
				timeCount += config.frameDelay;
				if (timeCount >= config.waitTime * 1000)
				{
					timeCount = 0;
					changeLayount = true;
				}
			}
			for (int i = 0; i < animationsNum; ++i)
			{
				if (changeLayount)
					animations[i].changeLayout();
				animations[i].step();
			}
			InvalidateRect(hWnd, NULL, FALSE);
			break;
		}
	case WM_PAINT:
		{
			PAINTSTRUCT ps;
			HDC hDC = BeginPaint(hWnd, &ps);
			painter.create(hDC, clientRect.right, clientRect.bottom);
			for (int i = 0; i < animationsNum; ++i)
				animations[i].paint(painter);
			painter.show(hDC, 0, 0, clientRect.right, clientRect.bottom);
			EndPaint(hWnd, &ps);
			break;
		}
	case WM_DESTROY:
		KillTimer(hWnd, timerID);
		for (int i = 0; i < animationsNum; ++i)
			animations[i].destroy();
		delete [] animations;
		animations = NULL;
		painter.destroy();
		bufferedTilePainter.destroy();
		break;
	}
	return DefScreenSaverProc(hWnd,message,wParam,lParam);
}

BOOL WINAPI RegisterDialogClasses(HANDLE)
{
	return TRUE;
}

void setDialogState(HWND const hWnd)
{
	bool const matrixZoomIn = IsDlgButtonChecked(hWnd, IDC_RADIO_MATRIX_ZOOM_IN) == BST_CHECKED;
	bool const matrixFixed = IsDlgButtonChecked(hWnd, IDC_RADIO_MATRIX_FIXED) == BST_CHECKED;
	EnableWindow(GetDlgItem(hWnd, IDC_EDIT_PLAYFIELD_LINES), matrixFixed ? TRUE : FALSE);
	EnableWindow(GetDlgItem(hWnd, IDC_EDIT_WAIT_TIME), matrixZoomIn ? TRUE : FALSE);
	EnableWindow(GetDlgItem(hWnd, IDC_EDIT_PLAYFIELD_LINES_MAX), matrixZoomIn ? TRUE : FALSE);
}

void initDialog(HWND const hWnd, Config const config)
{
	CheckDlgButton(hWnd, IDC_RADIO_MATRIX_ZOOM_IN, config.matrixLayout == Config::ZOOM_IN ? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDC_RADIO_MATRIX_FIXED, config.matrixLayout == Config::FIXED_SIZE ? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(hWnd, IDC_RADIO_MATRIX_ONLY_ONE_FIELD, config.matrixLayout == Config::ONLY_ONE_FIELD ? BST_CHECKED : BST_UNCHECKED);
	SetDlgItemInt(hWnd, IDC_EDIT_PLAYFIELD_LINES, config.playFieldLines, FALSE);
	SetDlgItemInt(hWnd, IDC_EDIT_PLAYFIELD_LINES_MAX, config.playFieldLinesMax, FALSE);
	SetDlgItemInt(hWnd, IDC_EDIT_WAIT_TIME, config.waitTime, FALSE);
	SetDlgItemInt(hWnd, IDC_EDIT_FRAME_DELAY, config.frameDelay, FALSE);
	setDialogState(hWnd);
}

BOOL WINAPI ScreenSaverConfigureDialog(HWND const hWnd, UINT const message, WPARAM const wParam, LPARAM)
{
	switch (message)
	{
	case WM_INITDIALOG:
		{
			Config const config = limitConfig(read());
			initDialog(hWnd, config);
			SendMessage(GetDlgItem(hWnd, IDC_EDIT_PLAYFIELD_LINES), EM_SETLIMITTEXT, 4, 0);
			SendMessage(GetDlgItem(hWnd, IDC_EDIT_PLAYFIELD_LINES_MAX), EM_SETLIMITTEXT, 4, 0);
			SendMessage(GetDlgItem(hWnd, IDC_EDIT_WAIT_TIME), EM_SETLIMITTEXT, 4, 0);
			SendMessage(GetDlgItem(hWnd, IDC_EDIT_FRAME_DELAY), EM_SETLIMITTEXT, 4, 0);
			return TRUE;
		}
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			{
				bool const matrixZoomIn = IsDlgButtonChecked(hWnd, IDC_RADIO_MATRIX_ZOOM_IN) == BST_CHECKED;
				bool const matrixFixed = IsDlgButtonChecked(hWnd, IDC_RADIO_MATRIX_FIXED) == BST_CHECKED;
				int const matrixLayout = matrixZoomIn ? static_cast<int>(Config::ZOOM_IN) :
					matrixFixed ? static_cast<int>(Config::FIXED_SIZE) : static_cast<int>(Config::ONLY_ONE_FIELD);
				Config config(
					GetDlgItemInt(hWnd, IDC_EDIT_FRAME_DELAY, NULL, FALSE),
					matrixLayout,
					GetDlgItemInt(hWnd, IDC_EDIT_PLAYFIELD_LINES, NULL, FALSE),
					GetDlgItemInt(hWnd, IDC_EDIT_PLAYFIELD_LINES_MAX, NULL, FALSE),
					GetDlgItemInt(hWnd, IDC_EDIT_WAIT_TIME, NULL, FALSE));
				Config const limited = limitConfig(config);
				if (config == limited)
				{
					write(limited);
					EndDialog(hWnd, 0);
				}
				else
				{
					MessageBox(hWnd, "Some values are out of range!\nThe values will now be limited.", "Tetris Matrix", MB_ICONEXCLAMATION);
					initDialog(hWnd, limited);
				}
				return TRUE;
			}
		case IDCANCEL:
			EndDialog(hWnd, 0);
			return TRUE;
		case IDDEFAULT:
			initDialog(hWnd, defaultConfig());
			return TRUE;
		case IDC_RADIO_MATRIX_ZOOM_IN:
		case IDC_RADIO_MATRIX_FIXED:
		case IDC_RADIO_MATRIX_ONLY_ONE_FIELD:
			setDialogState(hWnd);
			return TRUE;
		}
		break;
	case WM_CLOSE:
		EndDialog(hWnd, 0);
		return TRUE;
	}
	return FALSE;
}
