// Author: Hannes Pabst

#include "ai_player.h"
#include "ai_viewer.h"
#include "system.h"
#include <allegro.h>

namespace
{
	unsigned long volatile timeCounter = 0;
	Event timerEvent;

	void timerCallback()
	{
		++timeCounter;
		timerEvent.set();
	}
	END_OF_FUNCTION(timerCallback);

	unsigned long volatile displaySwitchCounter = 0;

	void displaySwitchCallback()
	{
		++displaySwitchCounter;
	}
	END_OF_FUNCTION(displaySwitchCallback)
}

int main(int const argc, char **)
{
	set_uformat(U_ASCII);
	allegro_init();
	install_keyboard();
	set_keyboard_rate(0, 0);

	set_color_depth(8);
	if (set_gfx_mode(argc > 1 ? GFX_AUTODETECT_WINDOWED : GFX_AUTODETECT, 640, 480, 0, 0))
	{
		allegro_message("Error setting graphic mode: %s!\n", allegro_error);
		return 1;
	}

	LOCK_VARIABLE(timeCounter);
	LOCK_FUNCTION(timerCallback);
	install_int_ex(timerCallback, BPS_TO_TIMER(AiViewer::FRAMES_PER_SECOND));

	set_display_switch_mode(SWITCH_BACKGROUND);
	set_display_switch_callback(SWITCH_IN, displaySwitchCallback);

	AiPlayer aiPlayer;
	AiViewer aiViewer(aiPlayer);

	unsigned int time = 0;
	unsigned int curDisplaySwitch = 0;
	int startPage = 1;

	for (;;)
	{
		unsigned int const t = timeCounter;
		unsigned int const deltaTime = t - time;
		time = t;

		if (keypressed())
		{
			int const keyCode = readkey() >> 8;
			if (aiViewer.getPage() == 0)
			{
				if (keyCode == KEY_ESC)
					return 0;
				else
					aiViewer.setPage(startPage);
			}
			else
			{
				switch (keyCode)
				{
				case KEY_ESC:
					startPage = aiViewer.getPage();
					aiViewer.setPage(0);
					break;
				case KEY_SPACE:
					aiPlayer.setFullSpeed(!aiPlayer.isFullSpeed());
					break;
				case KEY_BACKSPACE:
					aiPlayer.start();
					break;
				case KEY_F1:
					aiViewer.setPage(1);
					break;
				case KEY_F2:
					aiViewer.setPage(2);
					break;
				case KEY_F3:
					aiViewer.setPage(3);
					break;
				case KEY_F4:
					aiViewer.setPage(4);
					break;
				case KEY_F5:
					aiViewer.setPage(5);
					break;
				case KEY_F6:
					aiViewer.setPage(6);
					break;
				case KEY_F7:
					aiViewer.setPage(7);
					break;
				case KEY_F8:
					aiViewer.setPage(8);
					break;
				case KEY_PGUP:
					aiViewer.scrollUp();
					break;
				case KEY_PGDN:
					aiViewer.scrollDown();
					break;
				case KEY_F12:
					saveToFile(aiPlayer);
					break;
				}
			}
		}

 		if (aiViewer.getPage() != 0)
			aiPlayer.execute(deltaTime);

		if (deltaTime != 0)
		{
			unsigned int const ds = displaySwitchCounter;
			if (curDisplaySwitch != ds)
			{
				curDisplaySwitch = ds;
				aiViewer.invalidate();
			}
			acquire_screen();
			aiViewer.paint();
			release_screen();
		}

		if (!aiPlayer.isFullSpeed() || aiViewer.getPage() == 0)
			timerEvent.wait();
	}
}
END_OF_MAIN();
