// Author: Hannes Pabst

#include "ai_viewer.h"
#include "statistic.h"
#include "statistic_out.h"
#include "grid_canvas.h"
#include <stdio.h>
#include <math.h>

namespace
{
	void paintTitle()
	{
		char const VERSION_TEXT[] = "VERSION " VERSION;

		GridCanvas::drawText("TETRIS AI PLAYER" , 0, 0);
		GridCanvas::drawText("CREATED BY HANNES PABST" , 0, 2);
		GridCanvas::drawText(VERSION_TEXT, 0, 4);
		GridCanvas::drawText("ESC          VIEW TITLE SCREEN" , 0, 8);
		GridCanvas::drawText("F1           VIEW PLAYFIELD" , 0, 10);
		GridCanvas::drawText("F2           VIEW STATISTICS: ROTATION OF BLOCK" , 0, 12);
		GridCanvas::drawText("F3           VIEW STATISTICS: TYPE OF BLOCK" , 0, 14);
		GridCanvas::drawText("F4           VIEW STATISTICS: NUMBER OF CLEARD LINES" , 0, 16);
		GridCanvas::drawText("F5           VIEW STATISTICS: PILE HEIGHT" , 0, 18);
		GridCanvas::drawText("F6           VIEW STATISTICS: SUM OF TILES AND GAPS" , 0, 20);
		GridCanvas::drawText("F7           VIEW STATISTICS: NUMBER OF TILES" , 0, 22);
		GridCanvas::drawText("F8           VIEW STATISTICS: NUMBER OF GAPS" , 0, 24);
		GridCanvas::drawText("PAGE UP/DOWN SCROLL LIST" , 0, 26);
		GridCanvas::drawText("F12          SAVE STATISTICS TO FILE" , 0, 28);
		GridCanvas::drawText("SPACE        TOGGLE FULL SPEED ON/OFF" , 0, 30);
		GridCanvas::drawText("BACKSPACE    RESTART GAME" , 0, 32);
		GridCanvas::drawText("PRESS ESC TO QUIT, PRESS ANY OTHER KEY TO CONTINUE" , 0, 36);
	}

	void paintGame(AiPlayer const &aiPlayer)
	{
		int const FIELD_X = (GridCanvas::COLS - AiPlayer::Base::PlayField::WIDTH) / 2;
		int const FIELD_Y = (GridCanvas::ROWS - AiPlayer::Base::PlayField::HEIGHT) / 2;
		int const NEXT_FIELD_X = FIELD_X - 5;
		int const NEXT_FIELD_Y = FIELD_Y + 1;
		int const DESCRIPTION_WIDTH = 16;
		int const VALUE_WIDTH = 16;

		for (int y = 0; y < AiPlayer::Base::NextField::HEIGHT; ++y)
			for (int x = 0; x < AiPlayer::Base::NextField::WIDTH; ++x)
				GridCanvas::drawTile(aiPlayer.getNextField().at(x, y).getColor(), NEXT_FIELD_X + x, NEXT_FIELD_Y + y);

		for (int y = 0; y < AiPlayer::Base::PlayField::HEIGHT; ++y)
			for (int x = 0; x < AiPlayer::Base::PlayField::WIDTH; ++x)
				GridCanvas::drawTile(aiPlayer.getPlayField().at(x, y).getColor(), FIELD_X + x, FIELD_Y + y);

		sprintf(buffer, "%-*s %*.*f", DESCRIPTION_WIDTH, "BLOCKS", VALUE_WIDTH, 0, aiPlayer.getBlocks());
		GridCanvas::drawText(buffer, 0, 0);
 		if (aiPlayer.isFullSpeed())
			sprintf(buffer, "%-*s %*.*f", DESCRIPTION_WIDTH, "BLOCKS/SECOND", VALUE_WIDTH, 0, aiPlayer.getBlocksPerFrame() * AiViewer::FRAMES_PER_SECOND);
		else
			sprintf(buffer, "%-*s", GridCanvas::COLS, "");
		GridCanvas::drawText(buffer, 0, 1);
	}

	void paintStonesPercent(BlockHistograms const &blockHistograms)
	{
		GridCanvas::drawText("ROTATION OF BLOCK PERCENT", 0, 0);
		for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
		{
			int rotatedBlock = block;
			BlockHistograms::RotationsFrequencies rotationsFrequencies;
			blockHistograms.calculateRotationsFrequencies(rotationsFrequencies, rotatedBlock);
			Statistic::Buffer<BlockHistograms::RotationsFrequencies::SIZE> statisticBuffer;
			Statistic const statistic(rotationsFrequencies, statisticBuffer);
			int rotationCount = 0;
			do
			{
				for (int tile = 0; tile < Block::TILE_NUM; ++tile)
				{
					Point const tilePosition = Block::getTilePosition(rotatedBlock, tile);
					GridCanvas::drawTile(block + 1, tilePosition.x + block * 11, tilePosition.y + rotationCount * 7 + 2);
				}
				sprintf(buffer, "%*.*f", 11, 11 - 4, statistic.frequencyTable[rotationCount].percent);
				GridCanvas::drawText(buffer, block * 11, rotationCount * 7 + 7);
				++rotationCount;
				rotatedBlock = Block::rotate(rotatedBlock);
			}
			while (rotatedBlock != block);
		}
	}

	void paintStatistics(char const * const description, Statistic const &statistic, int const page, int const linesPerPage, char const *names[] = NULL)
	{
		int const VALUE_WIDTH = 16;
		int const FREQUENCY_WIDTH = 16;
		int const PERCENT_WIDTH = 16;
		int const PERCENT_PRECISION = 12;
		int const COMMA_VALUE_WIDTH = 16;
		int const COMMA_VALUE_PRECISION = 12;
		int const DESCRIPTION_WIDTH = 16;

		GridCanvas::drawText(description, 0, 0);
		int const offset = sprintf(buffer, "%-*s %-*s %-*s", VALUE_WIDTH, "VALUE", FREQUENCY_WIDTH, "FREQUENCY", PERCENT_WIDTH, "PERCENT");
		if (names == NULL)
			sprintf(buffer + offset, " %-*s", PERCENT_WIDTH, "CUMULATIVE PRCNT");
		GridCanvas::drawText(buffer, 0, 2);

		int const startValue = linesPerPage * page;
		int const endValue = min(startValue + linesPerPage, statistic.frequencyTable.getSize() - 1);
		int y = 3;
		for (int value = startValue; value <= endValue; ++value)
		{
			Statistic::Entry const entry = statistic.frequencyTable[value];

			int offset = names != NULL ? sprintf(buffer, "%*s", VALUE_WIDTH, names[value]) : sprintf(buffer, "%*d", VALUE_WIDTH, value);
			offset += sprintf(buffer + offset, " %*.*f %*.*f", FREQUENCY_WIDTH, 0, entry.frequency, PERCENT_WIDTH, PERCENT_PRECISION, entry.percent);
			if (names == NULL)
				sprintf(buffer + offset, " %*.*f", PERCENT_WIDTH, PERCENT_PRECISION, entry.cumulativePercent);
			GridCanvas::drawText(buffer, 0, y++);
		}
		sprintf(buffer, "%-*s %*.*g", VALUE_WIDTH, "TOTAL", FREQUENCY_WIDTH, FREQUENCY_WIDTH - 6, statistic.total);
		GridCanvas::drawText(buffer, 0, linesPerPage + 5);
		if (names == NULL)
		{
			sprintf(buffer, "%-*s %*.*f", DESCRIPTION_WIDTH, "MAXIMUM", COMMA_VALUE_WIDTH, COMMA_VALUE_PRECISION, statistic.maximum);
			GridCanvas::drawText(buffer, 0, linesPerPage + 7);
			sprintf(buffer, "%-*s %*.*f", DESCRIPTION_WIDTH, "MEAN", COMMA_VALUE_WIDTH, COMMA_VALUE_PRECISION, statistic.mean);
			GridCanvas::drawText(buffer, 0, linesPerPage + 8);
			sprintf(buffer, "%-*s %*.*f", DESCRIPTION_WIDTH, "STD DEV", COMMA_VALUE_WIDTH, COMMA_VALUE_PRECISION, sqrt(statistic.variance));
			GridCanvas::drawText(buffer, 0, linesPerPage + 9);
		}
	}
}

int const AiViewer::scrollPagesMax[PAGES_NUM] = {0, 0, 0, 0, 0, 0, 3, 3, 3};

AiViewer::AiViewer(AiPlayer const &aiPlayer)
: aiPlayer(&aiPlayer)
, page(0)
, clear(true)
{
	for (int i = 0; i < PAGES_NUM; ++i)
		scrollPages[i] = 0;
}

void AiViewer::scrollUp()
{
	if (scrollPages[page] > 0)
	{
		--scrollPages[page];
		clear = true;
	}
}

void AiViewer::scrollDown()
{
	if (scrollPages[page] < scrollPagesMax[page])
	{
		++scrollPages[page];
		clear = true;
	}
}

void AiViewer::setPage(int page)
{
	if (page != this->page)
	{
		this->page = page;
		clear = true;
	}
}

void AiViewer::invalidate()
{
	clear = true;
}

void AiViewer::paint()
{
	if (clear)
	{
		GridCanvas::clearScreen();
		clear = false;
	}
	switch (page)
	{
	case 0:
		paintTitle();
		break;
	case 1:
		paintGame(*aiPlayer);
		break;
	case 2:
		paintStonesPercent(aiPlayer->getHistograms());
		break;
	case 3:
		{
			Statistic::Buffer<AiPlayer::Histograms::AlignedBlockFrequencies::SIZE> statisticBuffer;
			paintStatistics("TYPE OF BLOCK", Statistic(aiPlayer->getHistograms().getAlignedBlockFrequencies(), statisticBuffer), 0, Block::BLOCK_ALIGNED_NUM - 1, blockNames);
		}
		break;
	case 4:
		{
			Statistic::Buffer<AiPlayer::Histograms::ClearFrequencies::SIZE> statisticBuffer;
			paintStatistics("NUMBER OF CLEARED LINES", Statistic(aiPlayer->getHistograms().getClearFrequencies(), statisticBuffer), 0, Block::BLOCK_HEIGHT_MAX);
		}
		break;
	case 5:
		{
			Statistic::Buffer<AiPlayer::Histograms::HeightFrequencies::SIZE> statisticBuffer;
			paintStatistics("PILE HEIGHT", Statistic(aiPlayer->getHistograms().getPileHeights(), statisticBuffer), 0, AiPlayer::Histograms::HeightFrequencies::SIZE - 1);
		}
		break;
	case 6:
		{
			Statistic::Buffer<AiPlayer::Histograms::TileFrequencies::SIZE> statisticBuffer;
			paintStatistics("SUM OF TILES AND GAPS", Statistic(aiPlayer->getHistograms().getSumTileGapFrequencies(), statisticBuffer), scrollPages[page], 50);
		}
		break;
	case 7:
		{
			Statistic::Buffer<AiPlayer::Histograms::TileFrequencies::SIZE> statisticBuffer;
			paintStatistics("NUMBER OF TILES", Statistic(aiPlayer->getHistograms().getTileFrequencies(), statisticBuffer), scrollPages[page], 50);
		}
		break;
	case 8:
		{
			Statistic::Buffer<AiPlayer::Histograms::TileFrequencies::SIZE> statisticBuffer;
			paintStatistics("NUMBER OF GAPS", Statistic(aiPlayer->getHistograms().getGapFrequencies(), statisticBuffer), scrollPages[page], 50);
		}
		break;
	}
}
