// Author: Hannes Pabst

#include "ai_player.h"
#include "statistic.h"
#include "statistic_out.h"
#include <stdio.h>
#include <math.h>

namespace
{
	typedef Statistic::Buffer<AiPlayer::Histograms::TileFrequencies::SIZE> StatisticBuffer;

	void saveBlockLegend(FILE &f)
	{
		fprintf(&f, "Tetris AI Player by Hannes Pabst\n\n\n");
		fprintf(&f, "Blocks (Type / Rotation)\n\n");
		fprintf(&f, " ");
		int blocks[Block::BLOCK_ALIGNED_NUM];
		for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
		{
			blocks[block] = block;
			for (int col = -1; col < Block::TILE_NUM; ++col)
			{
				if (col == 2)
					fprintf(&f, "%s", blockNames[block]);
				else
					fprintf(&f, " ");
			}
		}
		fprintf(&f, "\n");
		int tiles[Block::BLOCK_ALIGNED_NUM];
		for (int rotation = 0; rotation < Block::ROTATE_CYCLE_MAX; ++rotation)
		{
			for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
				tiles[block] = 0;
			for (int row = -1; row < Block::TILE_NUM; ++row)
			{
				if (row == 2)
					fprintf(&f, "%d", rotation);
				else
					fprintf(&f, " ");
				for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
				{
					for (int col = -1; col < Block::TILE_NUM; ++col)
					{
						Point const tilePosition = Block::getTilePosition(blocks[block], tiles[block]);
						if (blocks[block] == -1 ||
							tiles[block] >= Block::TILE_NUM ||
							tilePosition.y != row ||
							tilePosition.x != col)
						{
							fprintf(&f, " ");
						}
						else
						{
							fprintf(&f, "X");
							++tiles[block];
						}
					}
				}
				fprintf(&f, "\n");
			}
			for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
			{
				if (blocks[block] != -1)
				{
					int rotatedBlock = Block::rotate(blocks[block]);
					if (rotatedBlock < Block::BLOCK_ALIGNED_NUM)
						blocks[block] = -1;
					else
						blocks[block] = rotatedBlock;
				}
			}
		}
		fprintf(&f, "\n\n\n");
	}

	void saveSeed(FILE &f, unsigned long const seed)
	{
		fprintf(&f, "Random number generator\n");
		fprintf(&f, "Name: MT19937\n");
		fprintf(&f, "Seed: %lu\n\n\n\n", seed);
	}

	void saveStatistic(FILE &f, char const * const description, Statistic const &statistic, char const *names[] = NULL, int const valueBound = -1)
	{
		int const PRECISION = 12;

		fprintf(&f, "%s\n\n", description);
		fprintf(&f, "Value\tFrequency\tPercent%s\n", (names == NULL) ? "\tCumulative Percent" : "");
		int const bound = valueBound == -1 ? statistic.frequencyTable.getSize() : valueBound;
		for (int value = 0; value < bound; ++value)
		{
			Statistic::Entry const entry = statistic.frequencyTable[value];
			if (names != NULL)
				fprintf(&f, "%s", names[value]);
			else
				fprintf(&f, "%d", value);
			fprintf(&f, "\t%.*f\t%.*f", 0, entry.frequency, PRECISION, entry.percent);
			if (names == NULL)
				fprintf(&f, "\t%.*f", PRECISION, entry.cumulativePercent);
			fprintf(&f, "\n");
		}
		fprintf(&f, "\n");
		fprintf(&f, "Total\t%.*f\n", 0, statistic.total);
		if (names == NULL)
		{
			fprintf(&f, "\n");
			fprintf(&f, "Maximum\t%.*f\n", 0, statistic.maximum);
			fprintf(&f, "Mean\t%.*f\n", PRECISION, statistic.mean);
			fprintf(&f, "Std Dev\t%.*f\n", PRECISION, sqrt(statistic.variance));
		}
		fprintf(&f, "\n\n\n");
	}

	void saveRotationStatistic(FILE &f, BlockHistograms const &blockHistograms, StatisticBuffer &statisticBuffer)
	{
		for (int block = 0; block < Block::BLOCK_ALIGNED_NUM; ++block)
		{
			BlockHistograms::RotationsFrequencies rotationsFrequencies;
			int const rotationCount = blockHistograms.calculateRotationsFrequencies(rotationsFrequencies, block);
			sprintf(buffer, "Rotation of block type %s", blockNames[block]);
			saveStatistic(f, buffer, Statistic(rotationsFrequencies, statisticBuffer), NULL, rotationCount);
		}
	}
}

AiPlayer::AiPlayer()
: fullSpeed(false)
{
	resetFullSpeedMeasure();
	start();
}

void AiPlayer::start()
{
	seed = MtRandomNumber::timeSeed();
	MtRandomNumber::seed(seed);
	histograms = Histograms();
	blocks = 0;
	Base::start();
}

void AiPlayer::execute(int frames)
{
	Message const msg = aiPlayer();
	bool const handled = handle(msg);
	if (!handled || msg == DROP)
	{
		Event event = Base::execute();
		switch (event)
		{
		case LINE_CLEARED:
		case BLOCK_FIXED:
			if (getClearInfo().lineIdx == getClearInfo().lineNum)
			{
				histograms.addField(getFieldInfo());
				histograms.addBlock(getPlayBlock(), getClearInfo().lineNum);
				++blocks;
				if (fullSpeed)
					++fullSpeedBlocks;
			}
			break;
		case GAME_ENDED:
			saveToFile(*this);
			fullSpeed = false;
			break;
        default:
            break;
		}
	}
	if (fullSpeed && getState() != GAME_OVER)
		fullSpeedFrames += frames;
}

void AiPlayer::setFullSpeed(bool const fullSpeed)
{
	this->fullSpeed = fullSpeed;
	if (fullSpeed)
		resetFullSpeedMeasure();
}

void AiPlayer::resetFullSpeedMeasure()
{
	fullSpeedBlocks = 0;
	fullSpeedFrames = 0;
}

void saveToFile(AiPlayer const &aiPlayer)
{
	sprintf(buffer, "%0*.*f-%0*lu.txt", 16, 0, aiPlayer.getBlocks(), 10, aiPlayer.getSeed());
	FILE * const f = fopen(buffer, "wt");
	if (f != NULL)
	{
		AiPlayer::Histograms const &histograms = aiPlayer.getHistograms();
		saveBlockLegend(*f);
		saveSeed(*f, aiPlayer.getSeed());
		StatisticBuffer statisticBuffer;
		saveStatistic(*f, "Type of block", Statistic(histograms.getAlignedBlockFrequencies(), statisticBuffer), blockNames);
		saveRotationStatistic(*f, histograms, statisticBuffer);
		saveStatistic(*f, "Number of cleared lines", Statistic(histograms.getClearFrequencies(), statisticBuffer));
		saveStatistic(*f, "Number of gaps (empty cells below a tile)", Statistic(histograms.getGapFrequencies(), statisticBuffer));
		saveStatistic(*f, "Number of tiles", Statistic(histograms.getTileFrequencies(), statisticBuffer));
		saveStatistic(*f, "Sum of tiles and gaps", Statistic(histograms.getSumTileGapFrequencies(), statisticBuffer));
		saveStatistic(*f, "Pile height", Statistic(histograms.getPileHeights(), statisticBuffer));
		fclose(f);
	}
}
