// Author: Hannes Pabst

#ifndef TETRIS_H
#define TETRIS_H

#include "tetris_base.h"
#include "tile.h"
#include "random_number.h"
#include "piece_counter.h"
#include "dirty_field.h"
#include "dirty_flag.h"
#include "result.h"
#include <stdio.h>

class TetrisBaseImpl
	: public TetrisBase<TetrisBaseImpl, Tile, DirtyField>
{
public:
	typedef ::PieceCounter<1000> PieceCounter;
	typedef ::RandomNumber RandomNumber;

	TetrisBaseImpl(DirtyState &dirtyState);

	RandomNumber &getRandomNumber();
	PieceCounter &getPieceCounter();
	DirtyState &getDirtyState();

private:
	RandomNumber randomNumber;
	PieceCounter pieceCounter;
	DirtyState &dirtyState;

	// no copy
	TetrisBaseImpl(TetrisBaseImpl const &);
	TetrisBaseImpl &operator=(TetrisBaseImpl const &);
};

class Tetris
	: private TetrisBaseImpl
{
public:
	enum Mode { RECORD, REPLAY, DEMO };

	struct InfoDirtyFlag
		: DirtyFlag
	{
		static int const DEMO_MODE = 1 << 1;
		static int const GAME_RUNNING = 1 << 2;
		static int const BLOCKS_SCORE = 1 << 3;
	};

	typedef TetrisBaseImpl Base;

	struct DirtyState
	{
		Base::DirtyState fields;
		InfoDirtyFlag info;

		void validate();
		void invalidate();
	};

	struct SoundPlayer
	{
		virtual void sound(Tetris const &tetris, Base::Event event, bool normalGameSpeed) = 0;
	};

	struct Input
	{
		bool turn, drop, left, right;
	};

	Tetris(DirtyState &dirtyState, SoundPlayer &soundPlayer, Input const &input, char const *tempFile);
	~Tetris();

	void start(Mode mode, FILE *data = NULL);

	bool execute();

	FILE *getData() const;
	Result const &getResult() const;
	Mode getDemoMode() const;
	bool isGameRunning() const;

	using Base::getNextField;
	using Base::getPlayField;

private:
	DirtyState &dirtyState;
	SoundPlayer &soundPlayer;
	Input const &input;
	char const *tempFile;

	Result result;
	Mode mode;

	FILE *writeFile, *readFile;
	int keyCode, runLength;
	long saveSeed;
	bool seedDirty;

	bool turnFlag, dropFlag;
	int aiTimer, moveTimer, moveDir, executeTimer, stepDelay;
	unsigned int dscore;

	void writeCode();

	// no copy
	Tetris(Tetris const &);
	Tetris &operator=(Tetris const &);
};


inline TetrisBaseImpl::RandomNumber &TetrisBaseImpl::getRandomNumber()
{
	return randomNumber;
}

inline TetrisBaseImpl::PieceCounter &TetrisBaseImpl::getPieceCounter()
{
	return pieceCounter;
}

inline TetrisBaseImpl::DirtyState &TetrisBaseImpl::getDirtyState()
{
	return dirtyState;
}

inline FILE *Tetris::getData() const
{
	return writeFile;
}

inline Result const &Tetris::getResult() const
{
	return result;
}

inline Tetris::Mode Tetris::getDemoMode() const
{
	return mode;
}

inline bool Tetris::isGameRunning() const
{
	return getState() != GAME_OVER;
}

#endif
