// Author: Hannes Pabst

#include "tetris.h"
#include "execute.h"

namespace
{
	int const SPEED_LEVELS = 25;
	int const BASE_DELAY = FRAMES_PER_SECOND / 15;
	int const MOVE_REPEAT_DELAY = FRAMES_PER_SECOND * 2 / 15;
	int const MOVE_START_DELAY = FRAMES_PER_SECOND * 3 / 15;
	int const PREPARE_DELAY = FRAMES_PER_SECOND;
	int const DROP_DELAY = FRAMES_PER_SECOND / 60;
	int const FLASH_DELAY = FRAMES_PER_SECOND / 60;
	int const FLASH_NUM = 7;
	int const WRITE_BUFFER_SIZE = 1024 * 16;
	int const KEY_CODE_SENTINEL = -1;
}

TetrisBaseImpl::TetrisBaseImpl(DirtyState &dirtyState)
	: TetrisBase<TetrisBaseImpl, Tile, DirtyField>(FLASH_NUM)
	, dirtyState(dirtyState)
{
}

void Tetris::DirtyState::validate()
{
	info.validate();
	fields.validate();
}

void Tetris::DirtyState::invalidate()
{
	info.invalidate();
	fields.invalidate();
}

Tetris::Tetris(DirtyState &dirtyState, SoundPlayer &soundPlayer, Input const &input, char const * const tempFile)
: TetrisBaseImpl(dirtyState.fields)
, dirtyState(dirtyState)
, soundPlayer(soundPlayer)
, input(input)
, tempFile(tempFile)
, mode(RECORD)
, writeFile(fopen(tempFile, "w+b"))
, saveSeed(long(time(NULL)))
, seedDirty(true)
{
	if (writeFile != NULL)
		setvbuf(writeFile, NULL, _IOFBF, WRITE_BUFFER_SIZE);
}

Tetris::~Tetris()
{
	if (writeFile != NULL)
	{
		rewind(writeFile);
		fclose(writeFile);
	}
	remove(tempFile);
}

void Tetris::start(Mode const mode, FILE * const data)
{
	result.blocks = 0;
	result.score = 0;
	this->mode = mode;
	turnFlag = false;
	dropFlag = false;
	aiTimer = 0;
	moveTimer = 0;
	moveDir	= 0;
	executeTimer = PREPARE_DELAY;
	stepDelay = (BASE_DELAY + (SPEED_LEVELS - 1));
	dirtyState.info.add(InfoDirtyFlag::DEMO_MODE | InfoDirtyFlag::GAME_RUNNING | InfoDirtyFlag::BLOCKS_SCORE);
	long seed = seedDirty ? saveSeed : static_cast<long>(getRandomNumber());
	seedDirty = false;
	switch (mode)
	{
	case RECORD:
		if (writeFile != NULL)
		{
			keyCode = KEY_CODE_SENTINEL;
			rewind(writeFile);
			fwrite(&seed, sizeof(seed), 1, writeFile);
			unsigned char const code = 240;
			fwrite(&code, sizeof(code), 1, writeFile);
		}
		break;
	case REPLAY:
		readFile = data;
		if (readFile != NULL)
		{
			saveSeed = seed;
			seedDirty = true;
			seed = 0;
			runLength = 0;
			fread(&seed, sizeof(seed), 1, readFile);
			unsigned char code;
			fread(&code, sizeof(code), 1, readFile);
		}
		break;
    default:
        break;
    }
	getRandomNumber() = RandomNumber(seed);
	TetrisBaseImpl::start();
}

void Tetris::writeCode()
{
	unsigned char const code = static_cast<unsigned char>(keyCode << 4 | runLength);
	fwrite(&code, sizeof(code), 1, writeFile);
}

bool Tetris::execute()
{
	bool gameEnded = false;
	Input input = this->input;
	int const dt = mode == RECORD ? 1 : input.drop ? 64 : input.right ? 16 : input.left ? 4 : input.turn ? 0: 1;
	for (int i = dt; isGameRunning() && i > 0; --i)
	{
		switch (mode)
		{
		case RECORD:
			if (writeFile != NULL)
			{
				int keyCodeOut = 0;
				if (input.turn)
					keyCodeOut |= 1 << 0;
				if (input.drop)
					keyCodeOut |= 1 << 1;
				if (input.left)
					keyCodeOut |= 1 << 2;
				if (input.right)
					keyCodeOut |= 1 << 3;
				if (keyCodeOut == keyCode && runLength < (1 << 4) - 1)
					++runLength;
				else
				{
					if (keyCode != KEY_CODE_SENTINEL)
						writeCode();
					keyCode = keyCodeOut;
					runLength = 0;
				}
			}
			break;
		case REPLAY:
			{
				int keyCodeIn = 0;
				if (readFile != NULL)
				{
					if (runLength > 0)
						--runLength;
					else
					{
						unsigned char code = 0;
						fread(&code, sizeof(code), 1, readFile);
						keyCode = code >> 4;
						runLength = code & ((1 << 4) - 1);
					}
					keyCodeIn = keyCode;
				}
				input.turn = (keyCodeIn & 1 << 0) != 0;
				input.drop = (keyCodeIn & 1 << 1) != 0;
				input.left = (keyCodeIn & 1 << 2) != 0;
				input.right = (keyCodeIn & 1 << 3) != 0;
			}
			break;
		case DEMO:
			if (!dropFlag || !turnFlag)
				aiTimer = MOVE_REPEAT_DELAY - 1;
			else if (aiTimer > 0)
				--aiTimer;
			input.drop = aiTimer <= 0 && aiPlayer(DROP);
			input.turn = aiTimer <= 0 && aiPlayer(TURN);
			input.left = aiPlayer(LEFT);
			input.right = aiPlayer(RIGHT);
			break;
		}
		if (!input.drop)
			dropFlag = true;
		else if (dropFlag && handle(DROP))
		{
			dropFlag = false;
			executeTimer = 1;
		}
		if (!input.turn)
			turnFlag = true;
		else if (turnFlag && handle(TURN))
			turnFlag = false;
		if (moveTimer < 0 || --moveTimer <= 0)
		{
			if ((input.left && !input.right && handle(LEFT)))
			{
				moveTimer = (moveTimer < 0 || moveDir == 1) ? MOVE_START_DELAY : MOVE_REPEAT_DELAY;
				moveDir = -1;
			}
			else if ((input.right && !input.left && handle(RIGHT)))
			{
				moveTimer = (moveTimer < 0 || moveDir == -1) ? MOVE_START_DELAY : MOVE_REPEAT_DELAY;
				moveDir = 1;
			}
		}
		if (executeTimer > 0 && --executeTimer <= 0)
		{
			Event const event = TetrisBaseImpl::execute();
			switch (event)
			{
			case NEW_BLOCK:
				executeTimer = stepDelay;
				moveTimer = MOVE_REPEAT_DELAY;
				moveDir = 0;
				dscore = 0;
				break;
			case STEP_DOWN:
				if (getState() == DROPPING)
				{
					executeTimer = DROP_DELAY;
					dscore += 2;
				}
				else
				{
					executeTimer = stepDelay;
					++dscore;
				}
				break;
			case BLOCK_FIXED:
				executeTimer = FLASH_DELAY;
				result.score += dscore;
				++result.blocks;
				if (stepDelay > BASE_DELAY && result.blocks % 10 == 0)
					--stepDelay;
				dirtyState.info.add(InfoDirtyFlag::BLOCKS_SCORE);
				break;
			case LINE_FLASHED:
			case LINE_CLEARED:
				executeTimer = FLASH_DELAY;
				break;
			case GAME_ENDED:
				if (mode == RECORD && writeFile != NULL)
					writeCode();
				_time64(&result.time);
				dirtyState.info.add(InfoDirtyFlag::GAME_RUNNING);
				gameEnded = true;
				break;
            default:
                break;
			}
			soundPlayer.sound(*this, event, dt == 1);
		}
	}
	return gameEnded;
}
