// Author: Hannes Pabst

#ifndef TABLE_H
#define TABLE_H

#include "result.h"
#include "path.h"
#include <stdio.h>

class Table
{
public:
	static int const FILE_NAME_SIZE = 40;
	static int const DIR_SIZE_MAX = PATH_SIZE - FILE_NAME_SIZE;
	static int const BLOCKS_RANGE = 100000;
	static int const SCORE_RANGE = 100000;
	static int const NAME_SIZE = 11;
	static int const ENTRY_TITLE = -1;
	static int const ENTRY_MAX = 10;

	typedef char Name[NAME_SIZE];

	struct Info
	{
		Result result;
		Name name;

		Info(Result result = Result());
	};

	class DirtyState
	{
	public:
		static int const NOTHING = 0;
		static int const ENTRY = 1;
		static int const TABLE = 2;
		static int const ALL = 3;

		DirtyState();

		void invalidate();
		void validate();
		void update(int dirtyState);
		int getDirtyState() const;

	private:
		int dirtyState;
	};

	Table(DirtyState &dirtyState, char const *tableDir);
	~Table();

	void insert(Result const &result);
	void addChar(char charCode);
	void deleteChar();
	void finishInput(FILE *data);
	void blinkCursor();

	void setEntry(int entry);
	FILE *getData(int entry);

	bool isInsertMode() const;
	int getEntryNum() const;
	int getEntry() const;
	int getCursorPos() const;
	bool isCursorBlink() const;
	Info getEntryInfo(int entry) const;

private:
	typedef char FileName[FILE_NAME_SIZE];

	struct Entry
	{
		Info info;
		FileName fileName;

		Entry(Info const &info = Info());
		Entry(FileName const fileName);
	};

	DirtyState &dirtyState;
	char const *tableDir;
	FILE *readFile;

	bool insertMode;
	FileName fileNameForDelete;

	Entry table[ENTRY_MAX];
	int entryNum, entry, cursorPos;
	bool cursorBlink;

	static void storeEntry(Path const path, Info const &info, FILE *data);
	static void write(FILE *file, Info const &info);
	static bool read(FILE *file, Info &info);

	void close();
	void makeFilePath(Path path, FileName const fileName) const;
	bool makeFileName(Path path, FileName fileName, Info const &info) const;
	int insertEntry(Entry const &e);
	void loadTable();

	// no copy
	Table(Table const &);
	Table &operator=(Table const &);
};


inline int Table::DirtyState::getDirtyState() const
{
	return dirtyState;
}

inline bool Table::isInsertMode() const
{
	return insertMode;
}

inline int Table::getEntryNum() const
{
	return entryNum;
}

inline int Table::getEntry() const
{
	return entry;
}

inline int Table::getCursorPos() const
{
	return cursorPos;
}

inline bool Table::isCursorBlink() const
{
	return cursorBlink;
}

inline Table::Info Table::getEntryInfo(int const entry) const
{
	return table[entry].info;
}

#endif
