// Author: Hannes Pabst

#include "table.h"
#include <string.h>
#include <io.h>
#include <errno.h>

#define FILE_EXTENSION ".rec"

namespace
{
	int const FILE_VERSION = 1;
	int const READ_BUFFER_SIZE = 1024 * 16;
	int const COPY_BUFFER_SIZE = 256;
	unsigned int const MAKE_FILE_NAME_LOOP_NUM = 1000;
	char const FILE_NAME_FORMAT[] = "%05u%03u" FILE_EXTENSION;
}

Table::Info::Info(Result result)
: result(result)
{
	name[0] = 0;
}

Table::DirtyState::DirtyState()
: dirtyState(ALL)
{
}

void Table::DirtyState::invalidate()
{
	dirtyState = ALL;
}

void Table::DirtyState::validate()
{
	dirtyState = NOTHING;
}

void Table::DirtyState::update(int dirtyState)
{
	if (this->dirtyState < dirtyState)
		this->dirtyState = dirtyState;
}

Table::Entry::Entry(Info const &info)
: info(info)
{
	fileName[0] = 0;
}

Table::Entry::Entry(FileName const fileName)
{
	strcpy(this->fileName, fileName);
}

Table::Table(DirtyState &dirtyState, char const * const tableDir)
: dirtyState(dirtyState)
, tableDir(tableDir)
, readFile(NULL)
, insertMode(false)
, entryNum(0)
, entry(ENTRY_TITLE)
, cursorBlink(false)
{
	loadTable();
}

Table::~Table()
{
	close();
}

void Table::close()
{
	if (readFile != NULL)
		fclose(readFile);
}

void Table::write(FILE * const file, Info const &info)
{
	int const version = FILE_VERSION;
	fwrite(&version, sizeof(version), 1, file);
	fwrite(&info.result.time, sizeof(info.result.time), 1, file);
	fwrite(&info.result.blocks, sizeof(info.result.blocks), 1, file);
	fwrite(&info.result.score, sizeof(info.result.score), 1, file);
	fwrite(&info.name, sizeof(info.name) - sizeof(*info.name), 1, file);
}

bool Table::read(FILE * const file, Info &info)
{
	int version;
	fread(&version, sizeof(version), 1, file);
	if (version != FILE_VERSION)
		return false;
	fread(&info.result.time, sizeof(info.result.time), 1, file);
	fread(&info.result.blocks, sizeof(info.result.blocks), 1, file);
	fread(&info.result.score, sizeof(info.result.score), 1, file);
	fread(&info.name, sizeof(info.name) - sizeof(*info.name), 1, file);
	info.name[NAME_SIZE - 1] = 0;
	return true;
}

void Table::makeFilePath(Path path, FileName const fileName) const
{
	sprintf(path, "%s\\%s", tableDir, fileName);
}

bool Table::makeFileName(Path path, FileName fileName, Info const &info) const
{
	for (unsigned int i = 0; i < MAKE_FILE_NAME_LOOP_NUM; ++i)
	{
		sprintf(fileName, FILE_NAME_FORMAT, info.result.score % SCORE_RANGE, i);
		makeFilePath(path, fileName);
		if (_access(path, 00) == -1 && errno == ENOENT)
			return true;
	}
	return false;
}

int Table::insertEntry(Entry const &e)
{
	int i = ENTRY_MAX - 1;
	while ((i >= 0 && e.info.result.score > table[i].info.result.score) || (e.info.result.score == table[i].info.result.score && e.info.result.time < table[i].info.result.time))
	{
		if (i < ENTRY_MAX - 1)
			table[i + 1] = table[i];
		--i;
	}
	++i;
	if (i < ENTRY_MAX)
	{
		table[i] = e;
		if (entryNum < ENTRY_MAX)
			++entryNum;
		return i;
	}
	return -1;
}

void Table::loadTable()
{
	Path path;
	makeFilePath(path, "*" FILE_EXTENSION);
	_finddata_t findData;
	intptr_t const hFile = _findfirst(path, &findData);
	if (hFile != -1L)
	{
		do
		{
			if (strlen(findData.name) < static_cast<size_t>(FILE_NAME_SIZE))
			{
 				makeFilePath(path, findData.name);
				FILE * const file = fopen(path, "rb");
				if (file != NULL)
				{
					Entry entry(findData.name);
					if (read(file, entry.info))
						insertEntry(entry);
					fclose(file);
				}
			}
		}
		while (_findnext(hFile, &findData) == 0);
		_findclose(hFile);
	}
}

void Table::storeEntry(Path const path, Info const &info, FILE * const data)
{
	FILE * const file = fopen(path, "wb");
	if (file != NULL)
	{
		write(file, info);
		if (data != NULL)
		{
			char buffer[COPY_BUFFER_SIZE];
			size_t size;
			rewind(data);
			while ((size = fread(&buffer, 1, sizeof(buffer), data)) > 0)
				fwrite(&buffer, 1, size, file);
		}
		fclose(file);
	}
}

FILE *Table::getData(int const entry)
{
	if (!insertMode && entry >= 0 && entry < entryNum)
	{
		Entry &e = table[entry];
		if (e.fileName[0] != '\0')
		{
			Path path;
			makeFilePath(path, e.fileName);
			close();
			readFile = fopen(path, "rb");
			if (readFile != NULL)
			{
				setvbuf(readFile, NULL, _IOFBF, READ_BUFFER_SIZE);
				Info info;
				read(readFile, info);
				return readFile;
			}
		}
	}
	return NULL;
}

void Table::insert(Result const &result)
{
	if (!insertMode)
	{
		strcpy(fileNameForDelete, table[ENTRY_MAX - 1].fileName);
		int const pos = insertEntry(Entry(result));
		if (pos >= 0)
		{
			insertMode = true;
			entry = pos;
			cursorPos = 0;
			cursorBlink = false;
			dirtyState.update(DirtyState::TABLE);
		}
	}
}

void Table::addChar(char const charCode)
{
	if (insertMode && cursorPos < NAME_SIZE - 1)
	{
		Name &name = table[entry].info.name;
		name[cursorPos++] = charCode;
		name[cursorPos] = '\0';
		dirtyState.update(DirtyState::ENTRY);
	}
}

void Table::deleteChar()
{
	if (insertMode && cursorPos > 0)
	{
		Name &name = table[entry].info.name;
		name[cursorPos--] = '\0';
		name[cursorPos] = '\0';
		dirtyState.update(DirtyState::ENTRY);
	}
}

void Table::finishInput(FILE * const data)
{
	if (insertMode)
	{
		Entry &e = table[entry];
		Path path;
		if (makeFileName(path, e.fileName, e.info))
		{
			storeEntry(path, e.info, data);
			if (fileNameForDelete[0] != '\0')
			{
				close();
				readFile = NULL;
				makeFilePath(path, fileNameForDelete);
				remove(path);
			}
		}
		insertMode = false;
		cursorBlink = false;
		dirtyState.update(DirtyState::ENTRY);
	}
}

void Table::blinkCursor()
{
	if (insertMode)
	{
		cursorBlink = !cursorBlink;
		dirtyState.update(DirtyState::ENTRY);
	}
}

void Table::setEntry(int const entry)
{
	if (!insertMode && this->entry != entry)
	{
		this->entry = entry;
		dirtyState.update(DirtyState::TABLE);
	}
}
