// Author: Hannes Pabst

#include "screen_canvas.h"
#include <assert.h>

namespace
{
	BITMAP *createBuffer()
	{
		bool const pageFlipping = get_config_int(NULL, "screen_update_mode", 1) != 0;
		BITMAP * const t = pageFlipping ? create_video_bitmap(SCREEN_W, SCREEN_H) : NULL;
		return t != NULL ? t : create_bitmap(SCREEN_W, SCREEN_H);
	}

	bool isAllegroWaitingForVsync()
	{
		// condition is equivalent to the code in allegro
		char const * const dv = get_config_string("graphics", "disable_vsync", NULL);
		return dv == NULL || (*dv != 'y' && *dv != 'Y' && *dv != '1');
	}
}

ScreenCanvas::ScreenCanvas()
: screen(create_video_bitmap(SCREEN_W, SCREEN_H))
, buffer(createBuffer())
, waitForVsync(isAllegroWaitingForVsync())
, dirtyRectsNum(0)
{
}

ScreenCanvas::~ScreenCanvas()
{
	destroy_bitmap(buffer);
	destroy_bitmap(screen);
}

bool ScreenCanvas::isGood() const
{
	return screen != NULL && buffer != NULL;
}

bool ScreenCanvas::isPageFlipping() const
{
	return is_video_bitmap(buffer) != FALSE;
}

void ScreenCanvas::acquire() const
{
	acquire_bitmap(buffer);
}

void ScreenCanvas::release() const
{
	release_bitmap(buffer);
}

void ScreenCanvas::show(bool const syncRefresh)
{
	if (syncRefresh || dirtyRectsNum > 0)
	{
		if (isPageFlipping())
		{
			BITMAP * const t = buffer;
			buffer = screen;
			screen = t;
			show_video_bitmap(screen);
			for (int i = 0; i < dirtyRectsNum; ++i)
				blit(screen, buffer, dirtyRects[i].x, dirtyRects[i].y, dirtyRects[i].x, dirtyRects[i].y, dirtyRects[i].w, dirtyRects[i].h);
		}
		else
		{
			if (waitForVsync)
				vsync();
			if (dirtyRectsNum > 0)
			{
				acquire_bitmap(screen);
				for (int i = 0; i < dirtyRectsNum; ++i)
					blit(buffer, screen, dirtyRects[i].x, dirtyRects[i].y, dirtyRects[i].x, dirtyRects[i].y, dirtyRects[i].w, dirtyRects[i].h);
				release_bitmap(screen);
			}
		}
		dirtyRectsNum = 0;
	}
}

BITMAP *ScreenCanvas::getBuffer()
{
	return buffer;
}

void ScreenCanvas::setDirty(int const x, int const y, int const w, int const h)
{
	assert(dirtyRectsNum < DIRTY_RECTS_MAX);
	if (dirtyRectsNum < DIRTY_RECTS_MAX)
	{
		dirtyRects[dirtyRectsNum].x = x;
		dirtyRects[dirtyRectsNum].y = y;
		dirtyRects[dirtyRectsNum].w = w;
		dirtyRects[dirtyRectsNum].h = h;
		++dirtyRectsNum;
	}
}
