// Author: Hannes Pabst

#include "renderer.h"

namespace
{
	int const TILE_WIDTH = 20;
	int const TILE_HEIGHT = 20;

	int const FIELD_START_X = 220;
	int const FIELD_START_Y = 40;

	int const NEXT_FIELD_START_X = 80;
	int const NEXT_FIELD_START_Y = 220;

	int const TX = 16;
	int const TY = 16;

	char const TOP_SCORE_TEXT[] = "TOP SCORE";
	int const TOP_SCORE_X = 28 * TX;
	int const TOP_SCORE_Y = 6 * TY;
	int const TOP_SCORE_W = 9 * TX;
	int const TOP_SCORE_H = 1 * TY;

	char const DEMO_TEXT[] = "DEMO";
	int const DEMO_X = 33 * TX;
	int const DEMO_Y = 6 * TY;
	int const DEMO_W = 4 * TX;
	int const DEMO_H = 1 * TY;

	char const REPLAY_TEXT[] = "REPLAY";
	int const REPLAY_X = 31 * TX;
	int const REPLAY_Y = 6 * TY;
	int const REPLAY_W = 6 * TX;
	int const REPLAY_H = 1 * TY;

	char const BLOCKS_HEADING_TEXT[] = "BLOCKS";
	int const BLOCKS_HEADING_X = 31 * TX;
	int const BLOCKS_HEADING_Y = 11 * TY;

	char const BLOCKS_FORMAT[] = "%5u";
	int const BLOCKS_X = 32 * TX;
	int const BLOCKS_Y = 13 * TY;
	int const BLOCKS_W = 5 * TX;
	int const BLOCKS_H = 1 * TY;

	char const SCORE_HEADING_TEXT[] = "SCORE";
	int const SCORE_HEADING_X = 32 * TX;
	int const SCORE_HEADING_Y = 16 * TY;

	char const SCORE_FORMAT[] = "%5u";
	int const SCORE_X = 32 * TX;
	int const SCORE_Y = 18 * TY;
	int const SCORE_W = 5 * TX;
	int const SCORE_H = 1 * TY;

	char const GAME_OVER_TEXT[] = "GAME OVER";
	int const GAME_OVER_X = 28 * TX;
	int const GAME_OVER_Y = 23 * TY;
	int const GAME_OVER_W = 9 * TX;
	int const GAME_OVER_H = 1 * TY;

	char const TABLE_TITLE_TEXT[] = "TOP SCORES";
	int const TABLE_TITLE_X = 15 * TX;
	int const TABLE_TITLE_Y = 3 * TY;

	char const TABLE_HEADING_TEXT[] = "NO DATE     PLAYER    BLOCKS SCORE";
	int const TABLE_HEADING_X = 3 * TX;
	int const TABLE_HEADING_Y = 6 * TY;

	int const TABLE_X = 3 * TX;
	int const TABLE_Y = 8 * TY;
	int const TABLE_W = 34 * TX;
	int const TABLE_DY = 2 * TY;

	char const TABLE_ENTRY_FORMAT[] = "%2u %2.2u/%2.2u/%2.2u %-11s%5u %5u";

	char const MENU_PLAY_TEXT[] = "PLAY";
	int const MENU_PLAY_X = 18 * TX;
	int const MENU_PLAY_Y = 18 * TY;

	char const MENU_DEMO_TEXT[] = "DEMO";
	int const MENU_DEMO_X = 18 * TX;
	int const MENU_DEMO_Y = 20 * TY;

	char const MENU_TOP_SCORES_TEXT[] = "TOP SCORES";
	int const MENU_TOP_SCORES_X = 15 * TX;
	int const MENU_TOP_SCORES_Y = 22 * TY;

	char const TITLE_TEXT[] = " ";
	int const TITLE_X = 11 * TX;
	int const TITLE_Y = 9 * TY + 12;

	char const CURSOR_EDIT = '\x5f';
	char const CURSOR_END = '\x5e';
}

Renderer::Renderer(Game const &game, Game::DirtyState &dirtyState, Canvas &canvas, Data data, char * const textBuffer)
: game(game)
, dirtyState(dirtyState)
, canvas(canvas)
, data(data)
, textBuffer(textBuffer)
{
}

void Renderer::paintTile(int const x, int const y, Tile const tile) const
{
	if (tile.isBlank())
		blit(data.getGameBackground(), graphicBuffer, x, y, x, y, TILE_WIDTH, TILE_HEIGHT);
	else
		blit(data.getTiles(), graphicBuffer, Block::align(tile.getBlock()) * TILE_WIDTH, tile.getEdge() * TILE_HEIGHT, x, y, TILE_WIDTH, TILE_HEIGHT);
}

template <typename Field>
void Renderer::paintField(int const fieldX, int const fieldY, Field const &field, Rect const &dirty, bool const repaint) const
{
	if (isLeftTop(dirty.leftTop, dirty.rightBottom))
	{
		if (!repaint)
		{
			int const x = fieldX + dirty.leftTop.x * TILE_WIDTH;
			int const y = fieldY + dirty.leftTop.y * TILE_HEIGHT;
			int const w = (dirty.rightBottom.x - dirty.leftTop.x + 1) * TILE_WIDTH;
			int const h = (dirty.rightBottom.y - dirty.leftTop.y + 1) * TILE_HEIGHT;
			canvas.setDirty(x, y, w, h);
		}
		for (int j = dirty.leftTop.y; j <= dirty.rightBottom.y; ++j)
		{
			for (int i = dirty.leftTop.x; i <= dirty.rightBottom.x; ++i)
			{
				int const x = fieldX + i * TILE_WIDTH;
				int const y = fieldY + j * TILE_HEIGHT;
				paintTile(x, y, field.at(i, j));
			}
		}
	}
}

void Renderer::paintTetris() const
{
	int const infoDirtyFlag = dirtyState.tetris.info.get();
	bool const repaint = (infoDirtyFlag & Tetris::InfoDirtyFlag::INVALID) != 0;
	if (repaint)
	{
		blit(data.getGameBackground(), graphicBuffer, 0, 0, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		canvas.setDirty(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		textout_ex(graphicBuffer, data.getFont(), BLOCKS_HEADING_TEXT, BLOCKS_HEADING_X, BLOCKS_HEADING_Y, data.getHeadingTextColor(), -1);
		textout_ex(graphicBuffer, data.getFont(), SCORE_HEADING_TEXT, SCORE_HEADING_X, SCORE_HEADING_Y, data.getHeadingTextColor(), -1);
	}
	Tetris const &tetris = game.getTetris();
	paintField(FIELD_START_X, FIELD_START_Y, tetris.getPlayField(), dirtyState.tetris.fields.playField.getDirtyRect(), repaint);
	paintField(NEXT_FIELD_START_X, NEXT_FIELD_START_Y, tetris.getNextField(), dirtyState.tetris.fields.nextField.getDirtyRect(), repaint);
	if ((infoDirtyFlag & Tetris::InfoDirtyFlag::DEMO_MODE) != 0 || dirtyState.table.getDirtyState())
	{
		Table const &table = game.getTable();
		if (table.isInsertMode())
		{
			if (!repaint)
			{
				blit(data.getGameBackground(), graphicBuffer, TOP_SCORE_X, TOP_SCORE_Y, TOP_SCORE_X, TOP_SCORE_Y, TOP_SCORE_W, TOP_SCORE_H);
				canvas.setDirty(TOP_SCORE_X, TOP_SCORE_Y, TOP_SCORE_W, TOP_SCORE_H);
			}
			if (table.isCursorBlink())
				textout_ex(graphicBuffer, data.getFont(), TOP_SCORE_TEXT, TOP_SCORE_X, TOP_SCORE_Y, data.getNormalTextColor(), -1);
		}
		else
		{
			switch (tetris.getDemoMode())
			{
			case Tetris::DEMO:
				if (!repaint)
				{
					blit(data.getGameBackground(), graphicBuffer, DEMO_X, DEMO_Y, DEMO_X, DEMO_Y, DEMO_W, DEMO_H);
					canvas.setDirty(DEMO_X, DEMO_Y, DEMO_W, DEMO_H);
				}
				textout_ex(graphicBuffer, data.getFont(), DEMO_TEXT, DEMO_X, DEMO_Y, data.getNormalTextColor(), -1);
				break;
			case Tetris::REPLAY:
				if (!repaint)
				{
					blit(data.getGameBackground(), graphicBuffer, REPLAY_X, REPLAY_Y, REPLAY_X, REPLAY_Y, REPLAY_W, REPLAY_H);
					canvas.setDirty(REPLAY_X, REPLAY_Y, REPLAY_W, REPLAY_H);
				}
				textout_ex(graphicBuffer, data.getFont(), REPLAY_TEXT, REPLAY_X, REPLAY_Y, data.getNormalTextColor(), -1);
				break;
			default:
				break;
			}
		}
	}
	if ((infoDirtyFlag & Tetris::InfoDirtyFlag::BLOCKS_SCORE) != 0)
	{
		if (!repaint)
		{
			blit(data.getGameBackground(), graphicBuffer, BLOCKS_X, BLOCKS_Y, BLOCKS_X, BLOCKS_Y, BLOCKS_W, BLOCKS_H);
			canvas.setDirty(BLOCKS_X, BLOCKS_Y, BLOCKS_W, BLOCKS_H);
			blit(data.getGameBackground(), graphicBuffer, SCORE_X, SCORE_Y, SCORE_X, SCORE_Y, SCORE_W, SCORE_H);
			canvas.setDirty(SCORE_X, SCORE_Y, SCORE_W, SCORE_H);
		}
		sprintf(textBuffer, BLOCKS_FORMAT, tetris.getResult().blocks % Table::BLOCKS_RANGE);
		textout_ex(graphicBuffer, data.getFont(), textBuffer, BLOCKS_X, BLOCKS_Y, data.getNormalTextColor(), -1);
		sprintf(textBuffer, SCORE_FORMAT, tetris.getResult().score % Table::SCORE_RANGE);
		textout_ex(graphicBuffer, data.getFont(), textBuffer, SCORE_X, SCORE_Y, data.getNormalTextColor(), -1);
	}
	if ((infoDirtyFlag & Tetris::InfoDirtyFlag::GAME_RUNNING) != 0)
	{
		if (!repaint)
		{
			blit(data.getGameBackground(), graphicBuffer, GAME_OVER_X, GAME_OVER_Y, GAME_OVER_X, GAME_OVER_Y, GAME_OVER_W, GAME_OVER_H);
			canvas.setDirty(GAME_OVER_X, GAME_OVER_Y, GAME_OVER_W, GAME_OVER_H);
		}
		if (!tetris.isGameRunning())
			textout_ex(graphicBuffer, data.getFont(), GAME_OVER_TEXT, GAME_OVER_X, GAME_OVER_Y, data.getNormalTextColor(), -1);
	}
}

void Renderer::drawMenuText(char const * const text, int const x, int const y, bool const selected) const
{
	if (selected)
	{
		int const GAP = 4;
		int const x0 = x - GAP;
		int const y0 = y - GAP;
		int const x1 = x + strlen(text) * TX + GAP - 1;
		int const y1 = y + TY + GAP - 1;
		rectfill(graphicBuffer, x0, y0, x1, y1, data.getSelectionBarColor());
		for (int i = 1; i <= 2; ++i)
			rect(graphicBuffer, x0 - i, y0 - i, x1 + i, y1 + i, data.getSelectionFrameColor());
	}
	textout_ex(graphicBuffer, data.getFont(), text, x, y, data.getNormalTextColor(), -1);
}

void Renderer::drawTableLine(int const row, Table::Info const &info, bool const selected, bool const cursorBlink) const
{
	unsigned int day, month, year;
	tm *t = _localtime64(&info.result.time);
	if (t != NULL)
	{
		day = t->tm_mday % 100;
		month = (t->tm_mon + 1) % 100;
		year = t->tm_year % 100;
	}
	else
	{
		day = 0;
		month = 0;
		year = 0;
	}
	char text[Table::NAME_SIZE + 1];
	int const length = sprintf(text, "%s", info.name);
	for (int i = 0; i < length; ++i)
		text[i] = checkCharCode(text[i], ' ');
	if (selected && cursorBlink)
		sprintf(text + length, "%c", length < Table::NAME_SIZE - 1 ? CURSOR_EDIT : CURSOR_END);
	sprintf(textBuffer, TABLE_ENTRY_FORMAT, row + 1, day, month, year, text, info.result.blocks % Table::BLOCKS_RANGE, info.result.score % Table::SCORE_RANGE);
	drawMenuText(textBuffer, TABLE_X, TABLE_Y + row * TABLE_DY, selected);
}

void Renderer::paintTable() const
{
	Table const &table = game.getTable();
	switch (dirtyState.table.getDirtyState())
	{
	case Table::DirtyState::ALL:
	case Table::DirtyState::TABLE:
		blit(data.getTextBackground(), graphicBuffer, 0, 0, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		canvas.setDirty(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		drawMenuText(TABLE_TITLE_TEXT, TABLE_TITLE_X, TABLE_TITLE_Y, table.getEntry() == Table::ENTRY_TITLE && table.getEntryNum() > 0);
		textout_ex(graphicBuffer, data.getFont(), TABLE_HEADING_TEXT, TABLE_HEADING_X, TABLE_HEADING_Y, data.getHeadingTextColor(), -1);
		for (int entry = 0; entry < table.getEntryNum(); ++entry)
			drawTableLine(entry, table.getEntryInfo(entry), entry == table.getEntry(), table.isCursorBlink());
		break;
	case Table::DirtyState::ENTRY:
		canvas.setDirty(TABLE_X, TABLE_Y + table.getEntry() * TABLE_DY, TABLE_W, TY);
		drawTableLine(table.getEntry(), table.getEntryInfo(table.getEntry()), true, table.isCursorBlink());
		break;
	}
}

void Renderer::paintMenu() const
{
	if (dirtyState.menu.get())
	{
		blit(data.getTitleBackground(), graphicBuffer, 0, 0, 0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		canvas.setDirty(0, 0, SCREEN_WIDTH, SCREEN_HEIGHT);
		textout_ex(graphicBuffer, data.getTitleText(), TITLE_TEXT, TITLE_X, TITLE_Y, data.getHeadingTextColor(), -1);
		Menu const &menu = game.getMenu();
		drawMenuText(MENU_PLAY_TEXT, MENU_PLAY_X, MENU_PLAY_Y, menu.getSelection() == Menu::PLAY);
		drawMenuText(MENU_DEMO_TEXT, MENU_DEMO_X, MENU_DEMO_Y, menu.getSelection() == Menu::DEMO);
		drawMenuText(MENU_TOP_SCORES_TEXT, MENU_TOP_SCORES_X, MENU_TOP_SCORES_Y, menu.getSelection() == Menu::TOP_SCORES);
	}
}

void Renderer::paint()
{
	graphicBuffer = canvas.getBuffer();
	if (dirtyState.screen.get())
		dirtyState.invalidate();
	Game::Screen const screen = game.getScreen();
	switch (screen)
	{
	case Game::TETRIS:
		paintTetris();
		break;
	case Game::TABLE:
		paintTable();
		break;
	case Game::MENU:
		paintMenu();
		break;
	default:
		break;
	}
	dirtyState.validate();
}
