// Author: Hannes Pabst

#include "data_file_owner.h"
#include "screen_canvas.h"
#include "sound_player.h"
#include "screenshot_saver.h"
#include "off_screen_canvas.h"
#include "config_key.h"
#include "system.h"
#include "execute.h"
#include "static_check.h"
#include "version.h"
#include <direct.h>

//#define TEST_CODE

#define ARRAY_SIZE(a) (sizeof(a) / sizeof(*a))
#define STRING_LEN(a) (ARRAY_SIZE(a) - 1)

namespace
{
	char const APPLICATION_DIR[] = "\\HP's Tetris";
	char const TABLE_DIR[] = "\\Records";
	char const SCREENSHOT_DIR[] = "\\Screenshots";
	char const TEMP_FILE_NAME[] = "\\record.tmp";
	char const DAT_FILE_NAME[] = "tetris.dat";
	int const DATA_DIR_SIZE = Table::DIR_SIZE_MAX - STRING_LEN(TABLE_DIR) - STRING_LEN(APPLICATION_DIR);
	int const APPLICATION_DIR_SIZE = DATA_DIR_SIZE + STRING_LEN(APPLICATION_DIR);
	int const TABLE_DIR_SIZE = APPLICATION_DIR_SIZE + STRING_LEN(TABLE_DIR);
	int const TEMP_FILE_SIZE = APPLICATION_DIR_SIZE + STRING_LEN(TEMP_FILE_NAME);
	int const SCREENSHOT_DIR_SIZE = APPLICATION_DIR_SIZE + STRING_LEN(SCREENSHOT_DIR);

	Event timerEvent;
	Flag displaySwitchFlag(false);

	bool isSyncRefreshEnabled(ScreenCanvas const &screenCanvas)
	{
		int const timingMode = get_config_int(NULL, "timing_mode", 1);
		return timingMode == 2 || (timingMode == 1 && screenCanvas.isWaitingForVsync() && get_refresh_rate() == FRAMES_PER_SECOND);
	}

	Game::KeyEvent keyEventFromKey(int const keyCode)
	{
		switch (keyCode)
		{
		case KEY_ESC:
			return Game::QUIT;
		case KEY_UP:
			return Game::UP;
		case KEY_DOWN:
			return Game::DOWN;
		case KEY_SPACE:
		case KEY_ENTER:
			return Game::SELECT;
		}
		return Game::KEY;
	}

	void timerCallback()
	{
		timerEvent.set();
	}
	END_OF_FUNCTION(timerCallback);

	void displaySwitchCallback()
	{
		displaySwitchFlag.set(true);
	}
	END_OF_FUNCTION(displaySwitchCallback);
}

int main(int const argc, char const * const * const argv)
{
	STATIC_CHECK(TABLE_DIR_SIZE <= Table::DIR_SIZE_MAX);
	STATIC_CHECK(SCREENSHOT_DIR_SIZE <= ScreenshotSaver::DIR_SIZE_MAX);

	set_uformat(U_ASCII);
	if (allegro_init())
		return 1;

	override_config_file("tetris.cfg");

	bool fullScreen = get_config_int(NULL, "screen_mode", 1) != 0;

	for (int i = 1; i < argc; ++i)
	{
		if (!strcmp(argv[i], "-f"))
			fullScreen = true;
		else if (!strcmp(argv[i], "-w"))
			fullScreen = false;
		else
		{
			allegro_message("HP's Tetris\nVersion " VERSION "\n\nCommand line options:\n-f ... full screen\n-w ... windowed");
			return 1;
		}
	}

	ConfigKey const keyTurn("key_turn", "key_turn_alt", KEY_ALT);
	ConfigKey const keyDrop("key_drop", "key_drop_alt", KEY_DOWN);
	ConfigKey const keyLeft("key_left", "key_left_alt", KEY_LEFT);
	ConfigKey const keyRight("key_right", "key_right_alt", KEY_RIGHT);

	char tempFile[TEMP_FILE_SIZE];
	char tableDir[TABLE_DIR_SIZE];
	char screenshotDir[SCREENSHOT_DIR_SIZE];

	{
		char applicationDir[APPLICATION_DIR_SIZE] = ".";
		int const dataFolder = get_config_int(NULL, "data_folder", 2);
		if (dataFolder != 0)
		{
			char const * const dataDir = getenv(dataFolder != 1 ? "USERPROFILE" : "ALLUSERSPROFILE");
			if (dataDir != NULL && strlen(dataDir) < static_cast<size_t>(DATA_DIR_SIZE))
			{
				sprintf(applicationDir, "%s%s", dataDir, APPLICATION_DIR);
				_mkdir(applicationDir);
			}
		}

		sprintf(tableDir, "%s%s", applicationDir, TABLE_DIR);
		sprintf(tempFile, "%s%s", applicationDir, TEMP_FILE_NAME);
		sprintf(screenshotDir, "%s%s", applicationDir, SCREENSHOT_DIR);
	}

	_mkdir(tableDir);
	_mkdir(screenshotDir);

	if (install_timer())
		return 1;

	if (install_keyboard())
		return 1;

	set_keyboard_rate(0, 0);

	if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL))
	{
		allegro_message("Error installing sound: %s!\n", allegro_error);
		return 1;
	}

	set_color_depth(get_config_int(NULL, "color_depth", 32));
	request_refresh_rate(get_config_int(NULL, "request_refresh_rate", FRAMES_PER_SECOND));

	if (set_gfx_mode(fullScreen ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED, Renderer::SCREEN_WIDTH, Renderer::SCREEN_HEIGHT, 0, 0))
	{
		allegro_message("Error setting graphic mode: %s!\n", allegro_error);
		return 1;
	}

	{
		PALLETE pal;
		generate_332_pallete(pal);
		set_pallete(pal);
		set_color_conversion(COLORCONV_TOTAL);
	}

	DataFileOwner const dataFileOwner(DAT_FILE_NAME);
	if (dataFileOwner.getDatafile() == NULL)
	{
		set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
		allegro_message("Error loading datafile %s!\n", DAT_FILE_NAME);
		return 1;
	}

	ScreenCanvas screenCanvas;
	if (!screenCanvas.isGood())
	{
		set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
		allegro_message("Error creating video buffer!\n");
		return 1;
	}

	bool const syncRefresh = isSyncRefreshEnabled(screenCanvas);
	Data const data(*dataFileOwner.getDatafile());
	Game::DirtyState dirtyState;
	SoundPlayer soundPlayer(data);
	Tetris::Input input;
	Game game(dirtyState, soundPlayer, input, tempFile, tableDir);
	char textBuffer[Renderer::TEXT_BUFFER_SIZE];
	Renderer screenRenderer(game, dirtyState, screenCanvas, data, textBuffer);
	OffScreenCanvas offScreenCanvas;
	ScreenshotSaver screenshotSaver(screenshotDir);

	LOCK_FUNCTION(displaySwitchCallback);
	set_display_switch_mode(SWITCH_PAUSE);
	set_display_switch_callback(SWITCH_IN, displaySwitchCallback);

	if (!syncRefresh)
	{
		LOCK_FUNCTION(timerCallback);
		install_int_ex(timerCallback, BPS_TO_TIMER(FRAMES_PER_SECOND));
	}

	for (;;)
	{

#ifdef TEST_CODE
		while ((key_shifts & KB_NUMLOCK_FLAG) && !keypressed()) ;
#endif

		while (keypressed())
		{
			int const readCode = readkey();
			int const keyCode = readCode >> 8;
			char const charCode = static_cast<char>(readCode);

			switch (keyCode)
			{
			case KEY_PRTSCR:
				{
					Game::DirtyState dirtyState;
					Renderer screenshotRenderer(game, dirtyState, offScreenCanvas, data, textBuffer);
					screenshotRenderer.paint();
					screenshotSaver.save(offScreenCanvas.getBuffer());
				}
				break;

#ifdef TEST_CODE
			case KEY_F5:
				clear_to_color(screen, makecol(128, 0, 0));
				break;
			case KEY_F6:
				dirtyState.invalidate();
				break;
#endif

			}
			if (game.handleKeyEvent(keyEventFromKey(keyCode), charCode))
				return 0;
		}

		input.turn = keyTurn.state();
		input.drop = keyDrop.state();
		input.left = keyLeft.state();
		input.right = keyRight.state();

		game.execute();

		if (displaySwitchFlag.set(false))
			dirtyState.invalidate();

		screenCanvas.acquire();
		screenRenderer.paint();

#ifdef TEST_CODE
		static unsigned int frames = 0;
		textprintf_ex(screenCanvas.getBuffer(), font, 0, 0, data.getNormalTextColor(), data.getSelectionBarColor(), "Frame: %010u  Refresh rate: %u HZ  Sync refresh: %s  Page flip: %s", ++frames, get_refresh_rate(), syncRefresh ? "on" : "off", screenCanvas.isPageFlipping() ? "on" : "off");
#endif

		screenCanvas.release();
		screenCanvas.show(syncRefresh);
		if (!syncRefresh)
			timerEvent.wait();
	}
}
END_OF_MAIN()
