// Author: Hannes Pabst

#include "game.h"
#include "execute.h"

namespace
{
	int const BLINK_DELAY = FRAMES_PER_SECOND / 4;
}

void Game::DirtyState::invalidate()
{
	screen.invalidate();
	tetris.invalidate();
	table.invalidate();
	menu.invalidate();
}

void Game::DirtyState::validate()
{
	screen.validate();
	tetris.validate();
	table.validate();
	menu.validate();
}

Game::Game(DirtyState &dirtyState, Tetris::SoundPlayer &soundPlayer, Tetris::Input &input, char const * const tempFile, char const * const tableDir)
: dirtyState(dirtyState)
, tetris(dirtyState.tetris, soundPlayer, input, tempFile)
, table(dirtyState.table, tableDir)
, menu(dirtyState.menu)
, screen(MENU)
{
}

void Game::execute()
{
	if (screen == TETRIS && tetris.execute() && tetris.getDemoMode() == Tetris::RECORD)
	{
		table.insert(tetris.getResult());
		if (table.isInsertMode())
			cursorTimer = BLINK_DELAY;
	}
	if (table.isInsertMode() && --cursorTimer <= 0)
	{
		cursorTimer = BLINK_DELAY;
		table.blinkCursor();
	}
}

void Game::setScreen(Screen const screen)
{
	this->screen = screen;
	dirtyState.screen.invalidate();
}

void Game::setMenu()
{
	menu.setSelection(Menu::PLAY);
	setScreen(MENU);
}

bool Game::handleKeyEvent(KeyEvent const keyEvent, char const charCode)
{
	switch (screen)
	{
	case TETRIS:
		switch (keyEvent)
		{
		case SELECT:
			if (tetris.getDemoMode() == Tetris::RECORD && tetris.isGameRunning())
				break;
			// fall through
		case QUIT:
			if (table.isInsertMode())
				setScreen(TABLE);
			else
				setMenu();
			break;
		default:
            break;
		}
		break;
	case TABLE:
		if (table.isInsertMode())
		{
			bool finishInput = false;
			if (charCode == '\r')
				finishInput = true;
			else if (charCode == '\b')
				table.deleteChar();
			else if (checkCharCode(charCode, 0) != 0)
				table.addChar(charCode);
			else
			{
				switch (keyEvent)
				{
				case QUIT:
				case SELECT:
					finishInput = true;
					break;
                default:
                    break;
				}
			}
			if (finishInput)
				table.finishInput(tetris.getData());
		}
		else
		{
			switch (keyEvent)
			{
			case UP:
			case DOWN:
				{
					int entry = table.getEntry() + (keyEvent == UP ? - 1 : 1);
					if (entry < Table::ENTRY_TITLE)
						entry = table.getEntryNum() - 1;
					else if (entry >= table.getEntryNum())
						entry = Table::ENTRY_TITLE;
					table.setEntry(entry);
				}
				break;
			case SELECT:
				{
					FILE * const data = table.getData(table.getEntry());
					if (data != NULL)
					{
						tetris.start(Tetris::REPLAY, data);
						setScreen(TETRIS);
						break;
					}
				}
				// fall through
			case QUIT:
				setMenu();
				break;
            default:
                break;
			}
		}
		break;
	case MENU:
		switch (keyEvent)
		{
		case QUIT:
			return true;
		case UP:
		case DOWN:
			{
				int selection = menu.getSelection() + (keyEvent == UP ? - 1 : 1);
				if (selection < Menu::PLAY)
					selection = Menu::TOP_SCORES;
				else if (selection > Menu::TOP_SCORES)
					selection = Menu::PLAY;
				menu.setSelection(static_cast<Menu::Selection>(selection));
			}
			break;
		case SELECT:
			switch (menu.getSelection())
			{
			case Menu::PLAY:
				tetris.start(Tetris::RECORD);
				setScreen(TETRIS);
				break;
			case Menu::DEMO:
				tetris.start(Tetris::DEMO);
				setScreen(TETRIS);
				break;
			case Menu::TOP_SCORES:
				table.setEntry(Table::ENTRY_TITLE);
				setScreen(TABLE);
				break;
            default:
                break;
			}
			break;
        default:
            break;
		}
		break;
    default:
        break;
	}
	return false;
}

char checkCharCode(char charCode, char const defaultCharCode)
{
	if (charCode >= 0x61 && charCode <= 0x7a)
		charCode -= 0x20;
	if (charCode >= 0x20 && charCode < 0x5e)
		return charCode;
	return defaultCharCode;
}
