// Author: Hannes Pabst

#include "color.h"
#include <allegro.h>
#include <stdio.h>

using namespace Color;

namespace
{
	int const COLORS_NUM = 8;

	class AlColor
	{
	public:
		AlColor(int r, int g, int b);
		AlColor(Rgb const &c);

		operator int() const;

	private:
		int c;

		static int toPal(double rgb);
	};


	AlColor::AlColor(int const r, int const g, int const b)
		: c(makecol(r, g, b))
	{
	}

	AlColor::AlColor(Rgb const &c)
	{
		*this = AlColor(toPal(c.r), toPal(c.g), toPal(c.b));
	}

	AlColor::operator int() const
	{
		return c;
	}

	int AlColor::toPal(double const rgb)
	{
		int const p = static_cast<int>(rgb * 256.0);
		if (p > 255)
			return 255;
		return p;
	}

	Hsl getTileColor(int const colorIdx, double const lightIdx)
	{
		double const hues[COLORS_NUM] = { 4, 0, 5, 0.5, 3, 1, 2, 3.5 };
		return Hsl(hues[colorIdx], 0.75, 0.5 + 0.15625 * (lightIdx - 2));
	}

	void makeTiles(char const * const dir, BITMAP * const bmp, int const dest_x, int const dest_y)
	{
		int const EDGESTYLE_NUM = 16;
		int const TILE_W = 20;
		int const TILE_H = 20;

		for (int edgestyle = 0; edgestyle < EDGESTYLE_NUM; ++edgestyle)
		{
			char edge_filename[512];
			sprintf(edge_filename, "%.256s\\EDGE%.2u.pcx", dir, edgestyle);
			BITMAP * const edge_bmp = load_bitmap(edge_filename, NULL);
			if (edge_bmp != NULL)
			{
				for (int colorIdx = 0; colorIdx < COLORS_NUM; ++colorIdx)
				{
					for (int y = 0; y < TILE_H; ++y)
					{
						for (int x = 0; x < TILE_W; ++x)
						{
							int const lightIdx = edge_bmp->line[y][x];
							AlColor const col = static_cast<Rgb>(getTileColor(colorIdx, lightIdx));
							putpixel(bmp, dest_x + x + colorIdx * TILE_W, dest_y + y + edgestyle * TILE_H, col);
						}
					}
				}
				destroy_bitmap(edge_bmp);
			}
		}
	}

	void makeColors(BITMAP * const bmp, int const dest_x, int const dest_y)
	{
		putpixel(bmp, dest_x    , dest_y, AlColor(Rgb(1, 1, 1))); // normal
		putpixel(bmp, dest_x + 1, dest_y, AlColor(Rgb(1, 1, 0.25))); // heading
		putpixel(bmp, dest_x + 2, dest_y, AlColor(Rgb(1, 0.25, 0.25))); // selection frame
		putpixel(bmp, dest_x + 3, dest_y, AlColor(Rgb(0, 0, 0))); // selection bar
		putpixel(bmp, dest_x + 4, dest_y, AlColor(getTileColor(7, 0.5))); // background
	}
}

int main(int const argc, char const * const * const argv)
{
	set_uformat(U_ASCII);
	allegro_init();

	char const *dir = "creategfx";
	if (argc >= 2)
		dir = argv[1];

	set_color_depth(32);
	set_color_conversion(COLORCONV_NONE);

	install_keyboard();
	set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0);

	BITMAP *bmp = create_bitmap(300, 600);
	clear_bitmap(bmp);

	makeColors(bmp, 0, 320);

	makeTiles(dir, bmp, 0, 0);

	blit(bmp, screen, 0, 0, 0, 0, bmp->w, bmp->h);

	while (readkey() >> 8 != KEY_ESC) ;

	//static PALLETE pal;
	//for (int i = 0; i < PAL_SIZE; ++i)
	//	pal[i] = black_palette[i];
	//signed char const rsvd[PAL_SIZE] = { 0 };
	//generate_optimized_palette(bmp, pal, rsvd);
	//set_pallete(pal);
	//BITMAP *bmp8 = create_bitmap_ex(8, bmp->w, bmp->h);
	//blit(bmp, bmp8, 0, 0, 0, 0, bmp->w, bmp->h);
	//save_bitmap("gfx8.pcx", bmp8, pal);
	//destroy_bitmap(bmp8);

	save_bitmap("gfx.pcx", bmp, NULL);
	destroy_bitmap(bmp);

	return 0;
}
END_OF_MAIN();
