// Author: Hannes Pabst

#include "color.h"
#include <algorithm>

namespace Color
{
	namespace
	{
		double maxComponent(Rgb const &c)
		{
			return std::max(c.r, std::max(c.g, c.b));
		}

		double minComponent(Rgb const &c)
		{
			return std::min(c.r, std::min(c.g, c.b));
		}

		double partHue(double const max, double const delta, Rgb const &c, double const off)
		{
			return max == c.r && max != c.g ? (c.g - c.b) / delta + off : 0;
		}

		double hue(double const max, double const delta, Rgb const &c)
		{
			double const h = partHue(max, delta, c, 0) +
			                 partHue(max, delta, Rgb(c.g, c.b, c.r), 2) +
			                 partHue(max, delta, Rgb(c.b, c.r, c.g), 4);
			return h < 0 ? h + 6 : h;
		}
	}

	Rgb::Rgb(double const h)
	{
		*this = h <= 2 ? h <= 1 ? Rgb(1, h, 0)     : Rgb(2 - h, 1, 0) :
		        h <= 4 ? h <= 3 ? Rgb(0, 1, h - 2) : Rgb(0, 4 - h, 1) :
		                 h <= 5 ? Rgb(h - 4, 0, 1) : Rgb(1, 0, 6 - h);
	}

	double hue(Rgb const &c)
	{
		double const max = maxComponent(c);
		double const min = minComponent(c);
		double const delta = max - min;
		return delta > 0 ? hue(max, delta, c) : 0;
	}

	Hsl::Hsl(Rgb const &rgb)
	{
		double const max = maxComponent(rgb);
		double const min = minComponent(rgb);
		double const delta = max - min;
		double const sum = (min + max);
		double const mid = sum * 0.5;
		*this = delta > 0 && sum < 2 ?
			Hsl(hue(max, delta, rgb), delta / (sum <= 1 ? sum : 2 - sum), mid) :
			Hsl(0, 0, mid);
	}

	double Hsl::rgbComponent(double const c) const
	{
		double const t = 2 * s * c + (1 - s);
		return l <= 0.5 ? l * t : (1 - l) * t + 2 * l - 1;
	}

	Hsl::operator Rgb() const
	{
		Rgb const sat(h);
		return Rgb(rgbComponent(sat.r), rgbComponent(sat.g), rgbComponent(sat.b));
	}

	Hsv::Hsv(Rgb const &rgb)
	{
		double const max = maxComponent(rgb);
		double const min = minComponent(rgb);
		double const delta = max - min;
		*this = delta > 0 ?
			Hsv(hue(max, delta, rgb), delta / max, max) :
			Hsv(0, 0, max);
	}

	double Hsv::rgbComponent(double const c) const
	{
		return (1 - s + s * c) * v;
	}

	Hsv::operator Rgb() const
	{
		Rgb const sat(h);
		return Rgb(rgbComponent(sat.r), rgbComponent(sat.g), rgbComponent(sat.b));
	}
}
