// Author: Hannes Pabst

#ifndef TETRIS_BASE_H
#define TETRIS_BASE_H

#include "ai_piece.h"
#include "hp_evaluator_scores.h"
#include "field.h"
#include "block_edges.h"

template <typename Impl, typename Tile, template<typename> class DirtyField>
class TetrisBase
{
public:
	enum Event { NO_EVENT, NEW_BLOCK, STEP_DOWN, BLOCK_FIXED, LINE_FLASHED, LINE_CLEARED, GAME_ENDED };
	enum Message { NO_MESSAGE, TURN, LEFT, RIGHT, DROP };
	enum State { GAME_OVER, PREPARING, READY, DROPPING, CLEARING };

	typedef Field<Tile, Block::BLOCK_ALIGNED_WIDTH_MAX, Block::BLOCK_ALIGNED_HEIGHT_MAX> NextField;
	typedef Field<Tile, 10, 20> PlayField;
	typedef ::FieldInfo<PlayField> FieldInfo;

	struct DirtyState
	{
		DirtyField<NextField> nextField;
		DirtyField<PlayField> playField;

		void invalidate();
		void validate();
	};

	struct ClearInfo
	{
		int lineYs[Block::TILE_NUM];
		int lineNum;
		int lineIdx;

		ClearInfo();
	};

	TetrisBase(int flashNum);

	void start();
	Event execute();
	bool handle(Message msg);
	Message aiPlayer();
	bool aiPlayer(Message msg);

	State getState() const;
	NextField const &getNextField() const;
	PlayField const &getPlayField() const;
	FieldInfo const &getFieldInfo() const;
	ClearInfo const &getClearInfo() const;
	int getPlayBlock() const;
	int getNextBlock() const;

private:
	typedef HpEvaluatorScores<PlayField> EvaluatorScores;

	EvaluatorScores evaluatorScores;
	State state;
	NextField nextField;
	PlayField playField;
	FieldInfo fieldInfo;
	ClearInfo clearInfo;
	int flashNum, flash;
	Piece nextPiece, playPiece, aiPiece;
	bool aiPieceDirty;

	Impl &getImpl();
	Event executeClearing();
	int makeRandomBlock();
	bool newPiece();
	bool turnPiece();
	bool movePiece(Point const &deltaPosition);
	void fixPiece();
	void addEdge(int lineY, int edge);
	void cutEdges(int lineY);
	void flashLine(int lineY);
	void clearFullLine(int lineY);
	bool setAiPiece();
};

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::DirtyState::invalidate()
{
	nextField.invalidate();
	playField.invalidate();
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::DirtyState::validate()
{
	nextField.validate();
	playField.validate();
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
TetrisBase<Impl, Tile, DirtyField>::ClearInfo::ClearInfo()
: lineNum(0)
, lineIdx(0)
{
	for (int i = 0; i < Block::TILE_NUM; ++i)
		lineYs[i] = 0;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
TetrisBase<Impl, Tile, DirtyField>::TetrisBase(int const flashNum)
: state(GAME_OVER)
, flashNum(flashNum)
, flash(0)
, nextPiece(0, Point((NextField::WIDTH - Block::BLOCK_WIDTH_MAX) / 2, (NextField::HEIGHT - Block::BLOCK_HEIGHT_MAX) / 2))
, playPiece(0, Point(0, 0))
, aiPiece(0, Point(0, 0))
, aiPieceDirty(false)
{
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline Impl &TetrisBase<Impl, Tile, DirtyField>::getImpl()
{
	return static_cast<Impl &>(*this);
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::start()
{
	state = PREPARING;
	nextField = NextField();
	playField = PlayField();
	fieldInfo = FieldInfo();
	clearInfo = ClearInfo();
	flash = 0;
	nextPiece.block = makeRandomBlock();
	aiPieceDirty = true;
	getImpl().getPieceCounter().reset();
	getImpl().getDirtyState().invalidate();
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
typename TetrisBase<Impl, Tile, DirtyField>::Event TetrisBase<Impl, Tile, DirtyField>::executeClearing()
{
	Event event;
	if (clearInfo.lineIdx < clearInfo.lineNum)
	{
		state = CLEARING;
		int const lineY = clearInfo.lineYs[clearInfo.lineIdx];

		if (PlayField::Elem::isCutEdgesEnabled() && flash == 0)
			cutEdges(lineY);
		if (flash < flashNum)
		{
			++flash;
			flashLine(lineY);
			event = LINE_FLASHED;
		}
		else
		{
			flash = 0;
			clearFullLine(lineY);
			++clearInfo.lineIdx;
			event = LINE_CLEARED;
		}
	}
	else
	{
		if (newPiece())
		{
			state = READY;
			event = NEW_BLOCK;
		}
		else
		{
			state = GAME_OVER;
			event = GAME_ENDED;
		}
	}
	return event;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
typename TetrisBase<Impl, Tile, DirtyField>::Event TetrisBase<Impl, Tile, DirtyField>::execute()
{
	Event event = NO_EVENT;
	switch (state)
	{
	case PREPARING:
	case CLEARING:
		event = executeClearing();
		break;
	case READY:
	case DROPPING:
		if (movePiece(Point(0, 1)))
			event = STEP_DOWN;
		else
		{
			fixPiece();
			state = CLEARING;
			event = BLOCK_FIXED;
		}
		break;
    default:
        break;
	}
	return event;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::handle(Message const msg)
{
	if (state == READY)
	{
		switch (msg)
		{
		case TURN:
			return turnPiece();
		case LEFT:
			return movePiece(Point(-1, 0));
		case RIGHT:
			return movePiece(Point(1, 0));
		case DROP:
			state = DROPPING;
			return true;
        default:
            break;
		}
	}
	return false;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
int TetrisBase<Impl, Tile, DirtyField>::makeRandomBlock()
{
	return getImpl().getRandomNumber().next(Block::BLOCK_ALIGNED_NUM - 1);
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::newPiece()
{
	playPiece.block = nextPiece.block;
	playPiece.position = Point((PlayField::WIDTH - Block::BLOCK_WIDTH_MAX) / 2, 0);
	if (getImpl().getPieceCounter().hasMorePieces() && canPutPiece(playField, playPiece))
	{
		getImpl().getPieceCounter().takePiece();
		putPiece(playField, playPiece);
		removePiece(nextField, nextPiece);
		if (getImpl().getPieceCounter().hasMorePieces())
		{
			nextPiece.block = makeRandomBlock();
			putPiece(nextField, nextPiece);
		}
		aiPieceDirty = true;
		getImpl().getDirtyState().playField.addRect(getBound(playPiece));
		getImpl().getDirtyState().nextField.addRect(bound(getBound(playPiece.block), getBound(nextPiece.block)) + nextPiece.position);
		return true;
	}
	return false;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::turnPiece()
{
	int const prevPlayPieceBlock = playPiece.block;
	removePiece(playField, playPiece);
	playPiece.block = Block::rotate(playPiece.block);
	if (canPutPiece(playField, playPiece))
	{
		putPiece(playField, playPiece);
		getImpl().getDirtyState().playField.addRect(bound(getBound(prevPlayPieceBlock), getBound(playPiece.block)) + playPiece.position);
		return true;
	}
	playPiece.block = prevPlayPieceBlock;
	putPiece(playField, playPiece);
	return false;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::movePiece(Point const &deltaPosition)
{
	Point const prevPlayPiecePoint = playPiece.position;
	removePiece(playField, playPiece);
	playPiece.position = playPiece.position + deltaPosition;
	if (canPutPiece(playField, playPiece))
	{
		putPiece(playField, playPiece);
		getImpl().getDirtyState().playField.addRect(bound(getBound(playPiece.block), getBound(playPiece.block) + deltaPosition) + prevPlayPiecePoint);
		return true;
	}
	playPiece.position = prevPlayPiecePoint;
	putPiece(playField, playPiece);
	return false;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::fixPiece()
{
	clearInfo.lineIdx = 0;
	clearInfo.lineNum = 0;
	for (int tile = 0; tile < Block::TILE_NUM; ++tile)
	{
		Point const tilePosition = getTilePosition(playPiece, tile);
		fieldInfo.putTile(tilePosition.x, tilePosition.y);
		if (fieldInfo.getTileNum(tilePosition.y) == PlayField::WIDTH)
		{
			clearInfo.lineYs[clearInfo.lineNum] = tilePosition.y;
			++clearInfo.lineNum;
		}
	}
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::addEdge(int const lineY, int const edge)
{
	for (int x = 0; x < PlayField::WIDTH; ++x)
		playField.at(x, lineY).addEdge(edge);
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::cutEdges(int const lineY)
{
	bool const cutAbove = lineY > 0;
	int const aboveY = cutAbove ? lineY - 1 : lineY;
	if (cutAbove)
		addEdge(aboveY, BlockEdges::EDGE_B);
	bool const cutBelow = lineY < PlayField::HEIGHT - 1;
	int const belowY = cutBelow ? lineY + 1 : lineY;
	if (cutBelow)
		addEdge(belowY, BlockEdges::EDGE_T);
	getImpl().getDirtyState().playField.addRect(Rect(Point(0, aboveY), Point(PlayField::WIDTH - 1, belowY)));
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::flashLine(int const lineY)
{
	for (int x = 0; x < PlayField::WIDTH; ++x)
		playField.at(x, lineY).flash(makeRandomBlock());
	getImpl().getDirtyState().playField.addRect(Rect(Point(0, lineY), Point(PlayField::WIDTH - 1, lineY)));
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
void TetrisBase<Impl, Tile, DirtyField>::clearFullLine(int const lineY)
{
	int const dirtyTop = ::clearFullLine(playField, fieldInfo, lineY);
	getImpl().getDirtyState().playField.addRect(Rect(Point(0, dirtyTop), Point(PlayField::WIDTH - 1, lineY)));
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::setAiPiece()
{
	if (state == READY)
	{
		if (aiPieceDirty)
		{
			removePiece(playField, playPiece);
			aiPiece = calculateAiPiece(evaluatorScores, playField, fieldInfo, playPiece.block, nextPiece.block);
			putPiece(playField, playPiece);
			aiPieceDirty = false;
		}
		return true;
	}
	return false;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
typename TetrisBase<Impl, Tile, DirtyField>::Message TetrisBase<Impl, Tile, DirtyField>::aiPlayer()
{
	if (setAiPiece())
	{
		if (playPiece.block != aiPiece.block)
		{
			int const rotatedBlock = Block::rotate(playPiece.block);
			if (playPiece.position.x < blockLeft(rotatedBlock))
				return RIGHT;
			if (playPiece.position.x > blockRight<PlayField>(rotatedBlock))
				return LEFT;
			return TURN;
		}
		if (playPiece.position.x > aiPiece.position.x)
			return LEFT;
		if (playPiece.position.x < aiPiece.position.x)
			return RIGHT;
		return DROP;
	}
	return NO_MESSAGE;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
bool TetrisBase<Impl, Tile, DirtyField>::aiPlayer(Message const msg)
{
	if (setAiPiece())
	{
		int targetX = aiPiece.position.x;
		int rotatedBlock = playPiece.block;
		while (rotatedBlock != aiPiece.block)
		{
			rotatedBlock = Block::rotate(rotatedBlock);
			int const left = blockLeft(rotatedBlock);
			int const right = blockRight<PlayField>(rotatedBlock);
			if (targetX < left)
				targetX = left;
			else if (targetX > right)
				targetX = right;
		}
		switch(msg)
		{
		case TURN:
			return playPiece.block != aiPiece.block;
		case LEFT:
			return playPiece.position.x > targetX;
		case RIGHT:
			return playPiece.position.x < targetX;
		case DROP:
			return playPiece.block == aiPiece.block && playPiece.position.x == targetX;
        default:
            return false;
		}
	}
	return msg == NO_MESSAGE;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline typename TetrisBase<Impl, Tile, DirtyField>::State TetrisBase<Impl, Tile, DirtyField>::getState() const
{
	return state;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline typename TetrisBase<Impl, Tile, DirtyField>::NextField const &TetrisBase<Impl, Tile, DirtyField>::getNextField() const
{
	return nextField;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline typename TetrisBase<Impl, Tile, DirtyField>::PlayField const &TetrisBase<Impl, Tile, DirtyField>::getPlayField() const
{
	return playField;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline typename TetrisBase<Impl, Tile, DirtyField>::FieldInfo const &TetrisBase<Impl, Tile, DirtyField>::getFieldInfo() const
{
	return fieldInfo;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline typename TetrisBase<Impl, Tile, DirtyField>::ClearInfo const &TetrisBase<Impl, Tile, DirtyField>::getClearInfo() const
{
	return clearInfo;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline int TetrisBase<Impl, Tile, DirtyField>::getPlayBlock() const
{
	return playPiece.block;
}

template <typename Impl, typename Tile, template<typename> class DirtyField>
inline int TetrisBase<Impl, Tile, DirtyField>::getNextBlock() const
{
	return nextPiece.block;
}

#endif
