// Author: Hannes Pabst

#ifndef FIELD_HISTOGRAMS_H
#define FIELD_HISTOGRAMS_H

#include "frequency_array.h"
#include "field_info.h"

template <typename Field>
class FieldHistograms
{
public:
	typedef FrequencyArray<Field::WIDTH * Field::HEIGHT + 1> TileFrequencies;
	typedef FrequencyArray<Field::HEIGHT + 1> HeightFrequencies;

	void addField(FieldInfo<Field> const &fieldInfo);

	TileFrequencies const &getTileFrequencies() const;
	TileFrequencies const &getGapFrequencies() const;
	TileFrequencies const &getSumTileGapFrequencies() const;
	HeightFrequencies const &getPileHeights() const;

private:
	TileFrequencies tileFrequencies, gapFrequencies, sumTileGapFrequencies;
	HeightFrequencies pileHeightFrequencies;
};


template <typename Field>
void FieldHistograms<Field>::addField(FieldInfo<Field> const &fieldInfo)
{
	int tileNum = 0;
	for (int y = 0; y < Field::HEIGHT; ++y)
		tileNum += fieldInfo.getTileNum(y);
	int dirtyNum = 0;
	int pileHeight = 0;
	for (int x = 0; x < Field::WIDTH; ++x)
	{
		int const columnHeight = Field::HEIGHT - fieldInfo.getTopTileY(x);
		dirtyNum += columnHeight;
		if (columnHeight > pileHeight)
			pileHeight = columnHeight;
	}
	++tileFrequencies[tileNum];
	++gapFrequencies[dirtyNum - tileNum];
	++sumTileGapFrequencies[dirtyNum];
	++pileHeightFrequencies[pileHeight];
}


template <typename Field>
typename FieldHistograms<Field>::TileFrequencies const &FieldHistograms<Field>::getTileFrequencies() const
{
	return tileFrequencies;
}

template <typename Field>
typename FieldHistograms<Field>::TileFrequencies const &FieldHistograms<Field>::getGapFrequencies() const
{
	return gapFrequencies;
}

template <typename Field>
typename FieldHistograms<Field>::TileFrequencies const &FieldHistograms<Field>::getSumTileGapFrequencies() const
{
	return sumTileGapFrequencies;
}

template <typename Field>
typename FieldHistograms<Field>::HeightFrequencies const &FieldHistograms<Field>::getPileHeights() const
{
	return pileHeightFrequencies;
}

#endif
