// Author: Hannes Pabst

#ifndef EVALUATOR_H
#define EVALUATOR_H

#include "field_transition.h"
#include "evaluator_scores.h"
#include <limits.h>

template <typename Field>
class Evaluator
{
public:
	Evaluator(EvaluatorScores<Field> const &evaluatorScores);

	bool evaluate(FieldTransition<Field> const &field);

private:
	EvaluatorScores<Field> const *evaluatorScores;
	int maxScore;

	int calculateFieldScore(FieldInfo<Field> const &fieldInfo) const;
};


template <typename Field>
Evaluator<Field>::Evaluator(EvaluatorScores<Field> const &evaluatorScores)
: evaluatorScores(&evaluatorScores)
, maxScore(INT_MIN)
{
}

template <typename Field>
__forceinline bool Evaluator<Field>::evaluate(FieldTransition<Field> const &fieldTransition)
{
	int totalScore = fieldTransition.getClearedLines() * evaluatorScores->getClearedLineScore() +
		fieldTransition.getGaps() * evaluatorScores->getGapsScore();

	if (totalScore > maxScore)
	{
		totalScore += calculateFieldScore(fieldTransition.getFieldInfo());
		if (totalScore > maxScore)
		{
			maxScore = totalScore;
			return true;
		}
	}
	return false;
}

template <typename Field>
__forceinline int Evaluator<Field>::calculateFieldScore(FieldInfo<Field> const &fieldInfo) const
{
	int score =	evaluatorScores->getBorderScore(fieldInfo.getTopTileY(0) - fieldInfo.getTopTileY(1)) +
		evaluatorScores->getBorderScore(fieldInfo.getTopTileY(Field::WIDTH - 1) - fieldInfo.getTopTileY(Field::WIDTH - 2));
	for (int x = Field::WIDTH - 3; x >= 1; --x)
		score += evaluatorScores->getMidScore(fieldInfo.getTopTileY(x) - fieldInfo.getTopTileY(x + 1));
	return score;
}

#endif
