#include "Vehicle.h"
#include "Weapons/Weapon.h"

Vehicle::Vehicle(Game* game, Player* owner) {
    _game = game;
    _owner = owner;
    _pos.Set(1, 1);
    _angle = 0.0;
    _state = STATE_LIVING;
    _inventory = new Inventory(_game, this);
    Reset();
}

Vehicle::~Vehicle() {
    delete _inventory;
}

void Vehicle::Tick() {
    if (_state != STATE_DEAD) {
        for (int i = 0; i < AFFECTING_WEAPON_TYPES_COUNT; i++) {
            _affectedBy[i] = false;
        }
        for (int i = 0; i < _game->GetArena()->GetActiveWeaponsCount(); i++) {
            Weapon* wpn = _game->GetArena()->GetActiveWeapon(i);
            WEAPON_TYPE t = wpn->GetType();
            if (wpn->GetTarget() == this) {
                if (t <= AFFECTING_WEAPON_TYPES_COUNT - 1) {
                    _affectedBy[t] = true;
                }
            }
        }

        Move();
        PickupWeapons();
    }
}

void Vehicle::Draw() {
    if (_state != STATE_DEAD) {

        //vehicle (just an arrow)
        Vector2D vRight;
        Vector2D vLeft;
        vRight.SetPolar(_angle + PI * 14 / 16, 10.0);
        vLeft.SetPolar(_angle - PI * 14 / 16, 10.0);
        Vector2D aRight = _pos + vRight;
        Vector2D aLeft = _pos + vLeft;
        _game->GetArena()->DrawLine(_pos, aRight, GetColor());
        _game->GetArena()->DrawLine(_pos, aLeft, GetColor());

        _inventory->Draw();
    }
}

void Vehicle::TurnLeft() {
    if (!_affectedBy[WEAPON_STOP]) {
        _angle -= PI * 2 / (_angleCount);
        while (_angle < PI * 2) {
            _angle += PI * 2;
        }
    }
}

void Vehicle::TurnRight() {
    if (!_affectedBy[WEAPON_STOP]) {
        _angle += PI * 2 / (_angleCount);
        while (_angle >= PI * 2) {
            _angle -= PI * 2;
        }
    }
}

void Vehicle::ActivateItem() {
    if (!_affectedBy[WEAPON_STOP]) {
        if (!_inventory->IsEmpty()) {
            Weapon* takenWeapon = _inventory->Take();
            takenWeapon->Activate();
        }
    }
}

void Vehicle::SelectNextItem() {
    if(_inventory->CountItems() > 1) {
        _inventory->Rotate();
        _game->PlaySound(Game::SOUND_ITEM_SELECT, 0);
    }
}

void Vehicle::Move() {
    if (!_affectedBy[WEAPON_STOP]) {
        const double destinationDist = _game->GetNormalSpeed();
        const Vector2D scanLine(_angle, destinationDist, true);
        const double collisionDist = _game->GetArena()->CalcShortestCollisionDist(_pos, scanLine);
        double moveDist = -1;
        if(collisionDist <= (destinationDist > 1.5 ? destinationDist : 1.5)) {
            moveDist = collisionDist;
            _state = STATE_DYING;
        } else {
            moveDist = destinationDist;
        }
        Vector2D stepVector;
        stepVector.SetPolar(_angle, moveDist);
        Vector2D newPos = _pos + stepVector;
        _game->GetArena()->PlaceLine(_pos, newPos, GetColor());
        _pos = newPos;

        if(_state == STATE_DYING) {
            _game->GetArena()->PlaceCircle(_pos, 17, GetColor(), true);
            _game->GetArena()->PlaceSprite(Game::SPRITE_GRAVE, _pos, true);
        }
    }
}

void Vehicle::PickupWeapons() {
    if (_inventory->GetSize() > 0 && _state == STATE_LIVING) {
        Weapon* weapon;
        for (short i = 0; i < _game->GetArena()->GetFieldWeaponCount(); i++) {
            weapon = _game->GetArena()->GetFieldWeapon(i);
            if (_pos.x >= weapon->GetX() - 1 && _pos.x < weapon->GetX() + weapon->GetW() + 1 &&
                    _pos.y >= weapon->GetY() - 1 && _pos.y < weapon->GetY() + weapon->GetH() + 1) {
                _game->GetArena()->RemoveFieldWeapon(weapon);
                weapon->SetOwner(this);
                _inventory->Add(weapon);
                _game->PlaySound(Game::SOUND_ITEM_PICKUP, _game->GetArena()->GetPan(_pos.x));

                Vector2D weaponPos(weapon->GetX(), weapon->GetY());
                Vector2D weaponSize(weapon->GetW() - 1, weapon->GetH() - 1);
                _game->GetArena()->ClearRect(weaponPos, weaponSize);

                _game->GetArena()->SpawnWeapon();
            }
        }
    }
}

void Vehicle::Die() {
    _inventory->Clear();
    float pan = _game->GetArena()->GetPan(_pos.x);
    _game->PlaySound(Game::SOUND_COLLISION, pan);
    _state = STATE_DEAD;
}

void Vehicle::Reset() {
    _inventory->Clear();
    _state = STATE_LIVING;

    const int distFromBorder = 50;
    _pos.x = distFromBorder + rand() % ((int) _game->GetArena()->GetW() - distFromBorder * 2);
    _pos.y = distFromBorder + rand() % ((int) _game->GetArena()->GetH() - distFromBorder * 2);
    const double angleStep = PI * 2 / _angleCount;
    _angle = (rand() % _angleCount) * angleStep + angleStep / 2;
}

int Vehicle::GetColor() {
    return _owner->GetColor();
}
