#include "Player.h"
#include <math.h>

Player::Player(Game* game, int playerNumber) {
    _game = game;
    _idNumber = playerNumber;
    _totalScore = 0;
    _roundScore = 0;
    _aiCommandTimeCounter = 0;

    _holdingKeyForCommand[PLAYER_COMMAND_TURN_LEFT] = false;
    _holdingKeyForCommand[PLAYER_COMMAND_TURN_RIGHT] = false;
    _holdingKeyForCommand[PLAYER_COMMAND_SELECT_WEAPON] = false;
    _holdingKeyForCommand[PLAYER_COMMAND_USE_WEAPON] = false;

    _vehicle = new Vehicle(_game, this);
}

Player::~Player() {
    delete _vehicle;
}

//react on commands that the human player gives (with keyboard, gamepad, ...)

void Player::FollowCommands() {
    if (GetsKeyImpulseForCommand(PLAYER_COMMAND_TURN_RIGHT)) {
        _vehicle->TurnRight();
    }
    if (GetsKeyImpulseForCommand(PLAYER_COMMAND_TURN_LEFT)) {
        _vehicle->TurnLeft();
    }
    if (GetsKeyImpulseForCommand(PLAYER_COMMAND_SELECT_WEAPON)) {
        _vehicle->SelectNextItem();
    }
    if (GetsKeyImpulseForCommand(PLAYER_COMMAND_USE_WEAPON)) {
        _vehicle->ActivateItem();
    }
}

//choose commands for Computer Player

void Player::ComputeCommands() {
    if (_aiCommandTimeCounter > 0) {
        _aiCommandTimeCounter--;
    } else {
        PLAYER_TYPE type = GetType();

        //set timer for next possible command
        _aiCommandTimeCounter = (type == PLAYER_TYPE_COMPUTER_EASY) ? 2 : (type == PLAYER_TYPE_COMPUTER_NORMAL) ? 1 : (type == PLAYER_TYPE_COMPUTER_HARD) ? 0 : -1;

        //calculate direction with largest collision distance
        const double scanDist = (type == PLAYER_TYPE_COMPUTER_EASY) ? 50 : (type == PLAYER_TYPE_COMPUTER_NORMAL) ? 50 : (type == PLAYER_TYPE_COMPUTER_HARD) ? 50 : -1;
        const Vector2D scanPos = _vehicle->GetPos();
        short rotationSide = rand() % 2;
        double maxCollisionDist = 0.0;
        int maxCollisionDistIndex = 0;
        for (int i = 0; i < Vehicle::GetAngleCount(); i++) {
            const Vector2D scanLine(_vehicle->GetAngle(), scanDist, true);
            double currentCollisionDist = _game->GetArena()->CalcShortestCollisionDist(scanPos, scanLine);
            if (type == PLAYER_TYPE_COMPUTER_HARD) {
                //AI can turn right and left multiple times at once
                if (currentCollisionDist > maxCollisionDist) {
                    maxCollisionDist = currentCollisionDist;
                    maxCollisionDistIndex = i;
                }
            } else {
                //AI can turn right and left only one time at once
                if (currentCollisionDist > maxCollisionDist && (i <= 1 || i == Vehicle::GetAngleCount() - 1)) {
                    maxCollisionDist = currentCollisionDist;
                    maxCollisionDistIndex = i;
                }
            }
            if (rotationSide == 0) {
                _vehicle->TurnRight();
            } else {
                _vehicle->TurnLeft();
            }
        }

        //turn into calculated direction
        for (int i = 0; i < maxCollisionDistIndex; i++) {
            if (rotationSide == 0) {
                _vehicle->TurnRight();
            } else {
                _vehicle->TurnLeft();
            }
        }

        _vehicle->ActivateItem();
    }
}

void Player::Tick() {
    if (IsEnabled()) {
        if (GetType() == PLAYER_TYPE_HUMAN) {
            FollowCommands();
        } else {
            ComputeCommands();
        }

        _vehicle->Tick();
    }
}

void Player::Draw() {
    if (IsEnabled()) {
        _vehicle->Draw();
    }
}

void Player::Reset() {
    _vehicle->Reset();
    _roundScore = 0;
}

short Player::GetKeyForCommand(PLAYER_COMMAND command) {
    return _game->GetConfiguration()->GetPlayerKeyCode(_idNumber - 1, (short) command);
}

bool Player::IsEnabled() {
    return _game->GetConfiguration()->IsPlayerEnabled(_idNumber - 1);
}

int Player::GetColor() {
    return MakePlayerColor((PLAYER_COLOR) _game->GetConfiguration()->GetPlayerColor(_idNumber - 1));
}

PLAYER_TYPE Player::GetType() {
    return _game->GetConfiguration()->GetPlayerType(_idNumber - 1);
}

void Player::ApplyRoundScore(short _score) {
    _roundScore = _score;
    _totalScore += _roundScore;
}

const char* Player::GetPlayerTypeString(PLAYER_TYPE playerType) {
    if (playerType == PLAYER_TYPE_NONE) {
        return "None";
    } else if (playerType == PLAYER_TYPE_HUMAN) {
        return "Human";
    } else if (playerType == PLAYER_TYPE_COMPUTER_EASY) {
        return "AI-Easy";
    } else if (playerType == PLAYER_TYPE_COMPUTER_NORMAL) {
        return "AI-Normal";
    } else if (playerType == PLAYER_TYPE_COMPUTER_HARD) {
        return "AI-Hard";
    }
}

//Allegro:

bool AllegroPlayer::GetsKeyImpulseForCommand(PLAYER_COMMAND command) {
    int keyForCommand = GetKeyForCommand(command);
    if (!_holdingKeyForCommand[command]) {
        if (key[keyForCommand]) {
            _holdingKeyForCommand[command] = true;
            return true;
        } else {
            _holdingKeyForCommand[command] = false;
            return false;
        }
    } else {
        if (key[keyForCommand]) {
            _holdingKeyForCommand[command] = true;
            return false;
        } else {
            _holdingKeyForCommand[command] = false;
            return false;
        }
    }
}
