#ifndef LMENU_MENU_H
#define LMENU_MENU_H

#include <allegro.h>
#include <string>
#include <vector>
#include <algorithm>

#include "item.h"

namespace lmenu
{
    class Item;

    class Event
    {
    public:

        enum TYPE { CLICKED_LEFT, CLICKED_RIGHT, CLICKED_MIDDLE, MOUSE_OVER, MOUSE_LEAVES };

        Event(TYPE eventType, Item* source) { _eventType = eventType; _source = source; }
        TYPE GetType() { return _eventType; }
        Item* GetSource() { return _source; }

    private:

        TYPE _eventType;
        Item* _source;
    };

    class Menu
    {
    public:

        Menu();
        virtual ~Menu();

        enum LAYOUT { LAYOUT_CUSTOM_POS, LAYOUT_VERTICALLY_LEFT, LAYOUT_VERTICALLY_CENTERED, LAYOUT_HORIZONTALLY_CENTERED };

        void            SetTitle(const char *title);
        const char*     GetTitle() { return _title; };

        int	GetTextColor() { return _textColor; }
        void	SetTextColor(int red, int green, int blue) { _textColor = makecol(red, green, blue); };
        void	SetTextColor(int color) { _textColor = color; }

        LAYOUT	GetLayout() { return _layout; }
        void	SetLayout(LAYOUT layout) { _layout = layout; UpdateSize(); }

        bool	IsAutoSize() { return _autoSize; }
        void	SetAutoSize(bool autoSize);

        bool	HasBorderLine() { return _borderLine; }
        void	SetBorderLine(bool borderLine) { _borderLine = borderLine; }

        int	GetBorderSpace() { return _borderSpace; }
        void	SetBorderSpace(int borderSpace) { _borderSpace = borderSpace; UpdateSize(); }

        int	GetItemBorderSpace() { return _itemBorderSpace; }
        void	SetItemBorderSpace(int itemBorderSpace) { _itemBorderSpace = itemBorderSpace; UpdateSize(); }

        int	GetBorderLineColor() { return _borderLineColor; }
        void	SetBorderLineColor(int red, int green, int blue) { _borderLineColor = makecol(red, green, blue); SetBorderLine(true); }
        void	SetBorderLineColor(int borderLineColor) { _borderLineColor = borderLineColor; SetBorderLine(true); }

        bool	HasBackground() { return _background; }
        void	SetBackground(bool background) { _background = background; }

        int	GetBackgroundColor() { return _backgroundColor; }
        void	SetBackgroundColor(int red, int green, int blue) { _backgroundColor = makecol(red, green, blue); SetBackground(true); }
        void	SetBackgroundColor(int backgroundColor) { _backgroundColor = backgroundColor; SetBackground(true); }

        virtual void	Show() { _enabled = true; _visible = true; }
        virtual void	Hide() { _enabled = false; _visible = false; }

        void	SetPos(int x, int y) { _posX = x; _posY = y; };
        void	SetX(int x) { _posX = x; };
        void	SetY(int y) { _posY = y; };

        int	GetX() { return _posX; }
        int	GetY() { return _posY; }

        void	SetSize(int w, int h) { if(!_autoSize) { _sizeW = w; _sizeH = h; } }
        void	SetW(int w) { if(!_autoSize) _sizeW = w; }
        void	SetH(int h) { if(!_autoSize) _sizeH = h; }

        int	GetW() { return _sizeW; }
        int	GetH() { return _sizeH; }

        void	SetFont(FONT *f) { _font = f; UpdateSize(); }
        FONT*	GetFont() { return _font; };

        bool	IsTextMulticolored() { return _textMulticolored; }
        void	SetTextMulticolored(bool textMulticolored) { _textMulticolored = textMulticolored; }

        BITMAP*	GetDestinationBitmap() { return _destBitmap; }
        void	SetDestinationBitmap(BITMAP* destBitmap) { _destBitmap = destBitmap; }

        void	SetEnabled(bool enabled) { _enabled = enabled; }
        void	SetVisible(bool visible) { _visible = visible; }

        bool	IsEnabled() { return _enabled; }
        bool	IsVisible() { return _visible; }

        int	GetItemCount() { return (int)_items.size(); }
        int	GetItemIndex(Item* item);
        Item*	GetItemByIndex(int index);

        void	Add(Item *i);
        void	Remove(Item *i);

        void	UpdateSize();

        int	GetOffsetX() { return _offsetX; }
        int	GetOffsetY() { return _offsetY; }

        void	SetOffset(int offsetX, int offsetY) { _offsetX = offsetX; _offsetY = offsetY; }
        void	SetOffsetX(int offsetX) { _offsetX = offsetX; }
        void	SetOffsetY(int offsetY) { _offsetY = offsetY; }

        void	Draw();
        void	Tick();

        virtual void	EventOccurred(Event e) {};

    private:

        std::vector< Item* > _items;

        int	_posX;
        int	_posY;
        int	_sizeW;
        int	_sizeH;

        bool	_visible;
        bool	_enabled;

        bool	_autoSize;

        bool	_borderLine;
        bool	_background;

        int	_itemBorderSpace;
        int	_borderSpace;

        LAYOUT	_layout;

        int	_backgroundColor;
        int	_borderLineColor;
        int	_textColor;

        FONT*	_font;

        char	_title[64];

        bool	_textMulticolored;

        BITMAP*	_destBitmap;

        int	_offsetX;
        int	_offsetY;
    };

}

#endif //LMENU_MENU_H
